/*
 * Copyright (c) 2002, 2004, 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: strstr0.c,v 1.7 2005/06/02 19:00:37 ca Exp $")

#include "sm/assert.h"
#include "sm/magic.h"
#include "sm/error.h"
#include "sm/memops.h"
#include "sm/rpool.h"
#include "sm/varargs.h"
#include "sm/limits.h"
#include "sm/str.h"
#include "sm/str-int.h"

/*
**  The functions in this module are similar to those in strstr.c,
**  but they try to make sure that a trailing '\0' is in the created
**  str object, which however is not taken into account for the length.
**  See below for details.
*/

/*
**  SM_STR_SCPYN0 -- Create a str object and copy n bytes from src into it.
**	(including trailing '\0')
**
**	Parameters:
**		rpool -- The rpool in which to allocate.
**		src -- Byte array to copy from (should have trailing '\0').
**		n -- Number of bytes to copy from src (including '\0').
**		maxlen -- Maximum length of str data.
**
**	Returns:
**		New str object.
**		NULL on error.
*/

sm_str_P
sm_str_scpyn0(sm_rpool_P rpool, const char *src, uint n, uint maxlen)
{
	sm_str_P str;

	SM_REQUIRE(src != NULL);
	SM_REQUIRE(n > 0);
	str = sm_str_new(rpool, n, maxlen);
	if (str == NULL)
		return NULL;
	sm_memcpy(str->sm_str_base, src, n);
	str->sm_str_len = n - 1;
	return str;
}

/*
**  SM_STR_SCATN0 -- Append a copy of a sized char * to the end of a str
**			sm_str_P.
**
**	Parameters:
**		str -- sm_str_P object to append onto.
**		append -- array of bytes to append.
**		len -- number of bytes to append (including '\0')
**
**	Returns:
**		usual sm_error code; ENOMEM, SM_E_OVFLW_SC, SM_E_OVFLW_NS
*/

sm_ret_T
sm_str_scatn0(sm_str_P str, const char *append, uint len)
{
	uint new_alloc;

	SM_IS_BUF(str);
	SM_REQUIRE(append != NULL);

	new_alloc = str->sm_str_len + len;

	/* overflow? */
	if (new_alloc <= str->sm_str_len)
		return sm_error_perm(SM_EM_STR, SM_E_OVFLW_SC);
	if (new_alloc > str->sm_str_size)
		SM_STR_INCREASE_R(str, new_alloc);
	sm_memcpy(str->sm_str_base + str->sm_str_len, append, len);
	str->sm_str_len += len - 1;
	return SM_SUCCESS;
}

/*
**  SM_STR_SCAT0 -- Appends a copy of a char * onto the end of a sm_str_P
**	(including trailing '\0')
**
**	Parameters:
**		str -- sm_str_P object to append onto.
**		append -- '\0' terminated char * to append.
**
**	Returns:
**		str (modified) or NULL on error.
*/

sm_ret_T
sm_str_scat0(sm_str_P str, const char *append)
{
	SM_IS_BUF(str);
	SM_REQUIRE(append != NULL);

	return sm_str_scatn0(str, append, strlen(append) + 1);
}
