/*
 * Copyright (c) 2003-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: t-strexp.c,v 1.8 2005/06/06 16:23:21 ca Exp $")

#include "sm/assert.h"
#include "sm/magic.h"
#include "sm/error.h"
#include "sm/rpool.h"
#include "sm/test.h"
#include "sm/str.h"
#include "sm/strexp.h"
#include "sm/io.h"

extern char *optarg;
extern int optind;
extern int optopt;
extern int opterr;

#define SMAXLEN	1024
#define NMACROS	10

static int Verbose = 0;

static void
test_harness(sm_rpool_P a)
{
	uint argc;
	sm_str_P src, dst, argv[NMACROS];
	sm_ret_T ret;

	src = sm_str_new(a, SMAXLEN, SMAXLEN);
	SM_TEST(src != NULL);
	if (src == NULL)
		return;
	dst = sm_str_new(a, SMAXLEN, SMAXLEN);
	SM_TEST(dst != NULL);
	if (dst == NULL)
		return;

	for (argc = 0; argc < NMACROS; argc++)
	{
		argv[argc] = sm_str_new(a, SMAXLEN, SMAXLEN);
		SM_TEST(argv[argc] != NULL);
		if (argv[argc] == NULL)
			return;
		sm_strprintf(argv[argc], "arg-%u!", argc);
	}

	/* simple expansion */
	sm_str_clr(src);
	sm_str_clr(dst);
	ret = sm_str_scat0(src, "str%0");
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return;
	ret = sm_str_expdig(src, dst, '\0', 1, argv);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
		return;
	SM_TEST(strcmp((char *)sm_str_getdata(dst), "strarg-0!") == 0);
	if (Verbose > 2)
		sm_io_fprintf(smioerr, "dst='%S'\n", dst);

	/* two expansions and keep '%%' */
	sm_str_clr(src);
	sm_str_clr(dst);
	ret = sm_str_scat0(src, "%0-%%-%1");
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return;
	ret = sm_str_expdig(src, dst, '\0', 2, argv);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
		return;
	SM_TEST(strcmp((char *)sm_str_getdata(dst), "arg-0!-%-arg-1!") == 0);
	if (Verbose > 2)
		sm_io_fprintf(smioerr, "dst='%S'\n", dst);

	/* keep '%%', one expansion does not have an argument */
	sm_str_clr(src);
	sm_str_clr(dst);
	ret = sm_str_scat0(src, "%0-%%-%1");
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return;
	ret = sm_str_expdig(src, dst, '\0', 1, argv);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
		return;
	SM_TEST(strcmp((char *)sm_str_getdata(dst), "arg-0!-%-") == 0);
	if (Verbose > 2)
		sm_io_fprintf(smioerr, "dst='%S'\n", dst);

	/* two expansions and keep trailing '%' */
	sm_str_clr(src);
	sm_str_clr(dst);
	ret = sm_str_scat0(src, "%0-%1-%");
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return;
	ret = sm_str_expdig(src, dst, '\0', 2, argv);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
		return;
	SM_TEST(strcmp((char *)sm_str_getdata(dst), "arg-0!-arg-1!-%") == 0);
	if (Verbose > 2)
		sm_io_fprintf(smioerr, "dst='%S'\n", dst);

	/* many expansions */
	sm_str_clr(src);
	sm_str_clr(dst);
	ret = sm_str_scat0(src, "%0.%1.%2.%3.%4.%5.%6.%7.%8.%9.%A");
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return;
	ret = sm_str_expdig(src, dst, '\0', NMACROS, argv);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
		return;
	SM_TEST(strcmp((char *)sm_str_getdata(dst),
		"arg-0!.arg-1!.arg-2!.arg-3!.arg-4!.arg-5!.arg-6!.arg-7!.arg-8!.arg-9!.A") == 0);
	if (Verbose > 2)
		sm_io_fprintf(smioerr, "dst='%S'\n", dst);

	for (argc = 0; argc < NMACROS; argc++)
		sm_str_free(argv[argc]);
	sm_str_free(src);
	sm_str_free(dst);
}

int
main(int argc, char *argv[])
{
	int c;
	sm_rpool_P a;

	while ((c = getopt(argc, argv, "V")) != -1)
	{
		switch (c)
		{
		  case 'V':
			++Verbose;
			break;
		  default:
			/* usage(prg); */
			break;
		}
	}
	argc -= optind;
	argv += optind;
	sm_test_begin(argc, argv, "test strexpdig()");

	/* create an rpool for entire test */
	a = sm_rpool_new(NULL);
	SM_TEST(a != NULL);
	test_harness(a);
	sm_rpool_delete(a);

	/* test without rpool */
	test_harness(NULL);
	return sm_test_end();
}
