/*
 * Copyright (c) 2004, 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: pmilter_engine.c,v 1.18 2005/10/27 18:05:32 ca Exp $")
#include "sm/error.h"
#include "sm/assert.h"
#include "sm/ctype.h"
#include "sm/reccom.h"
#include "sm/pmilter.h"
#include "pmilter.h"
#include "sm/pmfdef.h"
#include "sm/pmfapi.h"
#include "util.h"

#if SM_USE_PMILTER

/*
**  How to implement session timeouts?
**  That is, when the server aborts a connection will pmilter be informed?
**
**  Note: currently a reply text is only sent if it
**  - has an SMTP reply code
**  - the reply code matches the return value of the function
**  This means it can't be used for other things right now, e.g.,
**  changing an address.
*/

/*
**  SM_PMILT_NSEID -- New SMTP session
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_nseid(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	bool sendreply;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	sendreply = false;

	if (pmilter->pmfi_connect != NULL &&
	    SM_IS_FLAG(pmss_ctx->pmss_pmcap, SM_SCAP_PM_CNNCT))
	{
		int rc;
		sm_sockaddr_T sm_sockaddr;

		sm_memzero(&sm_sockaddr, sizeof(sm_sockaddr));
		sm_sockaddr.sin.sin_addr.s_addr = pmse_ctx->pmse_cltipv4;
		sm_sockaddr.sin.sin_family = AF_INET;
		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_connect)(pmse_ctx,
				(char *) sm_str_getdata(pmse_ctx->pmse_arg1),
				&sm_sockaddr);
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);
	return ret;
}

/*
**  SM_PMILT_CSEID -- Close SMTP session
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_cseid(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx)
{
	sm_ret_T ret;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;

	if (pmilter->pmfi_close != NULL)
		ret = (*pmilter->pmfi_close)(pmse_ctx);
	return SM_SUCCESS;
}

/*
**  SM_PMILT_HELO -- EHLO/HELO
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_helo(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	bool sendreply;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	sendreply = false;

	if (pmilter->pmfi_helo != NULL)
	{
		int rc;

		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_helo)(pmse_ctx,
				(char *) sm_str_getdata(pmse_ctx->pmse_arg1));
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);
	return ret;
}

/*
**  SM_PMILT_MAIL -- MAIL
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_mail(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	unsigned int nelem;
	bool sendreply;
	char **argv;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	argv = NULL;
	sendreply = false;

	ret = args2argv(pmse_ctx->pmse_arg2, &nelem, &argv);
	if (sm_is_err(ret))
		goto error;
	if (pmilter->pmfi_mail != NULL)
	{
		int rc;

		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_mail)(pmse_ctx,
				(char *) sm_str_getdata(pmse_ctx->pmse_arg1),
				argv);
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);

  error:
	if (argv != NULL)
		sm_free(argv);
	return ret;
}

/*
**  SM_PMILT_RCPT -- RCPT
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_rcpt(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	unsigned int nelem;
	bool sendreply;
	char **argv;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	argv = NULL;
	sendreply = false;

	ret = args2argv(pmse_ctx->pmse_arg2, &nelem, &argv);
	if (sm_is_err(ret))
		goto error;
	if (pmilter->pmfi_rcpt != NULL)
	{
		int rc;

		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_rcpt)(pmse_ctx,
				(char *) sm_str_getdata(pmse_ctx->pmse_arg1),
				argv);
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);

  error:
	if (argv != NULL)
		sm_free(argv);
	return ret;
}

/*
**  SM_PMILT_DATA -- DATA
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_data(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	bool sendreply;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	sendreply = false;

	if (pmilter->pmfi_data != NULL)
	{
		int rc;

		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_data)(pmse_ctx);
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);
	return ret;
}

/*
**  SM_PMILT_MSG -- message chunck
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		buf -- message chunck
**		len -- size of message chunck
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_msg(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, uchar *buf, size_t len, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode;
	bool sendreply;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	sendreply = false;

	if (pmilter->pmfi_msg != NULL)
	{
		int rc;

		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_msg)(pmse_ctx, buf, len);
		sendreply = SM_IS_FLAG(pmss_ctx->pmss_pmcap,
					SM_SCAP_PM_MSG_RC) &&
			sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}

	if (SM_IS_FLAG(pmss_ctx->pmss_pmcap, SM_SCAP_PM_MSG_RC))
	{
		ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
			SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
			SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
			SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
			SM_RCBV_INT, RT_M2S_RCODE, rcode,
			SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
				pmse_ctx->pmse_reply,
			SM_RCBV_END);
	}

	return ret;
}

/*
**  SM_PMILT_DOT -- final message chunck
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**		buf -- message chunck
**		len -- size of message chunck
**		rcbe -- RCB for reply
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_dot(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx, uchar *buf, size_t len, sm_rcbe_P rcbe)
{
	sm_ret_T ret;
	int rcode, rc;
	bool sendreply;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;
	rcode = SMTP_R_OK;
	sendreply = false;

	if (pmilter->pmfi_msg != NULL)
	{
		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_msg)(pmse_ctx, buf, len);
		if (SM_IS_FLAG(pmss_ctx->pmss_pmcap, SM_SCAP_PM_MSG_RC))
		{
			sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
				IS_SMTP_REPLY(rcode) &&
				SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply,
					rcode, 0, rc);
		}
		else
			rcode = SMTP_R_OK;
	}
	if (!(SM_IS_FLAG(pmss_ctx->pmss_pmcap, SM_SCAP_PM_MSG_RC) &&
	      (IS_SMTP_REPLY(rcode) || rcode == SMTP_R_ACCEPT)) &&
	    pmilter->pmfi_eom != NULL)
	{
		sm_str_clr(pmse_ctx->pmse_reply);
		rcode = (*pmilter->pmfi_eom)(pmse_ctx);
		sendreply = sm_str_getlen(pmse_ctx->pmse_reply) > 0 &&
			IS_SMTP_REPLY(rcode) &&
			SMTP_REPLY_MATCHES_RCODE(pmse_ctx->pmse_reply, rcode, 0,
						rc);
	}
	ret = sm_rcb_putv(&(rcbe->rcbe_rcb), RCB_PUTV_FIRST,
		SM_RCBV_INT, RT_PROT_VER, PROT_VER_RT,
		SM_RCBV_INT, RT_M2S_ID, pmss_ctx->pmss_id,
		SM_RCBV_BUF, RT_M2S_SEID, pmse_ctx->pmse_se_id, SMTP_STID_SIZE,
		SM_RCBV_INT, RT_M2S_RCODE, rcode,
		SM_RCBV_STR, sendreply ? RT_M2S_STATT : RT_NOSEND,
			pmse_ctx->pmse_reply,
		SM_RCBV_END);

	return ret;
}

/*
**  SM_PMILT_ABORT_TA -- abort current transaction
**
**	Parameters:
**		pmss_ctx -- task context pmilter/SMTP server
**		pmse_ctx -- pmilter/SMTP server session context
**
**	Returns:
**		usual sm_error code
*/

sm_ret_T
sm_pmilt_abort_ta(pmss_ctx_P pmss_ctx, pmse_ctx_P pmse_ctx)
{
	sm_ret_T ret;
	pmilter_P pmilter;

	SM_IS_PMSE_CTX(pmse_ctx);
	SM_REQUIRE(pmse_ctx->pmse_pmg_ctx != NULL);
	pmilter = pmse_ctx->pmse_pmg_ctx->pmg_pmilter;
	SM_REQUIRE(pmilter != NULL);
	ret = SM_SUCCESS;

	if (pmilter->pmfi_abort != NULL)
		ret = (*pmilter->pmfi_abort)(pmse_ctx);

	return ret;
}
#endif /* SM_USE_PMILTER */
