/*
 * Copyright (c) 2004, 2005 Sendmail, Inc. and its suppliers.
 *      All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: dadbclose.c,v 1.7 2005/03/17 02:07:36 ca Exp $")
#include "sm/types.h"
#include "sm/assert.h"
#include "sm/magic.h"
#include "sm/str.h"
#include "sm/mta.h"
#include "sm/memops.h"
#include "sm/qmgr.h"
#include "sm/qmgr-int.h"
#include "sm/dadb.h"
#include "sm/occ.h"
#include "sm/da.h"
#include "dadb.h"
#include "occ.h"

/*
**  DADB_ENTRY_FREE -- free a DADB entry
**
**	Parameters:
**		dadb_entry -- DADB entry
**
**	Returns:
**		none
**
**	Last code review: 2005-03-16 06:13:26
**	Last code change: 2005-03-16 06:13:17
*/

static void
dadb_entry_free(dadb_entry_P dadb_entry)
{
	if (dadb_entry == NULL)
		return;
	SM_IS_DADBE(dadb_entry);
#if DADB_CHECK
	dadb_entry->sm_magic = SM_MAGIC_NULL;
#endif
	sm_free_size(dadb_entry, sizeof(*dadb_entry));
	return;
}

/*
**  DADB_CLOSE -- close a DADB; including all open sessions
**
**	Parameters:
**		qmgr_ctx -- QMGR context (only qmgr_conf is needed)
**		dadb_ctx -- DADB context
**		locktype -- kind of locking
**
**	Returns:
**		usual sm_error code
**
**	Locking: locks dadb_ctx if requested
**
**	Last code review: 2005-03-16 17:04:41; see comments below
**	Last code change: 2005-03-16 17:04:40
*/

sm_ret_T
dadb_close(qmgr_ctx_P qmgr_ctx, dadb_ctx_P dadb_ctx, thr_lock_T locktype)
{
	uint i;
	sm_ret_T ret;
	int r;
	dadb_entry_P dadb_entry;
	aq_ta_P aq_ta;

	if (dadb_ctx == NULL)
		return SM_SUCCESS;
	SM_IS_DADB(dadb_ctx);
	ret = SM_SUCCESS;
	if (thr_lock_it(locktype))
	{
		r = pthread_mutex_lock(&(dadb_ctx->dadb_mutex));
		SM_LOCK_OK(r);
		if (r != 0)
			return sm_error_perm(SM_EM_DA, r);
	}

	/*
	**  Notes:
	**  This loop checks dadb_ctx->dadb_entries != NULL to avoid an if
	**  around the entire loop (which requires further indentation).
	**  This isn't really optimal (checking the variable on each iteration)
	**  but shouldn't have any noticeable impact.
	**
	**  The return value (ret) of the last function call inside the loop
	**  is used as return value of this function; moreover, some return
	**  values are simply ignored. This isn't optimal but this is just
	**  some cleanup code hence any errors can be ignored.
	*/

	for (i = 0;
	     dadb_ctx->dadb_entries != NULL && i < dadb_ctx->dadb_entries_max;
	     i++)
	{
		dadb_entry = (dadb_ctx->dadb_entries)[i];
		if (dadb_entry == NULL)
			continue;
		if (!DADBE_IS_FREE(dadb_entry))
		{
			ret = aq_ta_find(qmgr_ctx->qmgr_aq,
					dadb_entry->dadbe_ss_ta_id, true,
					&aq_ta);
			if (sm_is_success(ret))
			{
				/* session must be closed */
				DADBE_SET_FLAG(dadb_entry, DADBE_FL_SE_CL);

				/* qda_update_ta_stat() will lock this mutex */
				if (thr_lock_it(locktype))
				{
					r = pthread_mutex_unlock(&(dadb_ctx->dadb_mutex));
					SM_ASSERT(r == 0);
					if (r != 0)
						ret = sm_error_perm(SM_EM_DA, r);
				}
				/*
				**  race condition?? can someone else access
				**  dadb_ctx between these calls?
				*/

				/* Update transaction status */
				ret = qda_update_ta_stat(qmgr_ctx,
					dadb_entry->dadbe_da_ta_id,
					SMTPC_DA_ST, DA_TERMINATED,
					dadb_ctx, dadb_entry,
					aq_ta, NULL, NULL, THR_LOCK_UNLOCK);
				/* fixme: complain on error? */

				if (thr_lock_it(locktype))
				{
					r = pthread_mutex_lock(&(dadb_ctx->dadb_mutex));
					SM_ASSERT(r == 0); /* use SM_LOCK_OK? */
					if (r != 0 && sm_is_success(ret))
						ret = sm_error_perm(SM_EM_DA, r);
				}
			}
			else
			{
				ret = dadb_sess_close_entry(qmgr_ctx, dadb_ctx,
					dadb_entry, true, NULL, THR_NO_LOCK);
			}
		}

		/* free DADB entry: this should be a function */
		dadb_entry_free(dadb_entry);
		(dadb_ctx->dadb_entries)[i] = NULL;
	}
	FREE_DADB_ENTRIES(dadb_ctx);

	if ((!sm_is_err(ret) && thr_unl_no_err(locktype))
	    || (sm_is_err(ret) && thr_unl_if_err(locktype)))
	{
		r = pthread_mutex_unlock(&(dadb_ctx->dadb_mutex));
		SM_ASSERT(r == 0);
		if (r != 0 && sm_is_success(ret))
			ret = sm_error_perm(SM_EM_DA, r);
	}
	return ret;
}
