# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: EPPConnection.pm,v 1.8 2003/02/21 00:17:31 lidl Exp $

package ISC::SRS::EPPConnection;

use strict;
use warnings;

use ISC::CC::Group::Connect;

use ISC::SRS::EPPRegistrar;
use ISC::SRS::EPPResultCode;

use ISC::Date ":all";

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.8 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

use constant DB_TIMEOUT => 60;

sub _check_required_args {
    my ($func, $args, $reqd) = @_;

    foreach my $i (@$reqd) {
	if (!exists($args->{$i})) {
	    die "$func: Missing argument: $i";
	}
    }
}

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    $self->{period} = 2;
    $self->{renew_period} = 1;

    _check_required_args('new', \%args, [ "msgbus", "feid" ]);

    $self->{srsdb_channel} = $args{srsdb_channel} || "srsdb";

    $self->{dbtimeout} = $args{dbtimeout} || DB_TIMEOUT;

    $self->{feid} = $args{feid};

    my $con = $args{msgbus};

    $self->{cc} = $con->join(group => $self->{srsdb_channel},
			     instance => "epp_req",
			     subtype => "meonly",
			     cb => \&_dummy_cb);

    return $self;
}

sub cancel {
    my ($self) = @_;

    $self->{cc} = undef;
}

sub _dummy_cb {
    warn "Odd, I got a message!\n";
}

sub dbtimeout {
    my ($self, $timeout) = @_;

    if ($timeout) {
	$self->{dbtimeout} = $timeout;
    }

    return $self->{dbtimeout};
}

sub period {
    my ($self) = @_;

    return $self->{period};
}

sub renew_period {
    my ($self) = @_;

    return $self->{renew_period};
}

sub _make_timeout {
    my $result = {
	resultcode => EPP_RF_FAIL,
	result => epp_result_totext(EPP_RF_FAIL),
	xresult => "database timeout",
    };

    return $result;
}

sub _make_addl {
    my ($self, $args) = @_;

    my $addl = {};

#    if (defined($self->{parent}->{feid})) {
#	$addl->{feid} = $self->{parent}->{feid}
#    } else {
#        print Data::Dumper->Dump([$self], ["self"]);
#    }

    $addl->{feid} = $self->{feid} if ($self->{feid});
    $addl->{sid} = $self->{sid} if ($self->{sid});

    foreach my $i (keys %$args) {
	$addl->{$i} = $args->{$i};
    }

#    print Data::Dumper->Dump([$addl], ["addl"]);

    return $addl;
}

sub _generic_sendreq {
    my ($self, $type, $addl, $args, $cb) = @_;

    $cb = \&_generic_resp unless (defined($cb));

    my $msg = {
        type => $type,
        args => $args->{req},
	addl => $self->_make_addl($addl),
    };

    my $data = {
        cb => $args->{cb},
        data => $args->{data},
        self => $self,
        msg => $msg,
    };

    $self->{cc}->send(timeout => $self->{dbtimeout},
		      cb => $cb,
		      data => $data,
		      msg => $msg);
}

sub hello {
    my ($self, %args) = @_;

    $self->_generic_sendreq('hello', undef, \%args);

    return 0;
}

sub fe_start {
    my ($self, $feid) = @_;
    my (%args);

    $self->_generic_sendreq('fe_start', { feid => $feid },
			    \%args, \&_fe_start_resp);

    return 0;  # XXX should probably check _sendreq result
}

sub _fe_start_resp {
    my ($channel, $msg, $wmsg) = @_;

    my $data = $channel->data;
    die "application botch: \$data is undef" if (!$data);
    my $self = $data->{self};
    my $cbdata = $data->{data};
    my $cb = $data->{cb};

    #
    # timeout
    #
    if (!$msg) {
	$cb->($self, $cbdata, _make_timeout);
	return;
    }

    # XXXKJL
    return;
}

sub login {
    my ($self, %args) = @_;

    $self->_generic_sendreq('login', { sid => $args{sid} },
			    \%args, \&_login_resp);

    return 0;  # XXXMLG should check _sendreq result
}

sub _generic_resp {
    my ($channel, $msg, $wmsg) = @_;

    my $data = $channel->data;
    die "application botch: \$data is undef" if (!$data);
    my $self = $data->{self};
    my $cbdata = $data->{data};
    my $cb = $data->{cb};

    #
    # timeout
    #
    if (!$msg) {
	$cb->($self, $cbdata, _make_timeout);
	return;
    }

    my $result = $msg->{result};
    delete($msg->{result});
    $result->{result} = epp_result_totext($result->{resultcode});

    print "Calling generic callback\n";
    $cb->($self, $cbdata, $result, $msg);
}

sub _login_resp {
    my ($channel, $msg, $wmsg) = @_;

    my $data = $channel->data;
    die "application botch: \$data is undef" if (!$data);
    my $self = $data->{self};
    my $cbdata = $data->{data};
    my $cb = $data->{cb};

    #
    # timeout
    #
    if (!$msg) {
	$cb->($self, $cbdata, _make_timeout);
	return;
    }

    my $result = $msg->{result};
    delete($msg->{result});
    $result->{result} = epp_result_totext($result->{resultcode});

    if ($result->{resultcode} == EPP_RS_SUCCESS) {
	my $registrar = new ISC::SRS::EPPRegistrar(parent => $self,
						   id => $msg->{id},
						   sid => $msg->{sid});
	$msg = {};
	$msg->{registrar} = $registrar;
	$self->{sid} = $registrar->{sid};
    }

    print "Calling login callback\n";
    $cb->($self, $cbdata, $result, $msg);
}

1;
