#!/usr/bin/env perl
#
# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: whois.pl,v 1.30.2.1 2003/02/21 22:34:12 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Carp;
use Data::Dumper;
use Event qw(loop unloop);
use Getopt::Long;
use POSIX qw(gmtime);
use Time::HiRes qw(gettimeofday);
use Sys::Hostname;

use ISC::CC::Group::Connect;
use ISC::Net::Listen;
use ISC::Net::CRLF;
use ISC::SRS::DB;
use ISC::Stats::Report;
use ISC::Log;

my $hostname = hostname;
my $log;

my $db = new ISC::SRS::DB(replication => "none");

my $tcp_port = $ENV{ISC_SRS_WHOISPORT} || 43;
my $tcp_addr = $ENV{ISC_SRS_WHOISADDR} || "0.0.0.0";
my $result = GetOptions("port=s", \$tcp_port,
			"addr=s", \$tcp_addr);

my $listen = new ISC::Net::Listen(cb => \&_listen_cb,
				  error_cb => \&_listen_err,
				  LocalPort => $tcp_port,
				  LocalAddr => $tcp_addr,
				  ReuseAddr => 1,
				  Proto => "tcp");

my $msgbus = new ISC::CC::Group::Connect(connect_cb => \&_msgbus_cb,
					 connect_error_cb => \&_msgbus_err,
					 error_cb => \&_msgbus_err);

my $stats = {
    ident => "$hostname/$tcp_addr/$tcp_port",
    start => time,

    #
    # The following are "counter" types.  That is, they start at 0 and
    # count up, without any reset.
    #
    counters => {
	tcp => 0,
	queries => 0,
	domain_hits => 0,
	contact_hits => 0,
	host_hits => 0,
	misses => 0,
    },

    #
    # The following are current stats, or "gauge" types.  These report the
    # number of things in use right now.
    #
    gauges => {
	tcp => 0,
    },
};

sub _msgbus_cb {
    my ($con) = @_;

    my $reporter = new ISC::Stats::Report(msgbus => $msgbus,
					  freq => 6,
					  group => "stats",
					  instance => "whois",
					  cb => \&_stats_msg);

    $log = new ISC::Log(facility => "whois($tcp_addr/$tcp_port)",

			maxlevel_stderr => ISC_LOG_DEBUG,

			msgbus => $con,
			group => "log",
			instance => "whois",
			maxlevel_msgbus => ISC_LOG_INFO,
			);

    $log->log(ISC_LOG_INFO,
	      "Connected to msgbus, local name is " . $con->myname);
    $log->log(ISC_LOG_INFO, "Setting up stats reporting.");
}

sub _msgbus_err {
    my ($msgbus, $msg) = @_;

    $log->log(ISC_LOG_ERR, "msgbus error: $msg");
    unloop();
}

sub _stats_msg {
    return $stats;
}

sub _listen_cb {
    my ($l, $sock) = @_;

    $log->log(ISC_LOG_INFO, "New connection");

    $stats->{counters}->{tcp}++;
    $stats->{gauges}->{tcp}++;

    my $crlf = new ISC::Net::CRLF(cb => \&_msg_cb,
				  error_cb => \&_msg_err,
				  socket => $sock,
				  oneshot => 1,
				  flush => 1,
				  idle_timeout => 10,
				  max_lines => 1,
				  max_line_length => 1024,
				  in_block_terminator => undef,
				  out_block_terminator => undef);
}

sub _listen_err {
    my ($l, $msg) = @_;

    die "listen error: $msg";
}

sub _msg_err {
    my ($crlf, $msg) = @_;

    $crlf->cancel;
    $stats->{gauges}->{tcp}--;

    $log->log(ISC_LOG_INFO, "Closing connection: $msg");
}

sub _msg_cb {
    my ($crlf, $lines) = @_;

    my $query = $lines->[0];

    $query = uc $query;

    $stats->{counters}->{queries}++;

    $log->log(ISC_LOG_DEBUG, "Query: $query");

    my $found = 0;

    my $dbi = $db->begin;

    my $apex = $db->apex;

    my $ret = [ "",
		"Whois Server Version 1.0",
		"",
		"This whois server contains a sample load of data from the $apex zone.",
		"All data returned is for example and testing only.",
		"" ];
#
# parse the incoming query, as much as we can, due to the variable nature
# of whois queries
#
    if ($query =~ /^DOM\s(\S+)$/) {
	$found = &domain_lookup($1,$ret);
    } elsif ($query =~ /^HOST\s(\S+)$/) {
	$found = &host_lookup($1,$ret);
    } elsif ($query =~ /^HANDLE\s(\S+)$/) {
	$found = &contact_lookup($1,$ret);
    } elsif ($query =~ /^DUMP\sCONTACT\s(\S+)$/) {
	$found = &contact_lookup($1,$ret);
    } elsif ($query =~ /^DUMP\sHOST\s(\S+)$/) {
	$found = &host_lookup($1,$ret);
    } elsif ($query =~ /^DUMP\sDOM\s(\S+)$/) {
	$found = &domain_dump($1,$ret);
    } else {
	$found = &domain_lookup($query,$ret);
	$found = &host_lookup($query,$ret) if (!$found);
	$found = &contact_lookup($query,$ret) if (!$found);
    }

    $db->commit;

    if (!$found) {
	$stats->{counters}->{misses}++;
	push(@$ret, "No match for \"$query\".");
    }

    push(@$ret, "", "This data is for testing and validation purposes ONLY");

    $crlf->send($ret);
    $crlf->shutdown(10);
}

#
# this only receives fqdn as lookup query
#
sub domain_dump {
    my ($query, $ret) = @_;
    my $found;

    my $domain = $db->_domain_get(fqdn => $query);
    if ($domain) {
	$found = 1;
	$stats->{counters}->{domain_hits}++;

	if (defined($domain->{owner_id})) {
	    my $contact = $db->_contact_get(id => "$domain->{owner_id}");
	    if ($contact) {
		$stats->{counters}->{contact_hits}++;
		push(@$ret, "Registrant:");
		my $postal = $db->_c_postal_get_byid(id => "$contact->{contact_id}");
		fmt_contact($contact, $postal, "", $ret);
	    }
	}
	push(@$ret, "");
	push(@$ret, "   Domain Name: " . $domain->{fqdn});
	push(@$ret, "");

	my $c_tech = $db->_domain_get_contacts(type => "tech",
	    domain_id => "$domain->{domain_id}");
	my $c_admin = $db->_domain_get_contacts(type => "admin",
	    domain_id => "$domain->{domain_id}");
	my $c_billing = $db->_domain_get_contacts(type => "billing",
	    domain_id => "$domain->{domain_id}");
        my $tret;
        if (scalar(@$c_admin) != 0) {
	    push(@$tret, "Administrative Contact(s): ");
	    foreach my $c (@$c_admin) {
		contact_lookup($c->{contact_id}, $tret, "   ");
	    }
	}
        if ($c_tech && @$c_tech) {
	    push(@$tret, "Technical Contact(s): ");
	    foreach my $c (@$c_tech) {
		contact_lookup($c->{contact_id}, $tret, "   ");
	    }
	}
        if ($c_billing && @$c_billing) {
	    push(@$tret, "Billing Contact(s): ");
	    foreach my $c (@$c_billing) {
		contact_lookup($c->{contact_id}, $tret, "   ");
	    }
	}
	# move over every entry in tret by three spaces
	foreach my $l (@$tret) {
		push(@$ret, "   " . $l);
	}

	push(@$ret, "");
	push(@$ret, "   Record expires on " . $domain->{expire_date});
	push(@$ret, "   Record created on " . $domain->{created_date});
	push(@$ret, "");
	push(@$ret, "   Domain name servers:");
	push(@$ret, "");

	my $ns = $db->_domain_get_hosts(domain_id => "$domain->{domain_id}");
	if ($ns) {
	    $stats->{counters}->{host_hits}++;
	    foreach my $n (@$ns) {
		push(@$ret, "   Name Server: " . $n->{fqdn});
	    }
	}
    }

    return $found;
}

#
# this receives fqdn or Dn as input
#
sub domain_lookup {
    my ($query, $ret) = @_;
    my $found = 0;

    my $domain = $db->_domain_get(fqdn => "$query");

    if ($domain) {
	$stats->{counters}->{domain_hits}++;
	$found = 1;
	my $reg = $db->registrar_get(registrar_id => "$domain->{registrar_id}");
	my $ns = $db->_domain_get_hosts(domain_id => "$domain->{domain_id}");
	push(@$ret, "   Domain Name: " . $domain->{fqdn});
	push(@$ret, "   Registrar: " . $reg->{name});
	push(@$ret, "   Whois Server: " . $reg->{whois})
	    if ($reg->{whois});
	push(@$ret, "   Referral URL: " . $reg->{url})
	    if ($reg->{url});
	if ($ns) {
	    foreach my $n (@$ns) {
		push(@$ret, "   Name Server: " . $n->{fqdn});
	    }
	}
    }
    return $found;
}

#
# this receives either fqdn or Hnnnn as input
#
sub host_lookup {
    my ($query, $ret) = @_;
    my $found = 0;

    my $ns = $db->_host_get(fqdn => $query);

    if ($ns) {
	$stats->{counters}->{host_hits}++;
	$found = 1;
	my $reg = $db->registrar_get(registrar_id => $ns->{registrar_id});
	my $addrs = $db->_host_get_addresses(id => $ns->{host_id});
	my $addrtype;
	push(@$ret, "   Server Name: " . $ns->{fqdn});
	if ($addrs) {
	    foreach my $n (@$addrs) {
		my ($type, $addr) = @$n;
		if ($type eq "6") {
		    $addrtype="IPv6";
		} else {
		    $addrtype="IP";
		}
		push(@$ret, "   $addrtype Address: " . $addr);
	    }
	}
	push(@$ret, "   Registrar: " . $reg->{name});
	push(@$ret, "   Whois Server: " . $reg->{whois}) if ($reg->{whois});
	push(@$ret, "   Referral URL: " . $reg->{url}) if ($reg->{url});
    }
    return $found;
}

#
# this receives either "n" or Cn as input
#
sub contact_lookup {
    my ($query, $ret, $indent) = @_;
    my $found = 0;
    my $contact;

    $contact = $db->_contact_get(handle => "$query");

    if ($contact) {
	$stats->{counters}->{contact_hits}++;
	$found = 1;
	my $postal = $db->_c_postal_get_byid(id => "$contact->{contact_id}");
	fmt_contact($contact, $postal, $indent, $ret);
    }
    return $found;
}

sub fmt_contact {
    my ($contact, $postal, $indent, $ret) = @_;

    foreach my $type (keys %$postal) {
        my $line;

	$line = $postal->{$type}->{name};
	$line .= "  (C" . $contact->{contact_id} . ")";
	$line .= "\t\t" . $contact->{email};
	push(@$ret, $indent . $line);
	push(@$ret, "   $postal->{$type}->{org}") if ($postal->{$type}->{org});
        my $streets = $postal->{$type}->{street};
	push(@$ret, "   " . shift(@$streets)) if (scalar(@$streets) >= 1);
	push(@$ret, "   " . shift(@$streets)) if (scalar(@$streets) >= 1);
	push(@$ret, "   " . shift(@$streets)) if (scalar(@$streets) >= 1);
	$line = "   $postal->{$type}->{city}";
	$line .= ", $postal->{$type}->{sp}" if ($postal->{$type}->{sp});
	$line .= " $postal->{$type}->{pc}" if ($postal->{$type}->{pc});
	$line .= "  $postal->{$type}->{cc}";
	push(@$ret, $line);
	$line = "   ";
	$line .= $contact->{voice} if ($contact->{voice});
	$line .= " x" . $contact->{voice_ext} if ($contact->{voice_ext});
	$line .= " (FAX) " . $contact->{fax} if ($contact->{fax});
	push(@$ret, $line);
    }
    return $ret;
}

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

$SIG{PIPE} = "IGNORE";

Event::loop();
