// Copyright (C) 2017 Internet Systems Consortium, Inc. ("ISC")
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include <config.h>

#include <cc/command_interpreter.h>
#include <cc/data.h>
#include <cc/simple_parser.h>
#include <cc/cfg_to_element.h>
#include <dhcp/tests/iface_mgr_test_config.h>
#include <dhcpsrv/cfgmgr.h>
#include <dhcp4/tests/dhcp4_test_utils.h>
#include <dhcp4/tests/get_config_unittest.h>
#include <dhcp4/dhcp4_srv.h>
#include <dhcp4/json_config_parser.h>
#include <dhcp4/simple_parser4.h>

#include <boost/algorithm/string.hpp>
#include <gtest/gtest.h>

#include <iostream>
#include <string>
#include <sstream>
#include <list>

using namespace isc::config;
using namespace isc::data;
using namespace isc::dhcp;
using namespace isc::dhcp::test;

namespace {

/// @name How to fill configurations
///
/// Copy get_config_unittest.cc.skel into get_config_unittest.cc
///
/// For the extracted configurations define the EXTRACT_CONFIG and
/// recompile this file. Run dhcp4_unittests on Dhcp4ParserTest
/// redirecting the standard error to a temporary file, e.g. by
/// @code
///    ./dhcp4_unittests --gtest_filter="Dhcp4Parser*" > /dev/null 2> x
/// @endcode
///
/// Update EXTRACTED_CONFIGS with the file content
///
/// When configurations have been extracted the corresponding unparsed
/// configurations must be generated. To do that define GENERATE_ACTION
/// and recompile this file. Run dhcp4_unittests on Dhcp4GetConfigTest
/// redirecting the standard error to a temporary file, e.g. by
/// @code
///    ./dhcp4_unittests --gtest_filter="Dhcp4GetConfig*" > /dev/null 2> u
/// @endcode
///
/// Update UNPARSED_CONFIGS with the file content, recompile this file
/// without EXTRACT_CONFIG and GENERATE_ACTION.
///
/// @note Check for failures at each step!
/// @note The tests of this file do not check if configs returned
/// by @ref isc::dhcp::CfgToElement::ToElement() are complete.
/// This has to be done manually.
///
///@{
/// @brief extracted configurations
const char* EXTRACTED_CONFIGS[] = {
    // CONFIGURATION 0
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [ ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 1
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 2
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 3
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 4
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 0,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101 - 192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 5
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 1024,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 100,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 1,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 34,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101 - 192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 6
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"next-server\": \"1.2.3.4\",\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 7
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 8
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"next-server\": \"192.0.0.1\",\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 9
"{\n"
"        \"echo-client-id\": false,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 10
"{\n"
"        \"echo-client-id\": true,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 11
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"match-client-id\": true,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"match-client-id\": false,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 12
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"match-client-id\": true,\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"match-client-id\": false,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1 - 192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 13
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"renew-timer\": 1,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 14
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.2.200-192.0.2.255\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/25\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.3.128/25\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 15
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.128/28\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 16
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 17
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"uint16, ipv4-address, ipv6-address, string\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"record\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 18
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 101,\n"
"                \"name\": \"foo-2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 19
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": true,\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 20
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"sub-opts-space\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 21
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 109,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 22
"{\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 213,\n"
"                \"name\": \"access-network-domain-name\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 23
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\"\n"
"            },\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"01\",\n"
"                \"name\": \"default-ip-ttl\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 24
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    },\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 25
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 56,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 26
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 27
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"11\",\n"
"                \"name\": \"base-option\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 222,\n"
"                \"encapsulate\": \"isc\",\n"
"                \"name\": \"base-option\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"uint8\"\n"
"            },\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 28
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"data\": \"AB\",\n"
"                \"name\": \"dhcp-message\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    },\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 29
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"0102030405060708090A\",\n"
"                        \"name\": \"dhcp-message\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"FF\",\n"
"                        \"name\": \"default-ip-ttl\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 30
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 31
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"csv-format\": false,\n"
"                \"name\": \"vendor-encapsulated-options\"\n"
"            },\n"
"            {\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 3000\n"
"    }\n",
    // CONFIGURATION 32
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"option-one\",\n"
"                \"space\": \"vendor-4491\"\n"
"            },\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"option-two\",\n"
"                \"space\": \"vendor-1234\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 33
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"data\": \"this is a string vendor-opt\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-4491\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 34
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"eth1\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 35
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"*\", \"eth1\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 36
"{\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": true,\n"
"            \"enable-updates\": true,\n"
"            \"generated-prefix\": \"test.prefix\",\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": true,\n"
"            \"override-no-update\": true,\n"
"            \"qualifying-suffix\": \"test.suffix.\",\n"
"            \"replace-client-name\": \"when-present\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 37
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"192.0.2.123\"\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 38
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 123,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 234,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0A\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.3.15\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"32\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    },\n"
"                    {\n"
"                        \"hostname\": \"\",\n"
"                        \"hw-address\": \"01:02:03:04:05:06\",\n"
"                        \"ip-address\": \"192.0.3.120\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.3.95\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"11\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"id\": 542,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101 - 192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"0A:09:08:07:06:05:04:03:02:01\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.101\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"192.0.4.11\",\n"
"                                \"name\": \"name-servers\"\n"
"                            },\n"
"                            {\n"
"                                \"data\": \"95\",\n"
"                                \"name\": \"default-ip-ttl\"\n"
"                            }\n"
"                        ]\n"
"                    },\n"
"                    {\n"
"                        \"circuit-id\": \"060504030201\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.102\"\n"
"                    },\n"
"                    {\n"
"                        \"client-id\": \"05:01:02:03:04:05:06\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.103\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 39
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"option-def\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"id\": 234,\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101 - 192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0A\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"data\": \"123\",\n"
"                                \"name\": \"foo\",\n"
"                                \"space\": \"isc\"\n"
"                            }\n"
"                        ]\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 40
"{\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservation-mode\": \"all\",\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservation-mode\": \"out-of-pool\",\n"
"                \"subnet\": \"192.0.3.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"reservation-mode\": \"disabled\",\n"
"                \"subnet\": \"192.0.4.0/24\"\n"
"            },\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.5.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 41
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 42
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 43
"{\n"
"        \"decline-probation-period\": 12345,\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 44
"{\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 35,\n"
"            \"hold-reclaimed-time\": 1800,\n"
"            \"max-reclaim-leases\": 50,\n"
"            \"max-reclaim-time\": 100,\n"
"            \"reclaim-timer-wait-time\": 20,\n"
"            \"unwarned-reclaim-cycles\": 10\n"
"        },\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 45
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 46
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-subnet\": \"2001:db8::123/45\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 47
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 48
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-subnet\": \"2001:db8::543/21\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 49
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface-id\": \"vlan123\",\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 50
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"name\": \"one\"\n"
"            },\n"
"            {\n"
"                \"name\": \"two\"\n"
"            },\n"
"            {\n"
"                \"name\": \"three\"\n"
"            }\n"
"        ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1 - 192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 51
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 52
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": { }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 53
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n",
    // CONFIGURATION 54
"{\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"rebind-timer\": 2000,\n"
"        \"renew-timer\": 1000,\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0 - 192.0.2.15\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.2.0/24\"\n"
"            }\n"
"        ],\n"
"        \"valid-lifetime\": 4000\n"
"    }\n"
};

/// @brief unparsed configurations
const char* UNPARSED_CONFIGS[] = {
    // CONFIGURATION 0
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 1
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 900,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 2
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 1800,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 3
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 4
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 3,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 4,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101-192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 5
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1024,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 100,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 34,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.101-192.0.5.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 6
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 7
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 8
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"1.2.3.4\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 9
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": false,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 10
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 11
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": false,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 12
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": false,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.1-192.0.3.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 13
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 14
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.2.200-192.0.2.255\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/25\"\n"
"                    },\n"
"                    {\n"
"                        \"pool\": \"192.0.3.128/25\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 15
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.128/28\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 16
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 17
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"uint16, ipv4-address, ipv6-address, string\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"record\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 18
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 101,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo-2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 19
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": true,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 20
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"sub-opts-space\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 21
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 109,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 22
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 213,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"access-network-domain-name\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 23
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"code\": 23,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"01\",\n"
"                \"name\": \"default-ip-ttl\",\n"
"                \"space\": \"dhcp4\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 24
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 25
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"code\": 56,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 56,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 26
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 27
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 222,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"11\",\n"
"                \"name\": \"base-option\",\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"isc\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"isc\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 222,\n"
"                \"encapsulate\": \"isc\",\n"
"                \"name\": \"base-option\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"dhcp4\",\n"
"                \"type\": \"uint8\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 3000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 28
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 56,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"AB\",\n"
"                \"name\": \"dhcp-message\",\n"
"                \"space\": \"dhcp4\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"ABCDEF0105\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    },\n"
"                    {\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"01\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 29
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"code\": 56,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"0102030405060708090A\",\n"
"                        \"name\": \"dhcp-message\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [\n"
"                    {\n"
"                        \"code\": 23,\n"
"                        \"csv-format\": false,\n"
"                        \"data\": \"FF\",\n"
"                        \"name\": \"default-ip-ttl\",\n"
"                        \"space\": \"dhcp4\"\n"
"                    }\n"
"                ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 30
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 31
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 43,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"0104000004D20204C0A80201\",\n"
"                \"name\": \"vendor-encapsulated-options\",\n"
"                \"space\": \"dhcp4\"\n"
"            },\n"
"            {\n"
"                \"code\": 1,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"1234\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            },\n"
"            {\n"
"                \"code\": 2,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"192.168.2.1\",\n"
"                \"name\": \"foo2\",\n"
"                \"space\": \"vendor-encapsulated-options-space\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 1,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"uint32\"\n"
"            },\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 2,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo2\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-encapsulated-options-space\",\n"
"                \"type\": \"ipv4-address\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 3000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 32
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"1234\",\n"
"                \"space\": \"vendor-1234\"\n"
"            },\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": false,\n"
"                \"data\": \"ABCDEF0105\",\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.10\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 33
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [\n"
"            {\n"
"                \"code\": 100,\n"
"                \"csv-format\": true,\n"
"                \"data\": \"this is a string vendor-opt\",\n"
"                \"name\": \"foo\",\n"
"                \"space\": \"vendor-4491\"\n"
"            }\n"
"        ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"vendor-4491\",\n"
"                \"type\": \"string\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 34
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"eth0\", \"eth1\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 35
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\", \"eth0\", \"eth1\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 36
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": true,\n"
"            \"enable-updates\": true,\n"
"            \"generated-prefix\": \"test.prefix\",\n"
"            \"max-queue-size\": 2048,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": true,\n"
"            \"override-no-update\": true,\n"
"            \"qualifying-suffix\": \"test.suffix.\",\n"
"            \"replace-client-name\": \"when-present\",\n"
"            \"sender-ip\": \"192.168.2.2\",\n"
"            \"sender-port\": 778,\n"
"            \"server-ip\": \"192.168.2.1\",\n"
"            \"server-port\": 777\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 37
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"192.0.2.123\"\n"
"                },\n"
"                \"renew-timer\": 1,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 38
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 123,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 234,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"hostname\": \"\",\n"
"                        \"hw-address\": \"01:02:03:04:05:06\",\n"
"                        \"ip-address\": \"192.0.3.120\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.3.95\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"11\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0a\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.3.15\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"32\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 542,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.101-192.0.4.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"client-id\": \"05010203040506\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.103\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [ ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"circuit-id\": \"060504030201\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.102\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [ ],\n"
"                        \"server-hostname\": \"\"\n"
"                    },\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"0a:09:08:07:06:05:04:03:02:01\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.4.101\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"code\": 5,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"192.0.4.11\",\n"
"                                \"name\": \"name-servers\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            },\n"
"                            {\n"
"                                \"code\": 23,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"95\",\n"
"                                \"name\": \"default-ip-ttl\",\n"
"                                \"space\": \"dhcp4\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 39
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [\n"
"            {\n"
"                \"array\": false,\n"
"                \"code\": 100,\n"
"                \"encapsulate\": \"\",\n"
"                \"name\": \"foo\",\n"
"                \"record-types\": \"\",\n"
"                \"space\": \"isc\",\n"
"                \"type\": \"uint32\"\n"
"            }\n"
"        ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 234,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.101-192.0.3.150\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [\n"
"                    {\n"
"                        \"boot-file-name\": \"\",\n"
"                        \"client-classes\": [ ],\n"
"                        \"duid\": \"01:02:03:04:05:06:07:08:09:0a\",\n"
"                        \"hostname\": \"\",\n"
"                        \"ip-address\": \"192.0.3.112\",\n"
"                        \"next-server\": \"0.0.0.0\",\n"
"                        \"option-data\": [\n"
"                            {\n"
"                                \"code\": 100,\n"
"                                \"csv-format\": true,\n"
"                                \"data\": \"123\",\n"
"                                \"name\": \"foo\",\n"
"                                \"space\": \"isc\"\n"
"                            }\n"
"                        ],\n"
"                        \"server-hostname\": \"\"\n"
"                    }\n"
"                ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 40
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 2,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.3.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"out-of-pool\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.3.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 3,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.4.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"disabled\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.4.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            },\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 4,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.5.0/24\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.5.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 41
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 42
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 43
"{\n"
"        \"decline-probation-period\": 12345,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 44
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 35,\n"
"            \"hold-reclaimed-time\": 1800,\n"
"            \"max-reclaim-leases\": 50,\n"
"            \"max-reclaim-time\": 100,\n"
"            \"reclaim-timer-wait-time\": 20,\n"
"            \"unwarned-reclaim-cycles\": 10\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [ ]\n"
"    }\n",
    // CONFIGURATION 45
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 46
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"2001:db8::123/45\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 47
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 48
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"ethX\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"2001:db8::543/21\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 49
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"vlan123\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 50
"{\n"
"        \"client-classes\": [\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"one\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"three\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            },\n"
"            {\n"
"                \"boot-file-name\": \"\",\n"
"                \"name\": \"two\",\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"server-hostname\": \"\"\n"
"            }\n"
"        ],\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.1-192.0.2.100\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 51
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\"\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 52
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": { }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 53
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n",
    // CONFIGURATION 54
"{\n"
"        \"decline-probation-period\": 86400,\n"
"        \"dhcp-ddns\": {\n"
"            \"always-include-fqdn\": false,\n"
"            \"enable-updates\": false,\n"
"            \"generated-prefix\": \"myhost\",\n"
"            \"max-queue-size\": 1024,\n"
"            \"ncr-format\": \"JSON\",\n"
"            \"ncr-protocol\": \"UDP\",\n"
"            \"override-client-update\": false,\n"
"            \"override-no-update\": false,\n"
"            \"qualifying-suffix\": \"\",\n"
"            \"replace-client-name\": \"never\",\n"
"            \"sender-ip\": \"0.0.0.0\",\n"
"            \"sender-port\": 0,\n"
"            \"server-ip\": \"127.0.0.1\",\n"
"            \"server-port\": 53001\n"
"        },\n"
"        \"dhcp4o6-port\": 0,\n"
"        \"echo-client-id\": true,\n"
"        \"expired-leases-processing\": {\n"
"            \"flush-reclaimed-timer-wait-time\": 25,\n"
"            \"hold-reclaimed-time\": 3600,\n"
"            \"max-reclaim-leases\": 100,\n"
"            \"max-reclaim-time\": 250,\n"
"            \"reclaim-timer-wait-time\": 10,\n"
"            \"unwarned-reclaim-cycles\": 5\n"
"        },\n"
"        \"hooks-libraries\": [ ],\n"
"        \"host-reservation-identifiers\": [ \"hw-address\", \"duid\", \"circuit-id\" ],\n"
"        \"interfaces-config\": {\n"
"            \"interfaces\": [ \"*\" ]\n"
"        },\n"
"        \"lease-database\": {\n"
"            \"type\": \"memfile\"\n"
"        },\n"
"        \"option-data\": [ ],\n"
"        \"option-def\": [ ],\n"
"        \"subnet4\": [\n"
"            {\n"
"                \"4o6-interface\": \"\",\n"
"                \"4o6-interface-id\": \"\",\n"
"                \"4o6-subnet\": \"\",\n"
"                \"id\": 1,\n"
"                \"interface\": \"\",\n"
"                \"match-client-id\": true,\n"
"                \"next-server\": \"0.0.0.0\",\n"
"                \"option-data\": [ ],\n"
"                \"pools\": [\n"
"                    {\n"
"                        \"pool\": \"192.0.2.0/28\",\n"
"                        \"user-context\": {\n"
"                            \"bool-param\": true,\n"
"                            \"integer-param\": 42,\n"
"                            \"string-param\": \"Sagittarius\"\n"
"                        }\n"
"                    }\n"
"                ],\n"
"                \"rebind-timer\": 2000,\n"
"                \"relay\": {\n"
"                    \"ip-address\": \"0.0.0.0\"\n"
"                },\n"
"                \"renew-timer\": 1000,\n"
"                \"reservation-mode\": \"all\",\n"
"                \"reservations\": [ ],\n"
"                \"subnet\": \"192.0.2.0/24\",\n"
"                \"valid-lifetime\": 4000\n"
"            }\n"
"        ]\n"
"    }\n"
};

/// @brief the number of configurations
const size_t max_config_counter = sizeof(EXTRACTED_CONFIGS) / sizeof(char*);
///@}

/// @brief the extraction counter
///
/// < 0 means do not extract, >= 0 means extract on extractConfig() calls
/// and increment
#ifdef EXTRACT_CONFIG
int extract_count = 0;
#else
int extract_count = -1;
#endif

/// @brief the generate action
/// false means do nothing, true means unparse extracted configurations
#ifdef GENERATE_ACTION
const bool generate_action = true;
#else
const bool generate_action = false;
static_assert(max_config_counter == sizeof(UNPARSED_CONFIGS) / sizeof(char*),
              "unparsed configurations must be generated");
#endif

/// @brief format and output a configuration
void
outputFormatted(const std::string& config) {
    // pretty print it
    ConstElementPtr json = parseJSON(config);
    std::string prettier = prettyPrint(json, 4, 4);
    // get it as a line array
    std::list<std::string> lines;
    boost::split(lines, prettier, boost::is_any_of("\n"));
    // add escapes using again JSON
    std::list<std::string> escapes;
    while (!lines.empty()) {
        const std::string& line = lines.front();
        ConstElementPtr escaping = Element::create(line + "\n");
        escapes.push_back(escaping->str());
        lines.pop_front();
    }
    // output them on std::cerr
    while (!escapes.empty()) {
        std::cerr << "\n" << escapes.front();
        escapes.pop_front();
    }
}

};

namespace isc {
namespace dhcp {
namespace test {

/// @ref isc::dhcp::test::extractConfig in the header
void
extractConfig(const std::string& config) {
    // skip when disable
    if (extract_count < 0) {
        return;
    }
    // mark beginning
    if (extract_count == 0) {
        // header (note there is no trailer)
        std::cerr << "/// put this after const char* EXTRACTED_CONFIGS[] = {\n";
    } else {
        // end of previous configuration
        std::cerr << ",\n";
    }
    std::cerr << "    // CONFIGURATION " << extract_count;
    try {
        outputFormatted(config);
    } catch (...) {
        // mark error
        std::cerr << "\n//// got an error\n";
    }
    ++extract_count;
}

};
};
};

namespace {

/// Test fixture class (code from Dhcp4ParserTest)
class Dhcp4GetConfigTest : public ::testing::TestWithParam<size_t> {
public:
    Dhcp4GetConfigTest()
    : rcode_(-1) {
        // Open port 0 means to not do anything at all. We don't want to
        // deal with sockets here, just check if configuration handling
        // is sane.
        srv_.reset(new Dhcpv4Srv(0));
        // Create fresh context.
        resetConfiguration();
    }

    ~Dhcp4GetConfigTest() {
        resetConfiguration();
    };

    /// @brief Parse and Execute configuration
    ///
    /// Parses a configuration and executes a configuration of the server.
    /// If the operation fails, the current test will register a failure.
    ///
    /// @param config Configuration to parse
    /// @param operation Operation being performed.  In the case of an error,
    ///        the error text will include the string "unable to <operation>.".
    ///
    /// @return true if the configuration succeeded, false if not.
    bool
    executeConfiguration(const std::string& config, const char* operation) {
        // clear config manager
        CfgMgr::instance().clear();

        // enable fake network interfaces
        IfaceMgrTestConfig test_config(true);

        // try JSON parser
        ConstElementPtr json;
        try {
            json = parseJSON(config);
        } catch (const std::exception& ex) {
            ADD_FAILURE() << "invalid JSON for " << operation
                          << " failed with " << ex.what()
                          << " on\n" << config << "\n";
            return (false);
        }

        // try DHCP4 parser
        try {
            json = parseDHCP4(config, true);
        } catch (...) {
            ADD_FAILURE() << "parsing failed for " << operation
                          << " on\n" << prettyPrint(json) << "\n";
            return (false);
        }

        // try DHCP4 configure
        ConstElementPtr status;
        try {
            status = configureDhcp4Server(*srv_, json);
        } catch (const std::exception& ex) {
            ADD_FAILURE() << "configure for " << operation
                          << " failed with " << ex.what()
                          << " on\n" << prettyPrint(json) << "\n";
            return (false);
        }

        // The status object must not be NULL
        if (!status) {
            ADD_FAILURE() << "configure for " << operation
                          << " returned null on\n"
                          << prettyPrint(json) << "\n";
            return (false);
        }

        // Returned value should be 0 (configuration success)
        comment_ = parseAnswer(rcode_, status);
        if (rcode_ != 0) {
            string reason = "";
            if (comment_) {
                reason = string(" (") + comment_->stringValue() + string(")");
            }
            ADD_FAILURE() << "configure for " << operation
                          << " returned error code "
                          << rcode_ << reason << " on\n"
                          << prettyPrint(json) << "\n";
            return (false);
        }
        return (true);
    }

    /// @brief Reset configuration database.
    ///
    /// This function resets configuration data base by
    /// removing all subnets and option-data. Reset must
    /// be performed after each test to make sure that
    /// contents of the database do not affect result of
    /// subsequent tests.
    void resetConfiguration() {
        string config = "{"
            "\"interfaces-config\": { \"interfaces\": [ \"*\" ] },"
            "\"valid-lifetime\": 4000, "
            "\"subnet4\": [ ], "
            "\"dhcp-ddns\": { \"enable-updates\" : false }, "
            "\"option-def\": [ ], "
            "\"option-data\": [ ] }";
        EXPECT_TRUE(executeConfiguration(config, "reset configuration"));
        CfgMgr::instance().clear();
        CfgMgr::instance().setFamily(AF_INET);
    }

    boost::scoped_ptr<Dhcpv4Srv> srv_;  ///< DHCP4 server under test
    int rcode_;                         ///< Return code from element parsing
    ConstElementPtr comment_;           ///< Reason for parse fail
};

/// Test a configuration
TEST_P(Dhcp4GetConfigTest, run) {
    // configurations have not been extracted yet
    if (max_config_counter == 0) {
        return;
    }

    // get the index of configurations to test
    size_t config_counter = GetParam();

    // emit unparsed header if wanted
    if ((config_counter == 0) && generate_action) {
        std::cerr << "///put this after const char* UNPARSED_CONFIGS[] = {\n";
    }

    // get the extracted configuration
    std::string config = EXTRACTED_CONFIGS[config_counter];
    std::ostringstream ss;
    ss << "extracted config #" << config_counter;

    // execute the extracted configuration
    ASSERT_TRUE(executeConfiguration(config, ss.str().c_str()));

    // unparse it
    ConstSrvConfigPtr extracted = CfgMgr::instance().getStagingCfg();
    ConstElementPtr unparsed;
    ASSERT_NO_THROW(unparsed = extracted->toElement());
    ConstElementPtr dhcp;
    ASSERT_NO_THROW(dhcp = unparsed->get("Dhcp4"));
    ASSERT_TRUE(dhcp);

    // dump if wanted else check
    std::string expected;
    if (generate_action) {
        if (config_counter > 0) {
            std::cerr << ",\n";
        }
        std::cerr << "    // CONFIGURATION " << config_counter;
        ASSERT_NO_THROW(expected = prettyPrint(dhcp));
        ASSERT_NO_THROW(outputFormatted(dhcp->str()));
    } else {
        expected = UNPARSED_CONFIGS[config_counter];
        ConstElementPtr json;
        ASSERT_NO_THROW(json = parseDHCP4(expected, true));
        EXPECT_TRUE(isEquivalent(dhcp, json));
        std::string current = prettyPrint(dhcp, 4, 4) + "\n";
        EXPECT_EQ(expected, current);
        if (expected != current) {
            expected = current;
        }
    }

    // execute the dhcp configuration
    ss.str("");
    ss << "unparsed config #" << config_counter;
    EXPECT_TRUE(executeConfiguration(expected, ss.str().c_str()));

    // is it a fixed point?
    ConstSrvConfigPtr extracted2 = CfgMgr::instance().getStagingCfg();
    ConstElementPtr unparsed2;
    ASSERT_NO_THROW(unparsed2 = extracted2->toElement());
    ASSERT_TRUE(unparsed2);
    EXPECT_TRUE(isEquivalent(unparsed, unparsed2));
}

/// Define the parametrized test loop
INSTANTIATE_TEST_CASE_P(Dhcp4GetConfigTest, Dhcp4GetConfigTest,
                        ::testing::Range(static_cast<size_t>(0), max_config_counter));

};
