#ifndef lint
static char *rcsid = "$Id: dude.c,v 1.1 2001/04/01 17:47:00 vixie Exp $";
#endif

/*
 * Copyright (c) 2000,2001 Japan Network Information Center.
 * All rights reserved.
 *  
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 * 			LICENSE TERMS AND CONDITIONS 
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Fuundo Bldg., 1-2 Kanda Ogawamachi, Chiyoda-ku,
 * Tokyo, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution.  For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) Japan Network Information Center.  All rights reserved."
 * 
 * 4. Neither the name of JPNIC may be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 6. Indemnification by Licensee
 *    Any person or entities using and/or redistributing this Software under
 *    this License Terms and Conditions shall defend indemnify and hold
 *    harmless JPNIC from and against any and all judgements damages,
 *    expenses, settlement liabilities, cost and other liabilities of any
 *    kind as a result of use and redistribution of this Software or any
 *    claim, suite, action, litigation or proceeding by any third party
 *    arising out of or relates to this License Terms and Conditions.
 * 
 * 7. Governing Law, Jurisdiction and Venue
 *    This License Terms and Conditions shall be governed by and and
 *    construed in accordance with the law of Japan. Any person or entities
 *    using and/or redistributing this Software under this License Terms and
 *    Conditions hereby agrees and consent to the personal and exclusive
 *    jurisdiction and venue of Tokyo District Court of Japan.
 */

#include <config.h>

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <mdn/result.h>
#include <mdn/assert.h>
#include <mdn/logmacro.h>
#include <mdn/converter.h>
#include <mdn/utf8.h>
#include <mdn/debug.h>
#include <mdn/dude.h>
#include <mdn/util.h>

#ifndef MDN_DUDE_PREFIX
#define MDN_DUDE_PREFIX		"dq--"
#endif
#define DUDE_PREFIX_LEN		(strlen(MDN_DUDE_PREFIX))

static unsigned long nibble_mask[] = {
	0,		/* dummy: this element is never referenced. */
	0xf,
	0xff,
	0xfff,
	0xffff,
	0xfffff,
	0xffffff,
};

static mdn_result_t	dude_l2u(const char *from, const char *end,
				 char *to, size_t tolen, size_t *clenp);
static mdn_result_t	dude_u2l(const char *from, const char *end,
				 char *to, size_t tolen, size_t *clenp);
static mdn_result_t	dude_decode(const char *from, size_t fromlen,
				    char *to, size_t tolen);
static mdn_result_t	dude_encode(const char *from, size_t fromlen,
				    char *to, size_t tolen);
static int		get_nibblelength(unsigned long v);
static int		dude_getwc(const char *s, size_t len,
				   unsigned long *vp, int *nlenp);
static int		dude_putwc(char *s, size_t len, unsigned long v,
				   int w);

/* ARGSUSED */
mdn_result_t
mdn__dude_open(mdn_converter_t ctx, mdn_converter_dir_t dir) {
	return (mdn_success);
}

/* ARGSUSED */
mdn_result_t
mdn__dude_close(mdn_converter_t ctx, mdn_converter_dir_t dir) {
	return (mdn_success);
}

mdn_result_t
mdn__dude_convert(mdn_converter_t ctx, mdn_converter_dir_t dir,
		    const char *from, char *toorg, size_t tolen)
{
	char *to = toorg;

	assert(ctx != NULL &&
	       (dir == mdn_converter_l2u || dir == mdn_converter_u2l));

	TRACE(("mdn__dude_convert(dir=%s,from=\"%s\")\n",
	       dir == mdn_converter_l2u ? "l2u" : "u2l",
	       mdn_debug_xstring(from, 20)));

	for (;;) {
		const char *end;
		size_t convlen;
		mdn_result_t r;

		/*
		 * Find the end of this component (label).
		 */
		if ((end = strchr(from, '.')) == NULL)
			end = from + strlen(from);

		/*
		 * Convert it.
		 */
		if (dir == mdn_converter_l2u)
			r = dude_l2u(from, end, to, tolen, &convlen);
		else
			r = dude_u2l(from, end, to, tolen, &convlen);
		if (r != mdn_success)
			return (r);

		/*
		 * Copy '.' or NUL.
		 */
		if (tolen <= convlen)
			return (mdn_buffer_overflow);

		to += convlen;
		*to++ = *end;
		tolen -= convlen + 1;

		if (*end == '\0')
			break;
		else if (*(end + 1) == '\0') {
			if (tolen < 1)
				return (mdn_buffer_overflow);
			*to++ = '\0';
			tolen--;
			break;
		}

		from = end + 1;
	}

	DUMP(("mdn__dude_convert: \"%s\"\n", mdn_debug_xstring(toorg, 70)));

	return (mdn_success);
}

static mdn_result_t
dude_l2u(const char *from, const char *end,
	 char *to, size_t tolen, size_t *clenp)
{
	size_t len = end - from;
	size_t prefix_len = DUDE_PREFIX_LEN;

	if (len >= prefix_len &&
	    mdn_util_casematch(from, MDN_DUDE_PREFIX, prefix_len)) {
		/*
		 * DUDE encoding prefix found.
		 */
		mdn_result_t r;

		r = dude_decode(from + prefix_len,
				len - prefix_len, to, tolen);
		if (r == mdn_invalid_encoding)
			goto copy;
		else if (r != mdn_success)
			return (r);

		len = strlen(to);
	} else {
		/*
		 * Not DUDE encoded.  Copy verbatim.
		 */
	copy:
		if (!mdn_util_validstd13(from, end)) {
			/* invalid character found */
			return (mdn_invalid_encoding);
		}

		if (tolen < len)
			return (mdn_buffer_overflow);

		(void)memcpy(to, from, len);
	}
	*clenp = len;
	return (mdn_success);
}

static mdn_result_t
dude_u2l(const char *from, const char *end,
	   char *to, size_t tolen, size_t *clenp) {
	size_t len = end - from;
	size_t prefix_len = DUDE_PREFIX_LEN;

	/*
	 * See if encoding is necessary.
	 */
	if (!mdn_util_validstd13(from, end)) {
		/*
		 * Conversion is necessary.
		 */
		mdn_result_t r;

		/* Set prefix. */
		if (tolen < prefix_len)
			return (mdn_buffer_overflow);
		(void)memcpy(to, MDN_DUDE_PREFIX, prefix_len);
		to += prefix_len;
		tolen -= prefix_len;

		r = dude_encode(from, len, to, tolen);
		if (r != mdn_success)
			return (r);

		len = prefix_len + strlen(to);
	} else {
		/*
		 * Conversion is NOT necessary.
		 * Copy verbatim.
		 */
		if (tolen < len)
			return (mdn_buffer_overflow);

		(void)memcpy(to, from, len);
	}
	*clenp = len;
	return (mdn_success);
}

static mdn_result_t
dude_decode(const char *from, size_t fromlen, char *to, size_t tolen) {
	size_t len;
	unsigned long prev, v, mask;
	char *to_org  = to;

	prev = 0;
	while (fromlen > 0) {
		if (from[0] == '-') {
			v = '-';
			from++;
			fromlen--;
		} else {
			int nlen;
			len = dude_getwc(from, fromlen, &v, &nlen);
			if (len == 0)
				return (mdn_invalid_encoding);
			from += len;
			fromlen -= len;
			mask = nibble_mask[nlen];
			v = (prev & ~mask) | v;

			/*
			 * Perform extra sanity checks.
			 */
			if (v == '-' || get_nibblelength(prev ^ v) != nlen)
				return (mdn_invalid_encoding);

			prev = v;
		}
		len = mdn_utf8_putwc(to, tolen, v);
		if (len == 0)
			return (mdn_buffer_overflow);
		to += len;
		tolen -= len;
	}

	/*
	 * Terminate with NUL.
	 */
	if (tolen <= 0)
		return (mdn_buffer_overflow);

	*to = '\0';
#if 0
	if (mdn_util_validstd13(to_org, strlen(to_org)))
		return (mdn_invalid_encoding);
#endif
	return (mdn_success);
}

static mdn_result_t
dude_encode(const char *from, size_t fromlen, char *to, size_t tolen) {
	size_t len;
	unsigned long prev, c, v, mask;

	prev = 0;
	while (fromlen > 0) {
		len = mdn_utf8_getwc(from, fromlen, &c);
		from += len;
		fromlen -= len;
		if (len == 0 || c > 0x10ffff)
			return (mdn_invalid_encoding);
		if (c == '-') {
			/*
			 * This special treatment of '-' may have some
			 * problem with STD13 conformance of the encoded
			 * name.  That is, if the input string ends with
			 * '-', the encoded name also ends with '-'.
			 */
			if (tolen < 1)
				return (mdn_buffer_overflow);
			*to++ = '-';
			tolen--;
		} else {
			/*
			 * We don't do DUDE extension #1 -- embedding
			 * case information in the encoded string --
			 * at least for now.  One of the reasons is that
			 * the draft is not clear if the character to be
			 * encoded is a titlecase character.
			 */
			int nlen = get_nibblelength(prev ^ c);
			mask = nibble_mask[nlen];
			v = c & mask;
			prev = c;
			len = dude_putwc(to, tolen, v, nlen);
			if (len == 0)
				return (mdn_buffer_overflow);
			to += len;
			tolen -= len;
		}
	}

	/*
	 * Terminate with NUL.
	 */
	if (tolen <= 0)
		return (mdn_buffer_overflow);

	*to = '\0';

	return (mdn_success);
}

static int
get_nibblelength(unsigned long v) {
	assert(v <= 0x10ffff);

	if (v <= 0xf)
		return 1;
	else if (v <= 0xff)
		return 2;
	else if (v <= 0xfff)
		return 3;
	else if (v <= 0xffff)
		return 4;
	else if (v <= 0xfffff)
		return 5;
	else
		return 6;
}

static int
dude_getwc(const char *s, size_t len, unsigned long *vp, int *nlenp) {
	size_t orglen = len;
	unsigned long v = 0;
	int dude_ext;		/* is DUDE extension #2 used? */
	int c;

	if (len < 1)
		return (0);

	c = *s++;
	len--;

	if ('G' <= c && c <= 'W')
		v = c - 'G';
	else if ('g' <= c && c <= 'w')
		v = c - 'g';
	else
		return (0);

	dude_ext = (v > 15);

	while (len > 0) {
		c = *s++;
		if ('0' <= c && c <= '9')
			c = c - '0';
		else if ('A' <= c && c <= 'F')
			c = c - 'A' + 10;
		else if ('a' <= c && c <= 'f')
			c = c - 'a' + 10;
		else
			break;
		v = (v << 4) + c;
		len--;
	}
	len = orglen - len;

	if (len > 5 || (dude_ext && len != 5))
		return (0);

	*vp = v;
	*nlenp = len + (dude_ext ? 1 : 0);
	return (len);
}

static int
dude_putwc(char *s, size_t len, unsigned long v, int w) {
	int off;		/* bit offset */

	assert(v <= 0x10ffff);
	assert(w > 0 && w <= 6 && v <= nibble_mask[w]);

	if (w == 6)
		w = 5;		/* for DUDE extension #2 */

	if (len < w)
		return (0);

	off = (w - 1) * 4;
	*s++ = 'g' + (v >> off);
	off -= 4;
	while (off >= 0) {
		int x = (v >> off) & 0xf;
		if (x < 10)
			*s++ = '0' + x;
		else
			*s++ = 'a' + x - 10;
		off -= 4;
	}
	return (w);
}
