/*
 * Copyright (c) 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses error recovery level zero functions used by the
 * iSCSI Initiator driver.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_ERL0_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/crypto.h>
#include <linux/blkdev.h>
#include <scsi.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_crc.h>
#include <iscsi_initiator_debug_opcodes.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_tmr.h>
#include <iscsi_initiator_util.h>

#undef ISCSI_INITIATOR_ERL0_C

extern iscsi_global_t *iscsi_global;
extern int iscsi_create_connection (iscsi_session_t *, iscsi_login_holder_t *, u16 *);
extern u32 iscsi_create_session (iscsi_login_holder_t *);
extern int iscsi_close_connection (iscsi_conn_t *);
extern int iscsi_close_session (iscsi_session_t *);
extern int iscsi_stop_session (iscsi_session_t *, int, int);
extern void iscsi_set_state_for_connections (iscsi_session_t *, u8);
static void iscsi_start_extra_conns_after_reinstatement (iscsi_channel_t *);

/*	iscsi_set_datain_sequence_values():
 *
 * 	Used to set the values in iscsi_cmd_t that iscsi_datain_check_sequence()
 * 	checks against to determine a PDU's Offset+Length is within the current
 * 	DataIN Sequence.  Used for DataSequenceInOrder=Yes only.
 */
extern void iscsi_set_datain_sequence_values (
	iscsi_cmd_t *cmd)
{
	iscsi_conn_t *conn = CONN(cmd);
	
	if (!cmd->seq_start_offset && !cmd->seq_end_offset) {
		cmd->seq_start_offset = 0;
		cmd->seq_end_offset = (cmd->data_length >
			SESS_OPS_C(conn)->MaxBurstLength) ?
			SESS_OPS_C(conn)->MaxBurstLength :
			cmd->data_length;
	} else {
		cmd->seq_start_offset = cmd->seq_end_offset;
		cmd->seq_end_offset = ((cmd->seq_end_offset +
			  SESS_OPS_C(conn)->MaxBurstLength) >=
			  cmd->data_length) ? cmd->data_length :
			  (cmd->seq_end_offset +
			   SESS_OPS_C(conn)->MaxBurstLength);
	}
		
	return;
}

/*	iscsi_datain_check_sequence():
 *
 *	Called from:	iscsi_check_pre_datain()
 */
static inline int iscsi_datain_check_sequence (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	u32 next_burst_len;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_seq_t *seq = NULL;
	struct iscsi_targ_scsi_data_in *hdr =
		(struct iscsi_targ_scsi_data_in *) buf;

	/*
	 * For DataSequenceInOrder=Yes: Check that the offset and offset+length
	 * is within range as defined by iscsi_set_datain_sequence_values().
	 * 
	 * For DataSequenceInOrder=No: Check that an iscsi_seq_t exists for
	 * offset+length tuple.
	 */
	if (SESS_OPS_C(conn)->DataSequenceInOrder) { 
		/*
		 * This check incorrectly assumes that the target uses data
		 * sequences that are equal to the maximum length when it is
		 * permissible to use shorter sequence lengths.  Leave the
		 * check disabled for now.
		 */
#if 0
		if ((hdr->offset < cmd->seq_start_offset) ||
		   ((hdr->offset + hdr->length) > cmd->seq_end_offset)) {
			TRACE_ERROR("Command ITT: 0x%08x with Offset: %u,"
			" Length: %u outside of Sequence: %u:%u while"
			" DataSequenceInOrder=Yes.\n", cmd->init_task_tag,
			hdr->offset, hdr->length, cmd->seq_start_offset,
					cmd->seq_end_offset);
			return(-1);
		}
#endif
		next_burst_len = (cmd->next_burst_len + hdr->length);
	} else {
		if (!(seq = iscsi_get_seq_holder(cmd, hdr->offset, hdr->length)))
			return(DATAIN_CANNOT_RECOVER);
		/*
		 * Set the iscsi_seq_t pointer to reuse in iscsi_check_post_datain().
		 */
		cmd->seq_ptr = seq;

		next_burst_len = (seq->next_burst_len + hdr->length);
	}

	if (next_burst_len > SESS_OPS_C(conn)->MaxBurstLength) {
		TRACE_ERROR("Command ITT: 0x%08x, NextBurstLength: %u and Length:"
			" %u exceed MaxBurstLength: %u. protocol error.\n",
			cmd->init_task_tag, (next_burst_len - hdr->length),
			hdr->length, SESS_OPS_C(conn)->MaxBurstLength);
		return(DATAIN_CANNOT_RECOVER);
	}

	/*
	 * Perform various MaxBurstLength and F_BIT sanity checks for the
	 * current DataIN Sequence.
	 */
	if (hdr->flags & F_BIT) {
		/*
		 * There checks are a bit overzealous and cause problems when there
		 * is an residual count by way of the underflow bit being set.  Leave
		 * them disabled for now.
		 */
#if 0
		if (SESS_OPS_C(conn)->DataSequenceInOrder) {
			if ((next_burst_len < SESS_OPS_C(conn)->MaxBurstLength) &&
		   	   ((cmd->read_data_done + hdr->length) < cmd->data_length)) {
				TRACE_ERROR("Command ITT: 0x%08x set F_BIT before"
				" end of DataIN sequence, protocol error.\n",
					cmd->init_task_tag);			
				return(DATAIN_CANNOT_RECOVER);
			}
		} else {
			if (next_burst_len < seq->xfer_len) {
				TRACE_ERROR("Command ITT: 0x%08x set F_BIT before"
				" end of DataIN sequence, protocol error.\n",
					cmd->init_task_tag);
				return(DATAIN_CANNOT_RECOVER);
			}
		}
#endif
	} else {
		if (SESS_OPS_C(conn)->DataSequenceInOrder) {
			if (next_burst_len == SESS_OPS_C(conn)->MaxBurstLength) {
				TRACE_ERROR("Command ITT: 0x%08x reached MaxBurstLength:"
				" %u, but F_BIT is not set, protocol error.",
				cmd->init_task_tag, SESS_OPS_C(conn)->MaxBurstLength);
				return(DATAIN_CANNOT_RECOVER);
			}
			if ((cmd->read_data_done + hdr->length) == cmd->data_length) {
				TRACE_ERROR("Command ITT: 0x%08x reached last DataIN PDU"
				" in sequence but F_BIT not set, protocol error.\n",
					cmd->init_task_tag);
				return(DATAIN_CANNOT_RECOVER);
			}
		} else {
			if (next_burst_len == seq->xfer_len) {
				TRACE_ERROR("Command ITT: 0x%08x reached last"
				" DataIN PDU in sequence but F_BIT not set,"
				" protocol error.\n", cmd->init_task_tag);
				return(DATAIN_CANNOT_RECOVER);
			}
		}
	}

	return(0);
}

/*	iscsi_datain_check_datasn():
 *
 *	Called from:	iscsi_check_pre_datain()
 */
static inline int iscsi_datain_check_datasn (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	int dump = 0, recovery = 0;
	iscsi_conn_t *conn = CONN(cmd);
	struct iscsi_targ_scsi_data_in *hdr =
		(struct iscsi_targ_scsi_data_in *) buf;
	
	/*
	 * When in within-command recovery, ignore all DataSNs until
	 * we receive the one that caused recovery to initially begin.
	 */
	if ((cmd->cmd_flags & ICF_WITHIN_COMMAND_RECOVERY) &&
	    (cmd->data_sn != hdr->data_sn)) {
		dump = 1;
		goto dump;
	}
	cmd->cmd_flags &= ~ICF_WITHIN_COMMAND_RECOVERY;
		
	/*
	 * If the DataSN is greater than expected, assume (Received DataSN -
	 * Expected DataSN) PDUs where lost to header digest falures, immediately
	 * send a Data SNACK.
	 *
	 * If the DataSN is less than expected assume this was a response to a
	 * proactive Data SNACK and dump the datain payload into an offload buffer.
	 */
	if (hdr->data_sn > cmd->data_sn) {
		TRACE_ERROR("Command ITT: 0x%08x, received DataSN: 0x%08x"
			" higher than expected: 0x%08x.\n", cmd->init_task_tag,
				hdr->data_sn, cmd->data_sn);
		recovery = 1;
		goto recover;
	} else if (hdr->data_sn < cmd->data_sn) {
		TRACE_ERROR("Command ITT: 0x%08x, received DataSN: 0x%08x"
			" lower than expected: 0x%08x, discarding payload.\n",
			cmd->init_task_tag, hdr->data_sn, cmd->data_sn);
		dump = 1;
		goto dump;
	}

	return(DATAIN_NORMAL);

recover:
	if (!SESS_OPS_C(conn)->ErrorRecoveryLevel) {
		TRACE_ERROR("Unable to perform within-command recovery"
				" while ERL=0.\n");
		return(DATAIN_CANNOT_RECOVER);
	}
dump:
	if (iscsi_dump_data_payload(conn, hdr->length, 1) < 0)
		return(DATAIN_CANNOT_RECOVER);

	return(DATAIN_CANNOT_RECOVER);
}

/*	iscsi_datain_pre_datapduinorder_yes():
 *
 *	Called from:	iscsi_check_pre_datain()
 */
static inline int iscsi_datain_pre_datapduinorder_yes (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	iscsi_conn_t *conn = CONN(cmd);
	struct iscsi_targ_scsi_data_in *hdr =
		(struct iscsi_targ_scsi_data_in *) buf;

	/*
	 * For DataSequenceInOrder=Yes: If the offset is greater than the global
	 * DataPDUInOrder=Yes offset counter in iscsi_cmd_t a protcol error has
	 * occured and fail the connection.
	 *
	 * For DataSequenceInOrder=No: If the offset is greater than the per
	 * sequence DataPDUInOrder=Yes offset counter in iscsi_seq_t a protocol
	 * error has occured and fail the connection.
	 */
	if (SESS_OPS_C(conn)->DataSequenceInOrder) {
		if (hdr->offset != cmd->read_data_done) {
			TRACE_ERROR("Command ITT: 0x%08x, received offset"
				" %u different than expected %u. Protocol error.\n",
				cmd->init_task_tag, hdr->offset, cmd->read_data_done);
			return(DATAIN_CANNOT_RECOVER);
		}
	} else {
		iscsi_seq_t *seq = cmd->seq_ptr;

		if (hdr->offset != seq->offset) {
			TRACE_ERROR("Command ITT: 0x%08x, received offset"
				" %u different than expected %u. Protcol error.\n",
				cmd->init_task_tag, hdr->offset, seq->offset);
			return(DATAIN_CANNOT_RECOVER);		
		}
	}

	return(DATAIN_NORMAL);
}

/*	iscsi_datain_pre_datapduinorder_no():
 *
 *	Called from:	iscsi_check_pre_datain()
 */
static inline int iscsi_datain_pre_datapduinorder_no (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	iscsi_pdu_t *pdu;
	struct iscsi_targ_scsi_data_in *hdr =
		(struct iscsi_targ_scsi_data_in *) buf;

	if (!(pdu = iscsi_get_pdu_holder(cmd, hdr->offset, hdr->length)))
		return(DATAIN_CANNOT_RECOVER);

	/*
	 * Set the iscsi_pdu_t pointer to reuse in iscsi_check_post_datain().
	 */
	cmd->pdu_ptr = pdu;
	
	switch (pdu->status) {
		case ISCSI_PDU_NOT_RECEIVED:
		case ISCSI_PDU_CRC_FAILED:
		case ISCSI_PDU_TIMED_OUT:
			break;
		case ISCSI_PDU_RECEIVED_OK:
			TRACE_ERROR("Command ITT: 0x%08x received already gotten"
				" Offset: %u, Length: %u\n", cmd->init_task_tag,
					hdr->offset, hdr->length);
			if (iscsi_dump_data_payload(CONN(cmd), hdr->length, 1) < 0)
				return(DATAIN_CANNOT_RECOVER);
			return(DATAIN_WITHIN_COMMAND_RECOVERY);
		default:
			return(DATAIN_CANNOT_RECOVER);
	}

	return(DATAIN_NORMAL);
}

/*	iscsi_datain_post_crc_passed()
 *
 *	Called from:	iscsi_check_post_datain()
 */
static inline int iscsi_datain_post_crc_passed (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	int send_to_scsi = 0;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_pdu_t *pdu;
	struct iscsi_targ_scsi_data_in *hdr =
		(struct iscsi_targ_scsi_data_in *) buf;

	/*
	 * It is OPTIONAL not to send a Data ACK if A_BIT is set and Status
	 * was sent ie: S_BIT on the the last DataIN.  If the
	 * !(hdr->flags & S_BIT) check is removed it will operate as it should,
	 * and will send the aforementioned OPTIONAL DataACK before
	 * acknowledgement of the StatSN on our side via iscsi_process_response().
	 *
	 * Since I do not see the point in sending a DataACK except
	 * on non S_BIT carring DataIN PDUs denoting the end of a
	 * MaxBurstLength with an F_BIT, we leave the !(hdr->flags & S_BIT)
	 * check in for now. (see iSCSI v19 10.7.2)
	 */
	if ((hdr->flags & F_BIT) && (SESS_OPS_C(conn)->ErrorRecoveryLevel > 0) &&
	    (hdr->flags & A_BIT) && !(hdr->flags & S_BIT)) {
		cmd->acked_datasn = hdr->data_sn;
		return(DATAIN_CANNOT_RECOVER);
	}
	
	/*
	 * Perform various sanity checks for S_BIT carring DataIN PDU.
	 */
	if (hdr->flags & S_BIT) {
		if (!(hdr->flags & U_BIT)) {
			if ((cmd->read_data_done + hdr->length) < cmd->data_length) {
				TRACE_ERROR("Command ITT: 0x%08x requesting phase"
					" collase but cmd->read_data_done: %u is less"
					" than cmd->data_length: %u\n", cmd->init_task_tag,
					(cmd->read_data_done + hdr->length), cmd->data_length);
				return(DATAIN_CANNOT_RECOVER);
			}
		}
		send_to_scsi = 1;
	}
				
	if (SESS_OPS_C(conn)->DataPDUInOrder)
		goto out;

	pdu = cmd->pdu_ptr;
	pdu->data_sn = hdr->data_sn;
	
	switch (pdu->status) {
	case ISCSI_PDU_NOT_RECEIVED:
		pdu->status = ISCSI_PDU_RECEIVED_OK;
		break;
	case ISCSI_PDU_CRC_FAILED:
		pdu->status = ISCSI_PDU_RECEIVED_OK;
		break;
	case ISCSI_PDU_TIMED_OUT:
		pdu->status = ISCSI_PDU_RECEIVED_OK;
		break;
	default:
		return(DATAIN_CANNOT_RECOVER);
	}

	if (hdr->flags & F_BIT) {
		if (iscsi_datain_datapduinorder_no_fbit(cmd, pdu) < 0)
			return(DATAIN_CANNOT_RECOVER);
	}

out:
	if (SESS_OPS_C(conn)->DataSequenceInOrder) {
		cmd->next_burst_len += hdr->length;

		if (hdr->flags & F_BIT) {
			cmd->next_burst_len = 0;
			iscsi_set_datain_sequence_values(cmd);
		}
	} else {
		iscsi_seq_t *seq = cmd->seq_ptr;

		seq->offset += hdr->length;
		seq->next_burst_len += hdr->length;

		if (hdr->flags & F_BIT)
			seq->next_burst_len = 0;

		/*
		 * We are covered here, but be extra extra paranoid in
		 * the DataSequenceInOrder=No case anyways.
		 */
		if ((hdr->flags & S_BIT) && !SESS_OPS_C(conn)->DataSequenceInOrder) {
			if (iscsi_datain_datasequenceinorder_no_sbit(cmd) < 0)
				return(DATAIN_CANNOT_RECOVER);
		}
	}

	cmd->data_sn++;
	cmd->read_data_done += hdr->length;

	return((send_to_scsi) ? DATAIN_SEND_TO_SCSI : DATAIN_NORMAL);
}

/*	iscsi_datain_post_crc_failed():
 *
 *	Called from:	iscsi_check_post_datain()
 */
static inline int iscsi_datain_post_crc_failed (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_pdu_t *pdu;

	if (SESS_OPS_C(conn)->DataPDUInOrder)
		goto out;
	
	/*
	 * The rest of this function is only called when DataPDUInOrder=No.
	 */
	pdu = cmd->pdu_ptr;
	
	switch (pdu->status) {
	case ISCSI_PDU_NOT_RECEIVED:
		pdu->status = ISCSI_PDU_CRC_FAILED;
		break;
	case ISCSI_PDU_CRC_FAILED:
		break;
	case ISCSI_PDU_TIMED_OUT:
		pdu->status = ISCSI_PDU_CRC_FAILED;
		break;
	default:
		return(DATAIN_CANNOT_RECOVER);
	}	
	
out:
	return(DATAIN_CANNOT_RECOVER);
}

/*	iscsi_check_pre_datain():
 *
 *	Called from iscsi_handle_data_in() before DataIN payload is received
 *	and CRC computed.
 */
extern int iscsi_check_pre_datain (
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	int ret;
	iscsi_conn_t *conn = CONN(cmd);

	ret = iscsi_datain_check_datasn(cmd, buf);
	if ((ret == DATAIN_WITHIN_COMMAND_RECOVERY) ||
	    (ret == DATAIN_CANNOT_RECOVER))
		return(ret);

	if (iscsi_datain_check_sequence(cmd, buf) < 0)
		return(DATAIN_CANNOT_RECOVER);
	
	return((SESS_OPS_C(conn)->DataPDUInOrder) ?
		iscsi_datain_pre_datapduinorder_yes(cmd, buf) :
		iscsi_datain_pre_datapduinorder_no(cmd, buf));
}

/*	iscsi_check_post_datain():
 *
 *	Called from iscsi_handle_data_in() after DataIN payload is received
 *	and CRC computed.
 */
extern int iscsi_check_post_datain (
	iscsi_cmd_t *cmd,
	unsigned char *buf,
	u8 data_crc_failed)
{
	iscsi_conn_t *conn = CONN(cmd);
	
	if (!data_crc_failed)
		return(iscsi_datain_post_crc_passed(cmd, buf));
	else {
		if (!SESS_OPS_C(conn)->ErrorRecoveryLevel) {
			TRACE_ERROR("Unable to recover from DataIN CRC failure"
				" while ERL=0, closing session.\n");
			return(-1);
		}
		return(iscsi_datain_post_crc_failed(cmd, buf));
	}
}

/*	iscsi_retry_init_logout_cmnd():
 *
 *
 */
static int iscsi_retry_init_logout_cmnd (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	struct iovec *iov = &cmd->iov_misc[0];
	struct iscsi_init_logout_cmnd *hdr = (struct iscsi_init_logout_cmnd *) cmd->pdu;
	
	hdr = hdr;
	iov[0].iov_base = cmd->pdu;
	iov[0].iov_len  = ISCSI_HDR_LEN;
	cmd->tx_size    = ISCSI_HDR_LEN;

	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		iov[0].iov_len	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
		cmd->tx_size	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
	}
	cmd->iov_misc_count = 1;

	TRACE(TRACE_ERL1, "Retrying Logout Request: ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpStatSN: 0x%08x, Reason: 0x%02x, CID: %hu, On CID: %hu\n",
		cmd->init_task_tag, cmd->cmdsn, ntohl(hdr->exp_stat_sn),
			hdr->flags & 0x7f, ntohs(hdr->cid), conn->cid);
	
	return(0);
}

/*	iscsi_retry_init_nop_out():
 *
 *
 */
static int iscsi_retry_init_nop_out (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	unsigned char *buf_ptr;
	u32 padding = 0, ping_data_len;
	struct iovec *iov = &cmd->iov_misc[0];
	struct iscsi_init_nop_out *hdr = (struct iscsi_init_nop_out *) cmd->pdu;
			
	hdr = hdr;
	iov[0].iov_base	= cmd->pdu;
	iov[0].iov_len	= ISCSI_HDR_LEN;
	cmd->tx_size	= ISCSI_HDR_LEN;

	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		iov[0].iov_len	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
		cmd->tx_size	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
	}
	cmd->iov_misc_count = 1;
	ping_data_len	= cmd->buf_ptr_size;

	if (ping_data_len) {
		buf_ptr = (unsigned char *) cmd->buf_ptr;
		if ((padding = ((-ping_data_len) & 3)) != 0)
			memset(buf_ptr + ping_data_len, 0, padding);

		iov[1].iov_base	= cmd->buf_ptr;
		iov[1].iov_len	= (ping_data_len + padding);
		cmd->tx_size	+= (ping_data_len + padding);
		cmd->iov_misc_count++;

		if (CONN_OPS(conn)->DataDigest) {
			iov[2].iov_base	= &cmd->data_checksum;
			iov[2].iov_len	= CRC_LEN;
			cmd->tx_size	+= CRC_LEN;
			cmd->iov_misc_count++;
		}
	}

	TRACE(TRACE_ERL1, "Retrying NOPOUT Ping ITT: 0x%08x, ExpStatSN:"
		" 0x%08x, Length: %u, CID: %hu\n", cmd->init_task_tag,
		ntohl(hdr->exp_stat_sn), ping_data_len, conn->cid);
	
	return(0);
}

/*	iscsi_retry_init_scsi_cmnd():
 *
 *
 */
static int iscsi_retry_init_scsi_cmnd (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	iscsi_unmap_sg_t *unmap_sg)
{
	int aidl = 0, iov_count = 0;
	int immediate_dlen = 0;
	u32 padding = 0;
	iscsi_r2t_t *r2t = NULL;
	struct iscsi_init_scsi_cmnd *hdr = (struct iscsi_init_scsi_cmnd *) cmd->pdu;
	struct scsi_cmnd *sc = cmd->scsi_cmnd;
	
	cmd->iov_data[0].iov_base = cmd->pdu;
	cmd->iov_data[0].iov_len = ISCSI_HDR_LEN;
	cmd->tx_size	= ISCSI_HDR_LEN;
	
	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		cmd->iov_data[0].iov_len += (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
		cmd->tx_size += (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
	}
	cmd->iov_data_count = 1;

	if (sc->sc_data_direction == DMA_TO_DEVICE) {
		/*
		 * Check for Immediate WRITE Data.
		 */
		if (SESS_OPS_C(conn)->ImmediateData) {
			struct iovec *iov = &cmd->iov_data[1];
			immediate_dlen = (CONN_OPS(conn)->MaxRecvDataSegmentLength < 
					  SESS_OPS_C(conn)->FirstBurstLength) ?
					  CONN_OPS(conn)->MaxRecvDataSegmentLength : 
					  SESS_OPS_C(conn)->FirstBurstLength;

			aidl = (sc->request_bufflen <  immediate_dlen) ? 
				sc->request_bufflen : immediate_dlen;

			if (sc->use_sg) {
				int iov_ret;
				iscsi_map_sg_t map_sg;
				
				memset((void *)&map_sg, 0, sizeof(iscsi_map_sg_t));
				map_sg.cmd = cmd;
				map_sg.map_flags |= MAP_SG_KMAP;
				map_sg.iov = &cmd->iov_data[1];
				map_sg.data_length = aidl;
				map_sg.data_offset = 0;

				if ((iov_ret = iscsi_map_scatterlists(
						(void *)&map_sg,
						(void *)unmap_sg)) < 0)
					return(-1);
				
				iov_count += iov_ret;
			} else {
				iov[iov_count].iov_len	= aidl;
				iov[iov_count++].iov_base = sc->request_buffer;
			}

			cmd->tx_size += aidl;

			if ((padding = ((-aidl) & 3)) != 0) {
				iov[iov_count].iov_base = cmd->buf_ptr;
				iov[iov_count++].iov_len = padding;
				cmd->tx_size += padding;
			}
			if (CONN_OPS(conn)->DataDigest) {
#ifndef CRYPTO_API_CRC32C
				int rc = 1;
#endif
				u32 counter = aidl;
				struct iovec *iov_ptr = &cmd->iov_data[1];
#ifdef CRYPTO_API_CRC32C
				crypto_digest_init(conn->conn_tx_tfm);
#endif				
				while (counter > 0) {
#ifdef CRYPTO_API_CRC32C
					ISCSI_CRC32C_TX_DATA_TX_PATH(iov_ptr->iov_base, iov_ptr->iov_len, conn);
#else
					crc32c(iov_ptr->iov_base, iov_ptr->iov_len,
						rc, &cmd->data_checksum);
					rc = 0;
#endif
					counter -= iov_ptr->iov_len;
					iov_ptr++;
				}
				
				if (padding) {
#ifdef CRYPTO_API_CRC32C
					ISCSI_CRC32C_TX_DATA_TX_PATH(cmd->buf_ptr, padding, conn);
#else
					crc32c((u8 *)cmd->buf_ptr, padding, rc, &cmd->data_checksum);
#endif
				}
#ifdef CRYPTO_API_CRC32C				
				ISCSI_CRC32C_TX_DATA_TX_PATH_FINAL(cmd->data_checksum, conn);
#endif			
				iov[iov_count].iov_base = &cmd->data_checksum;
				iov[iov_count++].iov_len = CRC_LEN;
			 	cmd->tx_size	+= CRC_LEN;
			}
			cmd->iov_data_count += iov_count;
		}

		/*
		 * We must retry the Unsolocited DataOUT as well.
		 */
		if (!(hdr->flags & F_BIT)) {
			u32 first_burst_left, total_burst_length;
				
			if (!(r2t = iscsi_allocate_r2t()))
				goto failure;

			total_burst_length = (SESS_OPS_C(conn)->FirstBurstLength >
					      cmd->data_length) ?
					      cmd->data_length :
					      SESS_OPS_C(conn)->FirstBurstLength;
			first_burst_left = (total_burst_length - hdr->length);
				
			r2t->offset = hdr->length;
			r2t->r2t_sn = r2t->targ_xfer_tag = 0xFFFFFFFF;
			r2t->xfer_length = first_burst_left;

			cmd->unsolicited_data_out = 1;
		}
	} else {
		/*
		 * Nothing extra to do for reads here.
		 */
		;
	}

	if (cmd->unsolicited_data_out) {
		if (!SESS_OPS_C(conn)->DataSequenceInOrder) {
			iscsi_seq_t *seq;
			if (!(seq = iscsi_get_seq_holder(cmd, r2t->offset,
					r2t->xfer_length)))
				goto failure;
			 r2t->seq = seq;
		}
		if (!SESS_OPS_C(conn)->DataPDUInOrder)
			if (iscsi_set_pdu_values_for_r2t(cmd, r2t) < 0)
				goto failure;

		iscsi_add_r2t_to_cmd(cmd, r2t);
	}

        TRACE(TRACE_ERL1, "Retrying SCSI Cmnd: ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpXferLen: %u, Length: %u, CID: %hu\n", cmd->init_task_tag,
			cmd->cmdsn, ntohl(hdr->exp_xfer_len),
			ntohl(hdr->length), conn->cid);
	
	return(0);

failure:
	if (aidl)
		iscsi_unmap_scatterlists((void *)&unmap_sg);
	if (r2t)
		kfree(r2t);
	return(-1);
}

/*	iscsi_retry_init_task_mgmt_cmnd():
 *
 *
 */
static int iscsi_retry_init_task_mgmt_cmnd (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	struct iovec *iov = &cmd->iov_misc[0];
	struct iscsi_init_task_mgt_cmnd *hdr = (struct iscsi_init_task_mgt_cmnd *) cmd->pdu;
	
	hdr = hdr;
	iov[0].iov_base = cmd->pdu;
	iov[0].iov_len	= ISCSI_HDR_LEN;
	cmd->tx_size	= ISCSI_HDR_LEN;

	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		iov[0].iov_len	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
		cmd->tx_size	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
	}
	cmd->iov_misc_count = 1;

	TRACE(TRACE_ERL1, "Retrying Task Management Request: ITT: 0x%08x,"
		" CmdSN: 0x%08x, ExpStatSN: 0x%08x, Function: 0x%02x on CID: %hu\n",
		cmd->init_task_tag, cmd->cmdsn, ntohl(hdr->exp_stat_sn),
			hdr->function, conn->cid);

	return(0);
}

/*	iscsi_retry_init_text_cmnd():
 *
 *
 */
static int iscsi_retry_init_text_cmnd (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	unsigned char *buf_ptr;
	u32 padding = 0, text_len;
	struct iovec *iov = &cmd->iov_misc[0];
	struct iscsi_init_text_cmnd *hdr = (struct iscsi_init_text_cmnd *) cmd->pdu;
	
	hdr = hdr;
        iov[0].iov_base	= cmd->pdu;
	iov[0].iov_len	= ISCSI_HDR_LEN;
	cmd->tx_size	= ISCSI_HDR_LEN;
	
	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		iov[0].iov_len	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
		cmd->tx_size	+= (CONN_OPS(conn)->HeaderDigest) ? CRC_LEN : 0;
	}
	cmd->iov_misc_count = 1;
	text_len	= cmd->buf_ptr_size;
	
	if (text_len) {
		buf_ptr = cmd->buf_ptr;
		if ((padding = ((-text_len) & 3)) != 0)
			memset(buf_ptr + text_len, 0, padding);

		iov[1].iov_base	= cmd->buf_ptr;
		iov[1].iov_len	= (text_len + padding);
		cmd->tx_size	+= (text_len + padding);
		cmd->iov_misc_count++;

		if (CONN_OPS(conn)->DataDigest) {
			iov[2].iov_base	= &cmd->data_checksum;
			iov[2].iov_len	= CRC_LEN;
			cmd->tx_size	+= CRC_LEN;
			cmd->iov_misc_count++;
		}
	}

	TRACE(TRACE_ERL1, "Retrying Text Request: ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpStatSN: 0x%08x, Length: %u\n", cmd->init_task_tag,
		cmd->cmdsn, ntohl(hdr->exp_stat_sn), text_len);

	return(0);
}

/*	iscsi_build_retry_command():
 *
 *
 */
extern int iscsi_build_retry_command (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	iscsi_unmap_sg_t *unmap_sg)
{
	switch (cmd->pdu[0]) {
	case ISCSI_INIT_LOGOUT_CMND:
		return(iscsi_retry_init_logout_cmnd(cmd, conn));
	case ISCSI_INIT_NOP_OUT:
		return(iscsi_retry_init_nop_out(cmd, conn));
	case ISCSI_INIT_SCSI_CMND:
		return(iscsi_retry_init_scsi_cmnd(cmd, conn, unmap_sg));
	case ISCSI_INIT_TASK_MGMT_CMND:
		return(iscsi_retry_init_task_mgmt_cmnd(cmd, conn));
	case ISCSI_INIT_TEXT_CMND:
		return(iscsi_retry_init_text_cmnd(cmd, conn));
	default:
		TRACE_ERROR("Cannot retry unknown iSCSI Opcode 0x%02x\n",
				cmd->pdu[0]);
		return(-1);
	}

	return(0);
}

/*	iscsi_cause_connection_reinstatement():
 *
 *
 */
extern void iscsi_cause_connection_reinstatement (iscsi_conn_t *conn, int sleep)
{
	spin_lock_bh(&conn->state_lock);
	if (atomic_read(&conn->connection_exit) ||
	    atomic_read(&conn->transport_failed) ||
	    atomic_read(&conn->connection_reinstatement)) {
		spin_unlock_bh(&conn->state_lock);
		return;
	}
	
	if (iscsi_thread_set_force_reinstatement(conn) < 0) {
		spin_unlock_bh(&conn->state_lock);
		return;
	}

	atomic_set(&conn->connection_reinstatement, 1);
	if (!sleep || atomic_read(&conn->sleep_on_conn_wait_sem)) {
		spin_unlock_bh(&conn->state_lock);
		return;
	}

	atomic_set(&conn->sleep_on_conn_wait_sem, 1);
	spin_unlock_bh(&conn->state_lock);

	down_interruptible(&conn->conn_wait_sem);

	return;
}

/*	iscsi_stop_connection_and_reinstatement():
 *
 *
 */
extern void iscsi_stop_connection_and_reinstatement (iscsi_conn_t *conn)
{
	iscsi_cause_connection_reinstatement(conn, 1);
	return;
}

/*	__iscsi_cause_session_reinstatement():
 *
 *
 */
extern void __iscsi_cause_session_reinstatement (iscsi_session_t *sess)
{
	if (atomic_read(&sess->session_logout))
		return;

	atomic_set(&sess->session_reinstatement, 1);
	iscsi_stop_session(sess, 0, 0);

	return;
}

/*	iscsi_cause_session_reinstatement():
 *
 *
 */
extern void iscsi_cause_session_reinstatement (iscsi_session_t *sess)
{
	if (atomic_read(&sess->session_logout))
		return;

	atomic_set(&sess->session_reinstatement, 1);
	iscsi_stop_session(sess, 0, 0);

	return;
}

/*	iscsi_stop_session_and_reinstatement():
 *
 *
 */
extern void iscsi_stop_session_and_reinstatement (iscsi_session_t *sess, int dec_session_usage)
{
	atomic_set(&sess->session_logout, 1);
	iscsi_set_state_for_connections(sess, INIT_CONN_STATE_IN_LOGOUT);

	iscsi_stop_session(sess, 1, 1);
	if (dec_session_usage)
		iscsi_dec_session_usage_count(sess);
	iscsi_close_session(sess);

	return;
}

/*	iscsi_start_extra_conns_after_reinstatement():
 *
 *
 */
static void iscsi_start_extra_conns_after_reinstatement (iscsi_channel_t *channel)
{
	u16 connection_count = 0;
	int ret;
	iscsi_channel_conn_t *chan_conn;
	iscsi_login_holder_t lh;
	iscsi_session_t *sess = channel->sess;
	
	spin_lock(&channel->chan_conn_lock);
	while (atomic_read(&channel->connection_count)) {
		if (!(chan_conn = iscsi_get_conn_from_channel(channel)))
			break;

		spin_unlock(&channel->chan_conn_lock);

		memset((void *)&lh, 0, sizeof(iscsi_login_holder_t));		
		lh.channel = channel;
		lh.ipv4_address = chan_conn->ipv4_address;
		lh.port = chan_conn->port;
		lh.network_transport = chan_conn->network_transport;
		lh.cc = chan_conn;
		strncpy(lh.net_dev, chan_conn->net_dev, ISCSI_NETDEV_NAME_SIZE);
						
		ret = iscsi_create_connection(sess, &lh, NULL);
		spin_lock(&channel->chan_conn_lock);

		if (!ret) {
			kfree(chan_conn);
			connection_count++;
			if (!atomic_read(&channel->connection_count))
				atomic_set(&channel->adding_connections, 0);
		} else
			iscsi_add_chan_conn_to_channel(channel, chan_conn);
		
		/*
		 * If the newly established session failed while still adding
		 * additional connections we exit immediately.
		 */
		if (signal_pending(current) || atomic_read(&channel->reinstatement_thread_stop)) {
			atomic_set(&channel->adding_connections, 0);
			spin_unlock(&channel->chan_conn_lock);
			up(&channel->reinstatement_thread_stop_sem);
			return;
		}
	}
	channel->reinstatement_thread = NULL;
	spin_unlock(&channel->chan_conn_lock);

	TRACE(TRACE_ISCSI, "Started %hu additional connection(s) after session"
			" reinstatement\n", connection_count);
		
	return;
}

/*	iscsi_session_reinstatement():
 *
 *
 */
extern int iscsi_session_reinstatement (
	iscsi_channel_t *channel)
{
	int session_reestablished = 0;
	iscsi_channel_conn_t *chan_conn;
	iscsi_login_holder_t lh;
	
	if (atomic_read(&channel->force_channel_offline) ||
	    atomic_read(&channel->stop_channel))
		return(0);
		
	TRACE(TRACE_ERL0, "Reestablishing failed iSCSI session for SCSI"
		" Channel %hu\n", channel->channel_id);

	/*
	 * Start a new session with a single FFP connection (we start the
	 * additional connections below) from one of the connection's IP/Port
	 * addresses from the failed session.
	 */
	spin_lock(&channel->chan_conn_lock);
	atomic_set(&channel->adding_connections, 1);
	while (!session_reestablished) {
		if (!(chan_conn = iscsi_get_conn_from_channel(channel)))
			break;
		
		spin_unlock(&channel->chan_conn_lock);

		memset((void *)&lh, 0, sizeof(iscsi_login_holder_t));
		lh.scsi_host_id = channel->host_id;
		lh.scsi_target_id = channel->target_id;
		lh.scsi_channel_id = channel->channel_id;
		lh.channel = channel;
		lh.session_reinstatement = 1;
		lh.ipv4_address = chan_conn->ipv4_address;
		lh.port = chan_conn->port;
		lh.network_transport = chan_conn->network_transport;
		lh.cc = chan_conn;
		strncpy(lh.net_dev, chan_conn->net_dev, ISCSI_NETDEV_NAME_SIZE);

		spin_lock_bh(&channel->channel_state_lock);
		if (channel->ch_target) {
			strncpy(lh.lh_targetname, channel->ch_target->targetname,
				ISCSI_MAX_TARGETNAME_SIZE);
			lh.lh_flags |= LH_TARGETNAME_PRESENT;
		}
		spin_unlock_bh(&channel->channel_state_lock);
		
		if (iscsi_create_session(&lh)) {
			session_reestablished = 1;
			spin_lock(&channel->chan_conn_lock);
			if (!atomic_read(&channel->connection_count)) {
				atomic_set(&channel->adding_connections, 0);
				channel->reinstatement_thread = NULL;
			}
			spin_unlock(&channel->chan_conn_lock);
		}
		spin_lock(&channel->chan_conn_lock);

		if (session_reestablished)
			kfree(chan_conn);
		else
			iscsi_add_chan_conn_to_channel(channel, chan_conn);

		if (signal_pending(current) || atomic_read(&channel->reinstatement_thread_stop))
			break;
	}
	spin_unlock(&channel->chan_conn_lock);

	if (signal_pending(current) || atomic_read(&channel->reinstatement_thread_stop)) {
		atomic_set(&channel->adding_connections, 0);
		up(&channel->reinstatement_thread_stop_sem);
		return(session_reestablished);
	}														
	
	/*
	 * If session reinstatement was successful start any additional
	 * connections from the previous failed session.
	 */
	if (session_reestablished) {
		if (atomic_read(&channel->connection_count))
			iscsi_start_extra_conns_after_reinstatement(channel);
		return(session_reestablished);
	}

	return(session_reestablished);	
}

/*      iscsi_handle_connection_cleanup():
 *
 *
 */
static int iscsi_handle_connection_cleanup (iscsi_conn_t *conn)
{
	int session_reinstatement = 0;
	iscsi_channel_t *c;
	iscsi_session_t *sess = SESS(conn);
	
	c = sess->channel;
		
	/*
	 * Perform Session Reinstatement by closing all
	 * existing connections and starting a new session.
	 */
	spin_lock(&sess->reinstatement_lock);
	printk("iCHANNEL[%d] - Performing Cleanup for failed iSCSI"
		" CID: %hu to %s\n", c->channel_id, conn->cid,
		SESS_OPS(sess)->TargetName);

	if (SESS_OPS(sess)->SessionType) {
		printk("iCHANNEL[%d] - Unable to reestablish failed"
			" iSCSI discovery session.\n", c->channel_id);
		spin_unlock(&sess->reinstatement_lock);
		up(&sess->discovery_sem);
		iscsi_close_connection(conn);
		iscsi_close_session(sess);
		return(0);
	}
		
	if (!sess->reinstatement_set) {
		atomic_set(&sess->session_reinstatement, 1);
		sess->reinstatement_set = 1;
		session_reinstatement = 1;
	}
	spin_unlock(&sess->reinstatement_lock);

	iscsi_close_connection(conn);

	if (!session_reinstatement || iscsi_global->in_shutdown)
		return(0);

	iscsi_stop_session(sess, 1, 1);
	iscsi_close_session(sess);
	
	iscsi_session_reinstatement(c);

	return(0);
}

/*	iscsi_take_action_for_connection_exit():
 *
 *
 */
extern void iscsi_take_action_for_connection_exit (iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->state_lock);	
	if (atomic_read(&conn->connection_exit)) {
		spin_unlock_bh(&conn->state_lock);
		return;
	}
	atomic_set(&conn->connection_exit, 1);

	if (conn->conn_state == INIT_CONN_STATE_IN_LOGOUT) {
		spin_unlock_bh(&conn->state_lock);
		iscsi_close_connection(conn);
		return;
	}

	if (conn->conn_state == INIT_CONN_STATE_CLEANUP_WAIT) {
		spin_unlock_bh(&conn->state_lock);
		return;
	}
	
	TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_CLEANUP_WAIT.\n");
	conn->conn_state = INIT_CONN_STATE_CLEANUP_WAIT;
	spin_unlock_bh(&conn->state_lock);

	iscsi_handle_connection_cleanup(conn);

	return;
}

