/*
 * Copyright (c) 2003-2004 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *  
 * This file houses the iSCSI Initiator specific iSCSI Channel Attribute functions.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_CHANATTRIB_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/blkdev.h>
#include <scsi/scsi.h>
#include <scsi/scsi_cmnd.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_ioctl_defs.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_lu.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_CHANATTRIB_C

extern iscsi_global_t *iscsi_global;

/*	iscsi_set_ca_closeconn_reinstatement():
 *
 *
 */
static int iscsi_set_ca_closeconn_reinstatement (
	iscsi_channel_t *c,
	u32 closeconn_reinstatement)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if ((closeconn_reinstatement != 0) && (closeconn_reinstatement != 1)) {
		TRACE_ERROR("Requested CloseConnection Reinstatement: %u not"
			" 0 or 1\n", closeconn_reinstatement);
		return(-1);
	}

	a->closeconn_reinstatement = closeconn_reinstatement;
	TRACE(TRACE_CHANNEL, "Set CloseConnection Reinstatement to %u for"
		" Channel %d\n", a->closeconn_reinstatement,
			c->channel_id);

	return(0);
}

/*	iscsi_set_ca_cmdsn_timeout():
 *
 *
 */
static int iscsi_set_ca_cmdsn_timeout (
	iscsi_channel_t *c,
	u32 cmdsn_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (cmdsn_timeout > CA_CMDSN_TIMEOUT_MAX) {
		TRACE_ERROR("Requested CmdSN Timeout %u larger than maximum"
			" %u\n", cmdsn_timeout, CA_CMDSN_TIMEOUT_MAX);
		return(-1);
	} else if (cmdsn_timeout < CA_CMDSN_TIMEOUT_MIN) {
		TRACE_ERROR("Requested CmdSN Timeout %u smaller than minimum"
			" %u\n", cmdsn_timeout, CA_CMDSN_TIMEOUT_MIN);
		return(-1);
	}

	a->cmdsn_timeout = cmdsn_timeout;
	TRACE(TRACE_CHANNEL, "Set CmdSN Timeout to %u for Channel %d\n",
		a->cmdsn_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_cmdsn_timeout_retries():
 *
 *
 */
static int iscsi_set_ca_cmdsn_timeout_retries (
	iscsi_channel_t *c,
	u32 cmdsn_timeout_retries)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (cmdsn_timeout_retries > CA_CMDSN_TIMEOUT_RETRIES_MAX) {
		TRACE_ERROR("Requested CmdSN Timeout Retries %u larger than"
			" maximum %u\n", cmdsn_timeout_retries,
				CA_CMDSN_TIMEOUT_RETRIES_MAX);
		return(-1);
	} else if (cmdsn_timeout_retries < CA_CMDSN_TIMEOUT_RETRIES_MIN) {
		TRACE_ERROR("Requested CmdSN Timeout Retries %u smaller than"
			" minimum %u\n", cmdsn_timeout_retries,
				CA_CMDSN_TIMEOUT_RETRIES_MIN);
		return(-1);
	}

	a->cmdsn_timeout_retries = cmdsn_timeout_retries;
	TRACE(TRACE_CHANNEL, "Set CmdSN Timeout Retries to %u for Channel %d\n",
		a->cmdsn_timeout_retries, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_datain_timeout():
 *
 *
 */
static int iscsi_set_ca_datain_timeout (
	iscsi_channel_t *c,
	u32 datain_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (datain_timeout > CA_DATAIN_TIMEOUT_MAX) {
		TRACE_ERROR("Requested Datain Timeout %u larger than maximum"
			" %u\n", datain_timeout, CA_DATAIN_TIMEOUT_MAX);
		return(-1);
	} else if (datain_timeout < CA_DATAIN_TIMEOUT_MIN) {
		TRACE_ERROR("Requested Datain Timeout %u smaller than minimum"
			" %u\n", datain_timeout, CA_DATAIN_TIMEOUT_MIN);
		return(-1);
	}

	a->datain_timeout = datain_timeout;
	TRACE(TRACE_CHANNEL, "Set DataIn Timeout to %u for Channel %d\n",
		a->datain_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_datain_timeout_retries():
 *
 *
 */
static int iscsi_set_ca_datain_timeout_retries (
	iscsi_channel_t *c,
	u32 datain_timeout_retries)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (datain_timeout_retries > CA_DATAIN_TIMEOUT_RETRIES_MAX) {
		TRACE_ERROR("Requested Datain Timeout Retries %u larger than"
			" maximum %u\n", datain_timeout_retries,
				CA_DATAIN_TIMEOUT_RETRIES_MAX);
		return(-1);
	} else if (datain_timeout_retries < CA_DATAIN_TIMEOUT_RETRIES_MIN) {
		TRACE_ERROR("Requested Datain Timeout Retries %u smaller than"
			" minimum %u\n", datain_timeout_retries,
				CA_DATAIN_TIMEOUT_RETRIES_MIN);
		return(-1);
	}

	a->datain_timeout_retries = datain_timeout_retries;
	TRACE(TRACE_CHANNEL, "Set DataIn Timeout Retries to %u for Channel"
		" %d\n", a->datain_timeout_retries, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_immediate_logout():
 *
 *
 */
static int iscsi_set_ca_immediate_logout (
	iscsi_channel_t *c,
	u32 immediate_logout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (immediate_logout != 0 && immediate_logout != 1) {
		TRACE_ERROR("Requested Immediate Logout Requests: %u not"
			" 0 or 1\n", immediate_logout);
		return(-1);
	}

	a->immediate_logout = immediate_logout;
	TRACE(TRACE_CHANNEL, "Set Immediate Logout Requests to %u for Channel"
		" %d\n", a->immediate_logout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_immediate_text():
 *
 *
 */
static int iscsi_set_ca_immediate_text (
	iscsi_channel_t *c,
	u32 immediate_text)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (immediate_text != 0 && immediate_text != 1) {
		TRACE_ERROR("Requested Immediate Text Requests: %u not"
			" 0 or 1\n", immediate_text);
		return(-1);
	}

	a->immediate_text = immediate_text;
	TRACE(TRACE_CHANNEL, "Set Immediate Text Requests to %u for Channel"
		" %d\n", a->immediate_text, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_login_retries():
 *
 *
 */
static int iscsi_set_ca_login_retries (
	iscsi_channel_t *c,
	u32 login_retries)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	a->login_retries = login_retries;
	TRACE(TRACE_CHANNEL, "Set Login Retries to %u for Channel %d\n",
		a->login_retries, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_login_retry_wait():
 *
 *
 */
static int  iscsi_set_ca_login_retry_wait (
	iscsi_channel_t *c,
	u32 login_retry_wait)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (login_retry_wait > CA_LOGIN_RETRY_WAIT_MAX) {
		TRACE_ERROR("Requested Login Retry Wait %u larger than maximum"
			" %u\n", login_retry_wait, CA_LOGIN_RETRY_WAIT_MAX);
		return(-1);
	} else if (login_retry_wait < CA_LOGIN_RETRY_WAIT_MIN) {
		TRACE_ERROR("Requested Login Retry Wait %u smaller than minimum"
			" %u\n", login_retry_wait, CA_LOGIN_RETRY_WAIT_MIN);
		return(-1);
	}

	a->login_retry_wait = login_retry_wait;
	TRACE(TRACE_CHANNEL, "Set Login Retry Wait to %u for Channel %d\n",
		a->login_retry_wait, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_login_timeout():
 *
 *
 */
static int iscsi_set_ca_login_timeout (
	iscsi_channel_t *c,
	u32 login_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (login_timeout > CA_LOGIN_TIMEOUT_MAX) {
		TRACE_ERROR("Requested Login Timeout %u larger than maximum"
			" %u\n", login_timeout, CA_LOGIN_TIMEOUT_MAX);
		return(-1);
	} else if (login_timeout < CA_LOGIN_TIMEOUT_MIN) {
		TRACE_ERROR("Requested Login Timeout %u smaller than minimum"
			" %u\n", login_timeout, CA_LOGIN_TIMEOUT_MIN);
		return(-1);
	}

	a->login_timeout = login_timeout;
	TRACE(TRACE_CHANNEL, "Set Login Timeout to %u for Channel %d\n",
		a->login_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_logout_timeout():
 *
 *
 */
static int iscsi_set_ca_logout_timeout (
	iscsi_channel_t *c,
	u32 logout_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (logout_timeout > CA_LOGOUT_TIMEOUT_MAX) {
		TRACE_ERROR("Requested Logout Timeout %u larger than maximum"
			" %u\n", logout_timeout, CA_LOGOUT_TIMEOUT_MAX);
		return(-1);
	} else if (logout_timeout < CA_LOGOUT_TIMEOUT_MIN) {
		TRACE_ERROR("Requested Logout Timeout %u smaller than minimum"
			" %u\n", logout_timeout, CA_LOGOUT_TIMEOUT_MIN);
		return(-1);
	}

	a->logout_timeout = logout_timeout;
	TRACE(TRACE_CHANNEL, "Set Logout Timeout to %u for Channel %d\n",
		a->logout_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_lu_scan_timeout():
 *
 *
 */
static int iscsi_set_ca_lu_scan_timeout (
	iscsi_channel_t *c,
	u32 lu_scan_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (lu_scan_timeout > CA_LU_SCAN_TIMEOUT_MAX) {
		TRACE_ERROR("Requested LU Scan Timeout %u larger than maximum"
			" %u\n", lu_scan_timeout, CA_LU_SCAN_TIMEOUT_MAX);
		return(-1);
	} else if (lu_scan_timeout < CA_LU_SCAN_TIMEOUT_MIN) {
		TRACE_ERROR("Requested LU Scan Timeout %u smaller than minimum"
			" %u\n", lu_scan_timeout, CA_LU_SCAN_TIMEOUT_MIN);
		return(-1);
	}

	a->lu_scan_timeout = lu_scan_timeout;
	TRACE(TRACE_CHANNEL, "Set  LU Scan Timeout to %u for Channel %d\n",
		a->lu_scan_timeout, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_netif_timeout():
 *
 *
 */
static int iscsi_set_ca_netif_timeout (
	iscsi_channel_t *c,
	u32 netif_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (netif_timeout > CA_NETIF_TIMEOUT_MAX) {
		TRACE_ERROR("Requested Network Interface Timeout %u larger"
			" than maximum %u\n", netif_timeout,
				CA_NETIF_TIMEOUT_MAX);
		return(-1);
	} else if (netif_timeout < CA_NETIF_TIMEOUT_MIN) {
		TRACE_ERROR("Requested Network Interface Timeout %u smaller"
			" than minimum %u\n", netif_timeout,
				CA_NETIF_TIMEOUT_MIN);
		return(-1);
	}

	a->netif_timeout = netif_timeout;
	TRACE(TRACE_CHANNEL, "Set Network Interface Timeout to %u for Channel"
		" %d\n", a->netif_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_nopout_timeout():
 *
 *
 */
static int iscsi_set_ca_nopout_timeout (
	iscsi_channel_t *c,
	u32 nopout_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (nopout_timeout > CA_NOPOUT_TIMEOUT_MAX) {
		TRACE_ERROR("Requested NOP Timeout %u larger than maximum %u\n",
			nopout_timeout, CA_NOPOUT_TIMEOUT_MAX);
		return(-1);
	} else if (nopout_timeout < CA_NOPOUT_TIMEOUT_MIN) {
		TRACE_ERROR("Requested NOP Timeout %u smaller than than minimum"
			" %u\n", nopout_timeout, CA_NOPOUT_TIMEOUT_MIN);
		return(-1);
	}

	a->nopout_timeout = nopout_timeout;
	TRACE(TRACE_CHANNEL, "Set NOPOUT Timeout to %u for Channel %d\n",
			a->nopout_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_nopout_response_timeout():
 *
 *
 */
static int iscsi_set_ca_nopout_response_timeout (
	iscsi_channel_t *c,
	u32 nopout_response_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (nopout_response_timeout > CA_NOPOUT_RESPONSE_TIMEOUT_MAX) {
		TRACE_ERROR("Requested NOP Response Timeout %u larger than"
			" maximum %u\n", nopout_response_timeout,
				CA_NOPOUT_RESPONSE_TIMEOUT_MAX);
		return(-1);
	} else if (nopout_response_timeout < CA_NOPOUT_RESPONSE_TIMEOUT_MIN) {
		TRACE_ERROR("Requested NOP Response Timeout %u smaller than"
			" minimum %u\n", nopout_response_timeout,
				CA_NOPOUT_RESPONSE_TIMEOUT_MIN);
		return(-1);
	}

	a->nopout_response_timeout = nopout_response_timeout;
	TRACE(TRACE_CHANNEL, "Set NOPOUT Response Timeout to %u for Channel"
		" %d\n", a->nopout_response_timeout, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_random_dataout_pdu_offsets():
 *
 *
 */
static int iscsi_set_ca_random_dataout_pdu_offsets (
	iscsi_channel_t *c,
	u32 random_dataout_pdu_offsets)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (random_dataout_pdu_offsets != 0 && random_dataout_pdu_offsets != 1) {
		TRACE_ERROR("Requested Random DataOUT PDU Offsets: %u not"
			" 0 or 1\n", random_dataout_pdu_offsets);
		return(-1);
	}

	a->random_dataout_pdu_offsets = random_dataout_pdu_offsets;
	TRACE(TRACE_CHANNEL, "Set Random DataOUT PDU Offsets to %u for"
		" Channel %d\n", a->random_dataout_pdu_offsets, c->channel_id);
		
	return(0);
}

/*	iscsi_set_ca_scsi_task_online_timeout():
 *
 *
 */
static int iscsi_set_ca_scsi_task_online_timeout (
	iscsi_channel_t *c,
	u32 scsi_task_online_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (scsi_task_online_timeout > CA_SCSI_TASK_ONLINE_TIMEOUT_MAX) {
		TRACE_ERROR("Requested SCSI Task Online Timeout %u larger than"
			" maximum %u\n", scsi_task_online_timeout,
				CA_SCSI_TASK_ONLINE_TIMEOUT_MAX);
		return(-1);
	} else if (scsi_task_online_timeout < CA_SCSI_TASK_ONLINE_TIMEOUT_MIN) {
		TRACE_ERROR("Requested SCSI Task Online Timeout %u smaller than"
			" minimum %u\n", scsi_task_online_timeout,
			CA_SCSI_TASK_ONLINE_TIMEOUT_MIN);
		return(-1);
	}

	a->scsi_task_online_timeout = scsi_task_online_timeout;
	TRACE(TRACE_CHANNEL, "Set SCSI Task Online Timeout to %u for"
		" Channel %d\n", a->scsi_task_online_timeout, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_scsi_task_offline_timeout():
 *
 *
 */
static int iscsi_set_ca_scsi_task_offline_timeout (
	iscsi_channel_t *c,
	u32 scsi_task_offline_timeout)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (scsi_task_offline_timeout > CA_SCSI_TASK_OFFLINE_TIMEOUT_MAX) {
		TRACE_ERROR("Requested SCSI Task Offline Timeout %u larger than"
			" maximum %u\n", scsi_task_offline_timeout,
				CA_SCSI_TASK_OFFLINE_TIMEOUT_MAX);
		return(-1);
	} else if (scsi_task_offline_timeout < CA_SCSI_TASK_OFFLINE_TIMEOUT_MIN) {
		TRACE_ERROR("Requested SCSI Task Offline Timeout %u smaller than"
			" minimum %u\n", scsi_task_offline_timeout,
			CA_SCSI_TASK_OFFLINE_TIMEOUT_MIN);
		return(-1);
	}

	a->scsi_task_offline_timeout = scsi_task_offline_timeout;
	TRACE(TRACE_CHANNEL, "Set SCSI Task Offline Timeout to %u for"
		" Channel %d\n", a->scsi_task_offline_timeout, c->channel_id);
	
	return(0);
}

/*	iscsi_set_ca_tpgfailover():
 *
 *
 */
static int iscsi_set_ca_tpgfailover (
	iscsi_channel_t *c,
	u32 tpgfailover)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;
	
	if (tpgfailover != 0 && tpgfailover != 1) {
		TRACE_ERROR("Requested TPG Failover: %u not 0 or 1\n",
				tpgfailover);
		return(-1);
	}

	a->tpgfailover = tpgfailover;
	TRACE(TRACE_CHANNEL, "Set TPG Failover to %u for Channel %d\n",
			a->tpgfailover, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_tpgfailover_attempts():
 *
 *
 */
static int iscsi_set_ca_tpgfailover_attempts (
	iscsi_channel_t *c,
	u32 tpgfailover_attempts)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	a->tpgfailover_attempts = tpgfailover_attempts;
	TRACE(TRACE_CHANNEL, "Set TPG Failover Attempts to %u for Channel %d\n",
		a->tpgfailover_attempts, c->channel_id);

	return(0);
}

/*	iscsi_set_ca_tpgfailover_login_retries():
 *
 *
 */
static int iscsi_set_ca_tpgfailover_login_retries (
	iscsi_channel_t *c,
	u32 tpgfailover_login_retries)
{
	iscsi_channel_attrib_t *a = &c->chan_attrib;

	if (tpgfailover_login_retries > CA_TPGFAILOVER_LOGIN_RETRIES_MAX) {
		TRACE_ERROR("Requested TPG Failover Login Retires %u larger"
			" than maximum %u\n", tpgfailover_login_retries,
				CA_TPGFAILOVER_LOGIN_RETRIES_MAX);
		return(-1);
	} else if (tpgfailover_login_retries < CA_TPGFAILOVER_LOGIN_RETRIES_MIN) {
		TRACE_ERROR("Requested TPG Failover Login Retires %u smaller"
			" than minimum %u\n", tpgfailover_login_retries,
				CA_TPGFAILOVER_LOGIN_RETRIES_MIN);
		return(-1);
	}

	a->tpgfailover_login_retries = tpgfailover_login_retries;
	TRACE(TRACE_CHANNEL, "Set TPG Failover Login Retries to %u for Channel"
		" %d\n", a->tpgfailover_login_retries, c->channel_id);
		
	return(0);
}

/*	iscsi_set_channel_attribute():
 *
 *
 */
extern int iscsi_set_channel_attribute (
	iscsi_channel_t *c,
	u32 ca,
	u32 ca_value)
{
	switch (ca) {
	case CA_SET_CLOSECONN_REINSTATEMENT:
		 return(iscsi_set_ca_closeconn_reinstatement(c, ca_value));
	case CA_SET_CMDSN_TIMEOUT:
		return(iscsi_set_ca_cmdsn_timeout(c, ca_value));
	case CA_SET_CMDSN_TIMEOUT_RETRIES:
		return(iscsi_set_ca_cmdsn_timeout_retries(c, ca_value));
	case CA_SET_DATAIN_TIMEOUT:
		return(iscsi_set_ca_datain_timeout(c, ca_value));
	case CA_SET_DATAIN_TIMEOUT_RETRIES:
		return(iscsi_set_ca_datain_timeout_retries(c, ca_value));
	case CA_SET_IMMEDIATE_LOGOUT:
		return(iscsi_set_ca_immediate_logout(c, ca_value));
	case CA_SET_IMMEDIATE_TEXT:
		return(iscsi_set_ca_immediate_text(c, ca_value));
	case CA_SET_LOGIN_RETRIES:
		return(iscsi_set_ca_login_retries(c, ca_value));
	case CA_SET_LOGIN_RETRY_WAIT:
		return(iscsi_set_ca_login_retry_wait(c, ca_value));
	case CA_SET_LOGIN_TIMEOUT:
		return(iscsi_set_ca_login_timeout(c, ca_value));
	case CA_SET_LOGOUT_TIMEOUT:
		return(iscsi_set_ca_logout_timeout(c, ca_value));
	case CA_SET_LU_SCAN_TIMEOUT:
		return(iscsi_set_ca_lu_scan_timeout(c, ca_value));
	case CA_SET_NETIF_TIMEOUT:
		return(iscsi_set_ca_netif_timeout(c, ca_value));
	case CA_SET_NOPOUT_TIMEOUT:
		return(iscsi_set_ca_nopout_timeout(c, ca_value));
	case CA_SET_NOPOUT_RESPONSE_TIMEOUT:
		return(iscsi_set_ca_nopout_response_timeout(c, ca_value));
	case CA_SET_RANDOM_DATAOUT_PDU_OFFSETS:
		return(iscsi_set_ca_random_dataout_pdu_offsets(c, ca_value));
	case CA_SET_SCSI_TASK_ONLINE_TIMEOUT:
		return(iscsi_set_ca_scsi_task_online_timeout(c, ca_value));
	case CA_SET_SCSI_TASK_OFFLINE_TIMEOUT:
		return(iscsi_set_ca_scsi_task_offline_timeout(c, ca_value));
	case CA_SET_TPGFAILOVER:
		return(iscsi_set_ca_tpgfailover(c, ca_value));
	case CA_SET_TPGFAILOVER_ATTEMPTS:
		return(iscsi_set_ca_tpgfailover_attempts(c, ca_value));
	case CA_SET_TPGFAILOVER_LOGIN_RETRIES:
		return(iscsi_set_ca_tpgfailover_login_retries(c, ca_value));
	default:
		TRACE_ERROR("Unknown channel attribute: %u\n", ca);
		return(-1);
	}
		
	return(0);
}

/*	iscsi_set_default_channel_attributes():
 *
 *
 */
extern void iscsi_set_default_channel_attributes (
	iscsi_channel_t *c)
{
	iscsi_channel_attrib_t *ca = &c->chan_attrib;
	
	ca->closeconn_reinstatement = CA_CLOSECONN_REINSTATEMENT;
	ca->cmdsn_timeout = CA_CMDSN_TIMEOUT;
	ca->cmdsn_timeout_retries = CA_CMDSN_TIMEOUT_RETRIES;
	ca->datain_timeout = CA_DATAIN_TIMEOUT;
	ca->datain_timeout_retries = CA_DATAIN_TIMEOUT_RETRIES;
	ca->immediate_logout = CA_IMMEDIATE_LOGOUT;
	ca->immediate_text = CA_IMMEDIATE_TEXT;
	ca->login_retries = CA_LOGIN_RETRIES;
	ca->login_retry_wait = CA_LOGIN_RETRY_WAIT;
	ca->login_timeout = CA_LOGIN_TIMEOUT;
	ca->logout_timeout = CA_LOGOUT_TIMEOUT;
	ca->lu_scan_timeout = CA_LU_SCAN_TIMEOUT;
	ca->netif_timeout = CA_NETIF_TIMEOUT;
	ca->nopout_timeout = CA_NOPOUT_TIMEOUT;
	ca->nopout_response_timeout = CA_NOPOUT_RESPONSE_TIMEOUT;
	ca->random_dataout_pdu_offsets = CA_RANDOM_DATAOUT_PDU_OFFSETS;
	ca->scsi_task_online_timeout = CA_SCSI_TASK_ONLINE_TIMEOUT;
	ca->scsi_task_offline_timeout = CA_SCSI_TASK_OFFLINE_TIMEOUT;
	ca->tpgfailover = CA_TPGFAILOVER;
	ca->tpgfailover_attempts = CA_TPGFAILOVER_ATTEMPTS;
	ca->tpgfailover_login_retries = CA_TPGFAILOVER_LOGIN_RETRIES;
		
	return;
}

