/*
 * Copyright (c) 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *  
 * This file houses the iSCSI Initiator specific utility functions
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_UTIL_C

#include <linux/version.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/in.h>
#include <linux/crypto.h>
#include <linux/blkdev.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <scsi.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_serial.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_scsi.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_crc.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_UTIL_C

extern iscsi_global_t *iscsi_global;
extern int iscsi_add_nopout (iscsi_conn_t *);
extern int iscsi_add_nopout_noqueue (iscsi_conn_t *);
extern int iscsi_add_text_req (iscsi_conn_t *, unsigned char *, int);
extern int iscsi_create_connection (iscsi_session_t *, iscsi_login_holder_t *, u16 *);
extern u32 iscsi_create_session (iscsi_login_holder_t *);
extern int iscsi_start_logout (iscsi_session_t *, int, int, u8, u16, u16);

/*	iscsi_add_conn_to_list():
 *
 *
 */
extern void iscsi_add_conn_to_list (iscsi_session_t *sess, iscsi_conn_t *conn)
{
	iscsi_channel_t *channel = sess->channel;
	unsigned long flags;
	
	spin_lock_bh(&sess->conn_lock);
	ADD_ENTRY_TO_LIST(conn, sess->conn_head, sess->conn_tail);	
	atomic_inc(&sess->nconn);
	printk("iCHANNEL[%d] - Incremented iSCSI connection count to %hu to node: %s\n",
		channel->channel_id, atomic_read(&sess->nconn), SESS_OPS(sess)->TargetName);

	spin_lock_irqsave(&sess->conn_schedule_lock, flags);
	if (!sess->schedule_conn)
		sess->schedule_conn = conn;

	if (sess->schedule_conn)
		up(&sess->schedule_conn->tx_sem);
	spin_unlock_irqrestore(&sess->conn_schedule_lock, flags);
	spin_unlock_bh(&sess->conn_lock);
	
	return;
}

/*	iscsi_remove_conn_from_list():
 *
 *	Called with sess->conn_lock held.
 */
extern int iscsi_remove_conn_from_list (iscsi_session_t *sess, iscsi_conn_t *conn)
{
	unsigned long flags;
	
	REMOVE_ENTRY_FROM_LIST(conn, sess->conn_head, sess->conn_tail);

	/*
	 * Set the scheduling point to the next connection in the list,
	 * if no other connections exist in the list then NULL the pointer.
	 */
	spin_lock_irqsave(&sess->conn_schedule_lock, flags);
	if (sess->schedule_conn == conn)
		sess->schedule_conn = (conn->next) ?
			conn->next : sess->conn_head;

	if (!sess->conn_head && !sess->conn_tail)
		sess->schedule_conn = NULL;

	spin_unlock_irqrestore(&sess->conn_schedule_lock, flags);

	return(0);
}

/*	iscsi_add_session_to_list ():
 *
 *	Adds the session struct to the global linked list of sessions,
 *	and increment session count.
 *
 *	Parameters:	iSCSI Session Pointer.
 *	Returns:	Nothing
 */
extern void iscsi_add_session_to_list (iscsi_session_t *sess)
{
	iscsi_channel_t *channel = sess->channel;
	
	spin_lock(&iscsi_global->session_lock);
	ADD_ENTRY_TO_LIST(sess, iscsi_global->sess_head, iscsi_global->sess_tail);
	printk("iCHANNEL[%d] - Established iSCSI session to node: %s\n",
			channel->channel_id, SESS_OPS(sess)->TargetName);
	iscsi_global->nsess++;
	printk("iSCSI Core Stack[1] - Incremented number of active iSCSI sessions"
			" to %u.\n", iscsi_global->nsess);
	spin_unlock(&iscsi_global->session_lock);

	return;
}

/*	iscsi_remove_session_from_list():
 *
 *
 */
extern int iscsi_remove_session_from_list (iscsi_session_t *sess)
{
	spin_lock(&iscsi_global->session_lock);
	REMOVE_ENTRY_FROM_LIST(sess, iscsi_global->sess_head, iscsi_global->sess_tail);
	spin_unlock(&iscsi_global->session_lock);
		
	return(0);
}

/*	iscsi_add_cmd_to_immediate_queue():
 *
 *
 */
extern void iscsi_add_cmd_to_immediate_queue (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	u8 state)
{
	iscsi_queue_req_t *qr;
	
	if (!(qr = kmalloc(sizeof(iscsi_queue_req_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_queue_req_t\n");
		return;
	}
	memset(qr, 0, sizeof(iscsi_queue_req_t));

	spin_lock_bh(&cmd->state_lock);
	qr->cmd = cmd;
	qr->state = state;
	cmd->cmd_flags |= ICF_CMD_STATE_ACTIVE;
	atomic_inc(&cmd->immed_queue_count);
	spin_unlock_bh(&cmd->state_lock);

	spin_lock_bh(&conn->immed_queue_lock);
	ADD_ENTRY_TO_LIST(qr, conn->immed_queue_head, conn->immed_queue_tail);
	atomic_set(&conn->check_immediate_queue, 1);
	spin_unlock_bh(&conn->immed_queue_lock);
		
	up(&conn->tx_sem);
	
	return;
}

/*	iscsi_get_cmd_from_immediate_queue():
 *
 *
 */
extern iscsi_queue_req_t *iscsi_get_cmd_from_immediate_queue (iscsi_conn_t *conn)
{
	iscsi_queue_req_t *qr;

	spin_lock_bh(&conn->immed_queue_lock);
	if (!conn->immed_queue_head) {
		spin_unlock_bh(&conn->immed_queue_lock);
		return(NULL);
	}

	qr = conn->immed_queue_head;
	REMOVE_ENTRY_FROM_LIST(qr, conn->immed_queue_head, conn->immed_queue_tail);
	spin_unlock_bh(&conn->immed_queue_lock);

	return(qr);
}

/*	iscsi_remove_cmd_from_immediate_queue():
 *
 *
 */
extern void iscsi_remove_cmd_from_immediate_queue (iscsi_cmd_t *cmd, iscsi_conn_t *conn)
{
	iscsi_queue_req_t *qr, *qr_next;

	spin_lock_bh(&cmd->state_lock);
	if (atomic_read(&cmd->immed_queue_count)) {
		printk("immed_queue_count: %d ITT: 0x%08x %p\n",
			atomic_read(&cmd->immed_queue_count),
				cmd->init_task_tag, cmd);

		spin_unlock_bh(&cmd->state_lock);

		spin_lock_bh(&conn->immed_queue_lock);
		qr = conn->immed_queue_head;
		while (qr) {
			qr_next = qr->next;

			if (qr->cmd != cmd) {
				qr = qr_next;
				continue;
			}

			REMOVE_ENTRY_FROM_LIST(qr, conn->immed_queue_head,
					conn->immed_queue_tail);
			atomic_dec(&cmd->immed_queue_count);

			kfree(qr);
			qr = qr_next;
		}
		spin_unlock_bh(&conn->immed_queue_lock);

		if (atomic_read(&cmd->immed_queue_count)) {
			TRACE_ERROR("Left over immed_queue_count: %d ITT:"
			" 0x%08x %p\n", atomic_read(&cmd->immed_queue_count),
				 cmd->init_task_tag, cmd);
		}

		return;
	}
	spin_unlock_bh(&cmd->state_lock);
	
	return;
}
		
/*	iscsi_get_cmd_from_pool():
 *
 *	Retrieve the next iscsi_cmd_t from the head of the session free pool,  
 *	else return NULL.
 */
extern iscsi_cmd_t *iscsi_get_cmd_from_pool (iscsi_session_t *sess)
{
	iscsi_cmd_t *cmd;
	unsigned long flags;

	spin_lock_irqsave(&sess->pool_lock, flags);
	if (!sess->pool_head) {
		spin_unlock_irqrestore(&sess->pool_lock, flags);
		return(NULL);
	}

	cmd		= sess->pool_head;
	sess->pool_head = sess->pool_head->next;

	/*
	 *  NOTE:  cmd->next MUST be NULL!
	 */
	cmd->next = NULL;

	if (!sess->pool_head)
		sess->pool_tail = NULL;
	spin_unlock_irqrestore(&sess->pool_lock, flags);

	atomic_dec(&sess->pool_count);

	return(cmd);
}

/*	iscsi_release_cmd_to_pool():
 *
 *	Release a iscsi_cmd_t back into the session free pool.
 */
extern void iscsi_release_cmd_to_pool (
	iscsi_cmd_t *cmd,
	iscsi_session_t *sess)
{
	u32 iov_data_count = 0;
	unsigned long flags;
	struct iovec *iov = NULL;
	
	if (cmd->orig_iov_data_count) {
		iov = cmd->iov_data;
		iov_data_count = cmd->orig_iov_data_count;
	}
		
	memset(cmd, 0, sizeof (iscsi_cmd_t));

	if (iov_data_count) {
		cmd->iov_data = iov;
		cmd->orig_iov_data_count = iov_data_count;
	}

	spin_lock_irqsave(&sess->pool_lock, flags);
	if (!sess->pool_head && !sess->pool_tail)
		sess->pool_head		= cmd;
	else
		sess->pool_tail->next	= cmd;
	sess->pool_tail			= cmd;
	spin_unlock_irqrestore(&sess->pool_lock, flags);

	atomic_inc(&sess->pool_count);
	
	return;
}	

/*	iscsi_release_all_cmds_in_pool():
 *
 *	Free the memory of all the iscsi_cmd_t's in the session free pool.
 */
extern void iscsi_release_all_cmds_in_pool (iscsi_session_t *sess)
{
	u32 release_cmd_count = 0;
	iscsi_cmd_t *cmd, *cmd_next;
	unsigned long flags;
	
	spin_lock_irqsave(&sess->pool_lock, flags);
	cmd = sess->pool_head;
	while (cmd) {
		cmd_next = cmd->next;

		if (cmd->iov_data)
			kfree(cmd->iov_data);
		kfree(cmd);

		cmd = cmd_next;
		release_cmd_count++;
	}
	spin_unlock_irqrestore(&sess->pool_lock, flags);

	if (release_cmd_count != atomic_read(&sess->pool_count)) {
		TRACE_ERROR("Released cmd pool count %d does not equal saved"
			" value %d.\n", release_cmd_count, 
				atomic_read(&sess->pool_count));
		return;
	}
	TRACE(TRACE_ISCSI, "Released %d cmds from iSCSI Session Pool.\n", 
			release_cmd_count);
	atomic_set(&sess->pool_count, 0);

	return;
}

/*	iscsi_add_cmd_to_session_queue():
 *	
 *
 */
extern void iscsi_add_cmd_to_session_queue (
	iscsi_cmd_t *cmd,
	iscsi_session_t *sess)
{
	unsigned long flags;
	
	spin_lock_irqsave(&sess->pending_lock, flags);
	ADD_ENTRY_TO_LIST(cmd, sess->pending_head, sess->pending_tail);
	spin_unlock_irqrestore(&sess->pending_lock, flags);
	
	return;
}

/*	iscsi_get_cmd_from_session_queue():
 *
 *
 */
extern iscsi_cmd_t *iscsi_get_cmd_from_session_queue (iscsi_session_t *sess)
{
	iscsi_cmd_t *cmd;
	unsigned long flags;
	
	spin_lock_irqsave(&sess->pending_lock, flags);
	if (!sess->pending_head) {
		spin_unlock_irqrestore(&sess->pending_lock, flags);
		return(NULL);
	}

	cmd			= sess->pending_head;
	sess->pending_head	= sess->pending_head->next;

	if (sess->pending_head)
		sess->pending_head->prev = NULL;
	
	/*
	 * NOTE:  cmd->next and cmd->prev MUST be NULL!
	 */
	cmd->next = NULL;
	cmd->prev = NULL;

	if (!sess->pending_head)
		sess->pending_tail = NULL;
	spin_unlock_irqrestore(&sess->pending_lock, flags);

	return(cmd);
}

/*	iscsi_allocate_iovecs_for_cmd():
 *
 *
 */
static int iscsi_allocate_iovecs_for_cmd (
	iscsi_cmd_t *cmd)
{
	int sg_count, req_sg_count;
	
	if (!(sg_count = cmd->scsi_cmnd->use_sg))
		sg_count = 1;
	else {
		req_sg_count = cmd->scsi_cmnd->request_bufflen / PAGE_SIZE;
		if (req_sg_count > sg_count)
			sg_count = req_sg_count;
	}
	
	sg_count += ISCSI_IOV_DATA_BUFFER;
		
	/*
	 * See if we already have enough iovecs from a previous iscsi_cmd_t
	 * to fulfill the request.
	 */	
	if (cmd->orig_iov_data_count >= sg_count)
		return(0);

	if (cmd->iov_data)
		kfree(cmd->iov_data);
	
	if (!(cmd->iov_data = (struct iovec *) kmalloc(
			sg_count * sizeof(struct iovec), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for"
			" iscsi_cmd_t->iov_data.\n");
		return(-1);
	}
	memset(cmd->iov_data, 0, sg_count * sizeof(struct iovec));
	
	cmd->orig_iov_data_count = sg_count;
	
	return(0);
}

/*	iscsi_allocate_cmd():
 *	
 *	First we try to grab a free iscsi_cmd_t from the session pool, if that
 *	fails go ahead an malloc a new one.  Also set the OS dependant SCSI
 *	pointer,  and fill in various other default values.
 */
extern iscsi_cmd_t *iscsi_allocate_cmd (
	iscsi_session_t *sess,
	struct scsi_cmnd *sc)
{
	iscsi_cmd_t *cmd;

	if (!(cmd = iscsi_get_cmd_from_pool(sess))) {
		if (!(cmd = (iscsi_cmd_t *) kmalloc(
				sizeof(iscsi_cmd_t), GFP_ATOMIC))) {
			TRACE_ERROR("Unable to allocate memory"
				" for iscsi_cmd_t.\n");
			return(NULL);
		}
		memset(cmd, 0, sizeof(iscsi_cmd_t));
	}
	
	if (sc) {
		cmd->scsi_cmnd = sc;
		
		if (iscsi_allocate_iovecs_for_cmd(cmd) < 0)
			return(NULL);
		
		cmd->iov_data[0].iov_base = cmd->pdu;
		cmd->iov_data[0].iov_len = ISCSI_HDR_LEN;
		cmd->iov_data_count = 1;
	} else {
		cmd->iov_misc[0].iov_base = cmd->pdu;
		cmd->iov_misc[0].iov_len = ISCSI_HDR_LEN;
		cmd->iov_misc_count = 1;
	}
	
	cmd->tx_size = ISCSI_HDR_LEN;
	
	init_MUTEX_LOCKED(&cmd->cmd_state_wait_sem);
	init_MUTEX_LOCKED(&cmd->cmd_waiting_on_uc_sem);
	init_MUTEX_LOCKED(&cmd->scsi_waiting_on_uc_sem);
	spin_lock_init(&cmd->cmd_usage_lock);
	spin_lock_init(&cmd->scsi_usage_lock);
	spin_lock_init(&cmd->datain_timeout_lock);
	spin_lock_init(&cmd->r2t_lock);
	spin_lock_init(&cmd->state_lock);
	
	return(cmd);
}

/*	iscsi_attach_cmd_to_conn():
 *
 *	Parameters: iSCSI Connection, iSCSI Command.
 *	Returns: Nothing
 */
extern void iscsi_attach_cmd_to_conn (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	if (cmd->cmd_flags & ICF_ATTACHED_TO_CONN) {
		TRACE_ERROR("ITT: 0x%08x already attached to connection"
			" CID: %hu\n", cmd->init_task_tag, conn->cid);
		return;
	}
	cmd->conn = conn;
	
	spin_lock_bh(&conn->cmd_lock);
	ADD_ENTRY_TO_LIST(cmd, conn->cmd_head, conn->cmd_tail);
	cmd->cmd_flags |= ICF_ATTACHED_TO_CONN;
	spin_unlock_bh(&conn->cmd_lock);

	return;
}

/*	__iscsi_remove_cmd_from_conn_list():
 *
 *	Called with conn->cmd_lock held.
 */
extern void __iscsi_remove_cmd_from_conn_list (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	REMOVE_ENTRY_FROM_LIST(cmd, conn->cmd_head, conn->cmd_tail);
	cmd->cmd_flags &= ~ICF_ATTACHED_TO_CONN;

	return;
}

/*	iscsi_remove_cmd_from_conn_list():
 *
 *
 */
extern void iscsi_remove_cmd_from_conn_list (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->cmd_lock);
	REMOVE_ENTRY_FROM_LIST(cmd, conn->cmd_head, conn->cmd_tail);
	cmd->cmd_flags &= ~ICF_ATTACHED_TO_CONN;
	spin_unlock_bh(&conn->cmd_lock);
	
	return;
}
	
/*	iscsi_free_command_from_conn():
 * 	
 *
 */
extern int iscsi_free_command_from_conn (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	iscsi_channel_t *c = SESS(conn)->channel;
	struct scsi_cmnd *sc = NULL;

	if (cmd->seq_list)
		kfree(cmd->seq_list);
	if (cmd->pdu_list)
		kfree(cmd->pdu_list);
	if (cmd->buf_ptr)
		kfree(cmd->buf_ptr);
	if (cmd->scsi_cmnd) {
		sc = cmd->scsi_cmnd;
		sc->SCp.ptr = NULL;
		iscsi_dec_scsi_usage_count(cmd);
		cmd->scsi_cmnd = NULL;
	}

	iscsi_remove_cmd_from_conn_list(cmd, conn);
	iscsi_release_cmd_to_pool(cmd, SESS(conn));
	
	if (sc) {
		iscsi_channel_lun_t *lu = &c->ch_lun_list[SCSI_LUN(sc)];
			
		lu->iscsi_current_tasks--;

		iscsi_complete_command(sc);
		/*
		 * The Linux SCSI Layer does not always send another
		 * SCSI Command down when an outstanding one is completed.
		 * Hence, we activate this connection's TX Thread to
		 * do more work.
		 */
		up(&conn->tx_sem);
	}
		
	return(0);	
}

/*	iscsi_remove_cmd_from_sess_list():
 *
 *
 */
static inline void iscsi_remove_cmd_from_sess_list (
	iscsi_cmd_t *cmd,
	iscsi_session_t *sess)
{
	unsigned long flags;
	
	spin_lock_irqsave(&sess->pending_lock, flags);
	REMOVE_ENTRY_FROM_LIST(cmd, sess->pending_head, sess->pending_tail);
	spin_unlock_irqrestore(&sess->pending_lock, flags);
		
	return;
}

/*	iscsi_free_command_from_sess():
 *
 *
 */
extern int iscsi_free_command_from_sess (
	iscsi_cmd_t *cmd,
	iscsi_session_t *sess)
{
	iscsi_channel_t *c = sess->channel;
	struct scsi_cmnd *sc = NULL;

	if (cmd->seq_list)
		kfree(cmd->seq_list);
	if (cmd->pdu_list)
		kfree(cmd->pdu_list);
	if (cmd->buf_ptr)
		kfree(cmd->buf_ptr);
	if (cmd->scsi_cmnd) {
		sc = cmd->scsi_cmnd;
		sc->SCp.ptr = NULL;
		iscsi_dec_scsi_usage_count(cmd);
		cmd->scsi_cmnd = NULL;
	}

	iscsi_remove_cmd_from_sess_list(cmd, sess);
	iscsi_release_cmd_to_pool(cmd, sess);

	if (sc) {
		iscsi_channel_lun_t *lu = &c->ch_lun_list[SCSI_LUN(sc)];
		
		lu->iscsi_current_tasks--;
		iscsi_complete_command(sc);
	}

	return(0);
}

/*	iscsi_stop_datain_timers_for_cmds():
 *
 *
 */
extern void iscsi_stop_datain_timers_for_cmds (iscsi_conn_t *conn)
{
	iscsi_cmd_t *cmd, *cmd_next;
	iscsi_channel_t *c = SESS(conn)->channel;
	iscsi_channel_req_t *cr = NULL;
	iscsi_lun_t *lun;
	struct scsi_cmnd *sc;

	spin_lock_bh(&conn->cmd_lock);
	cmd = conn->cmd_head;
	while (cmd) {
		cmd_next = cmd->next;
		if (!cmd->scsi_cmnd) {
			cmd = cmd_next;
			continue;
		}
		sc = cmd->scsi_cmnd;

		spin_unlock_bh(&conn->cmd_lock);
		lun = &cmd->lun;

		if (sc->sc_data_direction == DMA_FROM_DEVICE)
			iscsi_stop_datain_timer(cmd);

		if (!iscsi_OS_check_lu_online(lun->OS_SCSI_lu)) {
loop:
			if (!(cr = iscsi_allocate_ch_req(CREQ_STOP_SCSI_CMD,
					(void *)cmd, 1, NULL)))
				goto loop;

			ISCSI_ADD_CH_req(cr, c);
			down(&cr->cr_data_sem);
			iscsi_complete_ch_req(cr, c);
			
			/*
			 * cmd->scsi_cmnd NULL at this point, so it is safe
			 * to release the iscsi_cmd_t to the session pool now.
			 */
			iscsi_free_command_from_conn(cmd, conn);
		}
	
		spin_lock_bh(&conn->cmd_lock);
		cmd = cmd_next;
	}
	spin_unlock_bh(&conn->cmd_lock);
	
	return;						
}

/*	iscsi_free_all_commands_for_conn():
 *
 *	Loops through an iSCSI connection's command list,  calling
 *	iscsi_free_command_from_conn() on each iscsi_cmd_t present.
 */
extern int iscsi_free_all_commands_for_conn (iscsi_conn_t *conn)
{
	iscsi_cmd_t *cmd = NULL, *cmd_next = NULL;
	
	spin_lock_bh(&conn->cmd_lock);
	cmd = conn->cmd_head;
	while (cmd) {
		cmd_next = cmd->next;
		spin_unlock_bh(&conn->cmd_lock);
		iscsi_free_command_from_conn(cmd, conn);
		spin_lock_bh(&conn->cmd_lock);

		cmd = cmd_next;
	}
	spin_unlock_bh(&conn->cmd_lock);

	return(0);
}

/*	iscsi_free_all_commands_for_sess():
 *
 *	Loops through a iSCSI session's pending command list,  calling
 *	iscsi_free_command_from_sess() on each iscsi_cmd_t present.
 */
extern int iscsi_free_all_commands_for_sess (iscsi_session_t *sess)
{
	iscsi_cmd_t *cmd = NULL, *cmd_next = NULL;
	unsigned long flags;

	spin_lock_irqsave(&sess->pending_lock, flags);
	cmd = sess->pending_head;
	while (cmd) {
		cmd_next = cmd->next;

		spin_unlock_irqrestore(&sess->pending_lock, flags);
		iscsi_free_command_from_sess(cmd, sess);
		spin_lock_irqsave(&sess->pending_lock, flags);

		cmd = cmd_next;
	}
	spin_unlock_irqrestore(&sess->pending_lock, flags);

	return(0);
}

/*	iscsi_start_login_process():
 *
 *
 */
extern int iscsi_start_login_process (iscsi_login_holder_t *lh)
{
	u16 cid;
	u32 sid;
	int ret = 0;
	iscsi_conn_t *conn = NULL;
	iscsi_session_t *sess = NULL;
	iscsi_targetaddress_t *ta = NULL;
	
	if (!(ta = iscsi_search_for_login_targetaddress(lh))) {
		if (!(sid = iscsi_create_session(lh)))
			return(-1);
		if (!(sess = iscsi_get_session_from_sid(sid)))
			return(-1);
		if (!(conn = iscsi_get_conn(sess))) {
			iscsi_dec_session_usage_count(sess);
			return(-1);
		}

		if (iscsi_add_text_req(conn, "SendTargets=All", 0) < 0) {
			iscsi_dec_conn_usage_count(conn);
			iscsi_dec_session_usage_count(sess);
			return(-1);
		}
		cid = conn->cid;
		iscsi_dec_conn_usage_count(conn);

		down_interruptible(&sess->discovery_sem);
		if (signal_pending(current) || atomic_read(&sess->session_reinstatement)) {
			iscsi_dec_session_usage_count(sess);
			return(-1);
		}

		ret = iscsi_start_logout(sess, 0, 0, CLOSESESSION, cid, cid);
		if (ret != -2)
			iscsi_dec_session_usage_count(sess);

		if ((ta = iscsi_search_for_login_targetaddress(lh))) {
			if (!(iscsi_create_session(lh)))
				return(-1);
		} else {
			TRACE_ERROR("Unable to locate TargetAddress Entry for"
			" %s:%hu\n", iscsi_ntoa(lh->ipv4_address), lh->port);
			return(-1);
		}
	} else {
		if (!(iscsi_create_session(lh)))
			return(-1);
	}
		
	return(0);
}

/*	iscsi_process_response():
 *
 *	This function calls iscsi_update_statsn() to check for holes in
 *	ExpStatSN, and iscsi_free_command() to remove the iscsi_cmd_t from the
 *	per-connection command list, and complete the command to the
 *	OS'es SCSI layer.
 */
extern int iscsi_process_response (
	iscsi_conn_t *conn,
	iscsi_cmd_t *cmd,
	unsigned char *buf)
{
	int ret;
	struct iscsi_targ_scsi_rsp *hdr = (struct iscsi_targ_scsi_rsp *) buf;

	ret = iscsi_update_statsn(conn, cmd, hdr->stat_sn, 1);
	if (ret == STATSN_ERROR_CANNOT_RECOVER)
		return(-1);

	iscsi_check_cmd_usage_count(cmd);
	if (iscsi_cmd_state_wait(cmd))
		return(-1);
	iscsi_remove_cmd_from_immediate_queue(cmd, conn);
	
	if (iscsi_free_command_from_conn(cmd, conn) < 0)
		return(-1);

	return(0);
}

/*	iscsi_find_cmd_from_itt():
 *
 *	Searches through connections list of commands looking for a matching
 *	Initiator Task Tag.
 *
 *	Returns iscsi_cmd_t if found, NULL if not.
 */
extern iscsi_cmd_t *iscsi_find_cmd_from_itt (
	iscsi_conn_t *conn,
	u32 init_task_tag)
{
	iscsi_cmd_t *cmd;
	
	spin_lock_bh(&conn->cmd_lock);
	for (cmd = conn->cmd_head; cmd; cmd = cmd->next) {
		if (cmd->init_task_tag == init_task_tag)
			break;
	}
	spin_unlock_bh(&conn->cmd_lock);

	if (!cmd) {
		TRACE_ERROR("Unable to find ITT: 0x%08x on CID: %hu.\n",
			       	init_task_tag, conn->cid);
		return(NULL);
	}

	return(cmd);
}

/*	iscsi_find_cmd_from_itt_or_dump():
 *
 * 	Searches through connections list of commands looking for a matching
 * 	Initiator Task Tag.
 *
 * 	Returns iscsi_cmd_t if found, if not dump u32 length of data into
 * 	an offload buffer and return NULL.
 */
extern iscsi_cmd_t *iscsi_find_cmd_from_itt_or_dump (
	iscsi_conn_t *conn,
	u32 init_task_tag,
	u32 length)
{
	iscsi_cmd_t *cmd;

	spin_lock_bh(&conn->cmd_lock);
	for (cmd = conn->cmd_head; cmd; cmd = cmd->next)
		if (cmd->init_task_tag == init_task_tag)
			break;
	spin_unlock_bh(&conn->cmd_lock);

	if (!cmd) {
		TRACE_ERROR("Unable to find ITT: 0x%08x on CID: %hu,"
			" dumping payload\n", init_task_tag, conn->cid);
		if (length)
			iscsi_dump_data_payload(conn, length, 1);
		return(NULL);
	}
	
	return(cmd);
}

/*	iscsi_set_initiator_name():
 *
 *	Sets the global InitiatorName value for this iSCSI Node.
 */
//#warning FIXME: Do proper checking of InitiatorName
extern int iscsi_set_initiator_name (char *text_buf)
{
	if (strlen(text_buf) > ISCSI_MAX_INITIATORNAME_SIZE) {
		TRACE_ERROR("Requested iSCSI Initiator Name %d is greater than"
			" maximum size %d, please try again.\n", 
			(int)strlen(text_buf), ISCSI_MAX_INITIATORNAME_SIZE);
		return(-1);
	}

	iscsi_global->initname_set = 1;
	strncpy(iscsi_global->initiatorname, text_buf, strlen(text_buf));

	printk("iSCSI Core Stack[1] - Set iSCSI Node/Initiator Name to %s\n",
		iscsi_global->initiatorname);
	
	return(0);
}

/*	iscsi_determine_sync_and_steering_counts():
 *
 *	Used before iscsi_do[rx,tx]_data() to determine iov and [rx,tx]_marker
 *	array counts needed for sync and steering.
 */
static inline int iscsi_determine_sync_and_steering_counts (
	iscsi_conn_t *conn,
	iscsi_data_count_t *count)
{
	u32 length = count->data_length;
	u32 marker, markint;

	count->sync_and_steering = 1;
		
	marker = (count->type == ISCSI_RX_DATA) ? conn->if_marker : conn->of_marker;
	markint = (count->type == ISCSI_RX_DATA) ? (CONN_OPS(conn)->IFMarkInt * 4) :
					(CONN_OPS(conn)->OFMarkInt * 4);
	count->ss_iov_count = count->iov_count;

	while (length > 0) {
		if (length >= marker) {
			count->ss_iov_count += 3;
			count->ss_marker_count += 2;
			
			length -= marker;
			marker = markint;
		} else
			length = 0;
	}
	
	return(0);
}

/*	iscsi_set_sync_and_steering_values():
 *
 *	Setup conn->if_marker and conn->of_marker values based upon
 *	the initial marker-less interval. (see RFC 3720 Section A.2)
 */
extern int iscsi_set_sync_and_steering_values (iscsi_conn_t *conn)
{
	int login_ifmarker_count = 0, login_ofmarker_count = 0, next_marker = 0;
	/*
	 * IFMarkInt and OFMarkInt are negotiated as 32-bit words.
	 */
	u32 IFMarkInt = (CONN_OPS(conn)->IFMarkInt * 4);
	u32 OFMarkInt = (CONN_OPS(conn)->OFMarkInt * 4);
	
	if (CONN_OPS(conn)->OFMarker) {
		if (conn->of_marker <= OFMarkInt) {
			conn->of_marker = (OFMarkInt - conn->of_marker);
		} else {
			login_ofmarker_count = (conn->of_marker / OFMarkInt);
			next_marker = (OFMarkInt * (login_ofmarker_count + 1)) +
					(login_ofmarker_count * MARKER_SIZE);
			conn->of_marker = (next_marker - conn->of_marker);
		}
		printk("RFC-3720[1] - Setting OFMarker value to %u based on"
				" FIM\n", conn->of_marker);
	}
		
	if (CONN_OPS(conn)->IFMarker) {
		if (conn->if_marker <= IFMarkInt) {
			conn->if_marker = (IFMarkInt - conn->if_marker);
		} else {
			login_ifmarker_count = (conn->if_marker / IFMarkInt);
			next_marker = (IFMarkInt * (login_ifmarker_count + 1)) +
					(login_ifmarker_count * MARKER_SIZE);
			conn->if_marker = (next_marker - conn->if_marker);
		}
		conn->if_marker_offset = 0;
		printk("RFC-3720[1] - Setting IFMarker value to %u based on"
				" FIM\n", conn->if_marker);
	}

	return(0);
}

/*	iscsi_get_conn():
 *
 *
 */
extern iscsi_conn_t *iscsi_get_conn (iscsi_session_t *sess)
{
	iscsi_conn_t *conn;

	spin_lock_bh(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next)
		if (conn->conn_state == INIT_CONN_STATE_LOGGED_IN) {
			__iscsi_inc_conn_usage_count(conn);
			break;
		}
	spin_unlock_bh(&sess->conn_lock);

	if (!conn) {
		TRACE_ERROR("Unable to locate active connection.\n");
		return(NULL);
	}

	return(conn);
}

/*	iscsi_get_conn_from_cid():
 *
 *
 */
extern iscsi_conn_t *iscsi_get_conn_from_cid (iscsi_session_t *sess, u16 cid)
{
	iscsi_conn_t *conn;

	spin_lock_bh(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next) {
		if ((conn->cid == cid) &&
		    (conn->conn_state == INIT_CONN_STATE_LOGGED_IN)) {
			__iscsi_inc_conn_usage_count(conn);
			break;
		}
	}
	spin_unlock_bh(&sess->conn_lock);
	
	if (!conn)
		return(NULL);
		
	return(conn);
}

/*	iscsi_check_cmd_usage_count():
 *
 *
 */
extern void iscsi_check_cmd_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock(&cmd->cmd_usage_lock);
	if (atomic_read(&cmd->cmd_usage_count)) {
		atomic_set(&cmd->cmd_waiting_on_uc, 1);
		spin_unlock(&cmd->cmd_usage_lock);

		down_interruptible(&cmd->cmd_waiting_on_uc_sem);
		return;
	}
	spin_unlock(&cmd->cmd_usage_lock);

	return;
}

/*	iscsi_dec_cmd_usage_count():
 *
 *
 */
extern void iscsi_dec_cmd_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock(&cmd->cmd_usage_lock);	
	atomic_dec(&cmd->cmd_usage_count);

	if (!atomic_read(&cmd->cmd_usage_count) &&
	     atomic_read(&cmd->cmd_waiting_on_uc))
		up(&cmd->cmd_waiting_on_uc_sem);

	spin_unlock(&cmd->cmd_usage_lock);

	return;
}

/*	iscsi_inc_cmd_usage_count():
 *
 *
 */
extern void iscsi_inc_cmd_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock(&cmd->cmd_usage_lock);
	atomic_inc(&cmd->cmd_usage_count);
	spin_unlock(&cmd->cmd_usage_lock);

	return;
}

/*	iscsi_check_scsi_usage_count():
 *
 *
 */
extern void iscsi_check_scsi_usage_count (iscsi_cmd_t *cmd, int clear_scsi_cmnd)
{
	spin_lock_bh(&cmd->scsi_usage_lock);
	if (atomic_read(&cmd->scsi_usage_count)) {
		atomic_set(&cmd->scsi_waiting_on_uc, 1);
		spin_unlock_bh(&cmd->scsi_usage_lock);

		down(&cmd->scsi_waiting_on_uc_sem);
		if (clear_scsi_cmnd) {
			cmd->scsi_cmnd = NULL;
		}
		return;
	}
	if (clear_scsi_cmnd) {
		cmd->scsi_cmnd = NULL;
	}
	spin_unlock_bh(&cmd->scsi_usage_lock);

	return;
}

extern void __iscsi_dec_scsi_usage_count (iscsi_cmd_t *cmd)
{               
	spin_lock(&cmd->scsi_usage_lock);
	if (cmd->scsi_cmnd) {
		atomic_dec(&cmd->scsi_usage_count);

		if (!atomic_read(&cmd->scsi_usage_count) &&
		     atomic_read(&cmd->scsi_waiting_on_uc))
			up(&cmd->scsi_waiting_on_uc_sem);
	}
	spin_unlock(&cmd->scsi_usage_lock);

	return;
}


/*	iscsi_dec_scsi_usage_count():
 *
 *
 */
extern void iscsi_dec_scsi_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock_bh(&cmd->scsi_usage_lock);
	if (cmd->scsi_cmnd) {
		atomic_dec(&cmd->scsi_usage_count);

		if (!atomic_read(&cmd->scsi_usage_count) &&
		     atomic_read(&cmd->scsi_waiting_on_uc))
			up(&cmd->scsi_waiting_on_uc_sem);
	}
	spin_unlock_bh(&cmd->scsi_usage_lock);

	return;
}

extern void __iscsi_inc_scsi_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock(&cmd->scsi_usage_lock);
	if (cmd->scsi_cmnd) {
		atomic_inc(&cmd->scsi_usage_count);
	}
	spin_unlock(&cmd->scsi_usage_lock);
				        
	return;
}

/*	iscsi_inc_scsi_usage_count():
 *
 *
 */
extern void iscsi_inc_scsi_usage_count (iscsi_cmd_t *cmd)
{
	spin_lock_bh(&cmd->scsi_usage_lock);
	if (cmd->scsi_cmnd) {
		atomic_inc(&cmd->scsi_usage_count);
	}
	spin_unlock_bh(&cmd->scsi_usage_lock);

	return;
}

/*	iscsi_check_conn_usage_count():
 *
 *
 */
extern void iscsi_check_conn_usage_count (iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->conn_usage_lock);

	if (atomic_read(&conn->conn_usage_count)) {
		atomic_set(&conn->conn_waiting_on_uc, 1);
		spin_unlock_bh(&conn->conn_usage_lock);

		down(&conn->conn_waiting_on_uc_sem);
		return;
	}       
	spin_unlock_bh(&conn->conn_usage_lock);
			
	return;
}

/*	__iscsi_dec_conn_usage_count():
 *
 *
 */
extern void __iscsi_dec_conn_usage_count (iscsi_conn_t *conn)
{
	spin_lock(&conn->conn_usage_lock);
	atomic_dec(&conn->conn_usage_count);

	if (!atomic_read(&conn->conn_usage_count) &&
	     atomic_read(&conn->conn_waiting_on_uc))
		up(&conn->conn_waiting_on_uc_sem);

	spin_unlock(&conn->conn_usage_lock);

	return;
}

/*	iscsi_dec_conn_usage_count():
 *
 *
 */
extern void iscsi_dec_conn_usage_count (iscsi_conn_t *conn)
{	
	spin_lock_bh(&conn->conn_usage_lock);
	atomic_dec(&conn->conn_usage_count);

	if (!atomic_read(&conn->conn_usage_count) &&
	     atomic_read(&conn->conn_waiting_on_uc))
		up(&conn->conn_waiting_on_uc_sem);

	spin_unlock_bh(&conn->conn_usage_lock);
	
	return;
}

/*	__iscsi_inc_conn_usage_count():
 *
 *
 */
extern void __iscsi_inc_conn_usage_count (iscsi_conn_t *conn)
{
	spin_lock(&conn->conn_usage_lock);
	atomic_inc(&conn->conn_usage_count);
	spin_unlock(&conn->conn_usage_lock);

	return;
}

/*	iscsi_inc_conn_usage_count():
 *
 *
 */
extern void iscsi_inc_conn_usage_count (iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->conn_usage_lock);
	atomic_inc(&conn->conn_usage_count);
	spin_unlock_bh(&conn->conn_usage_lock);
	
	return;
}

/*	iscsi_update_cmdsn():
 *
 * 
 */
extern void iscsi_update_cmdsn (
	iscsi_session_t *sess,
	u32 exp_cmdsn,
	u32 max_cmdsn,
	int immediate)
{
	/* RFC-3720: Section 2.2.2.1  Command Numbering and Acknowledging
	 * 
	 *	MaxCmdSN and ExpCmdSN fields are processed by the
	 *	initiator as follows:
	 *
	 *	-If the PDU MaxCmdSN is less than the PDU ExpCmdSN-1
	 *		(in Serial Arithmetic Sense), they are both ignored.
	 *	-If the PDU MaxCmdSN is less than the local MaxCmdSN
	 *		(in Serial Arithmetic Sense), it is ignored; otherwise,
	 *		it updates the local MaxCmdSN.
	 *	-If the PDU ExpCmdSN is less than the local ExpCmdSN
	 *		(in Serial Arithmetic Sense), it is ignored; otherwise,
	 *		it updates the local ExpCmdSN.
	 */
	spin_lock_bh(&sess->cmdsn_lock);
	if (!serial_lt(max_cmdsn, (exp_cmdsn - 1))) {
		if (!serial_lt(max_cmdsn, sess->max_cmdsn)) {
			sess->max_cmdsn = max_cmdsn;
			TRACE(TRACE_ISCSI, "Updated local MaxCmdSN to"
				" 0x%08x\n", max_cmdsn);
		}
		if (!serial_lt(exp_cmdsn, sess->exp_cmdsn)) {
			sess->exp_cmdsn = exp_cmdsn;
			TRACE(TRACE_ISCSI, "Updated local ExpCmdSN to"
				" 0x%08x\n", exp_cmdsn);
			if (!immediate)
				iscsi_mod_cmdsn_timer(sess, &exp_cmdsn);
		}
	}
	spin_unlock_bh(&sess->cmdsn_lock);

	return;
}

/*	iscsi_update_statsn():
 *
 *
 */
extern int iscsi_update_statsn (
	iscsi_conn_t *conn,
	iscsi_cmd_t *cmd,
	u32 statsn,
	u8 update)
{
	if (cmd)
		cmd->statsn = statsn;
	if (statsn == conn->exp_statsn) {
		if (update) {
			conn->exp_statsn++;
			TRACE(TRACE_ISCSI, "iSCSI StatSN matches"
			" ExpStatSN, incremented ExpStatSN to:"
				" 0x%08x\n", conn->exp_statsn);
		} else {
			TRACE(TRACE_ISCSI, "iSCSI StatSN matches"
			" ExpStatSN, but not incrementing ExpStatSN.\n");
		}
		return(STATSN_NORMAL_OPERATION);
	} else if (statsn > conn->exp_statsn) {
		if (SESS_OPS_C(conn)->ErrorRecoveryLevel > 0) {
			TRACE_ERROR("StatSN: 0x%08x greater than"
			" ExpStatSN: 0x%08x.\n", statsn, 
				conn->exp_statsn);
			return(STATSN_ERROR_CANNOT_RECOVER);
		} else {
			TRACE_ERROR("StatSN: 0x%08x greater than"
			" ExpStatSN: 0x%08x, unable to recover while"
			" in ERL=0.\n", statsn, conn->exp_statsn);
			return(STATSN_ERROR_CANNOT_RECOVER);
		}
	} else {
		TRACE_ERROR("StatSN: 0x%08x less than ExpStatSN:"
		" 0x%08x, ignoring.\n", statsn, conn->exp_statsn);
		return(STATSN_NORMAL_OPERATION);
	}
}

/*	iscsi_add_r2t_to_cmd():
 *
 *
 */
extern void iscsi_add_r2t_to_cmd (
	iscsi_cmd_t *cmd,
	iscsi_r2t_t *r2t)
{
	spin_lock(&cmd->r2t_lock);
	ADD_ENTRY_TO_LIST(r2t, cmd->r2t_head, cmd->r2t_tail);
	spin_unlock(&cmd->r2t_lock);	
	
	return;
}

/*	iscsi_allocate_r2t():
 *
 *
 */
extern iscsi_r2t_t *iscsi_allocate_r2t (void)
{
	iscsi_r2t_t *r2t;
	
	if (!(r2t = kmalloc(sizeof(iscsi_r2t_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate iscsi_r2t_t.\n");
		return(NULL);
	}
	memset(r2t, 0, sizeof(iscsi_r2t_t));

	return(r2t);
}

/*	iscsi_get_pdu_holder_from_r2t():
 *
 *	Only called when DataPDUInOrder=No.
 */
extern iscsi_pdu_t *iscsi_get_pdu_holder_from_r2t (
	iscsi_cmd_t *cmd,
	iscsi_r2t_t *r2t)
{
	int i;
	iscsi_pdu_t *pdu;

	pdu = &cmd->pdu_list[r2t->pdu_start];
	
	/*
	 * For Normal R2Ts, increment the pdu_send_order which denotes
	 * which iscsi_pdu_t in the sequence to send next.
	 */
	for (i = 0; i < r2t->pdu_count; i++) {
		if (r2t->pdu_send_order == pdu[i].pdu_send_order) {
			r2t->pdu_send_order++;
			return(&pdu[i]);
		}
	}

	TRACE_ERROR("Unable to locate iscsi_pdu_t for r2t->pdu_send_order:"
			" %d.\n", r2t->pdu_send_order);
	
	return(NULL);
}

/*	iscsi_get_r2t_for_cmd():
 *
 *
 */
extern iscsi_r2t_t *iscsi_get_r2t_for_cmd (
	iscsi_cmd_t *cmd)
{
	if (!cmd->r2t_head) {
		TRACE_ERROR("Command ITT: 0x%08x r2t_head is NULL!\n",
				cmd->init_task_tag);
		return(NULL);
	}
		
	return(cmd->r2t_head);
}

/*	iscsi_set_pdu_values_for_r2t():
 *
 *	Sets up r2t->pdu_start, r2t->pdu_count, r2t->pdu_send_count values in the
 *	passed iscsi_r2t_t for sending Normal and Recovery DataOUT sequences.
 *	Only called when DataPDUInOrder=No.
 */
extern int iscsi_set_pdu_values_for_r2t (
	iscsi_cmd_t *cmd,
	iscsi_r2t_t *r2t)
{
	int i, pdu_start_found = 0, seq_pdu_count = 0;
	u32 seq_no = 0, xfer_len = 0;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_pdu_t *pdu = NULL;
	
	/*
	 * Determine the start of the sequence in cmd->pdu_list,
	 * and number of PDUs starting from the offset to the end
	 * of sequence.
	 */
	if (SESS_OPS_C(conn)->DataSequenceInOrder) {
		pdu = &cmd->pdu_list[0];
		/*
		 * Always do paranoid checks for DataSequenceInOrder=Yes
		 * and DataPDUInOrder=No.
		 */
		for (i = 0; i < cmd->pdu_count; i++) {
			if (pdu_start_found) {
				if (pdu[i].seq_no != seq_no)
					break;
				seq_pdu_count++;
				continue;
			}
			if (pdu[i].offset == r2t->offset) {
				pdu_start_found = 1;
				seq_pdu_count = 1;
				seq_no = pdu[i].seq_no;
				r2t->pdu_start = i;
			}
		}
	} else {
		iscsi_seq_t *seq = r2t->seq;
		/*
		 * Normal R2T, skip the paranoid checks.
		 */
		if ((r2t->offset == seq->offset) &&
		    (r2t->xfer_length == seq->xfer_len)) {
			r2t->pdu_start = seq->pdu_start;
			r2t->pdu_count = r2t->pdu_send_count = seq->pdu_count;
			goto out;
		}
		/*
		 * Recovery R2T, do the paranoid checks.
		 */
		pdu = &cmd->pdu_list[seq->pdu_start];
		for (i = 0; i < seq->pdu_count; i++) {
			if (pdu_start_found) {
				if (pdu[i].seq_no != seq_no)
					break;
				seq_pdu_count++;
				continue;
			}
			if (pdu[i].offset == r2t->offset) {
				pdu_start_found = 1;
				seq_pdu_count = 1;
				seq_no = pdu[i].seq_no;
				r2t->pdu_start = (seq->pdu_start + i);
			}		
		}
	}

	if (!pdu_start_found) {
		TRACE_ERROR("Unable to locate sequence in PDU list for ITT:"
			" 0x%08x, Offset: %u, Xfer Length: %u.\n",
			cmd->init_task_tag, r2t->offset, r2t->xfer_length);
		return(-1);
	}
	
	/*
	 * Determine how many PDUs are in the recovery DataOUT sequence.
	 */
	pdu = &cmd->pdu_list[r2t->pdu_start];
	for (i = 0; i < seq_pdu_count; i++) {
		r2t->pdu_count = r2t->pdu_send_count += 1;
		xfer_len += pdu[i].length;

		if (xfer_len == r2t->xfer_length)
			break;
	}

	if (xfer_len != r2t->xfer_length) {
		TRACE_ERROR("Unable to generate recovery R2T sequence for ITT:"
			" 0x%08x, Offset: %u, Xfer Length: %u.\n",
			cmd->init_task_tag, r2t->offset, r2t->xfer_length);
		return(-1);
	}

out:	
	return(0);
}

/*	iscsi_free_r2t():
 *
 *
 */
extern void iscsi_free_r2t (iscsi_cmd_t *cmd, iscsi_r2t_t *r2t)
{
	spin_lock(&cmd->r2t_lock);
	REMOVE_ENTRY_FROM_LIST(r2t, cmd->r2t_head, cmd->r2t_tail);
	spin_unlock(&cmd->r2t_lock);

	kfree(r2t);

	return;
}

/*	iscsi_get_session_from_sid();
 *
 *	Search through sessions looking for a matching PyX SID.
 *
 *	Parameters:	PyX SID.
 *	Returns:	Pointer to session struct if found, else NULL.
 */
extern iscsi_session_t *iscsi_get_session_from_sid (u32 sid)
{
	iscsi_session_t *sess;

	spin_lock(&iscsi_global->session_lock);
	for (sess = iscsi_global->sess_head; sess; sess = sess->next) {
		if (sess->sid == sid) {
			iscsi_inc_session_usage_count(sess);
			break;
		}
	}
	spin_unlock(&iscsi_global->session_lock);

	if (!sess) {
		TRACE_ERROR("Unable to locate PyX SID: %u, ignoring"
			" request.\n", sid);
		return(NULL);
	}

	return(sess);
}

/*	iscsi_check_session_usage_count():
 *
 *
 */
extern void iscsi_check_session_usage_count (iscsi_session_t *sess)
{
	spin_lock_bh(&sess->session_usage_lock);
	if (atomic_read(&sess->session_usage_count)) {
		atomic_set(&sess->session_waiting_on_uc, 1);
		spin_unlock_bh(&sess->session_usage_lock);
		down(&sess->session_waiting_on_uc_sem);
		return;
	}	
	spin_unlock_bh(&sess->session_usage_lock);

	return;
}

/*	__iscsi_dec_session_usage_count():
 *
 *
 */
extern void __iscsi_dec_session_usage_count (iscsi_session_t *sess)
{
	spin_lock(&sess->session_usage_lock);
	atomic_dec(&sess->session_usage_count);

	if (!atomic_read(&sess->session_usage_count) &&
	     atomic_read(&sess->session_waiting_on_uc))
		up(&sess->session_waiting_on_uc_sem);

	spin_unlock(&sess->session_usage_lock);

	return;
}

/*	__iscsi_inc_session_usage_count():
 *
 *
 */
extern void __iscsi_inc_session_usage_count (iscsi_session_t *sess)
{
	spin_lock(&sess->session_usage_lock);
	atomic_inc(&sess->session_usage_count);
	spin_unlock(&sess->session_usage_lock);

	return;
}

/*	iscsi_dec_session_usage_count():
 *
 *
 */
extern void iscsi_dec_session_usage_count (iscsi_session_t *sess)
{
	spin_lock_bh(&sess->session_usage_lock);
	atomic_dec(&sess->session_usage_count);
	if (!atomic_read(&sess->session_usage_count) &&
	     atomic_read(&sess->session_waiting_on_uc))
		up(&sess->session_waiting_on_uc_sem);

	spin_unlock_bh(&sess->session_usage_lock);
	
	return;
}

/*	iscsi_inc_session_usage_count():
 *
 *
 */
extern void iscsi_inc_session_usage_count (iscsi_session_t *sess)
{
	spin_lock_bh(&sess->session_usage_lock);
	atomic_inc(&sess->session_usage_count);
	spin_unlock_bh(&sess->session_usage_lock);

	return;
}

/*      iscsi_ntoa():
 *
 * 
 */
extern char *iscsi_ntoa (u32 ip)
{
	static char buf[18];

	memset((void *) buf, 0, 18);
	sprintf(buf, "%u.%u.%u.%u", ((ip >> 24) & 0xff), ((ip >> 16) & 0xff),
		       	((ip >> 8) & 0xff), (ip & 0xff));

	return(buf);
}

extern void iscsi_ntoa2 (unsigned char *buf, __u32 ip)
{
	memset((void *) buf, 0, 18);
	sprintf(buf, "%u.%u.%u.%u", ((ip >> 24) & 0xff), ((ip >> 16) & 0xff),
			((ip >> 8) & 0xff), (ip & 0xff));

	return;
}

/*	iscsi_put_lun():
 *	
 *	Assume flat space LUN addressing model.
 */
extern u64 iscsi_put_lun (unsigned int lun)
{
	u64 ret;

	ret = ((lun & 0xff) << 8);
	ret |= 0x00 | ((lun >> 8) & 0x3f);

	return(cpu_to_le64(ret));
}

/*	iscsi_get_lun():
 *
 * 	Assume flat space LUN addressing model.
 */
extern u32 iscsi_get_lun (unsigned char *buf)
{
	u32 ret;

	ret = *(buf+1);
	ret += ((*buf) & 0x3f) << 8;

	return(ret);
}

/*	print_sess_params():
 *
 *	Print parameter list for a given session.
 *
 *	Parameters:	iSCSI Session pointer.
 *	Returns:	0 on success, -1 on error.
 */
extern void iscsi_print_sess_params (iscsi_session_t *sess)
{
	iscsi_conn_t *conn;

	printk("-----------------------------[Session Params for SID: %u]"
				"-----------------------------\n", sess->sid);
	spin_lock_bh(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next) {
		iscsi_dump_conn_ops(conn->conn_ops);	
	}
	spin_unlock_bh(&sess->conn_lock);

	iscsi_dump_sess_ops(sess->sess_ops);
}

/*	iscsi_handle_netif_timeou():
 *
 *
 */
static void iscsi_handle_netif_timeout (
	unsigned long data)
{
	iscsi_conn_t *conn = (iscsi_conn_t *) data;
	
	iscsi_inc_conn_usage_count(conn);
		
	spin_lock_bh(&conn->netif_lock);
	if (conn->netif_timer_flags & NETIF_TF_STOP) {
		TRACE_ERROR("netif_timer_flags: 0x%02x\n", conn->netif_timer_flags);
		spin_unlock_bh(&conn->netif_lock);
		iscsi_dec_conn_usage_count(conn);
		return;
	}
	conn->netif_timer_flags &= ~NETIF_TF_RUNNING;
	
	if (iscsi_check_for_active_network_device((void *)conn)) {
		iscsi_start_netif_timer(conn);
		spin_unlock_bh(&conn->netif_lock);
		iscsi_dec_conn_usage_count(conn);
		return;
	}
	
	TRACE_ERROR("Detected PHY loss on Network Interface: %s for iSCSI"
		" CID: %hu on SID: %u\n", conn->net_dev, conn->cid,
			SESS(conn)->sid);
	
	spin_unlock_bh(&conn->netif_lock);
	
	iscsi_cause_connection_reinstatement(conn, 0);	
	iscsi_dec_conn_usage_count(conn);

	return;
}

/*	iscsi_start_netif_timer():
 *
 *	Called with conn->netif_lock held.
 */
extern void iscsi_start_netif_timer (
	iscsi_conn_t *conn)
{
	iscsi_channel_t *c;
	iscsi_session_t *sess = SESS(conn);
	
	if (!conn->net_if)
		return;

	if (conn->netif_timer_flags & NETIF_TF_RUNNING)
		return;
	
	c = sess->channel;
	
	init_timer(&conn->transport_timer);
	conn->transport_timer.expires = (jiffies + ISCSI_CA(c)->netif_timeout * HZ);
	conn->transport_timer.data = (unsigned long) conn;
	conn->transport_timer.function = iscsi_handle_netif_timeout;

	conn->netif_timer_flags |= NETIF_TF_RUNNING;
	add_timer(&conn->transport_timer);

	return;
}

/*	iscsi_stop_netif_timer():
 *
 *
 */
extern void iscsi_stop_netif_timer (
	iscsi_conn_t *conn)
{	
	spin_lock_bh(&conn->netif_lock);
	if (!(conn->netif_timer_flags & NETIF_TF_RUNNING)) {
		spin_unlock_bh(&conn->netif_lock);
		return;
	}
	conn->netif_timer_flags |= NETIF_TF_STOP;
	spin_unlock_bh(&conn->netif_lock);
		
	del_timer_sync(&conn->transport_timer);

	spin_lock_bh(&conn->netif_lock);
	conn->netif_timer_flags &= ~NETIF_TF_RUNNING;
	conn->netif_timer_flags &= ~NETIF_TF_STOP;
	spin_unlock_bh(&conn->netif_lock);
	
	return;
}

/*	iscsi_handle_nopout_response_timeout():
 *
 *
 */
static void iscsi_handle_nopout_response_timeout (
	unsigned long data)
{
	iscsi_conn_t *conn = (iscsi_conn_t *)data;
	iscsi_channel_t *channel;
	
	iscsi_inc_conn_usage_count(conn);

	spin_lock_bh(&conn->nopout_timer_lock);
	if (conn->nopout_response_timer_flags & NOPOUT_RESPONSE_TF_STOP) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		iscsi_dec_conn_usage_count(conn);
		return;
	}

	channel = SESS(conn)->channel;
	
	printk("iCHANNEL[%d] - No response to NOPOUT on CID: %hu on"
		" SID: %u, failing connection\n", channel->channel_id, conn->cid,
			SESS(conn)->sid);

	conn->nopout_response_timer_flags &= ~NOPOUT_RESPONSE_TF_RUNNING;
	spin_unlock_bh(&conn->nopout_timer_lock);
	
	iscsi_cause_connection_reinstatement(conn, 0);
	iscsi_dec_conn_usage_count(conn);

	return;
}

/*	iscsi_mod_nopout_response_timer():
 *
 *
 */
extern void iscsi_mod_nopout_response_timer (
	iscsi_conn_t *conn)
{
	iscsi_session_t *sess = SESS(conn);
	iscsi_channel_t *channel = sess->channel;
	
	spin_lock_bh(&conn->nopout_timer_lock);
	if (!(conn->nopout_response_timer_flags & NOPOUT_RESPONSE_TF_RUNNING)) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		return;
	}

	mod_timer(&conn->nopout_response_timer,
		jiffies + ISCSI_CA(channel)->nopout_response_timeout * HZ);
	spin_unlock_bh(&conn->nopout_timer_lock);

	return;
}

/*	iscsi_start_nopout_response_timer():
 *
 *
 */
extern void iscsi_start_nopout_response_timer (
	iscsi_conn_t *conn)
{
	iscsi_session_t *sess = SESS(conn);
	iscsi_channel_t *channel = sess->channel;
	
	spin_lock_bh(&conn->nopout_timer_lock);
	if (conn->nopout_response_timer_flags & NOPOUT_RESPONSE_TF_RUNNING) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		return;
	}
	
	init_timer(&conn->nopout_response_timer);
	conn->nopout_response_timer.expires = (jiffies + ISCSI_CA(channel)->nopout_response_timeout * HZ);
	conn->nopout_response_timer.data = (unsigned long) conn;
	conn->nopout_response_timer.function = iscsi_handle_nopout_response_timeout;
	
	conn->nopout_response_timer_flags |= NOPOUT_RESPONSE_TF_RUNNING;
	add_timer(&conn->nopout_response_timer);

	TRACE(TRACE_TIMER, "Started NOPOUT Response Timer on CID: %d to %u"
			" seconds\n", conn->cid,
		ISCSI_CA(channel)->nopout_response_timeout);
	spin_unlock_bh(&conn->nopout_timer_lock);

	return;
}

/*	iscsi_stop_nopout_response_timer():
 *
 *
 */
extern void iscsi_stop_nopout_response_timer (
	iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->nopout_timer_lock);
	if (!(conn->nopout_response_timer_flags & NOPOUT_RESPONSE_TF_RUNNING)) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		return;
	}
	conn->nopout_response_timer_flags |= NOPOUT_RESPONSE_TF_STOP;
	spin_unlock_bh(&conn->nopout_timer_lock);
	
	del_timer_sync(&conn->nopout_response_timer);

	spin_lock_bh(&conn->nopout_timer_lock); 
	conn->nopout_response_timer_flags &= ~NOPOUT_RESPONSE_TF_RUNNING;
	conn->nopout_response_timer_flags &= ~NOPOUT_RESPONSE_TF_STOP;
	spin_unlock_bh(&conn->nopout_timer_lock);

	return;
}

/*	iscsi_handle_nopout_timeout():
 *
 *	Call iscsi_add_nopout_noqueue() instead of iscsi_add_nopout() here
 *	to account for Bottom Half Locking Issues and let the connection's
 *	tx thread handle allocating and building the NopOut's iscsi_cmd_t.
 */
static void iscsi_handle_nopout_timeout (
	unsigned long data)
{
	iscsi_conn_t *conn = (iscsi_conn_t *) data;

	iscsi_inc_conn_usage_count(conn);
		
	spin_lock_bh(&conn->nopout_timer_lock);
	if (conn->nopout_timer_flags & NOPOUT_TF_STOP) {	
		spin_unlock_bh(&conn->nopout_timer_lock);
		iscsi_dec_conn_usage_count(conn);
		return;
	}
	
	conn->nopout_timer_flags &= ~NOPOUT_TF_RUNNING;
	spin_unlock_bh(&conn->nopout_timer_lock);
	
	iscsi_add_nopout_noqueue(conn);
	iscsi_dec_conn_usage_count(conn);

	return;
}

/*	iscsi_start_nopout_timer():
 *
 *
 */
extern void iscsi_start_nopout_timer (
	iscsi_conn_t *conn)
{
	iscsi_session_t *sess = SESS(conn);
	iscsi_channel_t *channel = sess->channel;
	
	spin_lock_bh(&conn->nopout_timer_lock);
	if (conn->nopout_timer_flags & NOPOUT_TF_RUNNING) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		return;
	}
	
	init_timer(&conn->nopout_timer);
	conn->nopout_timer.expires = (jiffies + ISCSI_CA(channel)->nopout_timeout * HZ);
	conn->nopout_timer.data = (unsigned long) conn;
	conn->nopout_timer.function = iscsi_handle_nopout_timeout;

	conn->nopout_timer_flags |= NOPOUT_TF_RUNNING;
	add_timer(&conn->nopout_timer);

	TRACE(TRACE_TIMER, "Started NOPOUT Timer on CID: %d at %u second"
		" interval\n", conn->cid, ISCSI_CA(channel)->nopout_timeout);
	spin_unlock_bh(&conn->nopout_timer_lock);

	return;
}

/*	iscsi_stop_nopout_timer():
 *
 *
 */
extern void iscsi_stop_nopout_timer (
	iscsi_conn_t *conn)
{
	spin_lock_bh(&conn->nopout_timer_lock);
	if (!(conn->nopout_timer_flags & NOPOUT_TF_RUNNING)) {
		spin_unlock_bh(&conn->nopout_timer_lock);
		return;
	}
	conn->nopout_timer_flags |= NOPOUT_TF_STOP;
	spin_unlock_bh(&conn->nopout_timer_lock);
	
	del_timer_sync(&conn->nopout_timer);

	spin_lock_bh(&conn->nopout_timer_lock);
	conn->nopout_timer_flags &= ~NOPOUT_TF_RUNNING;
	conn->nopout_timer_flags &= ~NOPOUT_TF_STOP;
	spin_unlock_bh(&conn->nopout_timer_lock);

	return;
}

/*	iscsi_cmd_state_check():
 *
 *	Called with cmd->state_lock held.
 */
extern void iscsi_cmd_state_check (
	iscsi_cmd_t *cmd)
{
	if (cmd->cmd_flags & ICF_CMD_STATE_WAIT_SEM)
		up(&cmd->cmd_state_wait_sem);

	return;
}

/*	iscsi_cmd_state_wait():
 *
 *
 */
extern int iscsi_cmd_state_wait (
	iscsi_cmd_t *cmd)
{
	int got_signal = 0;
	
	spin_lock_bh(&cmd->state_lock);
	if (cmd->cmd_flags & ICF_CMD_STATE_ACTIVE) {
		cmd->cmd_flags |= ICF_CMD_STATE_WAIT_SEM;
		spin_unlock_bh(&cmd->state_lock);
#if 0		
		printk("Caught iscsi_cmd_t ITT: 0x%08x with"
			" ICF_CMD_STATE_ACTIVE\n", cmd->init_task_tag);
#endif
		down_interruptible(&cmd->cmd_state_wait_sem);
		if (signal_pending(current))
			got_signal = 1;
		
		spin_lock_bh(&cmd->state_lock);
		cmd->cmd_flags &= ~ICF_CMD_STATE_WAIT_SEM;
	}
	spin_unlock_bh(&cmd->state_lock);

	return(got_signal);
} 

/*	iscsi_rx_data_payload():
 *
 *
 */
extern unsigned char *iscsi_rx_data_payload (
	iscsi_conn_t *conn,
	u32 payload_len)
{
	char data_crc_failed = 0;
	unsigned char *payload;
	int niov = 0, rx_size;
	u32 checksum, data_crc, padding, pad_bytes = 0;
	struct iovec iov[3];
	
	memset((void *)&iov[0], 0, 3 * sizeof(struct iovec));
		
	if (!(payload = (unsigned char *) kmalloc(payload_len, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for payload\n");
		return(NULL);
	}
	memset(payload, 0, payload_len);
	
	iov[niov].iov_base = payload;
	iov[niov++].iov_len = payload_len;
	rx_size = payload_len;

	if ((padding = ((-payload_len) & 3)) != 0) {
		TRACE(TRACE_ISCSI, "Adding %u padding bytes for payload\n", padding);
		iov[niov].iov_base = &pad_bytes;
		iov[niov++].iov_len = padding;
		rx_size += padding;
	}

	if (CONN_OPS(conn)->DataDigest) {
		iov[niov].iov_base = &checksum;
		iov[niov++].iov_len = CRC_LEN;
		rx_size += CRC_LEN;
	}

	if (rx_data(conn, &iov[0], niov, rx_size) != rx_size) {
		kfree(payload);
		return(NULL);
	}
		
	if (CONN_OPS(conn)->DataDigest) {
#ifdef CRYPTO_API_CRC32C
		crypto_digest_init(conn->conn_rx_tfm);
		ISCSI_CRC32C_RX_DATA_RX_PATH(payload, payload_len, conn);
#else
		int reset_crc = 1;

		crc32c((u8 *)payload, payload_len,
			reset_crc, &data_crc);
		reset_crc = 0;
#endif
		if (padding) {
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_RX_DATA_RX_PATH(&pad_bytes, padding, conn);
#else
			crc32c((u8 *)&pad_bytes, padding, reset_crc, &data_crc);
#endif
		}
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_RX_DATA_RX_PATH_FINAL(data_crc, conn);
#endif
		if (checksum != data_crc) {
			TRACE_ERROR("CRC32C DataDigest: 0x%08x does not match"
				" received 0x%08x for data payload\n", data_crc,
					checksum);
			data_crc_failed = 1;
			kfree(payload);
		} else {
			TRACE(TRACE_DIGEST, "Got CRC32C DataDigest 0x%08x for"
				" %u bytes of payload\n", data_crc,
					payload_len + padding);
		}
	}

	return((data_crc_failed) ? NULL : payload);
}

/*	iscsi_send_tx_data():
 *
 *
 */
extern int iscsi_send_tx_data (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	int use_misc)
{
	int iov_count, size, sent;
	struct iovec *iov;
	
	if (cmd->tx_size <= 0)
		return(0);

send_data:
	size = cmd->tx_size;

	if (cmd->scsi_cmnd && !use_misc) {
		iov = &cmd->iov_data[0];
		iov_count = cmd->iov_data_count;
	} else {
		iov = &cmd->iov_misc[0];
		iov_count = cmd->iov_misc_count;
	}
	
	sent = tx_data(conn, iov, iov_count, size);
	if (size != sent) {
		if (sent == -EAGAIN) {
			TRACE_ERROR("tx_data() returned -EAGAIN\n");
			goto send_data;
		} else
			return(-1);
	}

	cmd->tx_size = 0;

	return(0);
}

/*	iscsi_do_rx_data():
 *
 *
 */
static inline int iscsi_do_rx_data (
	iscsi_conn_t *conn,
	iscsi_data_count_t *count)
{
	int data = count->data_length, rx_loop = 0, total_rx = 0;
	u32 rx_marker_val[count->ss_marker_count], rx_marker_iov = 0;
	struct iovec iov[count->ss_iov_count];
	mm_segment_t oldfs;
	struct msghdr msg;

	memset(&msg, 0, sizeof(struct msghdr));
	msg.msg_name = &conn->s_addr;
	msg.msg_namelen = sizeof(struct sockaddr_in);
	
	if (count->sync_and_steering) {
		int size = 0;
		u32 i, orig_iov_count = 0;
		u32 orig_iov_len = 0, orig_iov_loc = 0;
		u32 iov_count = 0, per_iov_bytes = 0;
		u32 *rx_marker, old_rx_marker = 0;
		struct iovec *iov_record;
		
		memset((void *)&rx_marker_val, 0, count->ss_marker_count * sizeof(u32));
		memset((void *)&iov, 0, count->ss_iov_count * sizeof(struct iovec));
		
		iov_record = count->iov;
		orig_iov_count = count->iov_count;
		rx_marker = &conn->if_marker;

		i = 0;
		size = data;
		orig_iov_len = iov_record[orig_iov_loc].iov_len;
		while (size > 0) {
			if (orig_iov_len >= *rx_marker) {
				iov[iov_count].iov_len = *rx_marker;
				iov[iov_count++].iov_base = 
					(iov_record[orig_iov_loc].iov_base +
					 	per_iov_bytes);

				iov[iov_count].iov_len = (MARKER_SIZE / 2);
				iov[iov_count++].iov_base = 
					&rx_marker_val[rx_marker_iov++];
				iov[iov_count].iov_len = (MARKER_SIZE / 2);
				iov[iov_count++].iov_base =
					&rx_marker_val[rx_marker_iov++];
				old_rx_marker = *rx_marker;
				
				/*
				 * IFMarkInt is in 32-bit words.
				 */
				*rx_marker = (CONN_OPS(conn)->IFMarkInt * 4);
				size -= old_rx_marker;
				orig_iov_len -= old_rx_marker;
				per_iov_bytes += old_rx_marker;
			} else {
				iov[iov_count].iov_len = orig_iov_len;
				iov[iov_count++].iov_base =
					(iov_record[orig_iov_loc].iov_base +
					 	per_iov_bytes);

				per_iov_bytes = 0;
				*rx_marker -= orig_iov_len;
				size -= orig_iov_len;

				if (size)
					orig_iov_len =
					iov_record[++orig_iov_loc].iov_len;
			}
		}
		data += (rx_marker_iov * (MARKER_SIZE / 2));

		msg.msg_iov	= &iov[0];
		msg.msg_iovlen	= iov_count;

		if (iov_count > count->ss_iov_count) {
			TRACE_ERROR("iov_count: %d, count->ss_iov_count: %d\n",
				iov_count, count->ss_iov_count);
			return(-1);
		}
		if (rx_marker_iov > count->ss_marker_count) {
			TRACE_ERROR("rx_marker_iov: %d, count->ss_marker_count: %d\n",
				rx_marker_iov, count->ss_marker_count);
			return(-1);
		}
	} else {
		msg.msg_iov	= count->iov;
		msg.msg_iovlen	= count->iov_count;
	}
		
	while (total_rx < data) {
		oldfs = get_fs();
		set_fs(get_ds());

		conn->sock->sk->sk_allocation = GFP_ATOMIC;
		rx_loop = sock_recvmsg(conn->sock, &msg, (data - total_rx),
					MSG_WAITALL);
		set_fs(oldfs);
		if (rx_loop <= 0) {
			return(rx_loop);
		}

		total_rx += rx_loop;
	}

	if (count->sync_and_steering) {
		int j;
		for (j = 0; j < rx_marker_iov; j++) {
			conn->if_marker_offset = rx_marker_val[j];
		}
		total_rx -= (rx_marker_iov * (MARKER_SIZE / 2));
	}

	return(total_rx);
}

/*	tx_data()
 *
 *	
 */
static inline int iscsi_do_tx_data (
	iscsi_conn_t *conn,     
	iscsi_data_count_t *count)      
{
	int data = count->data_length, total_tx = 0, tx_loop = 0;
	u32 tx_marker_val[count->ss_marker_count], tx_marker_iov = 0;
	struct iovec iov[count->ss_iov_count];
	mm_segment_t oldfs;
	struct msghdr msg;

	memset(&msg, 0, sizeof(struct msghdr));
	msg.msg_name = &conn->s_addr;
	msg.msg_namelen = sizeof(struct sockaddr_in);

	if (count->sync_and_steering) {
		int size = 0;
		u32 i, orig_iov_count = 0;
		u32 orig_iov_len = 0, orig_iov_loc = 0;
		u32 iov_count = 0, per_iov_bytes = 0;
		u32 *tx_marker, old_tx_marker = 0;
		struct iovec *iov_record;

		memset((void *)&tx_marker_val, 0, count->ss_marker_count * sizeof(u32));
		memset((void *)&iov, 0, count->ss_iov_count * sizeof(struct iovec));
		
		iov_record = count->iov;
		orig_iov_count = count->iov_count;
		tx_marker = &conn->of_marker;

		i = 0;
		size = data;
		orig_iov_len = iov_record[orig_iov_loc].iov_len;
		while (size > 0) {
			if (orig_iov_len >= *tx_marker) {
				iov[iov_count].iov_len = *tx_marker;
				iov[iov_count++].iov_base =
					(iov_record[orig_iov_loc].iov_base +
					 	per_iov_bytes); 

				tx_marker_val[tx_marker_iov] =
						(size - *tx_marker);
				iov[iov_count].iov_len = (MARKER_SIZE / 2);
				iov[iov_count++].iov_base =
					&tx_marker_val[tx_marker_iov++];
				iov[iov_count].iov_len = (MARKER_SIZE / 2);
				iov[iov_count++].iov_base =
					&tx_marker_val[tx_marker_iov++];
				old_tx_marker = *tx_marker;
				
				/*
				 * OFMarkInt is in 32-bit words.
				 */
				*tx_marker = (CONN_OPS(conn)->OFMarkInt * 4);
				size -= old_tx_marker;
				orig_iov_len -= old_tx_marker;
				per_iov_bytes += old_tx_marker;
			} else {
				iov[iov_count].iov_len = orig_iov_len;
				iov[iov_count++].iov_base =
					(iov_record[orig_iov_loc].iov_base +
					 	per_iov_bytes);		

				per_iov_bytes = 0;
				*tx_marker -= orig_iov_len;
				size -= orig_iov_len;			

				if (size)
					orig_iov_len =
					iov_record[++orig_iov_loc].iov_len;
			}
		}
		data += (tx_marker_iov * (MARKER_SIZE / 2));

		msg.msg_iov	= &iov[0];
		msg.msg_iovlen	= iov_count;

		if (iov_count > count->ss_iov_count) {
			TRACE_ERROR("iov_count: %d, count->ss_iov_count: %d\n",
				iov_count, count->ss_iov_count);
			return(-1);
		}
		if (tx_marker_iov > count->ss_marker_count) {
			TRACE_ERROR("tx_marker_iov: %d, count->ss_marker_count: %d\n",
				tx_marker_iov, count->ss_marker_count);
			return(-1);
		}
	} else {
		msg.msg_iov	= count->iov;     
		msg.msg_iovlen	= count->iov_count;  
	}
		
	while (total_tx < data) {
		oldfs = get_fs();
		set_fs(get_ds());

		conn->sock->sk->sk_allocation = GFP_ATOMIC;
		tx_loop = sock_sendmsg(conn->sock, &msg, (data - total_tx));

		set_fs(oldfs);

		if (tx_loop <= 0) {
			TRACE_ERROR("tx_loop: %d total_tx %d\n",
				tx_loop, total_tx);
			return(tx_loop);
		}
		total_tx += tx_loop;
	}

	if (count->sync_and_steering)
		total_tx -= (tx_marker_iov * (MARKER_SIZE / 2));
	
	return(total_tx);
}

/*      rx_data():      
 *                      
 *                                      
 */                     
extern int rx_data (            
	iscsi_conn_t *conn,          
	struct iovec *iov,              
	int iov_count,                  
	int data)                       
{
	int ret;	
	iscsi_data_count_t c;                  
		                                        
	if (!conn || !CONN_OPS(conn))
		return(-1);     

	memset(&c, 0, sizeof(iscsi_data_count_t));
	c.iov = iov;
	c.iov_count = iov_count;
	c.data_length = data;   
	c.type = ISCSI_RX_DATA;
	
	if (CONN_OPS(conn)->IFMarker &&
	   (conn->conn_state >= INIT_CONN_STATE_LOGGED_IN)) {
		if (iscsi_determine_sync_and_steering_counts(conn, &c) < 0) {
			return(-1);
		}
	}
		
	ret = iscsi_do_rx_data(conn, &c);
	
	return(ret);
}

/*	tx_data():
 *
 *
 */
extern int tx_data (
	iscsi_conn_t *conn,
	struct iovec *iov,
	int iov_count,
	int data)
{
	int ret;
	iscsi_data_count_t c;

	if (!conn || !CONN_OPS(conn))
		return(-1);
	
	memset(&c, 0, sizeof(iscsi_data_count_t));
	c.iov = iov;
	c.iov_count = iov_count;
	c.data_length = data;
	c.type = ISCSI_TX_DATA;

	if (CONN_OPS(conn)->OFMarker &&
	   (conn->conn_state >= INIT_CONN_STATE_LOGGED_IN)) {
		if (iscsi_determine_sync_and_steering_counts(conn, &c) < 0) {
			return(-1);
		}
	}

	ret = iscsi_do_tx_data(conn, &c);	

	return(ret);
}
