/*
 * Copyright (c) 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses SYSFS related abstractions for exporting iSCSI Initiator Core
 * related data to userspace.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_SYSFS_C

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/in.h>
#include <linux/netdevice.h>
#include <linux/utsname.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_info.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_sysfs.h>

#undef ISCSI_INITIATOR_SYSFS_C

extern iscsi_global_t *iscsi_global;

static void iscsi_initiator_core_classdev_release (struct class_device *cd)
{
        iscsi_channel_t *c = class_dev_to_iscsi_channel(cd);

	put_device(&c->dev);
        complete(&c->class_dev_released);
        return;
}

struct class iscsi_initiator_core_class = {
        .name           = "iscsi_initiator_core",
        .release        = iscsi_initiator_core_classdev_release,
};

struct conn_sysfs_entry {
	struct attribute attr;
	ssize_t (*show)(iscsi_conn_t *, char *);
	ssize_t (*store)(iscsi_conn_t *, const char *, size_t);
};      

#define CONN_ATTR(_name, _mode, _show)							\
struct conn_sysfs_entry conn_attr_##_name = {						\
	.attr = { .name = __stringify(_name), .mode = _mode, .owner = THIS_MODULE },	\
	.show   = _show,								\
};

#define attr_to_conn_sysfs(atr) container_of((atr), struct conn_sysfs_entry, attr)
#define val_obj_to_conn(d) container_of((d), iscsi_conn_t, conn_val_obj)
#define ops_obj_to_conn(d) container_of((d), iscsi_conn_t, conn_ops_obj)

struct sess_sysfs_entry {
	struct attribute attr;
	ssize_t (*show)(iscsi_session_t *, char *);
	ssize_t (*store)(iscsi_session_t *, const char *, size_t);
};      

#define SESS_ATTR(_name, _mode, _show)							\
struct sess_sysfs_entry sess_attr_##_name = {						\
	.attr = { .name = __stringify(_name), .mode = _mode, .owner = THIS_MODULE },	\
	.show   = _show,								\
};

#define attr_to_sess_sysfs(atr) container_of((atr), struct sess_sysfs_entry, attr)
#define val_obj_to_sess(d) container_of((d), iscsi_session_t, sess_val_obj)
#define ops_obj_to_sess(d) container_of((d), iscsi_session_t, sess_ops_obj)

#define ATTR_FUNC(_iscsi_type, _struct_name, _name)					\
static ssize_t _iscsi_type##_attr_##_name##_show (struct kobject *kobj,			\
		struct attribute *attr,	char *buf)					\
{											\
	struct _iscsi_type##_sysfs_entry *entry = attr_to_##_iscsi_type##_sysfs(attr);	\
	_struct_name *var;								\
											\
	var = _name##_obj_to_##_iscsi_type(kobj);					\
	if (!entry->show)								\
		return(0);								\
											\
	return(entry->show(var, buf));							\
}											\
											\
static ssize_t _iscsi_type##_attr_##_name##_store (struct kobject *kobj,		\
		struct attribute *attr,	const char *buf, size_t length)			\
{											\
	struct _iscsi_type##_sysfs_entry *entry = attr_to_##_iscsi_type##_sysfs(attr);	\
	_struct_name *var;								\
											\
	var = _name##_obj_to_##_iscsi_type(kobj);					\
	if (!entry->store)								\
		return(-EINVAL);							\
											\
	return(entry->store(var, buf, length));						\
}											\
											\
static struct attribute *_iscsi_type##_default_##_name##_attrs[] = {			\
	NULL,                                                                           \
};                                                                                      \
											\
static struct sysfs_ops _iscsi_type##_##_name##_sysfs_ops = {				\
	.show	= _iscsi_type##_attr_##_name##_show,					\
	.store	= _iscsi_type##_attr_##_name##_store,					\
};											\
											\
struct kobj_type _iscsi_type##_##_name##_ktype = {					\
	.sysfs_ops      = &_iscsi_type##_##_name##_sysfs_ops,				\
	.default_attrs  = _iscsi_type##_default_##_name##_attrs,			\
};

ATTR_FUNC(conn, iscsi_conn_t, val);
ATTR_FUNC(conn, iscsi_conn_t, ops);
ATTR_FUNC(sess, iscsi_session_t, val);
ATTR_FUNC(sess, iscsi_session_t, ops);

/*
 * SYSFS - CONN ATTRIBUTE - Connection ID (CID)
 */
ssize_t iscsi_conn_cid_show (iscsi_conn_t *conn, char *buf)
{
	return(sprintf(buf, "%hu\n", conn->cid));
}
static CONN_ATTR(cid, S_IRUGO, iscsi_conn_cid_show);

/*
 * SYSFS - CONN ATTRIBUTE - Ipv4 Address
 */
ssize_t iscsi_conn_ipv4_address_show (iscsi_conn_t *conn, char *buf)
{
        return(sprintf(buf, "%s\n", iscsi_ntoa(conn->login_ip)));
}       
static CONN_ATTR(ipv4_address, S_IRUGO, iscsi_conn_ipv4_address_show);

/*
 * SYSFS - CONN ATTRIBUTE - Port
 */
ssize_t iscsi_conn_port_show (iscsi_conn_t *conn, char *buf)
{
	return(sprintf(buf, "%hu\n", conn->login_port));
}
static CONN_ATTR(port, S_IRUGO, iscsi_conn_port_show);

/*
 * SYSFS - CONN ATTRIBUTE - Network Transport
 */
ssize_t iscsi_conn_network_transport_show (iscsi_conn_t *conn, char *buf)
{
	return(sprintf(buf, "%u\n", conn->network_transport));
}
static CONN_ATTR(network_transport, S_IRUGO, iscsi_conn_network_transport_show);

/*
 * SYSFS - CONN ATTRIBUTE - ExpStatSN
 */
ssize_t iscsi_conn_expstatsn_show (iscsi_conn_t *conn, char *buf)
{
	return(sprintf(buf, "0x%08x\n", conn->exp_statsn));
}
static CONN_ATTR(expstatsn, S_IRUGO, iscsi_conn_expstatsn_show);

#define CONN_OPS_DIGEST_ATTR(_name)					\
ssize_t iscsi_conn_##_name##_show (iscsi_conn_t *conn, char *buf)	\
{									\
	switch (conn->conn_ops->_name) {				\
	case 1:								\
		return(sprintf(buf, "CRC32C\n"));			\
	default:							\
		return(sprintf(buf, "None\n"));				\
	}								\
									\
	return(0);							\
}									\
static CONN_ATTR(_name, S_IRUGO, iscsi_conn_##_name##_show);

/*
 * SYSFS - CONN OPS ATTRIBUTE - HeaderDigest
 */
CONN_OPS_DIGEST_ATTR(HeaderDigest);

/*
 * SYSFS - CONN OPS ATTRIBUTE - DataDigest
 */
CONN_OPS_DIGEST_ATTR(DataDigest);

#define CONN_OPS_VALUE_ATTR(_name, _format)				\
ssize_t iscsi_conn_##_name##_show (iscsi_conn_t *conn, char *buf)	\
{									\
	return(sprintf(buf, _format, conn->conn_ops->_name));		\
}									\
static CONN_ATTR(_name, S_IRUGO, iscsi_conn_##_name##_show);

/*
 * SYSFS - CONN OPS ATTRIBUTE - MaxRecvDataSegmentLength
 */
CONN_OPS_VALUE_ATTR(MaxRecvDataSegmentLength, "%u\n");

#define CONN_OPS_BOOL_ATTR(_name)					\
ssize_t iscsi_conn_##_name##_show (iscsi_conn_t *conn, char *buf)	\
{									\
	return(sprintf(buf, "%s\n", (conn->conn_ops->_name) ?		\
			"Yes" : "No"));					\
}									\
static CONN_ATTR(_name, S_IRUGO, iscsi_conn_##_name##_show);

/*
 * SYSFS - CONN OPS ATTRIBUTE - IFMarker
 */
CONN_OPS_BOOL_ATTR(IFMarker);

/*
 * SYSFS - CONN OPS ATTRIBUTE - OFMarker
 */
CONN_OPS_BOOL_ATTR(OFMarker);

/*
 * SYSFS - CONN OPS ATTRIBUTE - IFMarkInt
 */
CONN_OPS_VALUE_ATTR(IFMarkInt, "%hu\n");

/*
 * SYSFS - CONN OPS ATTRIBUTE - OFMarkInt
 */
CONN_OPS_VALUE_ATTR(OFMarkInt, "%hu\n");

/*	iscsi_sysfs_register_conn_attributes():
 *
 *	Called after a successful connection login to register SYSFS for
 *	exporting connection specific attributes to userspace.
 */
extern int iscsi_sysfs_register_conn_attributes(iscsi_conn_t *conn, iscsi_session_t *sess)
{
	kobject_set_name(&conn->conn_val_obj, "conn_%hu", conn->cid);
	conn->conn_val_obj.parent = kobject_get(&sess->sess_val_obj);
	conn->conn_val_obj.ktype = &conn_val_ktype;
	if (kobject_register(&conn->conn_val_obj) < 0) {
		kobject_put(&conn->conn_val_obj);
		return(-1);
	}

	if (sysfs_create_file(&conn->conn_val_obj, &conn_attr_cid.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_val_obj, &conn_attr_ipv4_address.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_val_obj, &conn_attr_port.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_val_obj, &conn_attr_network_transport.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_val_obj, &conn_attr_expstatsn.attr) < 0)
		goto out;

	kobject_set_name(&conn->conn_ops_obj, "conn_ops");
	conn->conn_ops_obj.parent = kobject_get(&conn->conn_val_obj);
	conn->conn_ops_obj.ktype = &conn_ops_ktype;
	if (kobject_register(&conn->conn_ops_obj) < 0)
		goto out;

	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_HeaderDigest.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_DataDigest.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_MaxRecvDataSegmentLength.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_IFMarker.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_OFMarker.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_IFMarkInt.attr) < 0)
		goto out;
	if (sysfs_create_file(&conn->conn_ops_obj, &conn_attr_OFMarkInt.attr) < 0)
		goto out;

	return(0);
out:
	kobject_unregister(&conn->conn_ops_obj);
	kobject_put(&conn->conn_val_obj);
	kobject_unregister(&conn->conn_val_obj);
	kobject_put(&sess->sess_val_obj);
	return(-1);
}

/*	iscsi_sysfs_unregister_conn_attributes():
 *
 *	Called during connection shutdown to unregister SYSFS connection specific
 *	attributes.
 */
extern void iscsi_sysfs_unregister_conn_attributes(iscsi_conn_t *conn, iscsi_session_t *sess)
{
	kobject_unregister(&conn->conn_ops_obj);
	kobject_put(&conn->conn_val_obj);

	kobject_unregister(&conn->conn_val_obj);
	kobject_put(&sess->sess_val_obj);
	return;
}

/*
 * SYSFS - SESSION ATTRIBUTE - ISID
 */
ssize_t iscsi_sess_isid_show (iscsi_session_t *sess, char *buf)
{
	return(sprintf(buf, "0x%02x %02x %02x %02x %02x %02x\n", sess->isid[0],
		sess->isid[1], sess->isid[2], sess->isid[3], sess->isid[4],
		sess->isid[5]));
}
static SESS_ATTR(isid, S_IRUGO, iscsi_sess_isid_show);

/*
 * SYSFS - SESSION ATTRIBUTE - Session ID (SID)
 */
ssize_t iscsi_sess_sid_show (iscsi_session_t *sess, char *buf)
{
	return(sprintf(buf, "%u\n", sess->sid));
}
static SESS_ATTR(sid, S_IRUGO, iscsi_sess_sid_show);

/*
 * SYSFS -  SESSION ATTRIBUTE - TSIH
 */
ssize_t iscsi_sess_tsih_show (iscsi_session_t *sess, char *buf)
{       
	return(sprintf(buf, "%u\n", sess->tsih));
}                       
static SESS_ATTR(tsih, S_IRUGO, iscsi_sess_tsih_show);
        
#define SESS_OPS_VALUE_ATTR(_name, _format)				\
ssize_t iscsi_sess_##_name##_show (iscsi_session_t *sess, char *buf)	\
{									\
        return(sprintf(buf, _format, sess->sess_ops->_name));		\
}									\
static SESS_ATTR(_name, S_IRUGO, iscsi_sess_##_name##_show);

#define SESS_OPS_BOOL_ATTR(_name)					\
ssize_t iscsi_sess_##_name##_show (iscsi_session_t *sess, char *buf)	\
{									\
	return(sprintf(buf, "%s\n", (sess->sess_ops->_name) ?		\
			"Yes" : "No"));					\
}									\
static SESS_ATTR(_name, S_IRUGO, iscsi_sess_##_name##_show);

/*
 * SYSFS - SESSION OPS ATTRIBUTE - InitialR2T
 */
SESS_OPS_BOOL_ATTR(InitialR2T);

/*
 * SYSFS - SESSION OPS ATTRIBUTE - ImmediateData
 */
SESS_OPS_BOOL_ATTR(ImmediateData);

/*
 * SYSFS - SESSION OPS ATTRIBUTE - MaxBurstLength
 */
SESS_OPS_VALUE_ATTR(MaxBurstLength, "%u\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - FirstBurstLength
 */
SESS_OPS_VALUE_ATTR(FirstBurstLength, "%u\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - TargetPortalGroup
 */
SESS_OPS_VALUE_ATTR(TargetPortalGroupTag, "%hu\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - SessionType
 */
ssize_t iscsi_sess_SessionType_show (iscsi_session_t *sess, char *buf)
{
	return(sprintf(buf, "%s\n", (!sess->sess_ops->SessionType) ?
			"Normal" : "Discovery"));
}
static SESS_ATTR(SessionType, S_IRUGO, iscsi_sess_SessionType_show);

/*
 * SYSFS - SESSION OPS ATTRIBUTE - InitiatorName
 */
SESS_OPS_VALUE_ATTR(InitiatorName, "%s\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - InitiatorAlias
 */
SESS_OPS_VALUE_ATTR(InitiatorAlias, "%s\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - TargetName
 */
SESS_OPS_VALUE_ATTR(TargetName, "%s\n");

/*
 * SYSFS - SESSION OPS ATTRIBUTE - TargetAlias
 */
SESS_OPS_VALUE_ATTR(TargetAlias, "%s\n");

/*	iscsi_sysfs_register_session_attributes():
 *
 *
 */
extern int iscsi_sysfs_register_session_attributes (iscsi_session_t *sess, iscsi_channel_t *c)
{
	kobject_set_name(&sess->sess_val_obj, "sess_%u", sess->sid);
	sess->sess_val_obj.parent = kobject_get(&c->ch_class_dev.kobj);
	sess->sess_val_obj.ktype = &sess_val_ktype;
	if (kobject_register(&sess->sess_val_obj) < 0) {
		kobject_put(&c->ch_class_dev.kobj);
		return(-1);
	}
	
	if (sysfs_create_file(&sess->sess_val_obj, &sess_attr_isid.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_val_obj, &sess_attr_sid.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_val_obj, &sess_attr_tsih.attr) < 0)
		goto out;

	kobject_set_name(&sess->sess_ops_obj, "sess_ops");
	sess->sess_ops_obj.parent = kobject_get(&sess->sess_val_obj);
	sess->sess_ops_obj.ktype = &sess_ops_ktype;	
	if (kobject_register(&sess->sess_ops_obj) < 0)
		goto out;
	
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_InitialR2T.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_ImmediateData.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_MaxBurstLength.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_FirstBurstLength.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_TargetPortalGroupTag.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_SessionType.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_InitiatorName.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_InitiatorAlias.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_TargetName.attr) < 0)
		goto out;
	if (sysfs_create_file(&sess->sess_ops_obj, &sess_attr_TargetAlias.attr) < 0)
		goto out;
	
	return(0);
out:
	kobject_unregister(&sess->sess_ops_obj);
	kobject_put(&sess->sess_val_obj);

	kobject_unregister(&sess->sess_val_obj);
	kobject_put(&c->ch_class_dev.kobj);
	return(-1);
}

/*	iscsi_sysfs_unregister_session_attributes():
 *
 *
 */
extern void iscsi_sysfs_unregister_session_attributes (iscsi_session_t *sess, iscsi_channel_t *c)
{
	kobject_unregister(&sess->sess_ops_obj);
	kobject_put(&sess->sess_val_obj);

	kobject_unregister(&sess->sess_val_obj);
	kobject_put(&c->ch_class_dev.kobj);
	return;
}
	
/*
 * SYSFS CLASS DEVICE ATTRIB - Channel ID
 */
ssize_t devattr_show_channel_id (struct class_device *cd, char *buf)
{
	iscsi_channel_t *c = class_dev_to_iscsi_channel(cd);

	return(sprintf(buf, "%d\n", c->channel_id));
}
static CLASS_DEVICE_ATTR(channel_id, S_IRUGO, devattr_show_channel_id, NULL);

/*      
 * SYSFS CLASS DEVICE ATTRIB - iSCSI
 */
ssize_t devattr_show_iscsi (struct class_device *cd, char *buf)
{
        iscsi_channel_t *c = class_dev_to_iscsi_channel(cd);

        return(iscsi_chan_get_iscsi_info(c->channel_id, buf, PAGE_SIZE));
}
static CLASS_DEVICE_ATTR(iscsi, S_IRUGO, devattr_show_iscsi, NULL);

/*
 * SYSFS CLASS DEVICE ATTRIBUTE - SCSI
 */
ssize_t devattr_show_scsi (struct class_device *cd, char *buf)
{
        iscsi_channel_t *c = class_dev_to_iscsi_channel(cd);

        return(iscsi_chan_get_scsi_info(c->channel_id, buf, PAGE_SIZE));
}
static CLASS_DEVICE_ATTR(scsi, S_IRUGO, devattr_show_scsi, NULL);

/*	iscsi_sysfs_register_channel_attributes():
 *
 *
 */
extern int iscsi_sysfs_register_channel_attributes (iscsi_channel_t *c)
{
	class_device_initialize(&c->ch_class_dev);
	c->ch_class_dev.dev = NULL;
	c->ch_class_dev.class = &iscsi_initiator_core_class;
	sprintf(c->ch_class_dev.class_id, "channel_%d", c->channel_id);
	if (class_device_register(&c->ch_class_dev) < 0)
		return(-1);

	if (class_device_create_file(&c->ch_class_dev, &class_device_attr_channel_id) < 0)
		goto out;
	if (class_device_create_file(&c->ch_class_dev, &class_device_attr_iscsi) < 0)
		goto out;
	if (class_device_create_file(&c->ch_class_dev, &class_device_attr_scsi) < 0)
		goto out;

	return(0);
out:
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_channel_id);
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_iscsi);
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_scsi);

	class_device_unregister(&c->ch_class_dev);
	return(-1);
}

/*	iscsi_sysfs_unregister_channel_attributes():
 *
 *
 */
extern void iscsi_sysfs_unregister_channel_attributes (iscsi_channel_t *c)
{
        init_completion(&c->class_dev_released);
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_channel_id);
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_iscsi);
	class_device_remove_file(&c->ch_class_dev, &class_device_attr_scsi);

	class_device_unregister(&c->ch_class_dev);

	wait_for_completion(&c->class_dev_released);

	return;
}

/*
 * SYSFS - CLASS ATTRIBUTE - iSCSI Initiator Node Name
 */
static ssize_t classattr_nodename_show (struct class *c, char *buf)
{
        if (iscsi_global->initname_set)
                return(sprintf(buf, "iSCSI InitiatorName: %s\n",
                        iscsi_global->initiatorname));

        return(sprintf(buf, "*** iSCSI Initiator node name is not set! ***\n"));
}

static ssize_t classattr_nodename_store (struct class *c, const char *buf, size_t count)
{
        return(0);
}

static CLASS_ATTR(initiator_nodename, 0644, classattr_nodename_show, classattr_nodename_store);

/*
 * SYSFS - CLASS ATTRIBUTE - iSCSI Initiator Core Version
 */
ssize_t classattr_version_show (struct class *c, char *buf)
{
        return(sprintf(buf, "Core-iSCSI Initiator Stack "
                        PYX_ISCSI_VERSION" on %s/%s on "UTS_RELEASE"\n",
                        system_utsname.sysname, system_utsname.machine));
}

static CLASS_ATTR(version_info, S_IRUGO, classattr_version_show, NULL);

/*	iscsi_sysfs_register_global_attributes():
 *
 *
 */
extern int iscsi_sysfs_register_global_attributes (void)
{
	int ret;

	if ((ret = class_register(&iscsi_initiator_core_class))) {
		TRACE_ERROR("class_register() failed for iscsi_initiator_core_class: %d\n", ret);
		goto out;
	}

        class_create_file(&iscsi_initiator_core_class, &class_attr_initiator_nodename);
        class_create_file(&iscsi_initiator_core_class, &class_attr_version_info);
	
	return(0);
out:
	class_remove_file(&iscsi_initiator_core_class, &class_attr_initiator_nodename);
	class_remove_file(&iscsi_initiator_core_class, &class_attr_version_info);
	class_unregister(&iscsi_initiator_core_class);

	return(-1);
}

/*	iscsi_sysfs_unregister_global_attributes():
 *
 *
 */
extern void iscsi_sysfs_unregister_global_attributes (void)
{
        class_remove_file(&iscsi_initiator_core_class, &class_attr_version_info);
	class_remove_file(&iscsi_initiator_core_class, &class_attr_initiator_nodename);
	class_unregister(&iscsi_initiator_core_class);

	return;
}
