/*
 * Copyright (c) 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses the LINUX OS dependant code for the iSCSI Initiator Core.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_LINUX_C

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/net.h>
#include <linux/miscdevice.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/in.h>
#include <linux/blkdev.h>
#include <linux/netdevice.h>
#include <linux/utsname.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <scsi.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_transport.h>
#ifdef USE_SCSI_TRANSPORT_ISCSI
#include <scsi/scsi_transport_iscsi.h>
#endif
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_class.h>
#include <iscsi_initiator_info.h>
#include <iscsi_initiator_ioctl_defs.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_lu.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_sysfs.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_parameters.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_linux_scsi_defs.h>

#undef ISCSI_INITIATOR_LINUX_C

extern iscsi_global_t *iscsi_global;
extern struct miscdevice iscsi_dev;

extern int iscsi_initiator_mod_init (void);
extern int iscsi_initiator_mod_fini (void);

extern int scsi_internal_device_block(struct scsi_device *);
extern int scsi_internal_device_unblock(struct scsi_device *);

#ifdef LINUX_64BITKERNEL_32BIT_USER
extern int register_ioctl32_conversion (unsigned int, int (*handler)
		(unsigned int, unsigned int, unsigned long, struct file *));
extern int unregister_ioctl32_conversion (unsigned int);
#endif

/*	iscsi_check_for_active_network_device():
 *
 *
 */
extern int iscsi_check_for_active_network_device (void *conn_ptr)
{
	iscsi_conn_t *conn = (iscsi_conn_t *) conn_ptr;
	struct net_device *net_dev;
	
	if (!conn->net_if) {
		TRACE_ERROR("iscsi_conn_t->net_if is NULL for CID:"
			" %hu\n", conn->cid);
		return(0);
	}
	net_dev = conn->net_if;

	return(netif_carrier_ok(net_dev));
}

/*	iscsi_complete_command():
 *
 *
 */
extern void iscsi_complete_command (struct scsi_cmnd *sc)
{
	unsigned long flags;
	
	spin_lock_irqsave(sc->device->host->host_lock, flags);
	sc->SCp.ptr = NULL;
	(*sc->scsi_done)(sc);
	spin_unlock_irqrestore(sc->device->host->host_lock, flags);

	return;
}

static void scsi_timeout_function (unsigned long arg)
{
	return;
}

/*	iscsi_set_result_from_action_and_complete():
 *
 *
 */
extern void iscsi_set_result_from_action_and_complete (struct scsi_cmnd *sc, u8 action)
{
	unsigned long flags;
	
	spin_lock_irqsave(sc->device->host->host_lock, flags);
	switch (action) {
	case ISCSI_COMMAND_RESET:
		sc->result = HOST_BYTE(DID_RESET);
		break;
	case ISCSI_COMMAND_RETRY:
		sc->result = HOST_BYTE(DID_BUS_BUSY);
		sc->retries--;
		break;
	case ISCSI_COMMAND_FAIL:
		sc->result = HOST_BYTE(DID_NO_CONNECT);
		iscsi_OS_set_SCSI_lun_failure((void *) sc);
		break;
	default:
		TRACE_ERROR("Unknown completion action"
			" 0x%02x!\n", action);
		sc->result = HOST_BYTE(DID_NO_CONNECT);
		iscsi_OS_set_SCSI_lun_failure((void *) sc);
		break;
	}

	/*
	 * This is completion of a given struct scsi_cmnd after an
	 * iSCSI exception occured.  Based upon iSCSI exceptions and/or
	 * passed action parameter, struct scsi_cmnd->eh_timeout may have
	 * been stopped, and needs to be rstarted before completion to
	 * the SCSI stack.
	 */
	if (!(timer_pending(&sc->eh_timeout))) {
		sc->eh_timeout.data = (unsigned long) sc;
		sc->eh_timeout.expires = (jiffies + sc->timeout_per_command);
		sc->eh_timeout.function = scsi_timeout_function;
		add_timer(&sc->eh_timeout);
	}
	
	sc->SCp.ptr = NULL;
	(*sc->scsi_done)(sc);
	spin_unlock_irqrestore(sc->device->host->host_lock, flags);
	
	return;
}

/*	iscsi_get_network_interface_from_socket():
 *
 *
 */
//#warning FIXME: Would be nice to get struct net_device from conn->conn_sock
extern void iscsi_get_network_interface_from_socket (void *conn_ptr)
{
	iscsi_conn_t *conn = (iscsi_conn_t *) conn_ptr;
	iscsi_channel_t *channel = SESS(conn)->channel;
#if 0
	struct sock *sk = conn->sock->sk;
#endif
	struct net_device *net_dev;

#if 0
	if (!(net_dev = __dev_get_by_index(sk->bound_dev_if))) {
		TRACE_ERROR("Unable to locate LINUX network_device for"
			" bound_dev_if: %d\n", sk->bound_dev_if);
		return;
	}
#else
	if (!(net_dev = __dev_get_by_name(conn->net_dev))) {
		printk("iCHANNEL[%d] - Unable to locate active network"
			" interface: %s\n", channel->channel_id,
			strlen((char *)conn->net_dev) ? (char *)conn->net_dev : "None");
		conn->net_if = NULL;
		return;
	}
#endif	
	conn->net_if = net_dev;
		
	return;
}

/*	iscsi_OS_get_SCSI_lu():
 *
 *
 */
extern void *iscsi_OS_get_SCSI_lu (void *OS_sc, int *lun)
{
	struct scsi_cmnd *sc = (struct scsi_cmnd *) OS_sc;

	*lun = sc->device->lun;
	
	return((void *)sc->device);
}

/*	iscsi_OS_check_device_online():
 *
 *
 */
extern int iscsi_OS_check_lu_online (void *OS_sd)
{
	int online = 0;
	struct scsi_device *sd = (struct scsi_device *) OS_sd;
	
	if (sd->sdev_state == SDEV_RUNNING)
		online = 1;
		
	return(online);
}

/*	iscsi_OS_get_SCSI_HOST_ID():
 *
 *
 */
extern int iscsi_OS_get_SCSI_HOST_ID (void *ch)
{
	iscsi_channel_t *c = (iscsi_channel_t *) ch;
	struct Scsi_Host *sh = c->scsi_host;

	if (!sh) {
		TRACE_ERROR("c->scsi_host pointer to SCSI Host is NULL"
			" for iSCSI Channel: %d!\n", c->channel_id);
		BUG();
	}

	printk("iCHANNEL[%d] - Allocated Linux SCSI Host with ID: %d\n",
			c->channel_id, sh->host_no);

	return(sh->host_no);
}

/*	iscsi_OS_get_SAM_lu():
 *
 *
 */
extern void *iscsi_OS_get_SAM_lu (int iscsi_lun, void *ch)
{
	iscsi_channel_t *c = (iscsi_channel_t *) ch;
	struct Scsi_Host *sh = c->scsi_host;

	if (!sh) {
		TRACE_ERROR("Scsi_Host is NULL for iSCSI Channel: %d\n",
				c->channel_id);
		return(NULL);
	}

	return(scsi_device_lookup(sh, c->channel_id, c->target_id, iscsi_lun));
}

/*     linux_scsi_forget_host():
 *
 *     From scsi_scan.c, but not exported. :/
 */
extern void linux_scsi_forget_host(struct Scsi_Host *shost)
{
	struct scsi_device *sdev, *tmp;
	unsigned long flags;

	spin_lock_irqsave(shost->host_lock, flags);
	list_for_each_entry_safe(sdev, tmp, &shost->__devices, siblings) {
		spin_unlock_irqrestore(shost->host_lock, flags);
		scsi_remove_device(sdev);
		spin_lock_irqsave(shost->host_lock, flags);
	}
	spin_unlock_irqrestore(shost->host_lock, flags);

	return;
}

/*	iscsi_scsi_block_devices():
 *
 *
 */
extern void iscsi_scsi_block_devices (iscsi_channel_t *c)
{
	struct scsi_device *sd;
	struct Scsi_Host *sh = c->scsi_host;

	shost_for_each_device(sd, sh)
		scsi_internal_device_block(sd);
	
	return;
}

/*	iscsi_scsi_unblock_devices():
 *
 *
 */
extern void iscsi_scsi_unblock_devices (iscsi_channel_t *c)
{
	struct scsi_device *sd;
	struct Scsi_Host *sh = c->scsi_host;

	shost_for_each_device(sd, sh)
		scsi_internal_device_unblock(sd);

	return;
}

/*	iscsi_scsi_offline_devices():
 *
 *
 */
extern void iscsi_scsi_offline_devices (iscsi_channel_t *c)
{
	struct scsi_device *sd;
	struct Scsi_Host *sh = c->scsi_host;
	
	shost_for_each_device(sd, sh)
		scsi_device_set_state(sd, SDEV_OFFLINE);

	return;
}

/*	iscsi_linux_calculate_map_segment():
 *
 *	Used from iscsi_map_scatterlists().
 */
static inline void iscsi_linux_calculate_map_segment (int do_kmap, u32 *data_length, iscsi_linux_map_t *lm)
{
	u32 sg_offset = 0;
	struct scatterlist *sg = lm->map_sg;
	
	/*
	 * Still working on pages in the current scatterlist.
	 */
	if (!lm->map_reset) {
		lm->iovec_length = (lm->sg_length > PAGE_SIZE) ?
				      PAGE_SIZE : lm->sg_length;

		if (*data_length < lm->iovec_length)
			lm->iovec_length = *data_length;

		lm->iovec_base = ((do_kmap) ? kmap(lm->sg_page) :
					      page_address(lm->sg_page)) +
							sg_offset;
		lm->sg_page++;
		return;
	} else if (lm->sg_last_page == sg->page) {
		lm->sg_last_page = NULL;
		do_kmap = 0;
	}

	/*
	 * First run of an iscsi_linux_map_t.
	 *
	 * OR:
	 * 
	 * Mapped all of the pages in the current scatterlist, move
	 * on to the next one.
	 */
	lm->map_reset = 0;
	sg_offset = sg->offset;
	lm->sg_page = sg->page;
	lm->sg_length = sg->length;

	/*
	 * Get the base and length of the current page for use with the iovec.
	 */
recalc:
	lm->iovec_length = (lm->sg_length > (PAGE_SIZE - sg_offset)) ?
			   (PAGE_SIZE - sg_offset) : lm->sg_length;

	/*
	 * See if there is any iSCSI offset we need to deal with.
	 */
	if (!lm->current_sg_offset) {
		lm->iovec_base = ((do_kmap) ? kmap(lm->sg_page++) :
					      page_address(lm->sg_page++)) +
							sg_offset;
		if (*data_length < lm->iovec_length)
			lm->iovec_length = *data_length;
		
		return;
	}
		
	/*
	 * We know the iSCSI offset is in the next page of the current
	 * scatterlist.  Increase the lm->sg_page pointer and try again.
	 */
	if (lm->current_sg_offset >= lm->iovec_length) {
		lm->current_sg_offset -= lm->iovec_length;
		lm->sg_length -= lm->iovec_length;
		lm->sg_page++;
		sg_offset = 0;
		goto recalc;
	}

	/*
	 * The iSCSI offset is in the current page, increment the iovec
	 * base and redeuce iovec length.
	 */
	lm->iovec_base = (do_kmap) ? kmap(lm->sg_page++) :
				     page_address(lm->sg_page++);
	lm->iovec_base += sg_offset;
	lm->iovec_base += lm->current_sg_offset;
	lm->iovec_length -= lm->current_sg_offset;
	lm->sg_length -= lm->current_sg_offset;
	
	lm->current_sg_offset = 0;

	return;
}

/*	iscsi_linux_unmap_SG_segments():
 *
 *
 */
static inline void iscsi_linux_unmap_SG_segments (u32 *data_length, iscsi_linux_map_t *lm)
{
	struct scatterlist *sg = lm->map_sg;
	u32 sg_offset = sg->offset, sg_page_length = 0;

	if (!lm->map_reset) {
		lm->iovec_length = (lm->sg_length > PAGE_SIZE) ?
				    PAGE_SIZE : lm->sg_length; 
		if (*data_length < lm->iovec_length) 
			lm->iovec_length = *data_length;

		kunmap(lm->sg_page++);
		return;
	} else if (lm->sg_last_page == sg->page) {
		lm->sg_last_page = NULL;
		return;
	}

	lm->map_reset = 0;
	sg_offset = sg->offset;
	lm->sg_page = sg->page;
	lm->sg_length = sg->length;

recalc:
	sg_page_length = (PAGE_SIZE - sg_offset);
	lm->iovec_length = (lm->sg_length > (PAGE_SIZE - sg_offset)) ?
			   (PAGE_SIZE - sg_offset) : lm->sg_length;

	if (!lm->current_sg_offset) {
		kunmap(lm->sg_page++);

		if (*data_length < lm->iovec_length)
			lm->iovec_length = *data_length;
		return;
	}

	if (lm->current_sg_offset >= lm->iovec_length) {
		lm->current_sg_offset -= lm->iovec_length;
		lm->sg_length -= lm->iovec_length;
		lm->sg_page++;
		sg_offset = 0;
		goto recalc;
	}

	kunmap(lm->sg_page++);

	lm->iovec_length -= lm->current_sg_offset;
	lm->sg_length -= lm->current_sg_offset;
	lm->current_sg_offset = 0;

	return;
}

/*	iscsi_unmap_scatterlists():
 *
 *
 */
extern void iscsi_unmap_scatterlists (void *unmap)
{
	u32 j = 0;
	iscsi_unmap_sg_t *unmap_sg = (iscsi_unmap_sg_t *) unmap;
	iscsi_cmd_t *cmd = (iscsi_cmd_t *) unmap_sg->cmd;
	iscsi_linux_map_t *lmap = &unmap_sg->lmap;
	struct scsi_cmnd *sc = (struct scsi_cmnd *) cmd->scsi_cmnd;

	if (!sc->use_sg)
		return;

	lmap->current_sg_offset = lmap->orig_sg_offset;
	lmap->map_sg = lmap->map_orig_sg;

	lmap->current_sg_count = 0;
	lmap->map_reset = 1;
	lmap->sg_page = lmap->sg_last_page = NULL;

	while (unmap_sg->data_length) {
		iscsi_linux_unmap_SG_segments(&unmap_sg->data_length, lmap);

		unmap_sg->data_length -= lmap->iovec_length;
		lmap->sg_length -= lmap->iovec_length;

		if (!lmap->sg_length || !unmap_sg->data_length) {
			lmap->map_sg = &lmap->map_orig_sg[++j];
			lmap->current_sg_count++;

			lmap->sg_last_page = --lmap->sg_page;
			lmap->sg_page = NULL;
			lmap->map_reset = 1;
		}
	}
	
	return;
}

/* 	iscsi_linux_get_iscsi_offset():
 *
 *	Used from iscsi_map_scatterlists()
 */
static inline u32 iscsi_linux_get_iscsi_offset (
	iscsi_linux_map_t *lmap,
	struct scsi_cmnd *sc)
{
	u32 current_sg_length = 0, current_iscsi_offset = lmap->iscsi_offset;
	u32 total_sg_offset = 0;
	struct scatterlist *sg = (struct scatterlist *) sc->request_buffer; 

	/*
	 * Locate the current scatterlist offset from the passed iSCSI Offset.
	 */
	while (lmap->iscsi_offset != current_sg_length) {
		/*
		 * The iSCSI Offset is within the current scatterlist.
		 *
		 * Or:
		 *
		 * The iSCSI Offset is outside of the current scatterlist,
		 * Recalcuate the values and increment the scatterlist
		 * pointer.
		 */
		total_sg_offset += sg[lmap->current_sg_count].length;
		
		if (total_sg_offset > lmap->iscsi_offset) {
			current_sg_length += current_iscsi_offset;
			lmap->orig_sg_offset = lmap->current_sg_offset = current_iscsi_offset;
		} else {
			current_sg_length += sg[lmap->current_sg_count].length;
			current_iscsi_offset -= sg[lmap->current_sg_count++].length;
		}
	}
	lmap->map_orig_sg = sg + lmap->current_sg_count;

	return(lmap->current_sg_count);
}

/*	iscsi_map_scatterlists():
 *
 *	Main engine to map some network iovector pairs to generic scatterlist structures.
 */
extern int iscsi_map_scatterlists (
	void *map,
	void *unmap)
{
	u32 i = 0 /* For iovecs */, j = 0 /* For scatterlists */;
	iscsi_map_sg_t *map_sg = (iscsi_map_sg_t *) map;
	iscsi_unmap_sg_t *unmap_sg = (iscsi_unmap_sg_t *) unmap;
	iscsi_cmd_t *cmd = (iscsi_cmd_t *) map_sg->cmd;
	iscsi_linux_map_t *lmap = &unmap_sg->lmap;
	struct iovec *iov = map_sg->iov;
	int do_kmap = (map_sg->map_flags & MAP_SG_KMAP);

	/*
	 * Set lmap->map_reset = 1 so the first call to iscsi_linux_map_SG_segment
	 * sets up the initial values for iscsi_linux_map_t.
	 */
	lmap->map_reset = 1;

	/*
	 * Get a pointer to the first used scatterlist based on the passed offset.
	 * Also set the rest of the needed values in iscsi_linux_map_t.
	 */
	lmap->iscsi_offset = map_sg->data_offset;
	if (map_sg->map_flags & MAP_SG_KMAP) {
		iscsi_linux_get_iscsi_offset(lmap, cmd->scsi_cmnd);
		unmap_sg->data_length = map_sg->data_length;
	} else {
		lmap->current_sg_offset = lmap->orig_sg_offset;
	}
	lmap->map_sg = lmap->map_orig_sg;

	while (map_sg->data_length) {
		/*
		 * Either kmapping actual pages or just getting virtual address
		 * for resetting iovec pointers.  This function will return the
		 * expected iovec_base address and iovec_length.  It will also
		 * perform any kmaps on pages if needed.
		 */
		 iscsi_linux_calculate_map_segment(do_kmap, &map_sg->data_length, lmap);

		/*
		 * Set the iov.iov_base and iov.iov_len from the current values
		 * in iscsi_linux_map_t.
		 */
		iov[i].iov_base = lmap->iovec_base;
		iov[i].iov_len = lmap->iovec_length;

		/*
		 * Subtract the final iovec length from the total length to be
		 * mapped, and the length of the current scatterlist.  Also
		 * perform the paranoid check to make sure we are not going to
		 * overflow the iovecs allocated for this command in the next
		 * pass.
		 */
		map_sg->data_length -= iov[i].iov_len;
		lmap->sg_length -= iov[i].iov_len;

		if ((++i + 1) > cmd->orig_iov_data_count) {
			TRACE_ERROR("Current iovec count %u is greater than"
				" iscsi_cmd_t->orig_data_iov_count %u, cannot"
				" continue.\n", i+1, cmd->orig_iov_data_count);
			BUG();
		}
		
		/*
		 * All done mapping this scatterlist's pages, move on to
		 * the next scatterlist by setting lmap.map_reset = 1;
		 */
		if (!lmap->sg_length || !map_sg->data_length) {
			lmap->map_sg = &lmap->map_orig_sg[++j];
			lmap->sg_last_page = --lmap->sg_page;
			lmap->current_sg_count++;
			lmap->sg_page = NULL;
			lmap->map_reset = 1;
		}
	}

	/*
	 * If these pages are being kmapped, save the count and offset
	 * for future kunmapping.
	 */
	if (do_kmap)
		unmap_sg->sg_count = j;

	return(i);
}
		
/*	iscsi_OS_set_SCSI_lun_failure():
 *
 *
 */
extern void iscsi_OS_set_SCSI_lun_failure (void *OS_sc)
{
	struct scsi_cmnd *sc = (struct scsi_cmnd *) OS_sc;

	sc->sense_buffer[0] = 0x70;
	sc->sense_buffer[2] = NOT_READY;
	sc->sense_buffer[7] = 0x6;
	sc->sense_buffer[12] = 0x08;
	sc->sense_buffer[13] = 0x00;

	return;
}

/*	iscsi_OS_set_SCSI_ptr():
 *
 *
 */
extern void iscsi_OS_set_SCSI_ptr (void *OS_sc, void *ptr)
{
	struct scsi_cmnd *sc = (struct scsi_cmnd *) OS_sc;

	sc->SCp.ptr = ptr;
	
	return;
}

/*	iscsi_OS_set_SCSI_status():
 *
 *
 */
extern void iscsi_OS_set_SCSI_status (void *OS_sc, int result)
{
	struct scsi_cmnd *sc = (struct scsi_cmnd *) OS_sc;

	sc->result = STATUS_BYTE(result);

	return;
}

/*	iscsi_set_SCSI_tag_attr():
 *
 *	Sets the Task Attribute (see SAM-2) in bits 5-7 of hdr->flags.
 *	See RFC-3720 10.3.1 Flags and Task Attributes.
 */
extern void iscsi_set_SCSI_tag_attr (struct scsi_cmnd *sc, struct iscsi_init_scsi_cmnd *iscsi_cmd)
{
	if (sc->device->tagged_supported) {
		switch (sc->tag) {
		case HEAD_OF_QUEUE_TAG:
			iscsi_cmd->flags |= (ISCSI_HEAD_OF_QUEUE & SAM2_ATTR);
			break;
		case ORDERED_QUEUE_TAG:
			iscsi_cmd->flags |= (ISCSI_ORDERED & SAM2_ATTR);
			break;
		default:
			iscsi_cmd->flags |= (ISCSI_SIMPLE & SAM2_ATTR);
			break;
		}
	} else
		iscsi_cmd->flags |= (ISCSI_UNTAGGED & SAM2_ATTR);

	return;
}

/*	iscsi_linux_init_scsi_host():
 *
 *
 */
static int iscsi_linux_init_scsi_host (void *ch, u32 max_sectors)
{
	int channel_id = 0, ret;
	struct Scsi_Host *sh = NULL;
	iscsi_channel_t *c = (iscsi_channel_t *) ch;
	channel_id = c->channel_id;

	if (!(sh = scsi_host_alloc(&iscsi_template, 0))) {
		TRACE_ERROR("scsi_host_alloc failed\n");
		return(-1);
	}

	snprintf(sh->shost_gendev.bus_id, BUS_ID_SIZE, "iscsi_%d", channel_id);
	snprintf(sh->shost_classdev.class_id, BUS_ID_SIZE, "iscsi_%d", channel_id);

	sh->max_channel		= SCSI_MAX_CHANNELS-1;
	sh->max_id		= SCSI_MAX_TARGET_IDS;
	sh->max_lun		= SCSI_MAX_LUNS;
	sh->max_cmd_len		= SCSI_MAX_CDB_LENGTH;

	sh->hostdata[0]		= (unsigned long) c;
	
	if (max_sectors) {
		sh->max_sectors	= max_sectors;
		printk("iCHANNEL[%d] - Setting max_sectors: %u\n",
			c->channel_id, sh->max_sectors);
	}

	if ((ret = scsi_add_host(sh, NULL))) {
		TRACE_ERROR("scsi_add_host() failed: %d\n", ret);
		scsi_host_put(sh);
		return(-1);
	}

	c->scsi_host = sh;

	return(0);
}

/*	iscsi_OS_init_iscsi_channel():
 *
 *
 */
extern int iscsi_OS_init_iscsi_channel (void *ch, u32 max_sectors)
{
	iscsi_channel_t *c = (iscsi_channel_t *) ch;

	if (iscsi_sysfs_register_channel_attributes(c) < 0)
		return(-1);		

	if (iscsi_linux_init_scsi_host(ch, max_sectors) < 0) {
		iscsi_sysfs_unregister_channel_attributes(c);
		return(-1);
	}
		
	return(0);
}

/*	iscsi_linux_free_scsi_host():
 *
 *
 */
static void iscsi_linux_free_scsi_host (struct Scsi_Host *sh)
{
	scsi_remove_host(sh);
	scsi_host_put(sh);
	return;
}

/*	iscsi_OS_free_scsi_host():
 *
 *
 */
extern void iscsi_OS_free_scsi_host (void *ch)
{
	iscsi_channel_t *c = (iscsi_channel_t *) ch;
	
	if (!c->scsi_host) {
		TRACE_ERROR("Scsi_Host is NULL for iSCSI Channel: %d\n",
				c->channel_id);
		return;
	}
	
	iscsi_sysfs_unregister_channel_attributes(c);

	iscsi_linux_free_scsi_host(c->scsi_host);
	c->scsi_host = NULL;

	return;
}

#ifdef LINUX_64BITKERNEL_32BIT_USER
static void iscsi_linux_unreg_ioctl_conv (void);

/*	iscsi_linux_reg_ioctl_conv():
 *
 *
 */
static int iscsi_linux_reg_ioctl_conv (void)
{
	if (register_ioctl32_conversion(ISCSI_INITIATOR_FULLINIT_CHANNEL, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_INIT_CHANNEL, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_FREE_CHANNEL, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_SET_CHANNEL_ATTRIB, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_STOP_CHANNEL, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LOGIN, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_SCANSCSI, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LOGOUTSESS, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LOGOUTCONN, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LISTCHANNELPARAMS, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LISTSESSPARAMS, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_SETCHANNELPARAM, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_SETSESSPARAM, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_SETINITNAME, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_LISTKNOWNTARGETS, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_ADDCONN, NULL) < 0)
		goto out;
	if (register_ioctl32_conversion(ISCSI_INITIATOR_DEBUG_ERL, NULL) < 0)
		goto out;

	return(0);

out:
	iscsi_linux_unreg_ioctl_conv();
	return(-1);
}

/*	iscsi_linux_unreg_ioctl_conv():
 *
 *
 */
static void iscsi_linux_unreg_ioctl_conv (void)
{
	unregister_ioctl32_conversion(ISCSI_INITIATOR_FULLINIT_CHANNEL);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_INIT_CHANNEL);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_FREE_CHANNEL);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_SET_CHANNEL_ATTRIB);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_STOP_CHANNEL);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LOGIN);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_SCANSCSI);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LOGOUTSESS);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LOGOUTCONN);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LISTCHANNELPARAMS);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LISTSESSPARAMS);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_SETCHANNELPARAM);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_SETSESSPARAM);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_SETINITNAME);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_LISTKNOWNTARGETS);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_ADDCONN);
	unregister_ioctl32_conversion(ISCSI_INITIATOR_DEBUG_ERL);

	return;
}
#endif /* LINUX_64BITKERNEL_32BIT_USER */

/*	iscsi_linux_proc_info(): (Part of Scsi_Host_Template)
 *
 *	Called when a file in /proc/scsi/iscsi_initiator/ directory is read.
 */
static int iscsi_linux_proc_info (struct Scsi_Host *sh, char *buffer, char **start,
				  off_t offset, int length, int inout)
{
	char *p = buffer;
	
	if (inout == 1)
		return(-ENOSYS);

	p += sprintf(p, "Use of Linux SCSI proc interface with iSCSI"
		" Initiator Core Stack is deprecated, please use"
			" /sys/class/iscsi_initiator_core/\n");
	
	return(p-buffer);
}

/*	iscsi_linux_init():
 *
 *	Called directly from init_iscsi_module().
 */
extern int iscsi_linux_init (struct scsi_host_template *sht)
{
#ifdef LINUX_64BITKERNEL_32BIT_USER
	int conv_set = 0;
#endif
	int ret, misc_set = 0;

	if ((ret = misc_register(&iscsi_dev))) {
		TRACE_ERROR("misc_register() failed: %d\n", ret);
		goto out;
	}
	misc_set = 1;
	
#ifdef USE_SCSI_TRANSPORT_ISCSI
	if (!(iscsi_global->iscsi_template = iscsi_attach_transport(&iscsi_fnt)))
		goto out;
#endif
	if (iscsi_sysfs_register_global_attributes() < 0)
		goto out;

#ifdef LINUX_64BITKERNEL_32BIT_USER
	if (iscsi_linux_reg_ioctl_conv() < 0) {
		TRACE_ERROR("iscsi_linux_reg_ioctl_conv() failed\n");
		goto out;
	}
	conv_set = 1;
#endif 

	return(0);
out:
#ifdef LINUX_64BITKERNEL_32BIT_USER
	if (conv_set)
		iscsi_linux_unreg_ioctl_conv();		
#endif
#ifdef USE_SCSI_TRANSPORT_ISCSI
	if (iscsi_global->iscsi_template)
		iscsi_release_transport(iscsi_global->iscsi_template);
#endif
	if (misc_set)
		misc_deregister(&iscsi_dev);

	return(-1);
}

/*	iscsi_linux_fini():
 *
 *	Called directly from exit_iscsi_module().
 */
extern int iscsi_linux_fini (void)
{
	int ret = 0;

	iscsi_initiator_mod_fini();

#ifdef LINUX_64BITKERNEL_32BIT_USER
	iscsi_linux_unreg_ioctl_conv();
#endif
	iscsi_sysfs_unregister_global_attributes();

#ifdef USE_SCSI_TRANSPORT_ISCSI
	iscsi_release_transport(iscsi_global->iscsi_template);
#endif
	if ((ret = misc_deregister(&iscsi_dev))) {
		TRACE_ERROR("misc_deregister() failed.\n");
	}

	kfree(iscsi_global);
	iscsi_global = NULL;

	return(0);
}

/*	iscsi_linux_info():
 *
 *	This function is called by the SCSI Mid-level to get info about the
 *	attached iSCSI HBA.
 *
 * 	Parameters:	Scsi_Host pointer.
 * 	Returns:	Size of buffer which gives info about the iSCSI device.
 */
static const char *iscsi_linux_info (struct Scsi_Host *host)
{
	static char buf[80];

	memset(buf, 0, 80);
	sprintf(buf, "Core-iSCSI Initiator Stack "PYX_ISCSI_VERSION" on %s/%s "UTS_RELEASE"",
			system_utsname.sysname, system_utsname.machine);

	return(buf);
}
 
/*	iscsi_linux_get_channel_from_sc():
 *
 *
 */
static iscsi_channel_t *iscsi_linux_get_channel_from_sc (struct scsi_cmnd *sc)
{
	return((iscsi_channel_t *)sc->device->host->hostdata[0]);
}

/*	iscsi_linux_queuecommand(): (Part of Scsi_Host_Template)
 *
 *	Parameter:	Pointer to struct scsi_cmnd,
 *				function pointer to completion function.
 *	Returns:	Always 0.  FIXME:  Is this correct?
 */	
static inline int iscsi_linux_queuecommand (struct scsi_cmnd *sc, void (*done)(struct scsi_cmnd *))
{
	int status;
	iscsi_channel_t *c;
	iscsi_channel_lun_t *lu = NULL;
	
	sc->scsi_done = done;

       /*
	* Get the assoicated iSCSI Channel for this active SCSI Task.
	*/
	spin_unlock_irq(sc->device->host->host_lock);
	c = iscsi_linux_get_channel_from_sc(sc);
	lu = iscsi_get_lun_from_channel(sc->device->lun, c);

//#warning FIXME: There has got to be a better way to get struct gendisk.
	/*
	 * Used for persisant mapping.  Note that TYPE_ROM does not get
	 * attached until IO is done across an assoicated CD/DVD device's
	 * /dev entry.
	 */
	if (!lu->OS_device_set) {
		switch (sc->device->type) {
		case TYPE_DISK:
			if (!sc->request->rq_disk)
				break;
			
			sprintf(lu->OS_device_name, "%s", sc->request->rq_disk->disk_name);
			lu->OS_device_set = 1;
			break;
		case TYPE_ROM:
			if (!sc->request->rq_disk)
				break;

			sprintf(lu->OS_device_name, "scd%d", sc->request->rq_disk->first_minor);
			lu->OS_device_set = 1;
			break;	
		default:
			break;
		}
	}
	
	lu->iscsi_tasks++;
	lu->iscsi_current_tasks++;
	lu->total_bytes += sc->request_bufflen;
	
	status = iscsi_queue_scsi_cmnd_to_channel(sc, c);

	spin_lock_irq(sc->device->host->host_lock);

	switch (status) {
	case OS_SCSI_CMD_OK:
		return(0);
	case OS_SCSI_CMD_RETRY:
		sc->result = HOST_BYTE(DID_BUS_BUSY);
		sc->retries--;
		break;
	case OS_SCSI_CMD_FAILED:
	default:
		sc->result = HOST_BYTE(DID_NO_CONNECT);
		iscsi_OS_set_SCSI_lun_failure((void *)sc);
		break;
	}

	(*done)(sc);

	return(0);
}

/*	iscsi_linux_eh_strategy_handler(): (Part of Scsi_Host_Template)
 *
 *	We disable the Scsi_Cmnd's timer and handle all SCSI related error
 *	recovery ourselves.  We need this present in our Scsi_Host_Template to
 *	prevent the SCSI stack in 2.6 from complaining.
 */
static int iscsi_linux_eh_strategy_handler (struct Scsi_Host *sh)
{
	panic("Huh? iscsi_linux_eh_strategy_handler called.\n");
}

/*	iscsi_linux_biosparam(): (Part of Scsi_Host_Template)
 *
 *
 */
static int iscsi_linux_biosparam (struct scsi_device *sd, struct block_device *n, sector_t capacity, int ip[])
{
	int size = capacity;
	
	ip[0] = 64;
	ip[1] = 32;
	ip[2] = size >> 11;

	if (ip[2] > 1024) {
		ip[0] = 255;
		ip[1] = 63;
		ip[2] = size / (ip[0] * ip[1]);
	}

	return (0);
}
