/*
 * Copyright (c) 2001, 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses the functions related to the Initiator IOCTL.
 * 
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_IOCTL_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <asm/uaccess.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_chanattrib.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_ioctl.h>
#include <iscsi_initiator_ioctl_defs.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_lu.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_IOCTL_C

extern iscsi_global_t *iscsi_global;
extern int iscsi_create_connection (iscsi_session_t *, iscsi_login_holder_t *, u16 *);
extern iscsi_session_t *iscsi_get_session_from_sid (u32);
extern int iscsi_start_logout (iscsi_session_t *, int, int, u8, u16, u16);

/*      iscsi_open():
 *
 *      Used by the IOCTL on LINUX.
 */
extern int iscsi_open (struct inode *inode, struct file *filp)
{
	return(0);
}

/*      iscsi_close():
 *
 *      Used by the IOCTL on LINUX.
 */
extern int iscsi_close (struct inode *inode, struct file *filp)
{
	return(0);
}

/*      iscsi_ioctl():
 *
 *      Accepts data via /dev/iscsi character device on LINUX and FreeBSD.
 */
extern int iscsi_ioctl (
	struct inode *inode,
	struct file *filp,
	unsigned int cmd,
	unsigned long arg)
{
	int ret = 0;
	u32 max_sectors = 0;
	iscsi_login_holder_t lh;
	iscsi_channel_t *channel = NULL;
	iscsi_session_t	*sess = NULL;
	struct iscsi_client *c;

	if (!(c = (struct iscsi_client *) kmalloc(sizeof(struct iscsi_client),
			GFP_KERNEL)))
		return(-ENOMEM);
	memset(c, 0, sizeof(struct iscsi_client));

	if (copy_from_user((void *)c, (void *)arg, sizeof(struct iscsi_client)) != 0) {
		if (c)
			kfree(c);
		TRACE_ERROR("Copy from User->Kernel failed.\n");
		return(-EFAULT);
	}

	if (!(iscsi_global->initname_set) &&
	     (cmd != ISCSI_INITIATOR_SETINITNAME)) {
		TRACE_ERROR("Please set iSCSI Initiator Node Name\n");
		goto dumpout;
	}

	switch (cmd) {
	case ISCSI_INITIATOR_FULLINIT_CHANNEL:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 1)))
			goto dumpout;
		if (c->params_set & PARAM_MAX_SECTORS)
			max_sectors = c->max_sectors;
		if (iscsi_init_channel(channel, max_sectors, 1) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_INIT_CHANNEL:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 1)))
			goto dumpout;
		if (c->params_set & PARAM_MAX_SECTORS)
			max_sectors = c->max_sectors;
		if (iscsi_init_channel(channel, max_sectors, 0) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_FREE_CHANNEL:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (iscsi_free_channel(channel) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (iscsi_force_channel_offline(channel) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_SET_CHANNEL_ATTRIB:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (iscsi_set_channel_attribute(channel, c->channel_attrib,
				c->ca_value) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_STOP_CHANNEL:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (iscsi_stop_channel(channel, 0) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_LOGIN:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;

		if (iscsi_check_channel_status_for_login(channel) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		
		memset((void *)&lh, 0, sizeof(iscsi_login_holder_t));
		lh.scsi_host_id = iscsi_OS_get_SCSI_HOST_ID(channel);
		lh.scsi_target_id = c->target;
		lh.scsi_channel_id = c->channel;
		lh.channel = channel;
		lh.conns = c->conns;
		lh.ipv4_address = c->ip;
		lh.port = c->port;

		if (c->params_set & PARAM_SCTP_TCP)
			lh.network_transport = ISCSI_SCTP_TCP;
		else if (c->params_set & PARAM_SCTP_UDP)
			lh.network_transport = ISCSI_SCTP_UDP;
		else
			lh.network_transport = ISCSI_TCP;
		strncpy(lh.net_dev, c->dev, strlen(c->dev));
		
		if (c->params_set & PARAM_TARGETNAME) {
			lh.lh_flags |= LH_TARGETNAME_PRESENT;
			strncpy(lh.lh_targetname, c->value,
				ISCSI_MAX_TARGETNAME_SIZE);
		}
		
		if (iscsi_start_login_process(&lh) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}

		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_ADDCONN:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (!(sess = iscsi_get_session_from_channel_id(channel))) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		memset((void *)&lh, 0, sizeof(iscsi_login_holder_t));
		lh.channel = channel;
		lh.conns = c->conns;
		lh.ipv4_address = c->ip;
		lh.port = c->port;
		if (c->params_set & PARAM_SCTP_TCP)
			lh.network_transport = ISCSI_SCTP_TCP;
		else if (c->params_set & PARAM_SCTP_UDP)
			lh.network_transport = ISCSI_SCTP_UDP;
		else
			lh.network_transport = ISCSI_TCP;
		strncpy(lh.net_dev, c->dev, strlen(c->dev));
		
		if (iscsi_create_connection(sess, &lh, NULL) < 0) {
			iscsi_dec_session_usage_count(sess);
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_dec_session_usage_count(sess);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_SCANSCSI:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		iscsi_lu_scan_full(channel);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_LOGOUTSESS:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (!(sess = iscsi_get_session_from_channel_id(channel))) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		if (c->lun_remove && c->force)
			c->lun_remove = 2;
		
		ret = iscsi_start_logout(sess, c->lun_remove, 0, CLOSESESSION, 0, 0);
		if (ret != 0) {
			if (ret != -2)
				iscsi_dec_session_usage_count(sess);
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_dec_session_usage_count(sess);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_LOGOUTCONN:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (!(sess = iscsi_get_session_from_channel_id(channel))) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		if (!(c->params_set & PARAM_ON_CONNECTION_ID))
			c->on_cid = c->cid;
		ret = iscsi_start_logout(sess, c->lun_remove, 0, CLOSECONNECTION,
				c->cid, c->on_cid);
		if (ret != 0) {
			if (ret != -2)
				iscsi_dec_session_usage_count(sess);
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_dec_session_usage_count(sess);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_SETINITNAME:
		if (iscsi_set_initiator_name(c->keytext) < 0)
			goto dumpout;
		break;
	case ISCSI_INITIATOR_LISTCHANNELPARAMS:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		iscsi_dump_channel_params(channel);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_LISTSESSPARAMS:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (!(sess = iscsi_get_session_from_channel_id(channel))) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_print_sess_params(sess);
		iscsi_dec_session_usage_count(sess);
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_SETCHANNELPARAM:
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (iscsi_change_param_value(c->keytext, SENDER_INITIATOR,
				channel->param_list) < 0) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_put_channel(channel);
		break;
	case ISCSI_INITIATOR_SETSESSPARAM:
//#warning FIXME: ISCSI_INITIATOR_SETSESSPARAM is stubbed
#if 0
		if (!(channel = iscsi_get_channel_from_id(c->channel, 0)))
			goto dumpout;
		if (!(sess = iscsi_get_session_from_channel_id(channel))) {
			iscsi_put_channel(channel);
			goto dumpout;
		}
		if (send_param_to_target(sess->conn_head, c->key, c->value) < 0) {
			iscsi_dec_session_usage_count(sess);
			iscsi_put_channel(channel);
			goto dumpout;
		}
		iscsi_dec_session_usage_count(sess);
		iscsi_put_channel(channel);
#endif
		break;
	case ISCSI_INITIATOR_LISTKNOWNTARGETS:
		iscsi_print_targets();
		break;
	default:
		TRACE_ERROR("Unknown IOCTL cmd\n");
		goto dumpout;
	}

	if (copy_to_user((void *)arg, (void *)c, sizeof(struct iscsi_client))) {
		if (c)
			kfree(c);
		TRACE_ERROR("Copy from Kernel->User failed.\n");
		return(-EFAULT);
	}
	
	if (c)
		kfree(c);
	return(0);
	
dumpout:
	if (c)
		kfree(c);

	return(-EINVAL);
}
