/*
 * Copyright (c) 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *  
 * This file houses the iSCSI Initiator specific iSCSI Channel functions.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_CHANNEL_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/random.h>
#include <linux/blkdev.h>
#include <scsi.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_chanattrib.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_ioctl_defs.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_lu.h>
#include <iscsi_initiator_scsi.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_CHANNEL_C

extern iscsi_global_t *iscsi_global;
extern int iscsi_start_logout (iscsi_session_t *, int, int, u8, u16, u16);
extern int iscsi_scsi_queue_cmd (void *, iscsi_session_t *);

/*	iscsi_init_channels():
 *
 *
 */
extern void iscsi_init_channels (void)
{
	int i;
	iscsi_channel_t *c;

	for (i = 0; i < ISCSI_MAX_CHANNELS; i++) {
		c = &iscsi_global->channels[i];

		c->channel_id = i;
		c->status = ISCSI_CHANNEL_FREE;
		init_MUTEX(&c->ch_access_sem);
		spin_lock_init(&c->channel_state_lock);
	}

	return;
}	

/*	iscsi_add_scsi_cmnd_to_channel():
 *
 *
 */
static void iscsi_add_scsi_cmnd_to_channel (
	struct scsi_cmnd *sc,
	iscsi_channel_t *channel)
{
	spin_lock_bh(&channel->scsi_cmnd_lock);
	if (!channel->scsi_cmnd_head && !channel->scsi_cmnd_tail)
		channel->scsi_cmnd_head	= sc;
	else {
		struct scsi_cmnd *sc_tail = channel->scsi_cmnd_tail;
		sc_tail->SCp.buffer = (struct scatterlist *) sc;
	}
	channel->scsi_cmnd_tail		= sc;
	spin_unlock_bh(&channel->scsi_cmnd_lock);
		
	return;
}

/*	iscsi_get_scsi_cmnd_from_channel():
 *
 *
 */
static struct scsi_cmnd *iscsi_get_scsi_cmnd_from_channel (iscsi_channel_t *channel)
{
	struct scsi_cmnd *sc;

	spin_lock_bh(&channel->scsi_cmnd_lock);
	if (!channel->scsi_cmnd_head) {
		spin_unlock_bh(&channel->scsi_cmnd_lock);
		return(NULL);
	}
	
	
	sc = channel->scsi_cmnd_head;
	channel->scsi_cmnd_head = (struct scsi_cmnd *)sc->SCp.buffer;

	sc->SCp.buffer = NULL;
	if (!channel->scsi_cmnd_head)
		channel->scsi_cmnd_tail = NULL;
	spin_unlock_bh(&channel->scsi_cmnd_lock);

	return(sc);
}

/*	iscsi_add_chan_conn_to_channel():
 *
 *	Called with channel->chan_conn_lock held.
 */
extern void iscsi_add_chan_conn_to_channel (iscsi_channel_t *channel, iscsi_channel_conn_t *chan_conn)
{
	ADD_ENTRY_TO_LIST(chan_conn, channel->chan_conn_head, channel->chan_conn_tail);
	atomic_inc(&channel->connection_count);
		
	return;
}

/*	iscsi_del_chan_conn_from_channel():
 *
 *	Called with channel->chan_conn_lock held.
 */
extern void iscsi_del_chan_conn_from_channel (iscsi_channel_t *channel, iscsi_channel_conn_t *chan_conn)
{
	REMOVE_ENTRY_FROM_LIST(chan_conn, channel->chan_conn_head, channel->chan_conn_tail);
	atomic_dec(&channel->connection_count);
	
	return;
}

/*	iscsi_free_chan_conns_for_channel():
 *
 *
 */
static void iscsi_free_chan_conns_for_channel (iscsi_channel_t *channel)
{
	iscsi_channel_conn_t *chan_conn, *chan_conn_next;
	
	spin_lock(&channel->chan_conn_lock);	
	chan_conn = channel->chan_conn_head;
	while (chan_conn) {
		chan_conn_next = chan_conn->next;

		kfree(chan_conn);
		chan_conn = chan_conn_next;
	}
	channel->chan_conn_head = channel->chan_conn_tail = NULL;

	atomic_set(&channel->connection_count, 0);
	spin_unlock(&channel->chan_conn_lock);
	
	TRACE(TRACE_CHANNEL, "Freed iscsi_channel_conn_t list for SCSI"
		" Channel ID: %d\n", channel->channel_id);
	
	return;
}

/*	iscsi_add_conn_to_channel_for_reinstatement():
 *
 *
 */
extern iscsi_channel_conn_t *iscsi_add_conn_to_channel_for_reinstatement (
	iscsi_conn_t *conn)
{
	iscsi_channel_conn_t *chan_conn = NULL;
	iscsi_channel_t *channel = SESS(conn)->channel;

	if (!(chan_conn = kmalloc(sizeof(iscsi_channel_conn_t), GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_channel_conn_t\n");
		return(NULL);
	}
	memset(chan_conn, 0, sizeof(iscsi_channel_conn_t));

	strncpy(chan_conn->net_dev, conn->net_dev, ISCSI_NETDEV_NAME_SIZE);
	chan_conn->cid = conn->cid;
	chan_conn->ipv4_address = (conn->conn_flags & CONNFLAG_TPGF_USE_ORIG) ?
		conn->orig_login_ip : conn->login_ip;
	chan_conn->port = (conn->conn_flags & CONNFLAG_TPGF_USE_ORIG) ?
		conn->orig_login_port : conn->login_port;
	chan_conn->network_transport = conn->network_transport;

	spin_lock(&channel->chan_conn_lock);
	iscsi_add_chan_conn_to_channel(channel, chan_conn);
	spin_unlock(&channel->chan_conn_lock);
	
	return(chan_conn);
}

/*	iscsi_get_cid_from_channel():
 *
 *
 */
extern iscsi_channel_conn_t *iscsi_get_cid_from_channel (
	iscsi_channel_t *chan,
	u16 cid)
{
	iscsi_channel_conn_t *cc;
	
	spin_lock(&chan->chan_conn_lock);
	cc = chan->chan_conn_head;
	while (cc) {
		if (cc->cid == cid) {
			spin_unlock(&chan->chan_conn_lock);
			return(cc);
		}

		cc = cc->next;
	}
	spin_unlock(&chan->chan_conn_lock);

	return(NULL);
}

/*	iscsi_get_conn_from_channel():
 *
 *	Called with iscsi_channel_t->chan_conn_lock held.
 */
extern iscsi_channel_conn_t *iscsi_get_conn_from_channel (
	iscsi_channel_t *chan)
{
	iscsi_channel_conn_t *cc;

	if (!(cc = chan->chan_conn_head))
		return(NULL);

	iscsi_del_chan_conn_from_channel(chan, cc);

	return(cc);
}

/*	iscsi_dump_channel_params():
 *
 *
 */
extern void iscsi_dump_channel_params (
	iscsi_channel_t *c)
{
	iscsi_print_params(c->param_list);
	return;
}

/*	iscsi_check_active_iSCSI_LUNs():
 *
 *
 */
extern int iscsi_check_active_iSCSI_LUNs (
	iscsi_channel_t *c,
	int force)
{
	int lc;

	if ((lc = iscsi_lu_check_access_counts(c))) {
		if (force) {
			TRACE_ERROR("Stopping iSCSI Channel: %d while %u"
			" active iSCSI LUNs remain.\n", c->channel_id, lc);
			return(1);
		}
		
		TRACE_ERROR("iSCSI Channel: %d contains %u active iSCSI LUNs,"
		" pass force=1 to force operation.\n", c->channel_id, lc);
		return(-1);
	}

	return(0);
}

/*	iscsi_release_logical_units():
 *
 *
 */
extern void iscsi_release_logical_units (iscsi_channel_t *c)
{
	iscsi_channel_req_t *cr;
	
	/*
	 * If the iSCSI Channel is being forced offline, this will
	 * be handled in iscsi_set_channel_status().
	 */
	if (atomic_read(&c->force_channel_offline))
		return;

	/*
	 * If the iSCSI Channel is being temporarily stopped, do
	 * not release any of the Logical Units.
	 */
	if (atomic_read(&c->stop_channel))
		return;
	
	/*
	 * Normal Operation: We need to release the Logical Units
	 * BEFORE logging out the iSCSI Session.  This is because
	 * more SCSI commands may need to come from the OS'es
	 * SCSI Stack, such as SYNCHRONIZE_CACHE.
	 */
loop:
	if (!(cr = iscsi_allocate_ch_req(CREQ_LUN_UNSCAN, NULL, 0, NULL)))
		goto loop;
	
	ISCSI_ADD_CH_scan(cr, c);

	down(&cr->cr_data_sem);
	iscsi_complete_ch_scan(cr, c);
	
	return;
}

/*	iscsi_free_all_channel_resources():
 *
 *
 */
extern void iscsi_free_all_channel_resources (void)
{
	int i;
	iscsi_channel_t *c;

	spin_lock(&iscsi_global->channel_lock);
	for (i = 0; i < ISCSI_MAX_CHANNELS; i++) {
		c = &iscsi_global->channels[i];

		if (c->status == ISCSI_CHANNEL_FREE)		
			continue;

		spin_unlock(&iscsi_global->channel_lock);

		atomic_set(&c->force_channel_offline, 1);
		
		iscsi_set_channel_status(c, ISCSI_CHANNEL_FAILED,
			SCS_FREE_CHANNEL_CONNS |
			SCS_RELEASE_LUNS |
			SCS_FAIL_OS_SCSI_CMDS);

		iscsi_OS_free_scsi_host((void *)c);
		
		iscsi_channel_stop_threads(c);
		
		if (c->param_list) {
			iscsi_release_param_list(c->param_list);
			c->param_list = NULL;
		}
		spin_lock(&iscsi_global->channel_lock);
	}
	spin_unlock(&iscsi_global->channel_lock);

	return;
}

/*	iscsi_activate_channel():
 *
 *
 */
//#warning FIXME: Disable LUN SCAN for logged out active channels
extern void iscsi_activate_channel (
	iscsi_conn_t *conn,
	iscsi_login_holder_t *lh)
{
	int scs_flags = 0;
	iscsi_channel_t *channel = lh->channel;
	iscsi_session_t *sess = SESS(conn);
	
	spin_lock_bh(&channel->channel_state_lock);
	channel->sess = sess;
	channel->host_id = lh->scsi_host_id;
	channel->target_id = lh->scsi_target_id;
	spin_unlock_bh(&channel->channel_state_lock);

	scs_flags |= SCS_RETRY_OS_SCSI_CMDS;
	if (!channel->active_luns && !lh->session_reinstatement)
		scs_flags |= SCS_LUN_SCAN;

	iscsi_dec_session_usage_count(sess);
	iscsi_dec_conn_usage_count(conn);
	
	iscsi_set_channel_status(channel, ISCSI_CHANNEL_ACTIVE, scs_flags);
		
	return;
}

/*	iscsi_force_channel_offline():
 *
 *
 */
extern int iscsi_force_channel_offline (
	iscsi_channel_t *channel)
{
	atomic_set(&channel->force_channel_offline, 1);

	if (iscsi_stop_channel(channel, 1) != 0)
		down(&channel->force_offline_sem);

	iscsi_set_channel_status(channel, ISCSI_CHANNEL_FAILED,
			SCS_FREE_CHANNEL_CONNS |
			SCS_RELEASE_LUNS |
			SCS_FAIL_OS_SCSI_CMDS);

	atomic_set(&channel->force_channel_offline, 0);
	
	return(0);
}

/*	iscsi_free_channel():
 *
 *
 */
extern int iscsi_free_channel (
	iscsi_channel_t *c)
{
	int lun_count, retry = 1;
	
	if ((c->status != ISCSI_CHANNEL_INACTIVE) &&
	    (c->status != ISCSI_CHANNEL_SESSION_LOGGED_OUT) &&
	    (c->status != ISCSI_CHANNEL_FAILED)) {
		TRACE_ERROR("Unable to free SCSI Channel %d while status %d\n",
			 c->channel_id, c->status);
		return(-1);
	}

check_LUNS:
	lun_count = iscsi_lu_check_access_counts(c);

	iscsi_release_all_lus(c);
	
	if (lun_count) {
		if (retry) {
			retry = 0;
			goto check_LUNS;
		}
		
		TRACE_ERROR("Unable to Free iSCSI Channel ID: %d while %d SCSI"
			" LUNs remain in use. Please umount all iSCSI LUNs and"
				" try again.\n", c->channel_id, lun_count);
		return(-1);
	}

	iscsi_set_channel_status(c, ISCSI_CHANNEL_FREE, SCS_FAIL_OS_SCSI_CMDS);

	iscsi_channel_stop_threads(c);
	
	if (c->param_list) {
		iscsi_release_param_list(c->param_list);
		c->param_list = NULL;
	}
	
	iscsi_OS_free_scsi_host((void *)c);
	
	TRACE(TRACE_CHANNEL, "Freed iSCSI Channel %d.\n", c->channel_id);
	
	return(0);
}

/*
 * Some saner parameter defaults than RFC 3720 defines.
 */
static unsigned char *full_params_array[] = {
        "AuthMethod=None",
	"DefaultTime2Wait=2",
	"DefaultTime2Retain=25",
	"HeaderDigest=CRC32C,None",
	"DataDigest=CRC32C,None",
	"MaxConnections=10",
	"MaxRecvDataSegmentLength=131072",
	"ErrorRecoveryLevel=0",
	"IFMarker=Yes",
	"OFMarker=Yes",
	"IFMarkInt=2048~8192",
	"OFMarkInt=2048~8192",
	"DataPDUInOrder=Yes",
	"DataSequenceInOrder=Yes",
	"FirstBurstLength=262144",
	"MaxBurstLength=16776192",
	"MaxOutstandingR2T=4",
	"ImmediateData=Yes",
	"InitialR2T=No"
};

/*	iscsi_setup_full_params():
 *
 *	Setup a little more reasonable values then the standard sets as defaults.
 */
static int iscsi_setup_full_params (iscsi_channel_t *c)
{
	unsigned char *buf;
	int i;
	
	if (!(buf = (unsigned char *) kmalloc(512, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate buf\n");
		return(-1);
	}

	for (i = 0; i < ARRAY_SIZE(full_params_array); i++) {
		memset(buf, 0, 512);
		sprintf(buf, "%s", full_params_array[i]);
		if (iscsi_change_param_value(buf, INITIATOR, c->param_list) < 0) {
			kfree(buf);
			return(-1);
		}
	}
	kfree(buf);
	
	return(0);
}

static void iscsi_initiator_build_channel_isid (iscsi_channel_t *channel)
{
	u16 qualifer;

	memset(channel->ch_isid, 0, 6);
	get_random_bytes(&channel->ch_isid[1], 3);
	qualifer = htons(iscsi_global->isid_qualifier++);

	channel->ch_isid[0] |= 0x80;
	channel->ch_isid[4] = qualifer & 0xff;
	channel->ch_isid[5] = (qualifer >> 8) & 0xff;

	printk("iCHANNEL[%u]: Generated iSID: 0x%02x %02x %02x %02x %02x %02x\n",
		channel->channel_id, channel->ch_isid[0], channel->ch_isid[1],
		channel->ch_isid[2], channel->ch_isid[3], channel->ch_isid[4],
		channel->ch_isid[5]);

	return;
}
	
/*	iscsi_init_channel():
 *
 *
 */
extern int iscsi_init_channel (iscsi_channel_t *channel, u32 max_sectors, int full_init)
{
	unsigned char *buf;
	
	spin_lock_bh(&channel->channel_state_lock);
	if (channel->status != ISCSI_CHANNEL_FREE) {
		TRACE_ERROR("iSCSI Channel ID: %u is already in use.\n",
			channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);	
		return(-1);
	}
	spin_unlock_bh(&channel->channel_state_lock);
		
	if (iscsi_create_default_params(&channel->param_list) < 0)
		return(-1);

	if (!(buf = (unsigned char *) kmalloc(512, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for buf\n");
		goto fail;
	}
	memset(buf, 0, 512);
	sprintf(buf, "InitiatorName=%s", iscsi_global->initiatorname);
	
	if (iscsi_change_param_value(buf, INITIATOR, channel->param_list) < 0) {
		kfree(buf);
		goto fail;
	}
	kfree(buf);
	
	if (full_init) {
		if (iscsi_setup_full_params(channel))
			goto fail;
	}

	iscsi_set_default_channel_attributes(channel);
	iscsi_initiator_build_channel_isid(channel);

	init_MUTEX_LOCKED(&channel->ct_req_sem);
	init_MUTEX_LOCKED(&channel->ct_scan_sem);
	init_MUTEX_LOCKED(&channel->ct_req_done_sem);
	init_MUTEX_LOCKED(&channel->ct_scan_done_sem);
	init_MUTEX_LOCKED(&channel->ct_req_start_sem);
	init_MUTEX_LOCKED(&channel->ct_scan_start_sem);
	init_MUTEX_LOCKED(&channel->force_offline_sem);
	init_MUTEX_LOCKED(&channel->reinstatement_thread_stop_sem);
	spin_lock_init(&channel->chan_conn_lock);
	spin_lock_init(&channel->cr_req_lock);
	spin_lock_init(&channel->cr_scan_lock);
	spin_lock_init(&channel->scsi_cmnd_lock);
	channel->status		= ISCSI_CHANNEL_INACTIVE;
	channel->ch_max_sectors	= max_sectors;

	iscsi_channel_start_threads(channel);
	
	if (iscsi_OS_init_iscsi_channel((void *)channel, max_sectors) < 0) {
		iscsi_channel_stop_threads(channel);
		goto fail;
	}

	TRACE(TRACE_CHANNEL, "%s initilization of iSCSI Channel: %d complete\n",
		(full_init) ? "Full" : "Basic", channel->channel_id);
	
	return(0);
fail:
	if (channel->param_list) {
		iscsi_release_param_list(channel->param_list);
		channel->param_list = NULL;
	}
	return(-1);
}

/*	iscsi_stop_channel():
 *
 *
 */
extern int iscsi_stop_channel (iscsi_channel_t *channel, int force)
{
	int ret = 0;
	iscsi_session_t *sess;
	
	atomic_set(&channel->stop_channel, 1);
		
	if (!(sess = iscsi_get_session_from_channel_id(channel))) {
		atomic_set(&channel->stop_channel, 0);
		goto check_reinstatement;
	}

	/*
	 * STOP_CHANNEL will never remove iSCSI/SCSI Logical Units.
	 * This can be achived after STOP_CHANNEL with FORCE_CHAN_OFFLINE.
	 */
	ret = iscsi_start_logout(sess, 0 /* lun_remove = 0 */, 0, CLOSESESSION, 0, 0);
	if (ret != -2)
		iscsi_dec_session_usage_count(sess);
	
	if (ret != -1) {
		printk("iCHANNEL[%d] - Stopped session on iSCSI Channel\n",
			channel->channel_id);
	}
	atomic_set(&channel->stop_channel, 0);
	
	printk("iCHANNEL[%d] - Stopped all action on iSCSI Channel\n", channel->channel_id);
	return(1);	
	
check_reinstatement:
	if (iscsi_set_channel_status(channel, ISCSI_CHANNEL_SESSION_LOGGED_OUT, 
			SCS_STOP_REINSTATEMENT_THREAD |
			SCS_FREE_CHANNEL_CONNS |
			SCS_REESTABLISHING_CHECK) < 0) {
		TRACE_ERROR("iSCSI Channel: %d does not have an active iSCSI"
			" Session, and is not in Session Reinstatement,"
			" ignoring request.\n", channel->channel_id);
		atomic_set(&channel->stop_channel, 0);
		return(0);
	}

	atomic_set(&channel->stop_channel, 0);
	printk("iCHANNEL[%d] - Stopped all action on iSCSI Channel\n", channel->channel_id);

	return(0);
}

/*	iscsi_get_channel_from_id():
 *
 *
 */
extern iscsi_channel_t *iscsi_get_channel_from_id (int channel_id, int initchan)
{
	int i;
	iscsi_channel_t *channel;

	spin_lock_bh(&iscsi_global->channel_lock);
	for (i = 0; i < ISCSI_MAX_CHANNELS; i++) {
		channel = &iscsi_global->channels[i];

		if (channel->channel_id != channel_id)
			continue;
		
		spin_lock(&channel->channel_state_lock);
		if ((channel->status == ISCSI_CHANNEL_FREE) && !initchan) {
			spin_unlock(&channel->channel_state_lock);
			break;
		}
		spin_unlock(&channel->channel_state_lock);
		spin_unlock_bh(&iscsi_global->channel_lock);

		down_interruptible(&channel->ch_access_sem);
		return((signal_pending(current)) ? NULL : channel);
	}
	spin_unlock_bh(&iscsi_global->channel_lock);
	
	TRACE_ERROR("Unable to locate available iSCSI Channel ID:"
			" %d\n", channel_id);
	
	return(NULL);
}

/*	iscsi_put_channel():
 *
 *
 */
extern void iscsi_put_channel (iscsi_channel_t *channel)
{
	up(&channel->ch_access_sem);
}

/*	__iscsi_get_session_from_channel_id():
 *
 *
 */
extern iscsi_session_t *__iscsi_get_session_from_channel_id (iscsi_channel_t *channel)
{
	spin_lock_bh(&channel->channel_state_lock);
	if (channel->status == ISCSI_CHANNEL_ACTIVE) {
		__iscsi_inc_session_usage_count(channel->sess);
		spin_unlock_bh(&channel->channel_state_lock);
		return(channel->sess);
	}
	spin_unlock_bh(&channel->channel_state_lock);

	return(NULL);
}

/*	iscsi_get_session_from_channel_id():
 *
 *
 */
extern iscsi_session_t *iscsi_get_session_from_channel_id (iscsi_channel_t *channel)
{
	spin_lock_bh(&channel->channel_state_lock);
	if (channel->status == ISCSI_CHANNEL_ACTIVE) {
		__iscsi_inc_session_usage_count(channel->sess);
		spin_unlock_bh(&channel->channel_state_lock);
		return(channel->sess);
	}
	spin_unlock_bh(&channel->channel_state_lock);

	TRACE_ERROR("Unable to locate active iSCSI Session on iSCSI Channel"
			" ID: %d\n", channel->channel_id);
	
	return(NULL);
}

/*	iscsi_get_lun_from_channel():
 *
 *
 */
extern iscsi_channel_lun_t *iscsi_get_lun_from_channel (int lun, iscsi_channel_t *c)
{
	return(&c->ch_lun_list[lun]);
}

/*	iscsi_queue_scsi_cmnd_to_channel():
 *
 *
 */
extern int iscsi_queue_scsi_cmnd_to_channel (struct scsi_cmnd *sc, iscsi_channel_t *c)
{
	int ret = 0, status;
	iscsi_session_t *s = NULL;
	
	spin_lock_bh(&c->channel_state_lock);
	switch (c->status) {
	case ISCSI_CHANNEL_ACTIVE:
		s = c->sess;
		__iscsi_inc_session_usage_count(s);
		spin_unlock_bh(&c->channel_state_lock);
		ret = iscsi_scsi_queue_cmd(sc, s);
		iscsi_dec_session_usage_count(s);
		status = (!ret) ? OS_SCSI_CMD_OK : OS_SCSI_CMD_RETRY;
		return(status);
	case ISCSI_CHANNEL_FAILED:
		TRACE(TRACE_CHANNEL, "iSCSI Channel %d is in a failed"
			" state, failing SCSI command.\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		return(OS_SCSI_CMD_FAILED);
	case ISCSI_CHANNEL_FREE:
		TRACE(TRACE_CHANNEL, "iSCSI Channel %d does not have an"
			" active iSCSI session, failing SCSI"
				" command.\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		return(OS_SCSI_CMD_FAILED);
	case ISCSI_CHANNEL_INACTIVE:
		TRACE(TRACE_CHANNEL, "iSCSI Channel %d in inactive state,"
			" failing SCSI command\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		return(OS_SCSI_CMD_FAILED);
	case ISCSI_CHANNEL_REESTABLISHING:
		/*
		 * Add this OS dependant SCSI command to the per channel queue
		 * that will be completed on the completion of session/connection
		 * reestablishment.
		 */
		TRACE(TRACE_CHANNEL, "iSCSI Channel %d is reestablishing"
			 " a new iSCSI session, retrying SCSI"
			 	" command.\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		iscsi_move_scsi_cmnd_to_channel(sc, c->channel_id);
		return(OS_SCSI_CMD_OK);
	case ISCSI_CHANNEL_SESSION_LOGGED_OUT:
		/*
		 * Add this OS dependant SCSI command to the per channel queue
		 * that will be completed once a new session is created for
		 * this iSCSI Channel.
		 */
		TRACE(TRACE_CHANNEL, "iSCSI Channel %d is unavailable as the"
			" session has been logged out.\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		iscsi_move_scsi_cmnd_to_channel(sc, c->channel_id);
		return(OS_SCSI_CMD_OK);
	default:
		TRACE_ERROR("Unknown channel->status for iSCSI Channel"
			" %d, failing SCSI Command.\n", c->channel_id);
		spin_unlock_bh(&c->channel_state_lock);
		return(OS_SCSI_CMD_FAILED);
	}
	
	return(status);
}

/*	iscsi_move_scsi_cmnd_to_channel():
 *
 *
 */
extern void iscsi_move_scsi_cmnd_to_channel (struct scsi_cmnd *sc, int channel_id)
{
	iscsi_channel_t *channel = NULL;
#if 0
	scsi_delete_timer(sc);
#endif	
	channel = &iscsi_global->channels[channel_id];
	iscsi_add_scsi_cmnd_to_channel(sc, channel);	
	
	return;
}	

/*	iscsi_move_conn_scsi_cmnd_to_channel():
 *
 *
 */
extern void iscsi_move_conn_scsi_cmnd_to_channel (iscsi_conn_t *conn, iscsi_channel_t *channel)
{
	iscsi_cmd_t *cmd;
	struct scsi_cmnd *sc;
	
	spin_lock_bh(&conn->cmd_lock);
	for (cmd = conn->cmd_head; cmd; cmd = cmd->next) {
		if (!cmd->scsi_cmnd)
			continue;

		sc = cmd->scsi_cmnd;
		spin_unlock_bh(&conn->cmd_lock);

		iscsi_dec_scsi_usage_count(cmd);
		cmd->scsi_cmnd = NULL;

		iscsi_add_scsi_cmnd_to_channel(sc, channel);

		spin_lock_bh(&conn->cmd_lock);
	}
	spin_unlock_bh(&conn->cmd_lock);
		
	return;
}

/*	iscsi_move_sess_scsi_cmnd_to_channel():
 *
 *
 */
extern void iscsi_move_sess_scsi_cmnd_to_channel (iscsi_session_t *sess, iscsi_channel_t *channel)
{
	iscsi_cmd_t *cmd;
	unsigned long flags;
	struct scsi_cmnd *sc;

	spin_lock_irqsave(&sess->pending_lock, flags);
	for (cmd = sess->pending_head; cmd; cmd = cmd->next) {
		if (!cmd->scsi_cmnd)
			continue;
		
		sc = cmd->scsi_cmnd;;
		spin_unlock_irqrestore(&sess->pending_lock, flags);

		iscsi_dec_scsi_usage_count(cmd);
		cmd->scsi_cmnd = NULL;
		
		iscsi_add_scsi_cmnd_to_channel(sc, channel);
		spin_lock_irqsave(&sess->pending_lock, flags);
	}
	spin_unlock_irqrestore(&sess->pending_lock, flags);
		
	return;
}

/*	iscsi_set_scsi_cmnd_results_for_channel():
 *
 *
 */
extern void iscsi_set_scsi_cmnd_results_for_channel (iscsi_channel_t *channel, u8 action)
{
	struct scsi_cmnd *sc;
	
	while ((sc = iscsi_get_scsi_cmnd_from_channel(channel)))
		iscsi_set_result_from_action_and_complete(sc, action);

	return;
}

/*	iscsi_set_conn_scsi_cmnd_results():
 *
 *	Used in iscsi_close_connection() when logging out iSCSI Connections
 *	on the fly.
 */
extern void iscsi_set_conn_scsi_cmnd_results (iscsi_conn_t *conn, u8 action)
{
	iscsi_cmd_t *cmd, *cmd_next;
	struct scsi_cmnd *sc;
	
	spin_lock_bh(&conn->cmd_lock);
	cmd = conn->cmd_head;
	while (cmd) {
		cmd_next = cmd->next;
		
		if (!cmd->scsi_cmnd) {
			cmd = cmd_next;
			continue;
		}
		sc = cmd->scsi_cmnd;
		spin_unlock_bh(&conn->cmd_lock);

		iscsi_dec_scsi_usage_count(cmd);
		cmd->scsi_cmnd = NULL;
		
		iscsi_set_result_from_action_and_complete(sc, action);		
		spin_lock_bh(&conn->cmd_lock);

		cmd = cmd_next;
	}
	spin_unlock_bh(&conn->cmd_lock);

	return;
}

/*	iscsi_stop_channel_reinstatement_process():
 *
 *
 */
extern void iscsi_stop_channel_reinstatement_process (
	iscsi_channel_t *channel,
	int reinstatement)
{
	spin_lock(&channel->chan_conn_lock);
	if (!channel->reinstatement_thread) {
		if (reinstatement) {
			channel->reinstatement_thread = current;
		}
		spin_unlock(&channel->chan_conn_lock);
		return;
	}

	if (atomic_read(&channel->reinstatement_thread_stop)) {
		spin_unlock(&channel->chan_conn_lock);
		return;
	}
	
	atomic_set(&channel->reinstatement_thread_stop, 1);

	send_sig(SIGKILL, channel->reinstatement_thread, 1);

	if (reinstatement)
		channel->reinstatement_thread = current;

	spin_unlock(&channel->chan_conn_lock);

	down(&channel->reinstatement_thread_stop_sem);

	spin_lock(&channel->chan_conn_lock);
	atomic_set(&channel->reinstatement_thread_stop, 0);
	if (!reinstatement)
		channel->reinstatement_thread = NULL;

	spin_unlock(&channel->chan_conn_lock);
		
	return;
}

/*	iscsi_dump_channel_attributes():
 *
 *
 */
extern void iscsi_dump_channel_attributes (
	iscsi_channel_t *c)
{
	return;
}

/*	iscsi_check_channel_status_for_login():
 *
 *
 */
extern int iscsi_check_channel_status_for_login (
	iscsi_channel_t *channel)
{
	spin_lock_bh(&channel->channel_state_lock);
	if (channel->status == ISCSI_CHANNEL_ACTIVE) {
		TRACE_ERROR("iSCSI Channel: %d is currently in a active state,"
			" ignoring request.\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);
		return(-1);
	}
	
	if (channel->status == ISCSI_CHANNEL_FAILED) {
		TRACE_ERROR("iSCSI Channel: %d currently in a failed state,"
			" unmount all active iSCSI LUNs and call freechan\n",
				channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);
		return(-1);
	}

	if (channel->status == ISCSI_CHANNEL_REESTABLISHING) {
		spin_unlock_bh(&channel->channel_state_lock);
		iscsi_stop_channel_reinstatement_process(channel, 0);
		return(0);
	}
	spin_unlock_bh(&channel->channel_state_lock);
	
	return(0);
}

/*	iscsi_set_channel_status():
 *
 *
 */
extern int iscsi_set_channel_status (
	iscsi_channel_t *channel,
	int channel_status,
	int scs_flags)
{
	spin_lock_bh(&channel->channel_state_lock);
	switch (channel_status) {
	case ISCSI_CHANNEL_FREE:
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to FREE for iSCSI Channel"
				" %d\n", channel->channel_id);
		channel->sess = NULL;
		spin_unlock_bh(&channel->channel_state_lock);

		if (scs_flags & SCS_FAIL_OS_SCSI_CMDS) {
			iscsi_scsi_offline_devices(channel);
			iscsi_set_scsi_cmnd_results_for_channel(channel,
				ISCSI_COMMAND_FAIL);
			iscsi_scsi_unblock_devices(channel);
		}

		break;
	case ISCSI_CHANNEL_INACTIVE:
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to INACTIVE for iSCSI Channel"
				" %d\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);
		break;
	case ISCSI_CHANNEL_ACTIVE:
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to ACTIVE for iSCSI Channel"
				" %d\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);

		if (scs_flags & SCS_RETRY_OS_SCSI_CMDS)
			iscsi_scsi_unblock_devices(channel);

		if (scs_flags & SCS_LUN_SCAN) {
			iscsi_channel_req_t *cr;
loop:
			if (!(cr = iscsi_allocate_ch_req(CREQ_LUN_SCAN, NULL, 0, NULL)))
				goto loop;

			ISCSI_ADD_CH_scan(cr, channel);
//#warning FIXME: Serial SCSI Scan stuff.
#if 0
			TRACE_ERROR("Scanning LUNS for iSCSI Channel ID: %d\n",
					channel->channel_id);
#endif
#if 1
			down(&cr->cr_data_sem);
			iscsi_complete_ch_scan(cr, channel);
#endif
		}
		break;
	case ISCSI_CHANNEL_REESTABLISHING:
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to REESTABLISHING for iSCSI"
			" Channel %d\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);

		if (scs_flags & SCS_STOP_REINSTATEMENT_THREAD_AND_SET)
			iscsi_stop_channel_reinstatement_process(channel, 1);
		if (scs_flags & SCS_STOP_REINSTATEMENT_THREAD)
			iscsi_stop_channel_reinstatement_process(channel, 0);
		break;
	case ISCSI_CHANNEL_SESSION_LOGGED_OUT:
		if ((scs_flags & SCS_REESTABLISHING_CHECK) &&
		    (channel->status != ISCSI_CHANNEL_REESTABLISHING)) {
			spin_unlock_bh(&channel->channel_state_lock);
			return(-1);
		}
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to SESSIONLOGGEDOUT for iSCSI"
			" Channel %d\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);

		if (scs_flags & SCS_STOP_REINSTATEMENT_THREAD)
			iscsi_stop_channel_reinstatement_process(channel, 0);

		if (scs_flags & SCS_FREE_CHANNEL_CONNS)
			iscsi_free_chan_conns_for_channel(channel);

		channel->sess = NULL;
		break;
	case ISCSI_CHANNEL_FAILED:
		channel->status = channel_status;
		TRACE(TRACE_CHANNEL, "Set status to FAILED for iSCSI Channel"
				" %d\n", channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);
		
		if (scs_flags & SCS_STOP_REINSTATEMENT_THREAD)
			iscsi_stop_channel_reinstatement_process(channel, 0);
		
		if (scs_flags & SCS_FREE_CHANNEL_CONNS)
			iscsi_free_chan_conns_for_channel(channel);

		if (scs_flags & SCS_FAIL_OS_SCSI_CMDS) {
			iscsi_scsi_offline_devices(channel);
			iscsi_set_scsi_cmnd_results_for_channel(channel,
				ISCSI_COMMAND_FAIL);
			iscsi_scsi_unblock_devices(channel);
		}

		if (scs_flags & SCS_RELEASE_LUNS)
			iscsi_release_all_lus(channel);

		break;
	default:
		TRACE_ERROR("Cannot set status to UNKNOWN: %d for iSCSI"
			" Channel %d\n", channel_status, channel->channel_id);
		spin_unlock_bh(&channel->channel_state_lock);
		break;
	}	

	return(0);
}

/*	iscsi_allocate_ch_req():
 *
 *
 */
extern iscsi_channel_req_t *iscsi_allocate_ch_req (int cr_action, void *cr_data, int atomic, iscsi_channel_req_t *cr_p)
{
	iscsi_channel_req_t *cr;

	if (cr_p) {
		cr = cr_p;
		goto init;
	}
	
	if (!(cr = kmalloc(sizeof(iscsi_channel_req_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for"
			" iscsi_channel_req_t\n");
		return(NULL);
	}

init:
	memset(cr, 0, sizeof(iscsi_channel_req_t));
	init_MUTEX_LOCKED(&cr->cr_data_sem);
	init_MUTEX_LOCKED(&cr->cr_stop_sem);
	spin_lock_init(&cr->cr_state_lock);

	cr->cr_data = cr_data;
	cr->cr_action = cr_action;
	
	return(cr);
}

#define BUILD_ISCSI_ADD_CH_XXX(name)						\
extern void ISCSI_ADD_CH_##name (iscsi_channel_req_t *cr, iscsi_channel_t *c)	\
{										\
	spin_lock_bh(&c->cr_##name##_lock);					\
	ADD_ENTRY_TO_LIST(cr, c->cr_##name##_head, c->cr_##name##_tail); 	\
	cr->cr_state = CH_REQ_STATE_ATTACHED;					\
	cr->cr_req_id = c->cr_req_id_counter++;					\
	spin_unlock_bh(&c->cr_##name##_lock);					\
	up(&c->ct_##name##_sem);						\
	return;									\
}

#define BUILD_ISCSI_GET_CH_XXX(name)					\
static iscsi_channel_req_t *ISCSI_GET_CH_##name (iscsi_channel_t *c)	\
{									\
	iscsi_channel_req_t *cr;					\
	spin_lock_bh(&c->cr_##name##_lock);				\
	if (!c->cr_##name##_head) {					\
		spin_unlock_bh(&c->cr_##name##_lock);			\
		return(NULL);						\
	}								\
	cr = c->cr_##name##_head;					\
	c->cr_##name##_head = c->cr_##name##_head->next;		\
	cr->next = cr->prev = NULL;					\
	if (!c->cr_##name##_head)					\
		c->cr_##name##_tail = NULL;				\
	else								\
		c->cr_##name##_head->prev = NULL;			\
	cr->cr_state = CH_REQ_STATE_ACTIVE;				\
	spin_unlock_bh(&c->cr_##name##_lock);				\
	return(cr);							\
}

BUILD_ISCSI_ADD_CH_XXX(req);
BUILD_ISCSI_GET_CH_XXX(req);
BUILD_ISCSI_ADD_CH_XXX(scan);
BUILD_ISCSI_GET_CH_XXX(scan);

/*	iscsi_finish_ch_req():
 *
 *
 */
extern void iscsi_finish_ch_req (iscsi_channel_req_t *cr, iscsi_channel_t *c)
{
	spin_lock_bh(&c->cr_req_lock);
	switch (cr->cr_state) {
	case CH_REQ_STATE_STOP_ACTIVE:
		cr->cr_state = CH_REQ_STATE_FREE;
		up(&cr->cr_stop_sem);
		spin_unlock_bh(&c->cr_req_lock);
		flush_signals(current);
		break;
	case CH_REQ_STATE_ACTIVE:
		cr->cr_state = CH_REQ_STATE_FREE;
		up(&cr->cr_data_sem);
		spin_unlock_bh(&c->cr_req_lock);
		break;
	default:
		spin_unlock_bh(&c->cr_req_lock);
		break;
	}

	return;
}

/*	iscsi_finish_ch_scan():
 *
 *
 */
extern void iscsi_finish_ch_scan (iscsi_channel_req_t *cr, iscsi_channel_t *c)
{
	spin_lock_bh(&c->cr_scan_lock);
	switch (cr->cr_state) {
	case CH_REQ_STATE_ACTIVE:
		cr->cr_state = CH_REQ_STATE_FREE;
		up(&cr->cr_data_sem);
		spin_unlock_bh(&c->cr_scan_lock);
		break;
	default:
		spin_unlock_bh(&c->cr_scan_lock);
		break;
	}

	return;	
}

/*	iscsi_complete_ch_req():
 *
 *
 */
extern void iscsi_complete_ch_req (iscsi_channel_req_t *cr, iscsi_channel_t *c)
{
	spin_lock_bh(&c->cr_req_lock);
	switch (cr->cr_state) {
	case CH_REQ_STATE_ATTACHED:
		REMOVE_ENTRY_FROM_LIST(cr, c->cr_req_head, c->cr_req_tail);
		spin_unlock_bh(&c->cr_req_lock);
		kfree(cr);
		break;
	case CH_REQ_STATE_ACTIVE:
		cr->cr_state = CH_REQ_STATE_STOP_ACTIVE;
		spin_unlock_bh(&c->cr_req_lock);
		send_sig(SIGKILL, c->channel_req_thread, 1);
		down(&cr->cr_stop_sem);
		kfree(cr);
		break;
	case CH_REQ_STATE_FREE:
		spin_unlock_bh(&c->cr_req_lock);
		kfree(cr);
		break;
	default:
		TRACE_ERROR("Unknown cr_state: %d\n", cr->cr_state);
		spin_unlock_bh(&c->cr_req_lock);
		break;
	}

	return;
}

/*	iscsi_complete_ch_scan():
 *
 *
 */
extern void iscsi_complete_ch_scan (iscsi_channel_req_t *cr, iscsi_channel_t *c)
{
	spin_lock_bh(&c->cr_scan_lock);
	switch (cr->cr_state) {
	case CH_REQ_STATE_FREE:
		spin_unlock_bh(&c->cr_scan_lock);
		kfree(cr);
		break;
	default:
		TRACE_ERROR("Unknown cr_state: %d\n", cr->cr_state);
		spin_unlock_bh(&c->cr_scan_lock);
		break;
	}

	return;
}

/*	iscsi_channel_req_thread():
 *
 *
 */
static int iscsi_channel_req_thread (void *arg)
{
	char name[16];
	iscsi_channel_t *c = (iscsi_channel_t *) arg;
	iscsi_channel_req_t *cr;
	iscsi_cmd_t *cmd;
	iscsi_login_holder_t *lh;

	sprintf(name, "iscsi_req_%d", c->channel_id);
	daemonize(name);
	spin_lock_irq(&current->sighand->siglock);
	siginitsetinv(&current->blocked, ISCSI_SHUTDOWN_SIGS);
	recalc_sigpending();
	c->channel_req_thread = current;
	spin_unlock_irq(&current->sighand->siglock);
	
	up(&c->ct_req_start_sem);
	
	while (1) {
		down_interruptible(&c->ct_req_sem);
		if (signal_pending(current))
			goto out;

		if (!(cr = ISCSI_GET_CH_req(c)))
			continue;

		switch (cr->cr_action) {
		case CH_REQ_ACTION_LOGIN:
			lh = (iscsi_login_holder_t *) cr->cr_data;
			iscsi_start_login(lh, cr);
			break;
		case CREQ_STOP_SCSI_CMD:
			cmd = (iscsi_cmd_t *) cr->cr_data;
			iscsi_lu_fail_scsi_task(cmd, &cmd->lun);
			break;
		case CREQ_STOP_SCSI_CMD_FROM_TIMER:
			cmd = (iscsi_cmd_t *) cr->cr_data;
			iscsi_lu_fail_scsi_task_from_timer(cmd, &cmd->lun);
			break;
		default:
			TRACE_ERROR("Unknown cr_action: %d on iSCSI Channel:"
				" %d\n", cr->cr_action, c->channel_id);
			break;
		}

		flush_signals(current);
		
		iscsi_finish_ch_req(cr, c);
	}

out:
	up(&c->ct_req_done_sem);
	return(0);
}

/*	iscsi_channel_scan_thread():
 *
 *
 */
static int iscsi_channel_scan_thread (void *arg)
{
	char name[16];
	iscsi_channel_t *c = (iscsi_channel_t *) arg;
	iscsi_channel_req_t *cr;

	sprintf(name, "iscsi_scan_%d", c->channel_id);
	daemonize(name);
	spin_lock_irq(&current->sighand->siglock);
	siginitsetinv(&current->blocked, ISCSI_SHUTDOWN_SIGS);
	recalc_sigpending();
	c->channel_scan_thread = current;
	spin_unlock_irq(&current->sighand->siglock);

	up(&c->ct_scan_start_sem);

	while (1) {
		down_interruptible(&c->ct_scan_sem);
		if (signal_pending(current))
			goto out;

		if (!(cr = ISCSI_GET_CH_scan(c)))
			continue;
		
		switch (cr->cr_action) {
		case CREQ_LUN_SCAN:
			iscsi_lu_scan_full(c);
			break;
		case CREQ_LUN_UNSCAN:
			iscsi_release_all_lus(c);
			break;
		default:
			TRACE_ERROR("Unknown cr_action: %d on iSCSI Channel:"
				" %d\n", cr->cr_action, c->channel_id);
			break;
		}
		
		flush_signals(current);

		iscsi_finish_ch_scan(cr, c);
	}

out:
	up(&c->ct_scan_done_sem);
	return(0);
}

/*	iscsi_channel_start_threads():
 *
 *
 */
extern void iscsi_channel_start_threads(iscsi_channel_t *c)
{
	kernel_thread(iscsi_channel_req_thread, (void *)c, 0);
	down(&c->ct_req_start_sem);
	
	kernel_thread(iscsi_channel_scan_thread, (void *)c, 0);
	down(&c->ct_scan_start_sem);

	return;
}

/*	iscsi_channel_stop_threads():
 *
 *
 */
extern void iscsi_channel_stop_threads (iscsi_channel_t *c)
{
	if (c->channel_req_thread) {
		send_sig(SIGKILL, c->channel_req_thread, 1);
		down(&c->ct_req_done_sem);
	}

	if (c->channel_scan_thread) {
		send_sig(SIGKILL, c->channel_scan_thread, 1);
		down(&c->ct_scan_done_sem);
	}

	return;
}
