/*
 * Copyright (c) 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <netinet/in.h>

#ifdef EZLICENSER
#define ISCSI_EZH_ENABLE
#include <iscsi_ezlicenser.h>
char * licensekey = "/etc/iscsi/initiator.key";
#endif /* EZLICENSER */

#include <iscsi_initiator_ioctl.h>
#include <iscsi_initiator_ioctl_defs.h>

#define VERSION		VERSION_IPYXD

#define PARAM_FAILURE_CHANNEL_ID(cmd) \
	printf("Parameter ERROR: %s command requires iSCSI Channel ID (channel=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_CONNECTION_COUNT(cmd) \
	printf("Parameter ERROR: %s command requires Connection Count (conns=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_IPV4_ADDRESS(cmd) \
	printf("Parameter ERROR: %s command requires IPv4 Address (ip=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_PORT(cmd) \
	printf("Parameter ERROR: %s command requires port (port=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_SETINITIATORNAME(cmd) \
	printf("Parameter ERROR: %s command requires iSCSI Initiator Node Name" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_ISCSI_PARAMETER(cmd) \
	printf("Parameter ERROR: %s command requires key=value" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_NETWORK_DEVICE(cmd) \
	printf("Parameter ERROR: %s command requires network device (dev=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_SESSION_ID(cmd) \
	printf("Parameter ERROR: %s command requires iSCSI Session ID (sid=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_CONNECTION_ID(cmd) \
	printf("Parameter ERROR: %s command requires iSCSI Connection ID (cid=)" \
		" parameter to be set!\n", cmd);

#define PARAM_FAILURE_DEBUGERL_TYPE(cmd) \
	printf("Parameter ERROR: %s command requires DebugERL Type (type=)" \
		" parameter to be set!\n", cmd);

int get_hostname(char *hostname)
{
	struct hostent *host;

	if ((host = gethostbyname(hostname)) == NULL)
		return(-1);

	return(ntohl(*(int *)host->h_addr));
}

void usage (char *argv)
{
	printf("PyX iSCSI Initiator CTL %s\n\n", VERSION);
	printf("usage: %s <cmd> <arg> <arg> ... \n\n", argv);
	printf("\n");
}

static int check_parameters_init_channel (struct iscsi_client *c)
{
	printf("Initializing iSCSI Channel:\n");
	
	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("initchan");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (c->params_set & PARAM_MAX_SECTORS)
		printf("\tMax Sectors: %u\n", c->max_sectors);
	
	return(0);
}

static int check_parameters_free_channel (struct iscsi_client *c)
{
	printf("Freeing iSCSI Channel:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("freechan");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	return(0);
}

static int check_parameters_force_channel_offline (struct iscsi_client *c)
{
	printf("Forcing iSCSI Channel Offline:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("forcechanoffline");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);

	return(0);
}

static int check_parameters_set_channel_attrib (struct iscsi_client *c)
{
	printf("Setting Channel Attribute:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("setchanattrib");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (c->ca_params_set & PARAM_CA_CLOSECONN_REINSTATEMENT) {
		printf("\tCloseConnection Reinstatement: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_CLOSECONN_REINSTATEMENT;
	} else if (c->ca_params_set & PARAM_CA_CMDSN_TIMEOUT) {
		printf("\tCmdSN Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_CMDSN_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_CMDSN_TIMEOUT_RETRIES) {
		printf("\tCmdSN Timeout Retries: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_CMDSN_TIMEOUT_RETRIES;
	} else if (c->ca_params_set & PARAM_CA_DATAIN_TIMEOUT) {
		printf("\tDataIn Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_DATAIN_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_DATAIN_TIMEOUT_RETRIES) {
		printf("\tDataIn Timeout Retries: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_DATAIN_TIMEOUT_RETRIES;
	} else if (c->ca_params_set & PARAM_CA_IMMEDIATE_LOGOUT) {
		printf("\tImmediate Logout Requests: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_IMMEDIATE_LOGOUT;
	} else if (c->ca_params_set & CA_SET_IMMEDIATE_TEXT) {
		printf("\tImmediate Text Requests: %u\n", c->ca_value);
		c->channel_attrib = PARAM_CA_IMMEDIATE_TEXT;	
	} else if (c->ca_params_set & PARAM_CA_LOGIN_RETRIES) {
		printf("\tLogin Retries: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_LOGIN_RETRIES;
	} else if (c->ca_params_set & PARAM_CA_LOGIN_RETRY_WAIT) {
		printf("\tLogout Retry Wait: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_LOGIN_RETRY_WAIT;
	} else if (c->ca_params_set & PARAM_CA_LOGIN_TIMEOUT) {
		printf("\tLogin Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_LOGIN_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_LOGOUT_TIMEOUT) {
		printf("\tLogout Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_LOGOUT_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_NETIF_TIMEOUT) {
		printf("\tNetwork Interface Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_NETIF_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_NOPOUT_TIMEOUT) {
		printf("\tNopOut Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_NOPOUT_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_NOPOUT_RESPONSE_TIMEOUT) {
		printf("\tNopOut Response Timeout: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_NOPOUT_RESPONSE_TIMEOUT;
	} else if (c->ca_params_set & PARAM_CA_RANDOM_DATAOUT_PDU_OFFSETS) {
		printf("\tRandom DataOUT PDU Offsets: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_RANDOM_DATAOUT_PDU_OFFSETS;
	} else if (c->ca_params_set & PARAM_CA_TPGFAILOVER) {
		printf("\tTPG Failover: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_TPGFAILOVER;
	} else if (c->ca_params_set & PARAM_CA_TPGFAILOVER_ATTEMPTS) {
		printf("\tTPG Failover Attempts: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_TPGFAILOVER_ATTEMPTS;
	} else if (c->ca_params_set & PARAM_CA_TPGFAILOVER_LOGIN_RETRIES) {
		printf("\tTPG Failover Login Retries: %u\n", c->ca_value);
		c->channel_attrib = CA_SET_TPGFAILOVER_LOGIN_RETRIES;
	} else {
		printf("No channel attribute passed, ignoring request.\n");
		return(-1);
	}

	if (c->ca_params_set_count > 1) {
		printf("Unable to set multiple channel attributes in"
			" single request, ignoring request\n");
		return(-1);
	}
			
	return(0);
}

static int check_parameters_stop_channel (struct iscsi_client *c)
{
	printf("Stopping all activity on iSCSI Channel:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("stopchan");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);

	if (c->params_set & PARAM_LUN_REMOVE)
		printf("\tRemoving iSCSI and SCSI Logical Units from Host.\n");

	return(0);
}	

static int check_parameters_login (struct iscsi_client *c)
{
	struct in_addr ip_addr, *ip_addr_pntr = &ip_addr;

	printf("Starting iSCSI Login Process:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("login");
		return(-1);
	}       
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (!(c->params_set & PARAM_CONNECTION_COUNT))
		c->conns = 1;
	printf("\tConnection Count: %hd\n", c->conns);
	
	if (c->params_set & PARAM_NETWORK_DEVICE)
		printf("\tNetwork Device: %s\n", c->dev);
	
	if (!(c->params_set & PARAM_IPV4_ADDRESS)) {
		PARAM_FAILURE_IPV4_ADDRESS("login");
		return(-1);
	}
	ip_addr.s_addr = htonl(c->ip);
	printf("\tIPv4 Address: %s\n", inet_ntoa(ip_addr));

	if (!(c->params_set & PARAM_PORT)) {
		PARAM_FAILURE_PORT("login");
		return(-1);
	}
	printf("\t%s Port: %hd\n", ((c->params_set & PARAM_SCTP_TCP) ||
		(c->params_set & PARAM_SCTP_UDP)) ? "SCTP" : "TCP", c->port);
	
	return(0);
}

static int check_parameters_scanscsi (struct iscsi_client *c)
{
	printf("Performing iSCSI Logical Unit Scan:\n");
	
	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("scanscsi");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	return(0);
}

static int check_parameters_logoutsess (struct iscsi_client *c)
{
	printf("Performing iSCSI Session Logout:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("logoutsess");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (c->params_set & PARAM_LUN_REMOVE)
		printf("\tRemoving iSCSI and SCSI Logical Units from Host.\n");
	
	return(0);
}

static int check_parameters_logoutconn (struct iscsi_client *c)
{
	printf("Performing iSCSI Connection Logout for iSCSI Session:\n");
	
	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("logoutconn");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (!(c->params_set & PARAM_CONNECTION_ID)) {
		PARAM_FAILURE_CONNECTION_ID("logoutconn");
		return(-1);
	}
	printf("\tiSCSI Connection ID: %hd\n", c->cid);
	
	if (c->params_set & PARAM_ON_CONNECTION_ID)
		printf("\tOn iSCSI Connection ID: %hd\n", c->on_cid);
	
	if (c->params_set & PARAM_LUN_REMOVE)
		printf("\tRemoving iSCSI and SCSI Logical Units from Host.\n");
	
	return(0);
}

static int check_parameters_listchannelparams (struct iscsi_client *c)
{
	printf("Dumping Default Channel Parameter List:\n");

	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("listchanparams");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	return(0);
}

static int check_parameters_listsessparams (struct iscsi_client *c)
{
	printf("Dumping Parameter List for iSCSI Session:\n");

	if (!(c->params_set & PARAM_SESSION_ID)) {
		PARAM_FAILURE_SESSION_ID("listsessparams");
		return(-1);
	}
	printf("\tiSCSI Session ID: %u\n", c->sid);
	
	return(0);
}

static int check_parameters_setchannelparam (struct iscsi_client *c)
{
	printf("Setting Channel Parameter:\n");
	
	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("setchanparam");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
	
	if (!(c->params_set & PARAM_ISCSI_PARAMETER)) {
		PARAM_FAILURE_ISCSI_PARAMETER("setchanparam");
		return(-1);
	}
	printf("\tiSCSI Parameter: %s\n", c->keytext);
	
	return(0);
}

static int check_parameters_setsessparam (struct iscsi_client *c)
{
	printf("Setting Parameter for iSCSI Session:\n");

	if (!(c->params_set & PARAM_SESSION_ID)) {
		PARAM_FAILURE_SESSION_ID("setsessparam");
		return(-1);
	}
	printf("\tiSCSI Session ID: %u\n", c->sid);

	if (!(c->params_set & PARAM_ISCSI_PARAMETER)) {
		PARAM_FAILURE_ISCSI_PARAMETER("setsessparam");
		return(-1);
	}
	printf("\tiSCSI Parameter: %s\n", c->keytext);
	
	return(0);
}

static int check_parameters_setinitiatorname (struct iscsi_client *c)
{
	printf("Setting InitiatorName for iSCSI Initiator Node:\n");
	
	if (!(c->params_set & PARAM_SETINITIATORNAME)) {
		PARAM_FAILURE_SETINITIATORNAME("initiatorname");
		return(-1);
	}
	printf("\tiSCSI Initiator Node Name: %s\n", c->keytext);
	
	return(0);
}

static int check_parameters_listknowntargets (struct iscsi_client *c)
{
	printf("Dumping List of Discovered iSCSI Target Nodes:\n");
	
	return(0);
}

static int check_parameters_addconnection (struct iscsi_client *c)
{
	struct in_addr ip_addr, *ip_addr_pntr = &ip_addr;
	
	printf("Adding iSCSI Connection to existing iSCSI Session:\n");

        if (!(c->params_set & PARAM_CONNECTION_COUNT))
		c->conns = 1;
	printf("\tConnection Count: %hd\n", c->conns);
	
	if (!(c->params_set & PARAM_CHANNEL_ID)) {
		PARAM_FAILURE_CHANNEL_ID("addconn");
		return(-1);
	}
	printf("\tiSCSI Channel ID: %d\n", c->channel);
		
	if (c->params_set & PARAM_NETWORK_DEVICE)
		printf("\tNetwork Device: %s\n", c->dev);

	if (!(c->params_set & PARAM_IPV4_ADDRESS)) {
		PARAM_FAILURE_IPV4_ADDRESS("addconn");
		return(-1);
	}
	ip_addr.s_addr = htonl(c->ip);
	printf("\tIPv4 Address: %s\n", inet_ntoa(ip_addr));

	if (!(c->params_set & PARAM_PORT)) {
		PARAM_FAILURE_PORT("addconn");
		return(-1);
	}
	printf("\t%s Port: %hd\n", ((c->params_set & PARAM_SCTP_TCP) ||
		(c->params_set & PARAM_SCTP_UDP)) ? "SCTP" : "TCP", c->port);
						
	return(0);
}

static int check_parameters_debug_erl (struct iscsi_client *c)
{
	printf("Starting Debug ErrorRecoveryLevel Call:\n");

	if (!(c->params_set & PARAM_SESSION_ID)) {
		PARAM_FAILURE_SESSION_ID("debugerl");
		return(-1);
	}
	printf("\tiSCSI Session ID: %u\n", c->sid);

	if (!(c->params_set & PARAM_CONNECTION_ID)) {
		PARAM_FAILURE_CONNECTION_ID("debugerl");
		return(-1);
	}
	printf("\tiSCSI Connection CID: %hd\n", c->cid);

	if (!(c->params_set & PARAM_DEBUGERL_TYPE)) {
		PARAM_FAILURE_DEBUGERL_TYPE("debugerl");
		return(-1);
	}
	printf("\tDebug ERL Type: %d\n", c->debug_type);
	
	return(0);
}

static int check_parameters_check_key (struct iscsi_client *c)
{
	printf("Checking License Key:\n");

	return(0);
}

static int check_ioctl_arguments (int action, struct iscsi_client *c)
{
	switch (action) {
	case ISCSI_INITIATOR_FULLINIT_CHANNEL:
	case ISCSI_INITIATOR_INIT_CHANNEL:
		return(check_parameters_init_channel(c));
	case ISCSI_INITIATOR_FREE_CHANNEL:
		return(check_parameters_free_channel(c));
	case ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE:
		return(check_parameters_force_channel_offline(c));
	case ISCSI_INITIATOR_SET_CHANNEL_ATTRIB:
		return(check_parameters_set_channel_attrib(c));
	case ISCSI_INITIATOR_STOP_CHANNEL:
		return(check_parameters_stop_channel(c));
	case ISCSI_INITIATOR_LOGIN:
		return(check_parameters_login(c));
	case ISCSI_INITIATOR_SCANSCSI:
		return(check_parameters_scanscsi(c));
	case ISCSI_INITIATOR_LOGOUTSESS:
		return(check_parameters_logoutsess(c));
	case ISCSI_INITIATOR_LOGOUTCONN:
		return(check_parameters_logoutconn(c));
	case ISCSI_INITIATOR_LISTCHANNELPARAMS:
		return(check_parameters_listchannelparams(c));
	case ISCSI_INITIATOR_LISTSESSPARAMS:
		return(check_parameters_listsessparams(c));
	case ISCSI_INITIATOR_SETCHANNELPARAM:
		return(check_parameters_setchannelparam(c));
	case ISCSI_INITIATOR_SETSESSPARAM:
		return(check_parameters_setsessparam(c));
	case ISCSI_INITIATOR_SETINITNAME:
		return(check_parameters_setinitiatorname(c));
	case ISCSI_INITIATOR_LISTKNOWNTARGETS:
		return(check_parameters_listknowntargets(c));
	case ISCSI_INITIATOR_ADDCONN:
		return(check_parameters_addconnection(c));
	case ISCSI_INITIATOR_DEBUG_ERL:
		return(check_parameters_debug_erl(c));
	case ISCSI_INITIATOR_CHECK_KEY:
		return(check_parameters_check_key(c));
	default:
		printf("Unknown action %u\n", action);
		return(-EINVAL);
	}

}

static int decode_init_channel (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_free_channel (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_force_channel_offline (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_set_channel_attrib (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_stop_channel (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_login (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_scanscsi (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_logoutsess (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");
	
	return(-1);
}

static int decode_logoutconn (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_listchannelparams (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_listsessparams (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_setchannelparam (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_setsessparam (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_setinitiatorname (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_listknowntargets (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_addconnection (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_debug_erl (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_check_key (struct iscsi_client *c, int ret)
{
	if (!ret) {
		printf("STATUS: OK\n");
		return(0);
	}

	printf("STATUS: FAILED\n");
	printf("REASON: \n");

	return(-1);
}

static int decode_ioctl_return (int action, struct iscsi_client *c, int ret)
{
	switch (action) {
	case ISCSI_INITIATOR_FULLINIT_CHANNEL:
	case ISCSI_INITIATOR_INIT_CHANNEL:
		return(decode_init_channel(c, ret));
	case ISCSI_INITIATOR_FREE_CHANNEL:
		return(decode_free_channel(c, ret));
	case ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE:
		return(decode_force_channel_offline(c, ret));
	case ISCSI_INITIATOR_SET_CHANNEL_ATTRIB:
		return(decode_set_channel_attrib(c, ret));
	case ISCSI_INITIATOR_STOP_CHANNEL:
		return(decode_stop_channel(c, ret));
	case ISCSI_INITIATOR_LOGIN:
		return(decode_login(c, ret));
	case ISCSI_INITIATOR_SCANSCSI:
		return(decode_scanscsi(c, ret));
	case ISCSI_INITIATOR_LOGOUTSESS:
		return(decode_logoutsess(c, ret));
	case ISCSI_INITIATOR_LOGOUTCONN:
		return(decode_logoutconn(c, ret));
	case ISCSI_INITIATOR_LISTCHANNELPARAMS:
		return(decode_listchannelparams(c, ret));
	case ISCSI_INITIATOR_LISTSESSPARAMS:
		return(decode_listsessparams(c, ret));
	case ISCSI_INITIATOR_SETCHANNELPARAM:
		return(decode_setchannelparam(c, ret));
	case ISCSI_INITIATOR_SETSESSPARAM:
		return(decode_setsessparam(c, ret));
	case ISCSI_INITIATOR_SETINITNAME:
		return(decode_setinitiatorname(c, ret));
	case ISCSI_INITIATOR_LISTKNOWNTARGETS:
		return(decode_listknowntargets(c, ret));
	case ISCSI_INITIATOR_ADDCONN:
		return(decode_addconnection(c, ret));
	case ISCSI_INITIATOR_DEBUG_ERL:
		return(decode_debug_erl(c, ret));
	case ISCSI_INITIATOR_CHECK_KEY:
		return(decode_check_key(c, ret));
	default:
		printf("Unknown action %u\n", action);
		return(-EINVAL);
	}
}

#define GETDEV(ISD)	((10)<<8 | (ISD))

int main (int argc, char *argv[])
{
	char dev[5], *delimptr, *endptr, *endvalptr, *paramptr, tmp[32];
	short int cid, on_cid, conns, type;
	int fd, i, ret;
	int count, port, sid;
	int channel, force, ip_address, lun_remove, sctp;
	static int action;	
	unsigned int closeconn_reinstatement;
	unsigned int cmdsn_timeout, cmdsn_timeout_retries, conn_recovery_timeout;
	unsigned int datain_timeout, datain_timeout_retries, immediate_logout;
	unsigned int immediate_text, login_retries;
	unsigned int login_retry_wait, login_timeout, logout_timeout, max_sectors;
	unsigned int netif_timeout, nopout_timeout, nopout_response_timeout;
	unsigned int random_dataout_pdu_offsets, tpgfailover, tpgfailover_count;
	unsigned int tpgfailover_login_retries;
	DIR *dir;
	struct dirent *dirv;
	struct in_addr ip_addr, *ip_addr_pntr = &ip_addr;
	struct iscsi_client c;

	if (argc < 2 || strncmp(argv[1], "help", sizeof(argv[1])) == 0) {
		usage(argv[0]);
		return(-1);
	}

	unlink(ISCSI_INITIATOR_DEVICE);

	if (mknod(ISCSI_INITIATOR_DEVICE, (S_IFCHR | 0600), GETDEV(ISCSI_INITIATOR_MINOR))) {
		printf("Failed to created %s!\n", ISCSI_INITIATOR_DEVICE);
		return(-1);
	}

	snprintf(tmp, sizeof(tmp), "%s", argv[1]);
	memset((void *)&c, 0, sizeof(struct iscsi_client));

	if (!strncmp(tmp, "fullinitchan", sizeof(tmp)))
		action = ISCSI_INITIATOR_FULLINIT_CHANNEL;
	else if (!strncmp(tmp, "initchan", sizeof(tmp)))
		action = ISCSI_INITIATOR_INIT_CHANNEL;
	else if (!strncmp(tmp, "freechan", sizeof(tmp)))
		action = ISCSI_INITIATOR_FREE_CHANNEL;
	else if (!strncmp(tmp, "forcechanoffline", sizeof(tmp)))
		action = ISCSI_INITIATOR_FORCE_CHANNEL_OFFLINE;
	else if (!strncmp(tmp, "setchanattrib", sizeof(tmp)))
		action = ISCSI_INITIATOR_SET_CHANNEL_ATTRIB;
	else if (!strncmp(tmp, "stopchan", sizeof(tmp)))
		action = ISCSI_INITIATOR_STOP_CHANNEL;
	else if (!strncmp(tmp, "login", sizeof(tmp)))
		action = ISCSI_INITIATOR_LOGIN;
	else if (!strncmp(tmp, "scanscsi", sizeof(tmp)))
		action = ISCSI_INITIATOR_SCANSCSI;
	else if (!strncmp(tmp, "logoutsess", sizeof(tmp)))
		action = ISCSI_INITIATOR_LOGOUTSESS;
	else if (!strncmp(tmp, "logoutconn", sizeof(tmp)))
		action = ISCSI_INITIATOR_LOGOUTCONN;
	else if (!strncmp(tmp, "setinitname", sizeof(tmp)))
		action = ISCSI_INITIATOR_SETINITNAME;
	else if (!strncmp(tmp, "listchanparams", sizeof(tmp)))
		action = ISCSI_INITIATOR_LISTCHANNELPARAMS;
	else if (!strncmp(tmp, "listsessparams", sizeof(tmp)))
		action = ISCSI_INITIATOR_LISTSESSPARAMS;
	else if (!strncmp(tmp, "setchanparam", sizeof(tmp)))
		action = ISCSI_INITIATOR_SETCHANNELPARAM;
	else if (!strncmp(tmp, "setsessparam", sizeof(tmp)))
		action = ISCSI_INITIATOR_SETSESSPARAM;
	else if (!strncmp(tmp, "listsessinfo", sizeof(tmp)))
		;
	else if (!strncmp(tmp, "listtargets", sizeof(tmp)))
		action = ISCSI_INITIATOR_LISTKNOWNTARGETS;
	else if (!strncmp(tmp, "addconn", sizeof(tmp)))
		action = ISCSI_INITIATOR_ADDCONN;
#ifdef DEBUG_ERL
	else if (!strncmp(tmp, "debugerl", sizeof(tmp)))
		action = ISCSI_INITIATOR_DEBUG_ERL;
#endif
#ifdef EZLICENSER
	else if (!strncmp(tmp, "unlock", sizeof(tmp)))
		action = ISCSI_INITIATOR_CHECK_KEY;
#endif
	else {
		printf("action is not valid!\n");
		usage(argv[0]);
		return -EINVAL;
	}

	for (i = 2; i < argc; i++) {
		if (strncmp(argv[i], "channel=", 8) == 0) {
			channel = atoi(&argv[i][8]);
			c.channel = channel;
			c.params_set |= PARAM_CHANNEL_ID;
		}

		if (strncmp(argv[i], "conns=", 6) == 0) {
			conns = atoi(&argv[i][6]);
			c.conns = conns;
			c.params_set |= PARAM_CONNECTION_COUNT;
		}
		
		if (strncmp(argv[i], "ip=", 3) == 0) {
			if (isalpha(argv[i][3])) {
				ip_address = get_hostname(&argv[i][3]);
				c.ip = ip_address;
			} else if (isdigit(argv[i][3])) {
				if (inet_aton(&argv[i][3], ip_addr_pntr) == 0)
					return(-EINVAL);

				ip_address = ntohl(ip_addr.s_addr);
				c.ip = ip_address;
			}
			c.params_set |= PARAM_IPV4_ADDRESS;
		}

		if (strncmp(argv[i], "port=", 5) == 0) {
			port = strtoul(&argv[i][5], &endptr, 0);
			c.port = port;
			c.params_set |= PARAM_PORT;
		}

		if (strncmp(argv[i], "sctp=", 5) == 0) {
			sctp = strtoul(&argv[i][5], &endptr, 0);
			if (sctp == 1)
				c.params_set |= PARAM_SCTP_TCP;
			else if (sctp == 2)
				c.params_set |= PARAM_SCTP_UDP;
			else {
				printf("Unknown sctp= value\n");
				return(-1);
			}
		}

		if (strncmp(argv[i], "dev=", 4) == 0) {
			if (strlen(&argv[i][4]) > 11) {
				printf("Network device name is invalid.\n");
				return(-1);
			}
			strncpy(c.dev, &argv[i][4], 12);
			c.params_set |= PARAM_NETWORK_DEVICE;
		}
		
		if (strncmp(argv[i], "sid=", 4) == 0) {
			sid = atoi(&argv[i][4]);
			c.sid = sid;
			c.params_set |= PARAM_SESSION_ID;
		}

		if (strncmp(argv[i], "cid=", 4) == 0) {
			cid = atoi(&argv[i][4]);
			c.cid = cid;
			c.params_set |= PARAM_CONNECTION_ID;
		}

		if (strncmp(argv[i], "on_cid=", 7) == 0) {
			on_cid = atoi(&argv[i][7]);
			c.on_cid = on_cid;
			c.params_set |= PARAM_ON_CONNECTION_ID;
		}
		
		if (strncmp(argv[i], "type=", 5) == 0) {
			type = atoi(&argv[i][5]);
			c.debug_type = type;
			c.params_set |= PARAM_DEBUGERL_TYPE;
		}

		if (strncmp(argv[i], "count=", 6) == 0) {
			count = atoi(&argv[i][6]);
			c.count = count;
			c.params_set |= PARAM_DEBUGERL_COUNT;
		}

		if (strncmp(argv[i], "force=", 6) == 0) {
			force = atoi(&argv[i][6]);
			c.force = force;
			c.params_set |= PARAM_FORCE;
		}

		if (strncmp(argv[i], "lr=", 3) == 0) {
			lun_remove = atoi(&argv[i][3]);
			c.lun_remove = lun_remove;
			c.params_set |= PARAM_LUN_REMOVE;
		}

		if (strncmp(argv[i], "max_sectors=", 12) == 0) {
			max_sectors = atoi(&argv[i][12]);
			c.max_sectors = max_sectors;
			c.params_set |= PARAM_MAX_SECTORS;
		}

		if (strncmp(argv[i], "closeconn_reinstatement=", 24) == 0) {
			closeconn_reinstatement = atoi(&argv[i][24]);
			c.ca_value = closeconn_reinstatement;
			c.ca_params_set |= PARAM_CA_CLOSECONN_REINSTATEMENT;
			c.ca_params_set_count++;
		}
		
		if (strncmp(argv[i], "cmdsn_timeout=", 14) == 0) {
			cmdsn_timeout = atoi(&argv[i][14]);
			c.ca_value = cmdsn_timeout;
			c.ca_params_set |= PARAM_CA_CMDSN_TIMEOUT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "cmdsn_timeout_retries=", 22) == 0) {
			cmdsn_timeout_retries = atoi(&argv[i][22]);
			c.ca_value = cmdsn_timeout_retries;
			c.ca_params_set |= PARAM_CA_CMDSN_TIMEOUT_RETRIES;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "conn_recovery_timeout=", 22) == 0) {
		       conn_recovery_timeout = atoi(&argv[i][22]);	
		       c.ca_value = conn_recovery_timeout;
		       c.ca_params_set |= PARAM_CA_CONN_RECOVERY_TIMEOUT;
		       c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "datain_timeout=", 15) == 0) {
			datain_timeout = atoi(&argv[i][15]);
			c.ca_value = datain_timeout;
			c.ca_params_set |= PARAM_CA_DATAIN_TIMEOUT;
			c.ca_params_set_count++;
		}			

		if (strncmp(argv[i], "datain_timeout_retries=", 23) == 0) {
			datain_timeout_retries = atoi(&argv[i][23]);
			c.ca_value = datain_timeout_retries;
			c.ca_params_set |= PARAM_CA_DATAIN_TIMEOUT_RETRIES;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "immediate_logout=", 17) == 0) {
			immediate_logout = atoi(&argv[i][17]);
			c.ca_value = immediate_logout;
			c.ca_params_set |= PARAM_CA_IMMEDIATE_LOGOUT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "immediate_text=", 15) == 0) {
			immediate_text = atoi(&argv[i][15]);
			c.ca_value = immediate_text;
			c.ca_params_set |= CA_SET_IMMEDIATE_TEXT;
			c.ca_params_set_count++;
		}
		
		if (strncmp(argv[i], "login_retries=", 14) == 0) {
			login_retries = atoi(&argv[i][14]);
			c.ca_value = login_retries;
	 		c.ca_params_set |= PARAM_CA_LOGIN_RETRIES;
			c.ca_params_set_count++;		
		}

		if (strncmp(argv[i], "login_retry_wait=", 17) == 0) {
			login_retry_wait = atoi(&argv[i][17]);
			c.ca_value = login_retry_wait;
			c.ca_params_set |= PARAM_CA_LOGIN_RETRY_WAIT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "login_timeout=", 14) == 0) {
		       login_timeout = atoi(&argv[i][14]);	
		       c.ca_value = login_timeout;
		       c.ca_params_set |= PARAM_CA_LOGIN_TIMEOUT;
		       c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "logout_timeout=", 15) == 0) {
			logout_timeout = atoi(&argv[i][15]);
			c.ca_value = logout_timeout;
			c.ca_params_set |= PARAM_CA_LOGOUT_TIMEOUT;
			c.ca_params_set_count++;
		}
			
		if (strncmp(argv[i], "netif_timeout=", 14) == 0) {
			netif_timeout = atoi(&argv[i][14]);
			c.ca_value = netif_timeout;
			c.ca_params_set |= PARAM_CA_NETIF_TIMEOUT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "nopout_timeout=", 15) == 0) {
			nopout_timeout = atoi(&argv[i][15]);
			c.ca_value = nopout_timeout;
			c.ca_params_set |= PARAM_CA_NOPOUT_TIMEOUT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "nopout_response_timeout=", 24) == 0) {
			nopout_response_timeout = atoi(&argv[i][24]);
			c.ca_value = nopout_response_timeout;
			c.ca_params_set |= PARAM_CA_NOPOUT_RESPONSE_TIMEOUT;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "random_dataout_pdu_offsets=", 27) == 0) {
			random_dataout_pdu_offsets = atoi(&argv[i][27]);	
			c.ca_value = random_dataout_pdu_offsets;
			c.ca_params_set |= PARAM_CA_RANDOM_DATAOUT_PDU_OFFSETS;
			c.ca_params_set_count++;
		}
		
		if (strncmp(argv[i], "tpgfailover=", 12) == 0) {
			tpgfailover = atoi(&argv[i][12]);
			c.ca_value = tpgfailover;
			c.ca_params_set |= PARAM_CA_TPGFAILOVER;
			c.ca_params_set_count++;
		}
		
		if (strncmp(argv[i], "tpgfailover_attempts=", 21) == 0) {
		       tpgfailover_count = atoi(&argv[i][21]);	
		       c.ca_value = tpgfailover_count;
		       c.ca_params_set |= PARAM_CA_TPGFAILOVER_ATTEMPTS;
		       c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "tpgfailover_login_retries=", 26) == 0) {
			tpgfailover_login_retries = atoi(&argv[i][26]);
			c.ca_value = tpgfailover_login_retries;
			c.ca_params_set |= PARAM_CA_TPGFAILOVER_LOGIN_RETRIES;
			c.ca_params_set_count++;
		}

		if (strncmp(argv[i], "targetname=", 11) == 0) {
			if (strlen(&argv[i][14]) > 255) {
				printf("targetname value exceeds 255 character maximum.\n");
				return(-EINVAL);
			}

			strncpy(c.value, &argv[i][11], strlen(&argv[i][11]));
			c.params_set |= PARAM_TARGETNAME;
		}
		
		if (action == ISCSI_INITIATOR_SETINITNAME) {
			if (argc < 1) {
				printf("setinitname requires 2 parameters\n");
				return(-EINVAL);
			}
			paramptr = argv[2];
			if (strlen(paramptr) > 255) {
				printf("InitiatorName value exceeds 255"
					" character maximum.\n");
				return(-EINVAL);
			}
			strncpy(c.keytext, paramptr, sizeof(c.keytext));
			c.params_set |= PARAM_SETINITIATORNAME;
		}

		if ((action == ISCSI_INITIATOR_SETCHANNELPARAM) ||
		    (action == ISCSI_INITIATOR_SETSESSPARAM)) {
			if (argc < 4) {
				printf("%s requires 3 parameters\n", argv[1]);
				return(-EINVAL);
			}
			paramptr = argv[3];

			if (strlen(paramptr) > 513) { /* key size + = + value size */
				printf("key=value exceeds maximum size\n");
				return(-EINVAL);
			}
			
			if ((delimptr = strchr(paramptr, '=')) == NULL) {
				printf("No Delimiting \"=\" found in parameter!\n");
				return(-EINVAL);
			}
			if ((endvalptr = strchr(paramptr, '\0')) == NULL) {
				printf("Error getting endvalptr\n");
				return(-EINVAL);
			}
			strncpy(c.keytext, paramptr, sizeof(c.keytext));
			c.params_set |= PARAM_ISCSI_PARAMETER;
		}			
	}

	printf("\n");

	if (check_ioctl_arguments(action, &c) < 0)
		return(-EINVAL);
	
	printf("\n");

	fd = open(ISCSI_INITIATOR_DEVICE, O_RDWR);
	if (fd < 0) {
		printf("open() to %s failed!\n", ISCSI_INITIATOR_DEVICE);
		close(fd);
		return(-1);
	}

	ret = ioctl(fd, action, &c);
	close(fd);

	return(decode_ioctl_return(action, &c, ret));
}
