/*
 * Copyright (c) 2003-2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 * Copyright (c) 2001 Cisco Systems (base64 decoding support)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This file houses functions related to iSCSI Initiator CHAP Authenication.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <md5.h>
#include <iscsi_chap.h>
#include <iscsi_auth.h>

unsigned char client_chap_id = 0;

#define DEBUG_CHAP

#ifdef DEBUG_CHAP
#define PRINT(x...)		printf(x)
#else
#define PRINT(x...)		do { } while(0);
#endif 
		
unsigned char chap_asciihex_to_binaryhex (char val[2])
{
	unsigned char result = 0;

	/*
	 * MSB
	 */
	if ((val[0] >= 'a') && (val[0] <= 'f'))
		result = ((val[0] - 'a' + 10) & 0xf) << 4;
	else
		if ((val[0] >= 'A') && (val[0] <= 'F'))
			result = ((val[0] - 'A' + 10) & 0xf) << 4;
		else // digit
			result = ((val[0] - '0') & 0xf) << 4;
	/*
	 * LSB
	 */
	if ((val[1] >= 'a') && (val[1] <= 'f'))
		result |= ((val[1] - 'a' + 10) & 0xf);
	else
		if ((val[1] >= 'A') && (val[1] <= 'F'))
			result |= ((val[1] - 'A' + 10) & 0xf);
       		else // digit
			result |= ((val[1] - '0') & 0xf);	

	return(result);
}

void chap_string_to_hex (char *dst, char *src, int len)
{
	int i = 0, j = 0;

	for (i = 0; i < len; i += 2)
		dst[j++] = (unsigned char) chap_asciihex_to_binaryhex(&src[i]);
	
	dst[j] = '\0';

	return;
}	

static const char acl_base64_string[] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static int              
acl_str_index(const char *s, int c)
{                       
        char *str = strchr(s, c); 
                        
        if (str)        
                return (str - s);
        else            
                return -1;
}       

int chap_string_to_base64 (char *dst, char *src, int len)
{
	int i;
	unsigned int n = 0, count = 0, data_length = len;

	while (*src != '\0' && *src != '=') {

		i = acl_str_index(acl_base64_string, *src++);
		if (i < 0)
			return(-1);
		
		n = (n << 6 | (unsigned int)i);
		count++;

		if (count >= 4) {
			if (data_length < 3)
				return(-1);

                        *dst++ = n >> 16;
                        *dst++ = n >> 8;
                        *dst++ = n;
                        data_length -= 3;
                        n = 0;
                        count = 0;
                }
        }

	while (*src != '\0')
		if (*src != '=')
			return(-1);

	if (count == 0) {
	
	} else if (count == 2) {
                if (data_length < 1)
                        return(-1);       /* error, too much data */
                n = n >> 4;
                *dst++ = n;
                data_length--;
        } else if (count == 3) {
                if (data_length < 2)
                        return(-1);       /* error, too much data */
                n = n >> 2;
                *dst++ = n >> 8;
                *dst++ = n;
                data_length -= 2;
	}

	return(len - data_length);
}

void chap_binaryhex_to_asciihex (char *dst, char *src, int src_len)
{
	int i;

	for (i = 0; i < src_len; i++)
		sprintf(&dst[i*2], "%02x", (int) src[i] & 0xff);
	
	return;
}

void chap_set_random (char *data, int length)
{
	long r;
	unsigned n;

	while (length > 0) {

		r = rand();
		r = r ^ (r >> 8);
		r = r ^ (r >> 4);
		n = r & 0x7;

		r = rand();
		r = r ^ (r >> 8);
		r = r ^ (r >> 5);
		n = (n << 3) | (r & 0x7);

		r = rand();
		r = r ^ (r >> 8);
		r = r ^ (r >> 5);
		n = (n << 2) | (r & 0x3);

		*data++ = n;
		 length--;
	}
}
	
static int chap_client_open (auth_conn_t *auth, char *out_text, unsigned int *out_len)
{
	iscsi_chap_t *iscsi_chap;	

	if (!(auth->auth_protocol = (iscsi_chap_t *) malloc(sizeof(iscsi_chap_t))))
		return(-1);

	iscsi_chap = (iscsi_chap_t *) auth->auth_protocol;
	memset(iscsi_chap, 0, sizeof(iscsi_chap_t));

	if (iscsi_get_client_auth_info(auth) < 0)
		return(-1);

	/*
	 * Currently only MD5 is supported.
	 */
	*out_len = sprintf(out_text, "CHAP_A=5");
	*out_len += 1;
	iscsi_chap->digest_type = CHAP_DIGEST_MD5;

	PRINT("[client] Sending CHAP_A=%d\n\n", iscsi_chap->digest_type);

	return(0);
}

void chap_close (auth_conn_t *auth)
{
	if (auth->auth_protocol)
		free(auth->auth_protocol);

	remove_auth_conn(auth->auth_id);
	return;
}

int chap_gen_challenge (auth_conn_t *auth, int caller, char *C_str, unsigned int *C_len)
{
	unsigned char challenge_asciihex[CHAP_CHALLENGE_LENGTH * 2 + 1];
	iscsi_chap_t *iscsi_chap = (iscsi_chap_t *) auth->auth_protocol;

	memset(challenge_asciihex, 0, CHAP_CHALLENGE_LENGTH * 2 + 1);
	
	chap_set_random(iscsi_chap->challenge, CHAP_CHALLENGE_LENGTH);
	chap_binaryhex_to_asciihex(challenge_asciihex, iscsi_chap->challenge,
					CHAP_CHALLENGE_LENGTH);

	/*
	 * Set CHAP_C, and copy the generated challenge into C_str.
	 */
	*C_len += sprintf(C_str+*C_len, "CHAP_C=0x%s", challenge_asciihex);
	*C_len += 1;

	PRINT("[%s] Sending CHAP_C=0x%s\n\n", (caller) ? "server" : "client",
			challenge_asciihex);

	return(0);
}

int chap_gen_response (auth_conn_t *auth, const char *AIC_ptr, char *NR_ptr, unsigned int *NR_len)
{
	unsigned char id, type = 0;
	unsigned char challenge[250], challenge_binhex[120];
	unsigned char digest[MD5_SIGNATURE_SIZE], identifier[10];
	unsigned char response[MD5_SIGNATURE_SIZE * 2 + 2];
	int chal_len = 0;
	iscsi_chap_t *iscsi_chap = (iscsi_chap_t *) auth->auth_protocol;
	
	memset(challenge, 0, sizeof(challenge));
	memset(challenge_binhex, 0, sizeof(challenge_binhex));
	memset(identifier, 0, sizeof(identifier));
	memset(response, 0, sizeof(response));
	
	/*
	 * Check CHAP_A is MD5.
	 */
	if (strncmp(AIC_ptr, "CHAP_A=5", 8)) {
		printf("Target does not support MD5.\n");
		return(-1);
	}

	PRINT("[client] Got CHAP_A=5\n");
	
	/*
	 * Get CHAP_I.
	 */
	if (extract_param(AIC_ptr, "CHAP_I", 10, identifier, &type) < 0) {
		printf("Could not find CHAP_I.\n");
		return(-1);
	}
	if (type == HEX) {
		id = atoi((char *)&identifier[2]);
	} else {
		id = atoi(identifier);
	}

	/*
	 * RFC 1994 says Identifier is no more than octet (8 bits).
	 */
	PRINT("[client] Got CHAP_I=%d\n", id);
	
	/*
	 * Get CHAP_C.
	 */
	if (extract_param(AIC_ptr, "CHAP_C", 250, challenge, &type) < 0) {
		printf("Could not find CHAP_C.\n");
		return(-1);
	}

	if (type == DECIMAL) {
		printf("Found DECIMAL CHAP_C\n");
		return(-1);
	}

	PRINT("[client] Got CHAP_C=%s\n", challenge);
	
	MD5Init(&iscsi_chap->md5_context);
	MD5Update(&iscsi_chap->md5_context, &id, 1);
	MD5Update(&iscsi_chap->md5_context, auth->password, strlen(auth->password));

	/*
	 * Convert received challenge to binary hex.
	 */
	if (type == HEX) {
		printf("[client] Found HEX CHAP_C.\n");
		chap_string_to_hex(challenge_binhex, challenge, strlen(challenge));
		chal_len = 16;
	} else if (type == BASE64) {
		printf("[client] Found BASE64 CHAP_C.\n");
		if ((chal_len = chap_string_to_base64(challenge_binhex, challenge, 12)) < 0) {
			printf("chap_string_to_base64() failed\n");
			return(-1);
		}
	}

	MD5Update(&iscsi_chap->md5_context, challenge_binhex, chal_len);
	MD5Final(digest, &iscsi_chap->md5_context);
	
	/*
	 * Generate CHAP_N and CHAP_R.
	 */
	*NR_len = sprintf(NR_ptr, "CHAP_N=%s", auth->userid); 
	*NR_len += 1;
	
	PRINT("[client] Sending CHAP_N=%s\n", auth->userid);

	/*
	 * Convert response from binary hex to ascii hext.
	 */
	chap_binaryhex_to_asciihex(response, digest, MD5_SIGNATURE_SIZE);

	*NR_len += sprintf(NR_ptr+*NR_len, "CHAP_R=0x%s", response);
	*NR_len += 1;

	PRINT("[client] Sending CHAP_R=0x%s\n", response);
	
	if (!auth->authenticate_target) {
		PRINT("\n");
		return(0);
	}
		
	/*
	 * Send our own CHAP_I and CHAP_C here for mutual authentication.
	 */
	iscsi_chap->id = client_chap_id++;
	*NR_len += sprintf(NR_ptr+*NR_len, "CHAP_I=%d", iscsi_chap->id);
	*NR_len += 1;

	PRINT("[client] Sending CHAP_I=%d\n", iscsi_chap->id);

	return(chap_gen_challenge(auth, 0, NR_ptr, NR_len));
}	

int chap_client_compute_md5 (auth_conn_t *auth, char *NR_in_ptr)
{
	unsigned char type;
	unsigned char client_digest[MD5_SIGNATURE_SIZE], server_digest[MD5_SIGNATURE_SIZE];
	unsigned char chap_n[MAX_CHAP_N_SIZE], chap_r[MAX_RESPONSE_LENGTH];
	iscsi_chap_t *iscsi_chap = (iscsi_chap_t *) auth->auth_protocol;

	memset(chap_n, 0, MAX_CHAP_N_SIZE);
	memset(chap_r, 0, MAX_RESPONSE_LENGTH);
	memset(client_digest, 0, MD5_SIGNATURE_SIZE);
	memset(server_digest, 0, MD5_SIGNATURE_SIZE);
	
	/*
	 * Extract CHAP_N.
	 */
	if (extract_param(NR_in_ptr, "CHAP_N", MAX_CHAP_N_SIZE, chap_n, &type) < 0) {
		printf("Could not find CHAP_N.\n");
		return(-1);
	}       

	if (type == HEX) {
		printf("Could not find CHAP_N.\n");
		return(-1);
	}

	if (memcmp(chap_n, auth->ma_userid, strlen(auth->ma_userid)) != 0) {
		printf("CHAP_N values do not match!\n");
		return(-1);
	}

	PRINT("[client] Got CHAP_N=%s\n", chap_n);
	
	/*
	 * Extract CHAP_R.
	 */
	if (extract_param(NR_in_ptr, "CHAP_R", MAX_RESPONSE_LENGTH, chap_r, &type) < 0) {
		printf("Could not find CHAP_R.\n");
		return(-1);
	}

	if (type != HEX) {
		printf("Could not find CHAP_R.\n");
		return(-1);
	}

	PRINT("[client] Got CHAP_R=%s\n", chap_r);
	
	chap_string_to_hex(server_digest, chap_r, strlen(chap_r));

	MD5Init(&iscsi_chap->md5_context);
	MD5Update(&iscsi_chap->md5_context, &iscsi_chap->id, 1);
	MD5Update(&iscsi_chap->md5_context, auth->ma_password, strlen(auth->ma_password));
	MD5Update(&iscsi_chap->md5_context, iscsi_chap->challenge, strlen(iscsi_chap->challenge));
	MD5Final(client_digest, &iscsi_chap->md5_context);
					        
	if (memcmp(client_digest, server_digest, MD5_SIGNATURE_SIZE) != 0) {
		PRINT("[client] MD5 Digests do not match!\n\n");
		return(-1);
	} else
		PRINT("[client] MD5 Digests match, mutual CHAP authetication successful.\n\n");
	
	return(0);
}

int chap_got_mutal_auth_response (auth_conn_t *auth, char *NR_in_ptr)
{
	iscsi_chap_t *iscsi_chap = (iscsi_chap_t *) auth->auth_protocol;

	switch (iscsi_chap->digest_type) {
	case CHAP_DIGEST_MD5:
		if (chap_client_compute_md5(auth, NR_in_ptr) < 0)
			return(-1);
		break;
	default:
		printf("Unknown CHAP digest type %d!\n", iscsi_chap->digest_type);
		return(-1);
	}

	return(0);
}

char chap_main_loop (auth_conn_t *auth, int chap_role, char *in_text, char *out_text, int *in_len, int *out_len)
{
	if (!auth->auth_state) {
		if (chap_client_open(auth, out_text, out_len) < 0) {
			printf("chap_client_open() failed!\n");
			chap_close(auth);
			return('2');
		}
		auth->auth_state = CHAP_STAGE_CLIENT_A;
		return('0');
	} else if (auth->auth_state == CHAP_STAGE_CLIENT_A) {
		convert_null_to_semi(in_text, *in_len);
		if (chap_gen_response(auth, in_text, out_text, out_len) < 0) {
			printf("chap_gen_response() failed!\n");
			chap_close(auth);
			return('2');
		}
		if (auth->authenticate_target) {
			auth->auth_state = CHAP_STAGE_CLIENT_NRIC;
			return('0');
		} else {
			auth->auth_state = CHAP_STAGE_CLIENT_NR;
			chap_close(auth);
			return('0');
		}
	} else if (auth->auth_state == CHAP_STAGE_CLIENT_NRIC) {
		convert_null_to_semi(in_text, *in_len);
		if (chap_got_mutal_auth_response(auth, in_text) < 0) {
			printf("chap_got_mutal_auth_response() failed!\n");
			chap_close(auth);
			return('2');
		}
		chap_close(auth);
		return('3');
	}

	return('2');
}

