! This file is part of xtb.
!
! Copyright (C) 2020 Sebastian Ehlert
!
! xtb is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! xtb is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with xtb.  If not, see <https://www.gnu.org/licenses/>.

!> A set of molecules for testing
module xtb_test_molstock
   use xtb_mctc_accuracy, only : wp
   use xtb_type_molecule, only : TMolecule, init
   implicit none
   private

   public :: getMolecule


contains


subroutine getMolecule(mol, name)
   type(TMolecule), intent(out) :: mol
   character(len=*), intent(in) :: name

   select case(name)
   case('mindless01');  call mindless01(mol)
   case('mindless02');  call mindless02(mol)
   case('mindless03');  call mindless03(mol)
   case('mindless04');  call mindless04(mol)
   case('mindless05');  call mindless05(mol)
   case('mindless06');  call mindless06(mol)
   case('mindless07');  call mindless07(mol)
   case('mindless08');  call mindless08(mol)
   case('mindless09');  call mindless09(mol)
   case('mindless10');  call mindless10(mol)
   case('caffeine');    call caffeine(mol)
   case('rivaroxaban'); call rivaroxaban(mol)
   case('grubbs');      call grubbs(mol)
   case('remdesivir');  call remdesivir(mol)
   case('taxol');       call taxol(mol)
   case('pdb-4qxx');    call pdb_4qxx(mol)
   case('bug332');      call bug332(mol)
   case('manganese');   call manganese(mol)
   case('vcpco4');      call vcpco4(mol)
   case('feco5');       call feco5(mol)
   case('co_cnx6');       call co_cnx6(mol)
   case('fe_cnx6');       call fe_cnx6(mol)
   end select

end subroutine getMolecule


subroutine mindless01(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "Na", "H", "O", "H", "F", "H", "H", "O", "N", "H", "H", "Cl", "B", "B", "N", "Al"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.85528263484662_wp,  3.58670515364616_wp, -2.41763729306344_wp, &
      &  4.40178023537845_wp,  0.02338844412653_wp, -4.95457749372945_wp, &
      & -2.98706033463438_wp,  4.76252065456814_wp,  1.27043301573532_wp, &
      &  0.79980886075526_wp,  1.41103455609189_wp, -5.04655321620119_wp, &
      & -4.20647469409936_wp,  1.84275767548460_wp,  4.55038084858449_wp, &
      & -3.54356121843970_wp, -3.18835665176557_wp,  1.46240021785588_wp, &
      &  2.70032160109941_wp,  1.06818452504054_wp, -1.73234650374438_wp, &
      &  3.73114088824361_wp, -2.07001543363453_wp,  2.23160937604731_wp, &
      & -1.75306819230397_wp,  0.35951417150421_wp,  1.05323406177129_wp, &
      &  5.41755788583825_wp, -1.57881830078929_wp,  1.75394002750038_wp, &
      & -2.23462868255966_wp, -2.13856505054269_wp,  4.10922285746451_wp, &
      &  1.01565866207568_wp, -3.21952154552768_wp, -3.36050963020778_wp, &
      &  2.42119255723593_wp,  0.26626435093114_wp, -3.91862474360560_wp, &
      & -3.02526098819107_wp,  2.53667889095925_wp,  2.31664984740423_wp, &
      & -2.00438948664892_wp, -2.29235136977220_wp,  2.19782807357059_wp, &
      &  1.12226554109716_wp, -1.36942007032045_wp,  0.48455055461782_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine mindless01


subroutine mindless02(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "H", "S", "B", "O", "Mg", "H", "H", "H", "Si", "H", "B", "Li", "F", "H", "H", "S"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.79537625851198_wp, -3.77866422935275_wp, -1.07883558363403_wp, &
      & -2.68278833302782_wp,  0.38892666265890_wp,  1.66214865238427_wp, &
      &  0.11484649791305_wp,  1.48857933226955_wp,  3.65660396510375_wp, &
      & -1.07998879593946_wp, -0.16259121615748_wp, -4.55703065871422_wp, &
      &  0.60302832999383_wp,  4.08816149622342_wp, -0.02589373148029_wp, &
      & -1.22534089315880_wp, -1.79981382478068_wp, -3.70773173318592_wp, &
      & -1.33460982049866_wp, -4.24819082475503_wp,  2.72791902701083_wp, &
      & -0.16278082578516_wp,  2.41267994179303_wp,  5.69030695190570_wp, &
      &  2.87802444057103_wp, -0.33120525058830_wp,  1.88311373530297_wp, &
      &  0.68489327931487_wp,  0.32790204044961_wp, -4.20547693710673_wp, &
      & -1.20919773588330_wp, -2.87253762561437_wp,  0.94064204223101_wp, &
      & -3.25572604597922_wp,  2.21241092990940_wp, -2.86715549314771_wp, &
      & -1.83147468262373_wp,  5.20527293771933_wp, -2.26976270603341_wp, &
      &  4.90885865772880_wp, -1.92576561961811_wp,  2.99069919443735_wp, &
      &  1.26806242248758_wp, -2.60409341782411_wp,  0.55162805282247_wp, &
      &  4.11956976339902_wp,  1.59892866766766_wp, -1.39117477789609_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 2
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless02


subroutine mindless03(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "O", "H", "Li", "Mg", "Al", "C", "H", "H", "H", "F", "S", "C", "H", "Na", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.02148551327524_wp, -0.67161751504297_wp, -4.75078512817560_wp, &
      &  1.37792545875526_wp, -3.24818416423144_wp,  3.83896600631495_wp, &
      & -2.23986953822894_wp,  1.64550402751694_wp,  3.42773272178522_wp, &
      & -0.87622711432790_wp, -2.74068400827752_wp,  1.43723692979592_wp, &
      &  1.29492470653815_wp,  1.86470311043681_wp, -1.04536500695239_wp, &
      & -3.65768365013010_wp,  0.45437052179208_wp, -1.41566056087159_wp, &
      & -0.23245910487384_wp, -1.83274112101585_wp, -2.43395808606122_wp, &
      &  0.30373451850419_wp, -3.84228931776777_wp, -2.44882782867802_wp, &
      & -3.36159503902161_wp,  4.20056392581975_wp,  1.63352684198071_wp, &
      &  0.49372989648081_wp, -1.56245253044952_wp, -6.53610501083288_wp, &
      &  4.38566058812996_wp,  1.86127331114460_wp,  0.56178822055152_wp, &
      & -1.17545963764009_wp,  2.49456345795141_wp, -4.90195191215762_wp, &
      & -1.86623614216854_wp,  2.76329843590746_wp,  1.71572598870213_wp, &
      &  1.02361259176985_wp, -4.24377370348987_wp,  5.32418288889440_wp, &
      &  4.71194535010347_wp, -1.03648125005561_wp,  3.35573062118779_wp, &
      & -0.16051737061546_wp,  3.89394681976155_wp,  2.23776331451663_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine mindless03


subroutine mindless04(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "H", "B", "H", "F", "B", "H", "H", "Si", "H", "H", "C", "Al", "Si", "O", "H", "B"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.34544890768411_wp,  2.85946545334720_wp,  3.11183388215396_wp, &
      & -0.36293929605305_wp,  4.15983774640545_wp,  1.36413101934678_wp, &
      & -3.36268280924844_wp,  4.92951597114402_wp, -3.59085684882314_wp, &
      &  3.78143178536443_wp, -4.97181356229699_wp,  1.59003443639387_wp, &
      &  3.44227417874042_wp, -3.46504338606415_wp,  3.62082644591507_wp, &
      &  1.88917586252014_wp,  3.42088101960529_wp,  1.28872629783483_wp, &
      & -0.32747529934233_wp, -4.29711514977711_wp, -3.55330460209973_wp, &
      & -3.58768360829779_wp, -1.39509759062952_wp, -1.10396714572410_wp, &
      & -0.39440896193088_wp,  6.31837673143592_wp,  1.99105318714945_wp, &
      &  4.34376903295874_wp, -4.12502353873667_wp,  5.57829602371555_wp, &
      & -1.39570266622309_wp, -2.60410756418652_wp, -4.03149806979915_wp, &
      &  0.21788515354592_wp,  0.28610741675369_wp,  1.29731097788136_wp, &
      & -2.00000183598828_wp,  3.04473467156937_wp, -2.00578147078785_wp, &
      &  2.12833842504876_wp, -1.30141517432227_wp,  3.38069910888504_wp, &
      & -2.48411958079522_wp, -2.81581487156584_wp, -5.76829803496286_wp, &
      & -0.54241147261516_wp, -0.04348817268188_wp, -3.16920520707912_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine mindless04


subroutine mindless05(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "B", "P", "H", "H", "B", "P", "H", "Cl", "N", "H", "P", "Si", "H", "H", "P", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.68391902268453_wp,  0.21679405065309_wp, -2.81441127558071_wp, &
      & -2.67199537993843_wp, -3.97743927106200_wp,  0.03497540139192_wp, &
      &  2.02325266152397_wp, -0.16048070975416_wp, -0.41980608052722_wp, &
      &  4.26224346168617_wp,  3.65384961705338_wp, -2.81836810458488_wp, &
      & -2.80378310343644_wp,  1.84796600006216_wp,  0.15107304476153_wp, &
      &  1.58317082705122_wp,  3.77079801391042_wp, -2.86230158107979_wp, &
      &  2.63670178694113_wp,  3.13142099211650_wp,  2.24139937019049_wp, &
      & -6.27112533979613_wp, -3.92471014080274_wp,  1.62562669834852_wp, &
      & -0.92594349239390_wp, -2.94451283088352_wp,  2.60616476876177_wp, &
      & -1.79532342290201_wp, -1.56841672860834_wp,  3.65515689388732_wp, &
      & -3.01460634915379_wp, -0.47748181717446_wp, -2.44834110183776_wp, &
      &  2.18249449208515_wp, -2.23505035804805_wp,  1.77725119258081_wp, &
      &  3.26068149442689_wp, -4.54078259646428_wp,  0.57204329987377_wp, &
      &  1.73744972267909_wp, -1.18654391698320_wp, -4.24063427353503_wp, &
      &  0.94405328902426_wp,  4.99525793054843_wp,  1.18501287451328_wp, &
      & -1.83118967048165_wp,  3.39933176543682_wp,  1.75515887283605_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless05


subroutine mindless06(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "B", "N", "H", "O", "B", "H", "Al", "H", "B", "Mg", "H", "H", "H", "H", "C", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.10912945825730_wp,  1.64180252123600_wp,  0.27838149792131_wp, &
      & -2.30085163837888_wp,  0.87765138232225_wp, -0.60457694150897_wp, &
      &  2.78083551168063_wp,  4.95421363506113_wp,  0.40788634984219_wp, &
      & -5.36229602768251_wp, -7.29510945515334_wp,  0.06097106408867_wp, &
      &  2.13846114572058_wp, -0.99012126457352_wp,  0.93647189687052_wp, &
      &  0.09330150731888_wp, -2.75648066796634_wp, -3.70294675694565_wp, &
      & -1.52684105316140_wp, -2.44981814860506_wp, -1.02727325811774_wp, &
      & -0.45240334635443_wp,  5.86105501765814_wp,  0.30815308772432_wp, &
      & -3.95419048213910_wp, -5.52061943693205_wp, -0.31702321028260_wp, &
      &  2.68706169520082_wp, -0.13577304635533_wp, -3.57041492458512_wp, &
      & -3.79914135008731_wp,  2.06429808651079_wp, -0.77285245656187_wp, &
      &  0.89693752015341_wp,  4.58640300917890_wp,  3.09718012019731_wp, &
      &  2.76317093138142_wp, -0.62928000132252_wp,  3.08807601371151_wp, &
      &  1.00075543259914_wp, -3.11885279872042_wp,  1.08659460804098_wp, &
      &  0.86969979951508_wp,  4.43363816376984_wp,  1.02355776570620_wp, &
      &  4.05637089597643_wp, -1.52300699610852_wp, -0.29218485610105_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless06


subroutine mindless07(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "H", "B", "H", "H", "Cl", "F", "N", "C", "H", "S", "H", "H", "O", "F", "Mg"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.75104222741336_wp, -5.81308736205268_wp, -1.22507366840233_wp, &
      & -1.45226572768296_wp, -3.01878767879831_wp,  2.38723142561073_wp, &
      & -1.99423317853240_wp, -3.52953889999752_wp, -1.30301724065129_wp, &
      & -4.33750965171233_wp, -6.65936981001909_wp,  0.55979831484564_wp, &
      & -4.51833920602637_wp, -6.72398616322561_wp, -2.90031439001886_wp, &
      & -1.25657105633503_wp, -2.39389339457851_wp, -4.58765484136593_wp, &
      & -0.14864209579028_wp,  4.40065007854051_wp,  1.35717716022989_wp, &
      & -0.91662354168326_wp, -2.22680612180354_wp,  0.71122632634918_wp, &
      &  1.83282041695179_wp,  5.36061635978157_wp,  3.22095765094686_wp, &
      &  0.66518416413161_wp,  6.30980889882630_wp,  4.62705414435961_wp, &
      &  3.68701623423530_wp,  2.79957532381681_wp,  4.21336212424745_wp, &
      &  1.69373321407504_wp,  0.01030275402386_wp, -3.74820290941150_wp, &
      &  3.35791986589808_wp,  2.52513229318111_wp, -3.46078430541625_wp, &
      &  2.79199182665654_wp,  1.01759578021447_wp, -2.59243571461852_wp, &
      &  3.05358934464082_wp,  7.15252337445235_wp,  1.82164153773112_wp, &
      &  1.29297161858681_wp,  0.78926456763834_wp,  0.91903438556425_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless07


subroutine mindless08(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "O", "B", "F", "H", "Al", "H", "H", "O", "B", "Be", "C", "H", "H", "B", "F"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.27823293129313_wp,  0.06442674490989_wp,  2.76980447300615_wp, &
      &  2.05039033278229_wp,  0.64690940303039_wp, -0.29571013189632_wp, &
      & -0.07388472989895_wp,  2.46033979750309_wp, -1.30590420482375_wp, &
      &  1.10019432741349_wp,  4.43501067437330_wp, -2.64796515354449_wp, &
      & -1.89008873387150_wp,  0.02064696008121_wp,  4.74727599156952_wp, &
      &  0.81013963557610_wp,  1.41165582964016_wp, -6.35835508532445_wp, &
      &  2.51638337449170_wp,  1.74086425451198_wp,  3.45340860505386_wp, &
      &  2.62048878651566_wp, -1.58024532804571_wp,  2.87415150030394_wp, &
      & -0.92472602392464_wp, -3.37659091509259_wp, -0.68138826965952_wp, &
      & -2.19962829538645_wp, -2.53092502025386_wp,  1.35654623095955_wp, &
      &  0.92594749614406_wp, -1.61669775704536_wp, -1.93872059141561_wp, &
      &  1.63141903847248_wp,  0.18081362275364_wp,  2.42899361614054_wp, &
      & -3.96336280784845_wp, -3.68611886004249_wp,  2.18920954455515_wp, &
      & -1.17097381446263_wp,  1.08303722364990_wp, -3.04753977323348_wp, &
      & -2.18263847972349_wp,  2.31604957286801_wp,  1.11461091308323_wp, &
      &  2.02857282501340_wp, -1.56917620284149_wp, -4.65841766477431_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless08


subroutine mindless09(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "H", "H", "H", "H", "Li", "H", "C", "B", "H", "H", "Si", "H", "Cl", "F", "H", "B"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.97360649552839_wp,  1.71723751297383_wp, -0.51862929250676_wp, &
      &  0.16903666216522_wp,  1.73154352333176_wp, -0.40099024352959_wp, &
      & -3.94463844105182_wp, -1.24346369608005_wp,  0.09565841726334_wp, &
      &  2.21647168119803_wp,  4.10625979391554_wp,  2.61391340002321_wp, &
      & -0.04488993380842_wp, -2.16288302687041_wp,  4.48488595610432_wp, &
      &  3.52287141817194_wp, -0.90500888687059_wp, -5.00916337263077_wp, &
      &  1.95336082370762_wp, -0.83849036872324_wp, -3.65515970516029_wp, &
      &  2.05706981818495_wp,  1.70095588601056_wp, -2.06303335904159_wp, &
      & -6.40097100472159_wp, -1.71072935987273_wp,  3.14621771036234_wp, &
      &  2.04751538182937_wp, -2.55691868000982_wp, -2.49926722310562_wp, &
      &  2.03251078714394_wp,  1.35094356516468_wp,  2.02150308748654_wp, &
      &  0.20477572129201_wp, -0.93291693232462_wp, -4.76431390827476_wp, &
      & -2.67673272939098_wp,  1.40764602033672_wp,  4.10347165469140_wp, &
      & -2.75901984658887_wp, -3.73954809548334_wp,  3.19373273207227_wp, &
      &  1.96938102642596_wp,  3.74070925169244_wp, -3.03185101883736_wp, &
      & -4.32034786008576_wp, -1.66533650719069_wp,  2.28302516508337_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine mindless09


subroutine mindless10(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "H", "Si", "H", "Cl", "C", "H", "F", "H", "C", "N", "B", "H", "Mg", "C", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.57062307661218_wp, -1.68792229443234_wp,  2.78939425857465_wp, &
      & -2.08994110527129_wp,  3.25317728228563_wp, -0.42147881550833_wp, &
      &  2.13532981939105_wp, -1.71356933061236_wp, -2.49234593851880_wp, &
      & -2.46885241522113_wp, -4.41076598859264_wp, -0.58746410797603_wp, &
      &  3.86605901148259_wp, -0.50808683490216_wp,  1.10929274542242_wp, &
      & -4.57284898019279_wp, -1.54920337824862_wp, -2.63711913350102_wp, &
      & -4.99945502320431_wp,  0.09990896897876_wp, -3.20268495970371_wp, &
      &  1.63618508154720_wp,  2.66791559582643_wp, -3.16904643876699_wp, &
      & -2.28445827511587_wp,  0.42792856662334_wp,  2.04433546457507_wp, &
      &  0.78486183614848_wp,  1.96692225005484_wp, -1.58921219981020_wp, &
      & -0.92003258313224_wp, -1.56076484060483_wp,  0.46494611026243_wp, &
      & -1.07970143095156_wp,  1.19037461384346_wp,  3.56880222429743_wp, &
      &  3.27327901654007_wp,  3.47628642644825_wp,  1.85050408639730_wp, &
      &  1.64922592697103_wp, -0.66726875777723_wp, -0.77306391492380_wp, &
      &  5.67004330685832_wp, -1.05218123504276_wp,  0.25282456342591_wp, &
      & -4.17031726246173_wp,  0.06724895615223_wp,  2.79231605575371_wp],&
      & shape(xyz))
   integer, parameter :: uhf = 1
   call init(mol, sym, xyz, uhf=uhf)
end subroutine mindless10


subroutine caffeine(mol)
   type(TMolecule), intent(inout) :: mol
   integer, parameter :: nat = 24
   integer, parameter :: at(nat) = &
      [6, 7, 6, 7, 6, 6, 6, 8, 7, 6, 8, 7, 6, 6, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.02799738646442_wp,  0.09231312124713_wp, -0.14310895950963_wp, &
      &  4.75011007621000_wp,  0.02373496014051_wp, -0.14324124033844_wp, &
      &  6.33434307654413_wp,  2.07098865582721_wp, -0.14235306905930_wp, &
      &  8.72860718071825_wp,  1.38002919517619_wp, -0.14265542523943_wp, &
      &  8.65318821103610_wp, -1.19324866489847_wp, -0.14231527453678_wp, &
      &  6.23857175648671_wp, -2.08353643730276_wp, -0.14218299370797_wp, &
      &  5.63266886875962_wp, -4.69950321056008_wp, -0.13940509630299_wp, &
      &  3.44931709749015_wp, -5.48092386085491_wp, -0.14318454855466_wp, &
      &  7.77508917214346_wp, -6.24427872938674_wp, -0.13107140408805_wp, &
      & 10.30229550927022_wp, -5.39739796609292_wp, -0.13672168520430_wp, &
      & 12.07410272485492_wp, -6.91573621641911_wp, -0.13666499342053_wp, &
      & 10.70038521493902_wp, -2.79078533715849_wp, -0.14148379504141_wp, &
      & 13.24597858727017_wp, -1.76969072232377_wp, -0.14218299370797_wp, &
      &  7.40891694074004_wp, -8.95905928176407_wp, -0.11636933482904_wp, &
      &  1.38702118184179_wp,  2.05575746325296_wp, -0.14178615122154_wp, &
      &  1.34622199478497_wp, -0.86356704498496_wp,  1.55590600570783_wp, &
      &  1.34624089204623_wp, -0.86133716815647_wp, -1.84340893849267_wp, &
      &  5.65596919189118_wp,  4.00172183859480_wp, -0.14131371969009_wp, &
      & 14.67430918222276_wp, -3.26230980007732_wp, -0.14344911021228_wp, &
      & 13.50897177220290_wp, -0.60815166181684_wp,  1.54898960808727_wp, &
      & 13.50780014200488_wp, -0.60614855212345_wp, -1.83214617078268_wp, &
      &  5.41408424778406_wp, -9.49239668625902_wp, -0.11022772492007_wp, &
      &  8.31919801555568_wp, -9.74947502841788_wp,  1.56539243085954_wp, &
      &  8.31511620712388_wp, -9.76854236502758_wp, -1.79108242206824_wp],&
      & shape(xyz))
   call init(mol, at, xyz)
end subroutine caffeine


subroutine rivaroxaban(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 47
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "Cl", "C", "C", "C", "C", "S", "C", "O", "N", "C", "C", "C", "N", &
      &  "C", "O", "O", "C", "C", "C", "C", "C", "C", "H", "H", "N", "C", &
      &  "O", "C", "O", "C", "C", "H", "H", "H", "H", "H", "H", "H", "H", &
      &  "H", "H", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  19.59759192978171_wp,  0.07558903797189_wp,  0.17650040366436_wp, &
      &  16.38864829528011_wp,  0.34638676650618_wp,  0.52326511536040_wp, &
      &  15.01935287241935_wp,  2.48858010262950_wp,  1.08300194154223_wp, &
      &  12.37657113232719_wp,  1.99214909574913_wp,  1.21792837432206_wp, &
      &  11.89828149456006_wp, -0.47337635029895_wp,  0.75683524269354_wp, &
      &  14.48985166142627_wp, -2.24896285225862_wp,  0.16251643163956_wp, &
      &   9.42330741876550_wp, -1.73344561329034_wp,  0.72282017560619_wp, &
      &   9.18671372991349_wp, -3.99714832795347_wp,  0.22997964802947_wp, &
      &   7.32779031358982_wp, -0.20068889581536_wp,  1.22548727811924_wp, &
      &   4.79045528146844_wp, -1.14725262381834_wp,  1.20205467634796_wp, &
      &   3.61995902847375_wp, -0.92237523585197_wp, -1.41634959899826_wp, &
      &   0.93711509825649_wp, -1.90616656505610_wp, -1.53974870348737_wp, &
      &  -0.56332730548550_wp,  0.34449704055688_wp, -1.27065172830745_wp, &
      &   0.93900482420579_wp,  2.46117907636470_wp, -1.70339897069651_wp, &
      &   0.33542635600026_wp,  4.68803213501653_wp, -1.83643567752704_wp, &
      &   3.40282951689950_wp,  1.70283205291172_wp, -2.05583286024044_wp, &
      &  -3.21669151089372_wp,  0.28364786498951_wp, -0.82637715762767_wp, &
      &  -4.59921501539957_wp,  2.52750845718503_wp, -0.79274003573018_wp, &
      &  -7.19834408606296_wp,  2.46798208978216_wp, -0.35753614960703_wp, &
      &  -8.40304437873993_wp,  0.16459513018379_wp,  0.04176294347947_wp, &
      &  -7.03242614771466_wp, -2.07983237979652_wp,  0.01020452012620_wp, &
      &  -4.43329707705127_wp, -2.01992806720379_wp, -0.42481039340201_wp, &
      &  -3.43608869360713_wp, -3.81176621232742_wp, -0.42821190011075_wp, &
      &  -7.88053515375925_wp, -3.90322894827340_wp,  0.40799183245327_wp, &
      & -11.09250234977973_wp,  0.10299006423670_wp,  0.49246258238685_wp, &
      & -12.58292920599045_wp, -1.97381875404094_wp, -0.22922375764975_wp, &
      & -11.87069149570033_wp, -3.81573463682094_wp, -1.46378172032562_wp, &
      & -15.30999272342126_wp, -1.91826081113160_wp,  0.69938757383490_wp, &
      & -16.35557809116741_wp,  0.48244703485558_wp,  1.17408673229836_wp, &
      & -14.75441329432788_wp,  1.88594649739862_wp,  2.80359741837735_wp, &
      & -12.32063524422799_wp,  2.42243969440410_wp,  1.39499569577121_wp, &
      & -11.05584166636337_wp,  3.45498595310010_wp,  2.66734817743302_wp, &
      & -12.72635940554211_wp,  3.57517252347540_wp, -0.27873457752134_wp, &
      & -15.70872489872297_wp,  3.65586382151040_wp,  3.28698931620758_wp, &
      & -14.42276639022622_wp,  0.83677065034881_wp,  4.55971974305925_wp, &
      & -16.48785890761821_wp, -2.82627412976892_wp, -0.74020565433972_wp, &
      & -15.42526600632839_wp, -3.07250542096234_wp,  2.41639257136635_wp, &
      &  -8.20934746893697_wp,  4.25282824889339_wp, -0.39249607966903_wp, &
      &  -3.77794011783500_wp,  4.37395968224334_wp, -1.09793077654168_wp, &
      &   0.56710675738409_wp, -3.30550863051069_wp, -0.06443965487104_wp, &
      &   0.53630422441055_wp, -2.72328406553222_wp, -3.40018390057048_wp, &
      &   4.80991945874621_wp, -1.85930136151353_wp, -2.82873077350300_wp, &
      &   4.78818761032929_wp, -3.11861473412520_wp,  1.83303417081830_wp, &
      &   3.73598820176060_wp, -0.02305465658143_wp,  2.58457818085380_wp, &
      &   7.55210078377140_wp,  1.66465958873592_wp,  1.59984198867502_wp, &
      &  10.99518146339092_wp,  3.43911225512600_wp,  1.63669164468632_wp, &
      &  15.86217064580591_wp,  4.32879523205513_wp,  1.38290144969570_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine rivaroxaban


subroutine grubbs(mol)
   type(TMolecule), intent(inout) :: mol
   integer, parameter :: nat = 75
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "Ru", "C", "C", "C", "C", "C", "C", "C", "O", "C", "C", "H", "H", &
      &  "H", "C", "H", "H", "H", "H", "H", "H", "H", "H", "H", "C", "N", &
      &  "C", "C", "C", "C", "C", "C", "C", "H", "H", "H", "H", "C", "H", &
      &  "H", "H", "H", "C", "H", "H", "H", "C", "C", "N", "C", "C", "C", &
      &  "C", "C", "C", "C", "H", "H", "H", "H", "C", "H", "H", "H", "H", &
      &  "C", "H", "H", "H", "H", "H", "H", "H", "Cl", "Cl"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  -4.48876242338909_wp, -2.09410736961751_wp,  6.48025767395976_wp, &
      &  -3.39542935008205_wp, -1.00934609321648_wp,  3.37297562635250_wp, &
      &  -0.89096799057465_wp, -1.65719139178430_wp,  2.52794687360526_wp, &
      &  -0.00187838759360_wp, -1.02726825411961_wp,  0.11235554604142_wp, &
      &   2.39977999054608_wp, -1.73257067017581_wp, -0.66001135422939_wp, &
      &   3.94523944585223_wp, -3.08713189980991_wp,  0.98469839765979_wp, &
      &   3.13084315074310_wp, -3.73573308875744_wp,  3.40383674083047_wp, &
      &   0.72574925082760_wp, -3.01450028294867_wp,  4.16236328824863_wp, &
      &  -0.34548725695431_wp, -3.51939537235595_wp,  6.43739490997780_wp, &
      &   1.10498701323636_wp, -4.64071528697208_wp,  8.49906135422373_wp, &
      &  -0.81899588806972_wp, -5.50999300310070_wp, 10.42067453918938_wp, &
      &  -2.16671442004141_wp, -6.83194323202777_wp,  9.58791577703093_wp, &
      &   0.14674477886673_wp, -6.44058098792832_wp, 11.99173033247876_wp, &
      &  -1.86211705317798_wp, -3.89003299196946_wp, 11.18088861188026_wp, &
      &   2.94345036670699_wp, -2.71461211315089_wp,  9.54109403718520_wp, &
      &   4.25175488455419_wp, -2.02822207439526_wp,  8.10113798162831_wp, &
      &   1.90191279194423_wp, -1.09894933882835_wp, 10.29213915957009_wp, &
      &   4.05380231191340_wp, -3.56092776926960_wp, 11.06322671537322_wp, &
      &   2.09442106412509_wp, -6.28009790141020_wp,  7.69876997360015_wp, &
      &   4.36447703742976_wp, -4.78998851915897_wp,  4.65115224339005_wp, &
      &   5.82373853328467_wp, -3.65349977434782_wp,  0.38695351345974_wp, &
      &   3.07317627367435_wp, -1.24186175346991_wp, -2.53223088233232_wp, &
      &  -1.25251791809799_wp,  0.01729855133987_wp, -1.13979009604457_wp, &
      &  -4.50256120227086_wp,  0.08700676215754_wp,  2.02024932061518_wp, &
      &  -7.98117439891502_wp, -0.91146395822073_wp,  6.61491387592883_wp, &
      &  -9.54120931046810_wp,  0.23196386027623_wp,  4.93017972843353_wp, &
      &  -8.81626697262086_wp,  1.20094351748977_wp,  2.52737617636858_wp, &
      &  -9.12263312297282_wp, -0.33655074294009_wp,  0.39778731232706_wp, &
      &  -8.32022713901198_wp,  0.60170386006383_wp, -1.92819510044706_wp, &
      &  -7.24034702473799_wp,  2.99722440832019_wp, -2.16074666549353_wp, &
      &  -6.95751930081047_wp,  4.46636539143802_wp,  0.00913871469080_wp, &
      &  -7.72222903153473_wp,  3.60494893525848_wp,  2.37913095509811_wp, &
      &  -7.20124859484703_wp,  5.10159865902023_wp,  4.72196405416212_wp, &
      &  -6.40848210295356_wp,  6.94912135869179_wp,  4.26394361692715_wp, &
      &  -8.90386710872648_wp,  5.41338265366883_wp,  5.85428028407721_wp, &
      &  -5.85075405545582_wp,  4.11205689375239_wp,  5.94471689883273_wp, &
      &  -6.07723204130125_wp,  6.31818721764424_wp, -0.13610562177218_wp, &
      &  -6.41560636978241_wp,  3.96790859834000_wp, -4.68988595617280_wp, &
      &  -5.00475020411875_wp,  5.46289536326637_wp, -4.51108764575409_wp, &
      &  -5.61432855221330_wp,  2.46525899468923_wp, -5.85859641814541_wp, &
      &  -8.01116812918227_wp,  4.75888551975949_wp, -5.74517506666859_wp, &
      &  -8.51402987346811_wp, -0.58379681696829_wp, -3.59647918355373_wp, &
      & -10.11190843688181_wp, -2.97112351350850_wp,  0.67258748042197_wp, &
      & -10.35742352194046_wp, -3.87297632555110_wp, -1.16520313061071_wp, &
      & -11.93170696715124_wp, -3.01882775537256_wp,  1.65559268363119_wp, &
      &  -8.80853799348823_wp, -4.13077840873208_wp,  1.79194019032488_wp, &
      & -12.00826165508321_wp,  0.89370998292708_wp,  6.00001216983511_wp, &
      & -11.99803445824562_wp, -0.48821258872689_wp,  8.52708221059990_wp, &
      &  -9.34971393139607_wp, -1.22700094860463_wp,  8.78345188151131_wp, &
      &  -8.45101507196284_wp, -2.63174385082231_wp, 10.90901184982892_wp, &
      &  -7.37601611981588_wp, -1.32890819987238_wp, 12.95669810150194_wp, &
      &  -6.33567852103093_wp, -2.74890119050089_wp, 14.91661225200796_wp, &
      &  -6.40297544153730_wp, -5.38068850502004_wp, 14.91783301497120_wp, &
      &  -7.66406515654131_wp, -6.60173870661033_wp, 12.95345155232105_wp, &
      &  -8.73996550796608_wp, -5.27286452180504_wp, 10.95168753094190_wp, &
      & -10.30065615442357_wp, -6.64280056176261_wp,  9.02763659950165_wp, &
      & -10.40423770288240_wp, -5.65022200689425_wp,  7.22532748289297_wp, &
      &  -9.53138840470961_wp, -8.51497473644276_wp,  8.63796754985277_wp, &
      & -12.23007579728577_wp, -6.89514322667607_wp,  9.74487452465361_wp, &
      &  -7.82480146633663_wp, -8.65107468903709_wp, 12.97505111992152_wp, &
      &  -5.13310794949881_wp, -6.86601499089360_wp, 16.96523202829993_wp, &
      &  -4.90164297687319_wp, -5.75406433753408_wp, 18.68816509613433_wp, &
      &  -6.18605568954342_wp, -8.57769474069993_wp, 17.43910348679760_wp, &
      &  -3.24350105293640_wp, -7.46973329881938_wp, 16.37032740220168_wp, &
      &  -5.44484281127273_wp, -1.76139654980639_wp, 16.48390182148039_wp, &
      &  -7.45351378099656_wp,  1.49347687389288_wp, 13.13898673574900_wp, &
      &  -7.68282068658808_wp,  2.38897831701990_wp, 11.29898325562969_wp, &
      &  -5.72170844983021_wp,  2.23715773425336_wp, 13.97522259365416_wp, &
      &  -9.03185824727238_wp,  2.07577135873148_wp, 14.35092845826368_wp, &
      & -12.58185017247340_wp,  0.70698805160297_wp, 10.11158718347043_wp, &
      & -13.22024495156878_wp, -2.16522909544526_wp,  8.51641281789017_wp, &
      & -13.53213099141864_wp,  0.28360062184078_wp,  4.74165688827974_wp, &
      & -12.17023762510128_wp,  2.95020235752383_wp,  6.21916935735296_wp, &
      &  -5.47317169297864_wp, -6.30777860711551_wp,  5.37945073452325_wp, &
      &  -2.91575454319409_wp,  1.17077593243519_wp,  9.10079167045084_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine grubbs


subroutine remdesivir(mol)
   type(TMolecule), intent(inout) :: mol
   integer, parameter :: nat = 77
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "P", "O", "N", "H", "C", "C", "O", "O", "C", "H", "C", "H", "C", &
      & "C", "H", "H", "H", "H", "H", "C", "C", "H", "H", "H", "H", "H", &
      & "H", "C", "H", "H", "H", "H", "O", "C", "C", "C", "C", "C", "C", &
      & "H", "H", "H", "H", "H", "O", "C", "C", "C", "C", "O", "H", "H", &
      & "C", "C", "N", "C", "C", "C", "C", "N", "N", "C", "N", "C", "N", &
      & "H", "H", "H", "H", "H", "O", "H", "O", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  -0.19132988942724_wp,  -0.40972414993236_wp,  -6.46745247268699_wp, &
      &  -0.88708370315348_wp,   1.57425420135113_wp,  -8.31530734472471_wp, &
      &  -2.26653281976549_wp,  -0.70381344345842_wp,  -4.09986618135019_wp, &
      &  -2.11173222089634_wp,  -2.37930900572134_wp,  -3.20423243529946_wp, &
      &  -4.86798224614609_wp,   0.18225289406478_wp,  -4.30097756552511_wp, &
      &  -6.27677559924320_wp,  -1.11525632742753_wp,  -6.46873009595223_wp, &
      &  -5.51905876432564_wp,  -3.00081288327976_wp,  -7.52001192348601_wp, &
      &  -8.47434896130393_wp,   0.03844979764335_wp,  -6.95747164587644_wp, &
      & -10.03544986524468_wp,  -0.96247151044850_wp,  -8.97893491976716_wp, &
      & -11.92553948403753_wp,  -1.01962112437940_wp,  -8.18393281287400_wp, &
      &  -9.91996893316031_wp,   0.72014934689054_wp, -11.29995764502249_wp, &
      & -10.49219335196244_wp,   2.62232232617395_wp, -10.71536445894101_wp, &
      &  -7.23550619143170_wp,   0.87644512354367_wp, -12.35847819278769_wp, &
      &  -6.87897439809205_wp,   2.83708304137020_wp, -14.42535601514885_wp, &
      &  -7.91274033059926_wp,   2.37928435595668_wp, -16.14387740068727_wp, &
      &  -7.50480813843883_wp,   4.69755740449728_wp, -13.79720095827741_wp, &
      &  -4.89131637876603_wp,   2.99533648526117_wp, -14.93249537526973_wp, &
      &  -6.68095383546727_wp,  -0.98905693330672_wp, -13.05009790708299_wp, &
      &  -5.92831603844048_wp,   1.31709220244427_wp, -10.83195473545779_wp, &
      & -11.81883395463374_wp,  -0.21638516352561_wp, -13.27120005704696_wp, &
      & -14.57820291988140_wp,  -0.13823440252655_wp, -12.46760468875125_wp, &
      & -15.09962799221286_wp,   1.73374750107546_wp, -11.78227793100761_wp, &
      & -15.81183209717192_wp,  -0.57984018774397_wp, -14.05459574765567_wp, &
      & -15.01320381133929_wp,  -1.48941780730427_wp, -10.97846457800362_wp, &
      & -11.30523991685855_wp,  -2.13790076670461_wp, -13.83308989189071_wp, &
      & -11.60212248492759_wp,   0.93505736051472_wp, -14.96289818215977_wp, &
      &  -9.42330047711975_wp,  -2.87490672821927_wp,  -9.41531345938043_wp, &
      &  -5.04282112259610_wp,   3.05026865184906_wp,  -4.32998130896150_wp, &
      &  -4.01002824733346_wp,   3.80062049846515_wp,  -2.72085258323442_wp, &
      &  -7.00023053390600_wp,   3.65000997903864_wp,  -4.20915769754319_wp, &
      &  -4.22198186637037_wp,   3.81134084487576_wp,  -6.04677579218044_wp, &
      &  -5.81121939607902_wp,  -0.47906832308640_wp,  -2.58584825044032_wp, &
      &   2.47068801183837_wp,  -0.11375678761847_wp,  -4.92928504669041_wp, &
      &   2.92927268291256_wp,   1.82311303997703_wp,  -3.23788147405311_wp, &
      &   4.31694667681501_wp,   1.21347428732154_wp,  -1.10001374569184_wp, &
      &   4.92882581296040_wp,   3.08964908400000_wp,   0.62256151918150_wp, &
      &   4.15333643337733_wp,   5.56437325171411_wp,   0.22064670072489_wp, &
      &   2.77450133560862_wp,   6.15249424200922_wp,  -1.93426422378554_wp, &
      &   2.16318169837946_wp,   4.29340911625344_wp,  -3.68128192214883_wp, &
      &   1.12254214460929_wp,   4.73864437690493_wp,  -5.37821980658322_wp, &
      &   2.16215515804734_wp,   8.07483114693414_wp,  -2.26709260331350_wp, &
      &   4.61493583816390_wp,   7.02200897692256_wp,   1.57707963977785_wp, &
      &   6.01367922680868_wp,   2.60712876084823_wp,   2.28626955764142_wp, &
      &   4.91847251835001_wp,  -0.71791976724209_wp,  -0.82106224185859_wp, &
      &   0.48862131509161_wp,  -3.15027245516979_wp,  -7.56487341150835_wp, &
      &  -0.28365251447257_wp,  -3.92900495641729_wp, -10.06103509313444_wp, &
      &   1.93931407221488_wp,  -5.01987573681761_wp, -11.47572727213072_wp, &
      &   3.46194780403375_wp,  -7.00005509254758_wp, -10.06109570800838_wp, &
      &   5.52898042059816_wp,  -5.44581508792227_wp,  -8.80420942773199_wp, &
      &   7.62409660046871_wp,  -6.96790008764107_wp,  -8.27975764432074_wp, &
      &   8.89063621220764_wp,  -5.81873810579412_wp,  -7.58099930932030_wp, &
      &   4.77873116045408_wp,  -4.55591296051726_wp,  -7.10871869864302_wp, &
      &   5.98488325193633_wp,  -3.28214046211090_wp, -10.78131744099112_wp, &
      &   8.04420042616590_wp,  -3.95737391706384_wp, -12.52727083968610_wp, &
      &   9.69988749587325_wp,  -4.39558589702530_wp, -13.88601683593368_wp, &
      &   6.54330482327131_wp,  -0.78383440890520_wp,  -9.57046610088104_wp, &
      &   5.52059329950252_wp,   1.57608902101080_wp, -10.08638609628190_wp, &
      &   6.54855685480143_wp,   3.33074126268125_wp,  -8.40250813030378_wp, &
      &   8.21214243564971_wp,   2.03687260366483_wp,  -6.84298150007518_wp, &
      &   8.18991948783700_wp,  -0.49591478301912_wp,  -7.62041011860608_wp, &
      &   9.62805038196471_wp,  -2.33392280927626_wp,  -6.55343562197677_wp, &
      &  10.98869135966597_wp,  -1.54613291279684_wp,  -4.63623967569048_wp, &
      &  11.07034938130244_wp,   0.76539452738098_wp,  -3.59472088345013_wp, &
      &   9.65919212948107_wp,   2.54832948990048_wp,  -4.66761767959168_wp, &
      &   9.68564318115148_wp,   4.89076718249344_wp,  -3.63920321849348_wp, &
      &  10.43606082735075_wp,   4.96688374705139_wp,  -1.89321506855591_wp, &
      &   8.13682851479651_wp,   5.96513261015273_wp,  -3.87978909063992_wp, &
      &  12.16148349338088_wp,  -2.98423232166298_wp,  -3.77895066773497_wp, &
      &   6.10221882852163_wp,   5.31162664510204_wp,  -8.30050533110730_wp, &
      &   4.11071649734023_wp,   1.92211959433765_wp, -11.50538869802556_wp, &
      &   3.68433105167984_wp,  -3.04094982338455_wp, -12.15047372108470_wp, &
      &   2.31364009015306_wp,  -7.97109672786193_wp,  -8.64122424792099_wp, &
      &   4.47494932080769_wp,  -8.73120728144298_wp, -11.82363408175874_wp, &
      &   6.13555180213229_wp,  -9.19639447872914_wp, -11.21105462844537_wp, &
      &   1.22248128516574_wp,  -5.84013634310975_wp, -13.22675900331083_wp, &
      &  -1.76661235190559_wp,  -5.33561121819292_wp,  -9.84945177134171_wp, &
      &  -1.03909744200315_wp,  -2.31955212534116_wp, -11.09039554947463_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine remdesivir


subroutine taxol(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 113
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "C", "C", "C", "H", "H", "H", "C", "C", "H", "H", "C", "O", "H", &
      & "C", "C", "H", "H", "H", "C", "H", "H", "H", "C", "H", "O", "C", &
      & "O", "C", "C", "C", "C", "C", "C", "H", "H", "H", "H", "H", "C", &
      & "H", "C", "C", "H", "H", "O", "C", "C", "H", "H", "C", "O", "H", &
      & "H", "C", "C", "O", "C", "H", "O", "C", "O", "C", "H", "H", "H", &
      & "C", "H", "H", "H", "H", "O", "C", "O", "C", "H", "H", "H", "H", &
      & "O", "C", "O", "C", "O", "H", "H", "C", "N", "C", "O", "C", "C", &
      & "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", "C", "C", "C", &
      & "C", "C", "C", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  -4.27437796868034_wp, -4.21928318528357_wp, -0.64611116061329_wp, &
      &  -1.80385866712198_wp, -4.49179267962028_wp, -1.09021540834022_wp, &
      &  -0.70059193593179_wp, -4.65691453257619_wp, -3.70642288207301_wp, &
      &  -0.77845617901805_wp, -2.78919392789312_wp, -4.59464301033223_wp, &
      &  -1.78874135988759_wp, -5.97283783030446_wp, -4.85744945419119_wp, &
      &   1.26314279112424_wp, -5.27006881075666_wp, -3.60817443221132_wp, &
      &  -0.00030627793660_wp, -4.28735666380398_wp,  1.08338125050305_wp, &
      &  -0.65818114921658_wp, -1.84867266482099_wp,  2.51351427841739_wp, &
      &   0.02097488681957_wp, -0.25608865473600_wp,  1.39332843896544_wp, &
      &   0.31820249428279_wp, -1.83782947791020_wp,  4.32582829373035_wp, &
      &  -3.51287249936110_wp, -1.53117587190408_wp,  3.01374411136104_wp, &
      &  -3.96408957494663_wp, -1.58852572013899_wp,  5.66082560828332_wp, &
      &  -3.85410979407103_wp,  0.13416152157481_wp,  6.35802848958120_wp, &
      &  -5.06473348475706_wp, -3.83511943726724_wp,  2.07224569636481_wp, &
      &  -7.82988726154316_wp, -3.24862641567655_wp,  2.66407218738313_wp, &
      &  -8.57751512507724_wp, -1.70211795076769_wp,  1.56233547447708_wp, &
      &  -9.02215566516809_wp, -4.89086427651674_wp,  2.38187413897498_wp, &
      &  -7.91832036862123_wp, -2.71727645474732_wp,  4.65300149439551_wp, &
      &  -4.43487627431995_wp, -6.19957849484587_wp,  3.68814161451424_wp, &
      &  -3.16666173986881_wp, -5.71060923581374_wp,  5.23126003699847_wp, &
      &  -3.58892170712183_wp, -7.67691934888030_wp,  2.53916899116474_wp, &
      &  -6.16327406414357_wp, -6.95263091914175_wp,  4.50886282119166_wp, &
      &  -4.43403718551753_wp,  1.02686240248629_wp,  1.94705639090826_wp, &
      &  -6.18131158908824_wp,  1.52853577257256_wp,  2.92988410174731_wp, &
      &  -2.51433610565894_wp,  2.84264996593750_wp,  2.48916823205561_wp, &
      &  -2.29619571371463_wp,  3.71544897188849_wp,  4.75974666499101_wp, &
      &  -3.45843813988640_wp,  3.04080573266905_wp,  6.64195764417307_wp, &
      &  -0.35881526898593_wp,  5.73730681871137_wp,  4.90275103505786_wp, &
      &   0.01573951184733_wp,  6.96668740241894_wp,  7.19753930207764_wp, &
      &   1.79861795419125_wp,  8.86461822319633_wp,  7.40872044489438_wp, &
      &   3.22863211733955_wp,  9.54690928081462_wp,  5.32072062248882_wp, &
      &   2.87462876765914_wp,  8.32936108834446_wp,  3.02759428398099_wp, &
      &   1.09032295554875_wp,  6.42966577861662_wp,  2.81949561911974_wp, &
      &   0.78730914442476_wp,  5.45812264610239_wp,  1.03840561302205_wp, &
      &   3.99945821048446_wp,  8.86620906391446_wp,  1.39999905542228_wp, &
      &   4.62745536175853_wp, 11.03585116854222_wp,  5.48333598031472_wp, &
      &   2.08201830967716_wp,  9.82038999682105_wp,  9.20020197650751_wp, &
      &  -1.12603658919603_wp,  6.40017217464386_wp,  8.80801167182970_wp, &
      &  -4.82300580852231_wp,  1.32127056984249_wp, -0.96367561160434_wp, &
      &  -3.52073853698517_wp,  0.02066439441838_wp, -1.87788685728542_wp, &
      &  -3.93099610201992_wp,  3.98397715268175_wp, -1.70297356757830_wp, &
      &  -4.83129941478753_wp,  6.17236489511779_wp, -0.04168869813259_wp, &
      &  -6.03297601335802_wp,  5.68932716662825_wp,  1.55628363178919_wp, &
      &  -3.30467629744770_wp,  7.42122862396423_wp,  0.56355038684552_wp, &
      &  -6.28073043143666_wp,  7.26433146083583_wp, -2.08479012831368_wp, &
      &  -5.21395584472362_wp,  5.46771978659285_wp, -3.85434809136400_wp, &
      &  -7.24257272340474_wp,  4.26754074250294_wp, -5.45907543555252_wp, &
      &  -6.85660584984287_wp,  4.52839739712925_wp, -7.46596056716137_wp, &
      &  -9.03669122217812_wp,  5.19318007824272_wp, -5.03172441000641_wp, &
      &  -7.53481090861879_wp,  1.45790703557441_wp, -4.93777427722140_wp, &
      &  -9.87400670403454_wp,  0.79685819491794_wp, -6.04025449816774_wp, &
      & -10.58886727469495_wp, -0.63731159892987_wp, -5.12172206442900_wp, &
      &  -5.96867901099894_wp,  0.43673079829201_wp, -5.83904305373725_wp, &
      &  -7.46204365640483_wp,  0.76234851320712_wp, -2.07457055170817_wp, &
      &  -8.15233562187160_wp, -2.01909322123343_wp, -2.33493227332548_wp, &
      & -10.33171427134996_wp, -2.59912399220692_wp, -2.68891419923086_wp, &
      &  -5.98931042872122_wp, -3.77993231188434_wp, -2.87481355631034_wp, &
      &  -4.83786147817200_wp, -2.68912398014316_wp, -4.21557536076896_wp, &
      &  -6.55145543086553_wp, -5.90936485696088_wp, -4.39209333361281_wp, &
      &  -8.06352762045317_wp, -7.73378679749750_wp, -3.86009139873076_wp, &
      &  -8.37272258086699_wp, -9.39367763262380_wp, -5.44161707867957_wp, &
      &  -9.41219366870205_wp, -7.91270734405018_wp, -1.34385079733592_wp, &
      & -10.54683450944101_wp, -9.62132856794695_wp, -1.32259528637126_wp, &
      & -10.63033710334819_wp, -6.28174032417005_wp, -1.09700506273163_wp, &
      &  -8.04441104797831_wp, -7.99321079444480_wp,  0.17916508190017_wp, &
      &  -9.59071075883635_wp,  2.20866231152695_wp, -0.76450678915921_wp, &
      &  -9.56645809232997_wp,  1.88636215498546_wp,  1.26622532847853_wp, &
      &  -9.42785063755489_wp,  4.23105970566003_wp, -1.10198452210804_wp, &
      & -11.38618948943588_wp,  1.53255140515293_wp, -1.51739726470692_wp, &
      &  -3.79061551084485_wp,  6.39309162438104_wp, -5.02734059999421_wp, &
      &  -1.27422200217135_wp,  4.10772123295000_wp, -1.98438268961729_wp, &
      &  -0.28743387275139_wp,  2.73731502671137_wp, -3.74861247521675_wp, &
      &  -1.37660416603083_wp,  1.25775077713349_wp, -5.13804670860084_wp, &
      &   2.52758817477529_wp,  3.20847969134492_wp, -3.93207943550893_wp, &
      &   3.30901888955614_wp,  2.04416350667181_wp, -5.43305639019368_wp, &
      &   3.43451544368503_wp,  2.71575231203533_wp, -2.15613114788026_wp, &
      &   2.88933025932021_wp,  5.19123884494056_wp, -4.32293373263776_wp, &
      &  -0.23087077739026_wp, -5.87614232880479_wp,  2.37349485145195_wp, &
      &   2.52061909703465_wp, -4.25684618955045_wp,  0.19387448907145_wp, &
      &   4.13063025163789_wp, -5.09244926377785_wp,  1.82395682951956_wp, &
      &   3.73233690195010_wp, -5.74198777973861_wp,  4.00397638723427_wp, &
      &   6.81305583067861_wp, -5.23478305505100_wp,  0.84181757850332_wp, &
      &   8.32651037080541_wp, -6.27761170028085_wp,  2.77826741369310_wp, &
      &   7.18930164181526_wp, -6.53483479739171_wp,  4.20703795330612_wp, &
      &   6.92150744269002_wp, -6.42423563881976_wp, -0.84600037439096_wp, &
      &   7.79700369915555_wp, -2.57312992561942_wp,  0.17536976028259_wp, &
      &   6.29203608472598_wp, -1.63892325608139_wp, -1.88656905005969_wp, &
      &   7.14424961136221_wp, -1.02265632319745_wp, -4.21522221586895_wp, &
      &   5.74884371666377_wp, -0.93993421959222_wp, -6.04415109821061_wp, &
      &   9.85346369971059_wp, -0.34493446296585_wp, -4.44925183904305_wp, &
      &  11.10369929531118_wp,  1.16050692440512_wp, -2.69466751087552_wp, &
      &  13.60102427963728_wp,  1.84094403320177_wp, -3.07750106640408_wp, &
      &  14.87071389256069_wp,  1.02742909067580_wp, -5.21818777486650_wp, &
      &  13.62867824577475_wp, -0.43970162177492_wp, -6.99900029232176_wp, &
      &  11.12857518862883_wp, -1.10058730663001_wp, -6.62215913615899_wp, &
      &  10.12544944506200_wp, -2.22935263919387_wp, -8.01239176157645_wp, &
      &  14.61399690104017_wp, -1.05501490837713_wp, -8.68805610640805_wp, &
      &  16.82969309541321_wp,  1.55352561784152_wp, -5.51038202215573_wp, &
      &  14.56336107391366_wp,  3.01617997030377_wp, -1.70078080272236_wp, &
      &  10.10364793716325_wp,  1.82779557593624_wp, -1.03565574792613_wp, &
      &   4.44789716772231_wp, -2.14638203181252_wp, -1.75377902279299_wp, &
      &   7.65649430197261_wp, -0.93019506774360_wp,  2.48608724258786_wp, &
      &   5.68606140567588_wp,  0.75277614320481_wp,  2.87641442326019_wp, &
      &   5.54105367548906_wp,  2.13612665304333_wp,  5.09291609602019_wp, &
      &   7.36899800970274_wp,  1.84496195213717_wp,  6.94420672264969_wp, &
      &   9.34031335281618_wp,  0.16048616574779_wp,  6.57432265205073_wp, &
      &   9.47492767933860_wp, -1.22058982660547_wp,  4.35683012302981_wp, &
      &  10.99066489411268_wp, -2.56873559858273_wp,  4.05860593053675_wp, &
      &  10.77232312283835_wp, -0.07823692377195_wp,  8.02123940448809_wp, &
      &   7.25497128542544_wp,  2.92907458474438_wp,  8.67906628566800_wp, &
      &   3.99004733324701_wp,  3.44000682122569_wp,  5.38097191476319_wp, &
      &   4.25346598365691_wp,  0.98335917704217_wp,  1.42864426000057_wp, &
      &   9.75526108547830_wp, -2.73430178671873_wp, -0.43650459888694_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine taxol


subroutine pdb_4qxx(mol)
   use xtb_mctc_filetypes, only : fileType
   use xtb_type_vendordata, only : pdb_data
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 76
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "N", "C", "C", "O", "H", "H", "H", "H", "H", "N", "C", "C", "O", &
      & "C", "C", "O", "N", "H", "H", "H", "H", "H", "H", "N", "C", "C", &
      & "O", "C", "C", "C", "C", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "N", "C", "C", "O", "C", "C", "C", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "N", "C", "C", "O", "C", "O", "O", &
      & "H", "H", "H", "H", "H", "O", "H", "H", "O", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -1.55146500437301_wp,  -3.91551216694383_wp,  31.38645829187747_wp, &
      & -3.22198274355175_wp,  -4.43140735110197_wp,  29.26618577676599_wp, &
      & -1.82925471891971_wp,  -5.68429565548603_wp,  27.10622901671928_wp, &
      &  0.48754929491868_wp,  -5.63516278080430_wp,  27.00796326735582_wp, &
      & -4.02511627200307_wp,  -2.65506495876259_wp,  28.60100224261338_wp, &
      & -4.74510185868531_wp,  -5.66728812194235_wp,  29.89357479193267_wp, &
      & -2.57758619484140_wp,  -3.29190260367575_wp,  32.86989316207578_wp, &
      & -0.28345889239458_wp,  -2.57947592079070_wp,  30.88568091531371_wp, &
      & -0.63116846706527_wp,  -5.51422032004928_wp,  31.87589731274545_wp, &
      & -3.25221835874051_wp,  -6.80868259531787_wp,  25.36957086931514_wp, &
      & -2.15617730814812_wp,  -8.16928527881187_wp,  23.22662164281209_wp, &
      & -3.30324095937153_wp,  -7.36993120225914_wp,  20.72651421189187_wp, &
      & -5.58414018017328_wp,  -6.95986067126165_wp,  20.54699024670863_wp, &
      & -2.55679920939913_wp, -11.01143310655488_wp,  23.51952916495316_wp, &
      & -1.28312391957281_wp, -12.07723854195851_wp,  25.85712016423382_wp, &
      &  0.98076776768526_wp, -11.72008033754134_wp,  26.25963179143412_wp, &
      & -2.73632317458237_wp, -13.39248780266937_wp,  27.41236462050542_wp, &
      & -5.15139293778421_wp,  -6.72175520165020_wp,  25.53397702690399_wp, &
      & -0.13228081645081_wp,  -7.79134008895242_wp,  23.17370931623177_wp, &
      & -1.78579102208587_wp, -11.95818580715279_wp,  21.86034978147020_wp, &
      & -4.57880597514716_wp, -11.39315774831292_wp,  23.62724354406310_wp, &
      & -4.58636487894435_wp, -13.64004190202731_wp,  27.00985299330512_wp, &
      & -1.99555060245786_wp, -14.14837818238826_wp,  28.99973441791509_wp, &
      & -1.71398143601258_wp,  -7.18662778517731_wp,  18.79143483981153_wp, &
      & -2.62293961762454_wp,  -6.75765999468685_wp,  16.22518700066590_wp, &
      & -1.47965541829972_wp,  -8.80612292372503_wp,  14.56789734313224_wp, &
      &  0.82581023984288_wp,  -9.04800784523507_wp,  14.44317543047863_wp, &
      & -1.84626225246338_wp,  -4.12905119921442_wp,  15.27087539627081_wp, &
      & -2.87994234672896_wp,  -3.15395260937705_wp,  12.72919399446605_wp, &
      & -2.31491428788909_wp,  -0.36093765631577_wp,  12.41549948688271_wp, &
      & -1.81791636322392_wp,  -4.55234981185699_wp,  10.47097148505588_wp, &
      &  0.16251643163956_wp,  -7.34725449086758_wp,  19.10323962144556_wp, &
      & -4.67707172451061_wp,  -6.91828670037711_wp,  16.19117193357855_wp, &
      & -2.42640811889763_wp,  -2.77600741951761_wp,  16.71084656963529_wp, &
      &  0.20975958037199_wp,  -4.08558750238058_wp,  15.16694046905946_wp, &
      & -4.92462582386855_wp,  -3.39772725683640_wp,  12.73108372041535_wp, &
      & -3.06702521570938_wp,   0.67841161579770_wp,  14.02743572163324_wp, &
      & -3.19552658026159_wp,   0.32692258922842_wp,  10.68451051732646_wp, &
      & -0.27778971454669_wp,  -0.08125821581978_wp,  12.30778510777277_wp, &
      & -2.20719990877915_wp,  -6.56679767380783_wp,  10.66372353188419_wp, &
      & -2.70041838154572_wp,  -3.84559230681984_wp,   8.74943114524611_wp, &
      &  0.21731848416918_wp,  -4.25188338591874_wp,  10.37270573569242_wp, &
      & -3.08970192710095_wp, -10.24987354898810_wp,  13.28288369761014_wp, &
      & -2.20153073093126_wp, -12.20762963246002_wp,  11.56323308374967_wp, &
      & -3.38449917519132_wp, -11.77299266412166_wp,   8.98564688890827_wp, &
      & -5.69563401118181_wp, -11.73330841918642_wp,   8.73053388575314_wp, &
      & -2.96120056254874_wp, -14.87592267286769_wp,  12.45896318371655_wp, &
      & -1.91240266068878_wp, -16.88281163102133_wp,  10.64482627239122_wp, &
      & -2.04657320308889_wp, -15.34457470829340_wp,  15.15182266146508_wp, &
      & -4.96619979475309_wp,  -9.98153246418790_wp,  13.50398163367791_wp, &
      & -0.15117807594378_wp, -12.09802552740078_wp,  11.40260637805940_wp, &
      & -5.01722239538411_wp, -15.00253431147060_wp,  12.44384537612217_wp, &
      & -2.46231291193427_wp, -18.75741977272417_wp,  11.30056117679736_wp, &
      & -2.67207249230626_wp, -16.56533767153940_wp,   8.75699004904330_wp, &
      &  0.14172944619729_wp, -16.75053081457053_wp,  10.58813449391230_wp, &
      & -2.60215263218227_wp, -17.23619038353991_wp,  15.74708633549371_wp, &
      &  0.00566917784789_wp, -15.17827882475524_wp,  15.21796306969049_wp, &
      & -2.88939097647544_wp, -13.95184668366135_wp,  16.41415959559562_wp, &
      & -1.82547526702111_wp, -11.43662144514676_wp,   7.06001614657440_wp, &
      & -2.88372179862755_wp, -11.12670638946201_wp,   4.54857035995840_wp, &
      & -2.28089922080174_wp, -13.38870835077078_wp,   2.88939097647544_wp, &
      & -0.82581023984288_wp, -15.07245417159460_wp,   3.59425875556331_wp, &
      & -1.94830745372543_wp,  -8.68518046297001_wp,   3.33914575240818_wp, &
      &  0.68219106769630_wp,  -8.79100511613065_wp,   2.91017796191771_wp, &
      & -3.28245397392927_wp, -13.56445286405542_wp,   0.81069243224851_wp, &
      &  0.06236095632681_wp, -11.39693720021151_wp,   7.33213668327320_wp, &
      & -4.93218472766574_wp, -11.00198447680840_wp,   4.73187377704023_wp, &
      & -2.91584713976561_wp,  -8.40739074842332_wp,   1.54201637462653_wp, &
      & -2.36971634041871_wp,  -7.10347984340824_wp,   4.58825460489364_wp, &
      &  1.23399104489108_wp,  -7.23954011175764_wp,   2.14861840435094_wp, &
      &  1.76689376259290_wp,  -9.77933178761310_wp,  31.18425761530267_wp, &
      &  1.50044240374199_wp, -10.43506669201923_wp,  29.51940905397182_wp, &
      &  3.15395260937705_wp,  -8.61904005474461_wp,  31.15969117796181_wp, &
      &  1.30580063096438_wp, -15.88881578169100_wp,  33.78641024748494_wp, &
      &  2.63049852142173_wp, -15.35402333803989_wp,  34.89567937972240_wp, &
      &  1.87649786765214_wp, -15.79055003232754_wp,  32.07242881147236_wp],&
      & shape(xyz))
   type(pdb_data), parameter :: pdb(nat) = [&
      & pdb_data(name=" N  ", residue="GLY", residue_number=1), &
      & pdb_data(name=" CA ", residue="GLY", residue_number=1), &
      & pdb_data(name=" C  ", residue="GLY", residue_number=1), &
      & pdb_data(name=" O  ", residue="GLY", residue_number=1), &
      & pdb_data(name=" HA2", residue="GLY", residue_number=1), &
      & pdb_data(name=" HA3", residue="GLY", residue_number=1), &
      & pdb_data(name=" H1 ", residue="GLY", residue_number=1), &
      & pdb_data(name=" H2 ", residue="GLY", residue_number=1), &
      & pdb_data(name=" H3 ", residue="GLY", residue_number=1), &
      & pdb_data(name=" N  ", residue="ASN", residue_number=2), &
      & pdb_data(name=" CA ", residue="ASN", residue_number=2), &
      & pdb_data(name=" C  ", residue="ASN", residue_number=2), &
      & pdb_data(name=" O  ", residue="ASN", residue_number=2), &
      & pdb_data(name=" CB ", residue="ASN", residue_number=2), &
      & pdb_data(name=" CG ", residue="ASN", residue_number=2), &
      & pdb_data(name=" OD1", residue="ASN", residue_number=2), &
      & pdb_data(name=" ND2", residue="ASN", residue_number=2), &
      & pdb_data(name=" H  ", residue="ASN", residue_number=2), &
      & pdb_data(name=" HA ", residue="ASN", residue_number=2), &
      & pdb_data(name=" HB2", residue="ASN", residue_number=2), &
      & pdb_data(name=" HB3", residue="ASN", residue_number=2), &
      & pdb_data(name="HD21", residue="ASN", residue_number=2), &
      & pdb_data(name="HD22", residue="ASN", residue_number=2), &
      & pdb_data(name=" N  ", residue="LEU", residue_number=3), &
      & pdb_data(name=" CA ", residue="LEU", residue_number=3), &
      & pdb_data(name=" C  ", residue="LEU", residue_number=3), &
      & pdb_data(name=" O  ", residue="LEU", residue_number=3), &
      & pdb_data(name=" CB ", residue="LEU", residue_number=3), &
      & pdb_data(name=" CG ", residue="LEU", residue_number=3), &
      & pdb_data(name=" CD1", residue="LEU", residue_number=3), &
      & pdb_data(name=" CD2", residue="LEU", residue_number=3), &
      & pdb_data(name=" H  ", residue="LEU", residue_number=3), &
      & pdb_data(name=" HA ", residue="LEU", residue_number=3), &
      & pdb_data(name=" HB2", residue="LEU", residue_number=3), &
      & pdb_data(name=" HB3", residue="LEU", residue_number=3), &
      & pdb_data(name=" HG ", residue="LEU", residue_number=3), &
      & pdb_data(name="HD11", residue="LEU", residue_number=3), &
      & pdb_data(name="HD12", residue="LEU", residue_number=3), &
      & pdb_data(name="HD13", residue="LEU", residue_number=3), &
      & pdb_data(name="HD21", residue="LEU", residue_number=3), &
      & pdb_data(name="HD22", residue="LEU", residue_number=3), &
      & pdb_data(name="HD23", residue="LEU", residue_number=3), &
      & pdb_data(name=" N  ", residue="VAL", residue_number=4), &
      & pdb_data(name=" CA ", residue="VAL", residue_number=4), &
      & pdb_data(name=" C  ", residue="VAL", residue_number=4), &
      & pdb_data(name=" O  ", residue="VAL", residue_number=4), &
      & pdb_data(name=" CB ", residue="VAL", residue_number=4), &
      & pdb_data(name=" CG1", residue="VAL", residue_number=4), &
      & pdb_data(name=" CG2", residue="VAL", residue_number=4), &
      & pdb_data(name=" H  ", residue="VAL", residue_number=4), &
      & pdb_data(name=" HA ", residue="VAL", residue_number=4), &
      & pdb_data(name=" HB ", residue="VAL", residue_number=4), &
      & pdb_data(name="HG11", residue="VAL", residue_number=4), &
      & pdb_data(name="HG12", residue="VAL", residue_number=4), &
      & pdb_data(name="HG13", residue="VAL", residue_number=4), &
      & pdb_data(name="HG21", residue="VAL", residue_number=4), &
      & pdb_data(name="HG22", residue="VAL", residue_number=4), &
      & pdb_data(name="HG23", residue="VAL", residue_number=4), &
      & pdb_data(name=" N  ", residue="SER", residue_number=5), &
      & pdb_data(name=" CA ", residue="SER", residue_number=5), &
      & pdb_data(name=" C  ", residue="SER", residue_number=5), &
      & pdb_data(name=" O  ", residue="SER", residue_number=5), &
      & pdb_data(name=" CB ", residue="SER", residue_number=5), &
      & pdb_data(name=" OG ", residue="SER", residue_number=5), &
      & pdb_data(name=" OXT", residue="SER", residue_number=5), &
      & pdb_data(name=" H  ", residue="SER", residue_number=5), &
      & pdb_data(name=" HA ", residue="SER", residue_number=5), &
      & pdb_data(name=" HB2", residue="SER", residue_number=5), &
      & pdb_data(name=" HB3", residue="SER", residue_number=5), &
      & pdb_data(name=" HG ", residue="SER", residue_number=5), &
      & pdb_data(name=" O  ", residue="HOH", residue_number=101, het=.true.), &
      & pdb_data(name=" H1 ", residue="HOH", residue_number=101, het=.true.), &
      & pdb_data(name=" H2 ", residue="HOH", residue_number=101, het=.true.), &
      & pdb_data(name=" O  ", residue="HOH", residue_number=102, het=.true.), &
      & pdb_data(name=" H1 ", residue="HOH", residue_number=102, het=.true.), &
      & pdb_data(name=" H2 ", residue="HOH", residue_number=102, het=.true.)]
   call init(mol, sym, xyz)
   mol%ftype = fileType%pdb
   mol%pdb = pdb
end subroutine pdb_4qxx


subroutine manganese(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 37
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "Mn", "S ", "C ", "C ", "Cl", "C ", "Cl", "C ", "Cl", "C ", "Cl", &
      & "C ", "S ", "S ", "C ", "C ", "Cl", "C ", "Cl", "C ", "Cl", "C ", &
      & "Cl", "C ", "S ", "S ", "C ", "C ", "Cl", "C ", "Cl", "C ", "Cl", &
      & "C ", "Cl", "C ", "S "]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 16.92179988988945_wp, 18.98986097777520_wp,  3.24087546770245_wp, &
      & 19.63586152587131_wp, 20.32013129211184_wp,  0.11672364757322_wp, &
      & 22.42490162937085_wp, 18.63498480641440_wp,  0.51123530893977_wp, &
      & 24.43533854799430_wp, 18.96679044749580_wp, -1.16619107933701_wp, &
      & 24.10320655124145_wp, 21.04804012175429_wp, -3.62199123975962_wp, &
      & 26.67304619249890_wp, 17.63473528700005_wp, -0.86283412858670_wp, &
      & 29.10641492795817_wp, 18.06077343535132_wp, -2.93237789545227_wp, &
      & 26.92967173230385_wp, 15.94170921101183_wp,  1.14409564764744_wp, &
      & 29.67819006432741_wp, 14.30306793606362_wp,  1.52713667250176_wp, &
      & 24.94495171617919_wp, 15.59872810861147_wp,  2.82212599233761_wp, &
      & 25.24296074249298_wp, 13.52507399883359_wp,  5.28918929736323_wp, &
      & 22.68267158721068_wp, 16.92807661463004_wp,  2.53303571661413_wp, &
      & 20.20875462411216_wp, 16.47769596146290_wp,  4.63913679888663_wp, &
      & 18.00855803970110_wp, 22.47123033971641_wp,  5.62523321580372_wp, &
      & 16.61135949039080_wp, 22.08499945745968_wp,  8.57009936519192_wp, &
      & 17.00003397727257_wp, 23.86543631901340_wp, 10.47904533917425_wp, &
      & 18.88891635290602_wp, 26.42371499485825_wp,  9.87802385915071_wp, &
      & 15.88411546633731_wp, 23.57816490451612_wp, 12.83421891560495_wp, &
      & 16.38872067178397_wp, 25.76827696639874_wp, 15.14417153904157_wp, &
      & 14.34348520569459_wp, 21.49009369752586_wp, 13.30913898303189_wp, &
      & 12.96097870872229_wp, 21.12987678268715_wp, 16.19802483576108_wp, &
      & 13.94509129133780_wp, 19.70979440802124_wp, 11.42632943070910_wp, &
      & 12.06113958865126_wp, 17.13940636829330_wp, 11.99208881349134_wp, &
      & 15.06583087111491_wp, 19.97579506723324_wp,  9.05128897640084_wp, &
      & 14.57279532382023_wp, 17.72326932566981_wp,  6.71893913204056_wp, &
      & 13.67978794100109_wp, 21.26749909850624_wp,  1.39841364309570_wp, &
      & 11.89907990377364_wp, 19.12401451264176_wp, -0.33337902690679_wp, &
      &  9.71226031141248_wp, 19.90517204008351_wp, -1.58848435777456_wp, &
      &  8.80692038807563_wp, 23.00512724224015_wp, -1.38200628651352_wp, &
      &  8.28045861229940_wp, 18.20608901448260_wp, -2.98007306663178_wp, &
      &  5.61062695702381_wp, 19.19334345546141_wp, -4.49071280954136_wp, &
      &  9.02964500024056_wp, 15.68171863620897_wp, -3.14110927593525_wp, &
      &  7.27555526478972_wp, 13.58529918689250_wp, -4.84826842332567_wp, &
      & 11.20095088742098_wp, 14.88598243998618_wp, -1.90593885307921_wp, &
      & 12.13599654679039_wp, 11.79301714514339_wp, -2.09252037776529_wp, &
      & 12.65253499366167_wp, 16.57942293287508_wp, -0.49502372796594_wp, &
      & 15.35828807156226_wp, 15.55907882360629_wp,  1.05562907219407_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -2.0_wp
   call init(mol, sym, xyz, chrg=charge)
end subroutine manganese


subroutine vcpco4(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 19
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "V", "C", "C", "C", "C", "C", "H", "H", "H", "H", "H", "C", "C", "C", &
      & "C", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.01825776267725_wp,  0.13110550404250_wp, -0.00041948324548_wp, &
      & -1.89470237617899_wp,  4.08642287701680_wp, -0.00030420996203_wp, &
      & -0.30782201031402_wp,  3.88425999400855_wp,  2.20719428591717_wp, &
      &  2.27116070415413_wp,  3.55786274705629_wp,  1.36798210568814_wp, &
      &  2.27143282469211_wp,  3.55811030115682_wp, -1.36783652495488_wp, &
      & -0.30737225553597_wp,  3.88461526248869_wp, -2.20751735722155_wp, &
      & -3.93427226450870_wp,  4.35896104443439_wp, -0.00049696200977_wp, &
      & -0.94195923983321_wp,  3.98951206070242_wp,  4.16170760886828_wp, &
      &  3.92482020592999_wp,  3.39076372920244_wp,  2.58034524280125_wp, &
      &  3.92534932919828_wp,  3.39117568946133_wp, -2.57987462920319_wp, &
      & -0.94109563507031_wp,  3.99017157506183_wp, -4.16214784318207_wp, &
      & -3.29850987066545_wp, -1.29654655258453_wp,  0.00027026672926_wp, &
      & -0.17610811102747_wp, -1.64797888903101_wp, -3.13400275137814_wp, &
      & -0.17602496308531_wp, -1.64828313491028_wp,  3.13280851640703_wp, &
      &  2.94080399470195_wp, -1.97941414538782_wp, -0.00023240037562_wp, &
      & -5.38907023452527_wp, -2.16777069054547_wp,  0.00282139677283_wp, &
      & -0.29329379714738_wp, -2.74172581747386_wp, -5.11399524942367_wp, &
      & -0.29324466427246_wp, -2.74226249964599_wp,  5.11269330007294_wp, &
      &  4.79632024388083_wp, -3.27840609987675_wp,  0.00107528998745_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine vcpco4


subroutine feco5(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 11
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "fe", "c", "c", "c", "c", "c", "o", "o", "o", "o", "o"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.00000000000000_wp,  0.00414196770719_wp,  0.00000003847382_wp, &
      &  0.00000000000000_wp,  3.39649674965194_wp, -0.00000185125214_wp, &
      &  2.92058469648008_wp, -1.71911047193131_wp,  0.00000192819978_wp, &
      & -2.92058469648008_wp, -1.71911047193131_wp,  0.00000192819978_wp, &
      &  0.00000000000000_wp,  0.02144618830643_wp,  3.39858524690276_wp, &
      & -0.00000000000000_wp,  0.02144618830643_wp, -3.39858516995512_wp, &
      &  0.00000000000000_wp,  5.64401208242199_wp, -0.00000185125214_wp, &
      &  4.85252146340858_wp, -2.86696025455742_wp,  0.00000003847382_wp, &
      & -4.85252146340858_wp, -2.86696025455742_wp,  0.00000003847382_wp, &
      &  0.00000000000000_wp,  0.03728587128806_wp,  5.63976243880901_wp, &
      & -0.00000000000000_wp,  0.03728776101402_wp, -5.63976236186137_wp],&
      & shape(xyz))
   call init(mol, sym, xyz)
end subroutine feco5


subroutine bug332(mol)
   use xtb_mctc_filetypes, only : fileType
   use xtb_type_vendordata, only : sdf_data
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 13
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "O", "C", "C", "C", "C", "C", "N", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.81493577407313_wp, -4.24016708503309_wp, -0.05196746360567_wp,  &
      &  1.60154274202939_wp, -2.27863154966258_wp, -0.01417294461973_wp,  &
      &  2.83099844464216_wp,  0.08806122923725_wp,  0.44597532403414_wp,  &
      &  1.46340377513576_wp,  2.31302456193979_wp,  0.48452573339981_wp,  &
      & -1.07355331179575_wp,  2.33456743776178_wp,  0.09675396860402_wp,  &
      & -2.54527188110842_wp, -0.01870828689804_wp, -0.38777176479579_wp,  &
      & -0.96848454901482_wp, -2.31018997301585_wp, -0.41441690068088_wp,  &
      &  4.85281623779525_wp,  0.06500657265582_wp,  0.75551243452903_wp,  &
      &  2.44870688509933_wp,  4.07557195484931_wp,  0.83261325326035_wp,  &
      & -2.07113964042975_wp,  4.12319304877160_wp,  0.14021766543785_wp,  &
      & -3.51470129309789_wp,  0.11451739252741_wp, -2.22137285339888_wp,  &
      & -4.00017188947235_wp, -0.25964834543344_wp,  1.07695481850448_wp,  &
      & -1.83832540347633_wp, -4.00659695769996_wp, -0.74266229807381_wp], &
      & shape(xyz))
   integer, parameter :: bonds(3, 13) = reshape([ &
      &  1,  2,  2,  &
      &  2,  3,  1,  &
      &  3,  4,  2,  &
      &  3,  8,  1,  &
      &  4,  5,  1,  &
      &  4,  9,  1,  &
      &  5,  6,  1,  &
      &  5, 10,  1,  &
      &  6,  7,  1,  &
      &  6, 11,  1,  &
      &  6, 12,  1,  &
      &  7,  2,  1,  &
      &  7, 13,  1], &
      & shape(bonds))
   integer, parameter :: charge_at = 5
   real(wp), parameter :: charge = 1.0_wp
   integer :: ibond

   call init(mol, sym, xyz, chrg=charge)
   mol%ftype = fileType%sdf

   allocate(mol%sdf(nat), source=sdf_data())
   mol%sdf(charge_at)%charge = nint(charge)

   call mol%bonds%allocate(size=size(bonds, 2), order=size(bonds, 1))
   do ibond = 1, size(bonds, 2)
      call mol%bonds%push_back(bonds(:, ibond))
   end do

end subroutine bug332

subroutine co_cnx6(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 13
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "co", "c", "c", "c", "c", "c", "c","n", "n", "n", "n", "n","n"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.94877666423787_wp,  0.19317347093307_wp, -0.18341814349628_wp, &
      &  1.81797069098334_wp,  2.09322297545391_wp, -0.13557960699323_wp, &
      &  3.84410529185707_wp,  0.32638512462790_wp, -0.32326617548606_wp, &
      &  0.05351867320654_wp,  0.05921970587159_wp, -0.04456772850566_wp, &
      &  2.07888638555554_wp, -1.70686814314262_wp, -0.23025590574619_wp, &
      &  2.09136372684550_wp,  0.15595598873397_wp,  1.71598181170762_wp, &
      &  1.80637191292446_wp,  0.23061626304675_wp, -2.08282495154360_wp, &
      &  5.00922648908409_wp,  0.40491540990094_wp, -0.40959258032660_wp, &
      & -1.11165037804906_wp, -0.02302956650778_wp,  0.03747983755938_wp, &
      &  1.73422256026807_wp,  3.26080060741577_wp, -0.10606602566732_wp, &
      &  2.15910094532487_wp, -2.87478868845240_wp, -0.25552216296531_wp, &
      &  2.18240567286263_wp,  0.13350207175117_wp,  2.88316823771709_wp, &
      &  1.71548136489902_wp,  0.25339478036763_wp, -3.25001660625405_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -3.0_wp
   call init(mol, sym, xyz, chrg=charge)
end subroutine co_cnx6

subroutine fe_cnx6(mol)
   type(TMolecule), intent(out) :: mol
   integer, parameter :: nat = 13
   character(len=*), parameter :: sym(nat) = [character(len=4) ::&
      & "fe", "c", "c", "c", "c", "c", "c", "n", "n", "n", "n", "n", "n"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.94840992315069_wp,  0.19244055581557_wp, -0.18393767617093_wp, &
      &  1.78272604464510_wp,  2.62293683490692_wp, -0.10723975007002_wp, &
      &  4.32689647736570_wp,  0.35980495482330_wp, -0.35944563157517_wp, &
      & -0.43003654695118_wp,  0.02471213925081_wp, -0.00863311985379_wp, &
      &  2.11416222449034_wp, -2.23778736891213_wp, -0.26000826782577_wp, &
      &  2.12989310575441_wp,  0.16148305276099_wp,  2.24615988473859_wp, &
      &  1.76690913095287_wp,  0.22401331347493_wp, -2.61356432743079_wp, &
      &  5.49819068007791_wp,  0.44208319690825_wp, -0.44642524573858_wp, &
      & -1.60132363118181_wp, -0.05824176614676_wp,  0.07779983551021_wp, &
      &  1.70181293250445_wp,  3.79667254862885_wp, -0.08319943830802_wp, &
      &  2.19561869833960_wp, -3.41158489904539_wp, -0.27871155631627_wp, &
      &  2.21856307276739_wp,  0.13465068747712_wp,  3.41927390294530_wp, &
      &  1.67795788808459_wp,  0.25531675005748_wp, -3.78654860990502_wp],&
      & shape(xyz))
   real(wp), parameter :: charge = -4.0_wp
   integer, parameter  :: uhf = 4
   call init(mol, sym, xyz, chrg=charge, uhf=4)
end subroutine fe_cnx6

end module xtb_test_molstock
