% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xgb.plot.multi.trees.R
\name{xgb.plot.multi.trees}
\alias{xgb.plot.multi.trees}
\title{Project all trees on one tree}
\usage{
xgb.plot.multi.trees(
  model,
  features_keep = 5,
  plot_width = NULL,
  plot_height = NULL,
  render = TRUE,
  ...
)
}
\arguments{
\item{model}{Object of class \code{xgb.Booster}. If it contains feature names (they can be set through
\link{setinfo}), they will be used in the output from this function.}

\item{features_keep}{Number of features to keep in each position of the multi trees,
by default 5.}

\item{plot_width, plot_height}{Width and height of the graph in pixels.
The values are passed to \code{\link[DiagrammeR:render_graph]{DiagrammeR::render_graph()}}.}

\item{render}{Should the graph be rendered or not? The default is \code{TRUE}.}

\item{...}{currently not used.}
}
\value{
The value depends on the \code{render} parameter:
\itemize{
\item If \code{render = TRUE} (default): Rendered graph object which is an htmlwidget of
class \code{grViz}. Similar to "ggplot" objects, it needs to be printed when not
running from the command line.
\item If \code{render = FALSE}: Graph object which is of DiagrammeR's class \code{dgr_graph}.
This could be useful if one wants to modify some of the graph attributes
before rendering the graph with \code{\link[DiagrammeR:render_graph]{DiagrammeR::render_graph()}}.
}
}
\description{
Visualization of the ensemble of trees as a single collective unit.
}
\details{
This function tries to capture the complexity of a gradient boosted tree model
in a cohesive way by compressing an ensemble of trees into a single tree-graph representation.
The goal is to improve the interpretability of a model generally seen as black box.

Note: this function is applicable to tree booster-based models only.

It takes advantage of the fact that the shape of a binary tree is only defined by
its depth (therefore, in a boosting model, all trees have similar shape).

Moreover, the trees tend to reuse the same features.

The function projects each tree onto one, and keeps for each position the
\code{features_keep} first features (based on the Gain per feature measure).

This function is inspired by this blog post:
\url{https://wellecks.wordpress.com/2015/02/21/peering-into-the-black-box-visualizing-lambdamart/}
}
\examples{

data(agaricus.train, package = "xgboost")

## Keep the number of threads to 2 for examples
nthread <- 2
data.table::setDTthreads(nthread)

bst <- xgboost(
  data = agaricus.train$data,
  label = agaricus.train$label,
  max_depth = 15,
  eta = 1,
  nthread = nthread,
  nrounds = 30,
  objective = "binary:logistic",
  min_child_weight = 50,
  verbose = 0
)

p <- xgb.plot.multi.trees(model = bst, features_keep = 3)
print(p)

\dontrun{
# Below is an example of how to save this plot to a file.
# Note that for export_graph() to work, the {DiagrammeRsvg} and {rsvg} packages
# must also be installed.

library(DiagrammeR)

gr <- xgb.plot.multi.trees(model = bst, features_keep = 3, render = FALSE)
export_graph(gr, "tree.pdf", width = 1500, height = 600)
}

}
