// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! Policy decisions in rpgpie.
//!
//! These include:
//! - Algorithm preferences.
//! - Which algorithms do we accept, for specific operations, depending on context?
//!
//! rpgpie currently couples policy limitations to signature creation timestamps:
//! Obsolete cryptographic mechanisms are considered valid for signatures that show a sufficiently
//! old creation time.
//!
//! This approach represents a tradeoff:
//!
//! - The upside is that old OpenPGP artifacts can be cryptographically validated and used
//!   seamlessly.
//! - The downside is that an attacker may trick users with weak, new (or newly modified) artifacts
//!   that show "old" signature creation timestamps.

use chrono::{DateTime, Utc};
use pgp::crypto::aead::AeadAlgorithm;
use pgp::crypto::hash::HashAlgorithm;
use pgp::crypto::sym::SymmetricKeyAlgorithm;
use pgp::types::{CompressionAlgorithm, EcdhPublicParams, PublicParams};

/// FIXME: where should this go? -> upstream to rpgp?
#[derive(Debug, Copy, Clone, PartialEq)]
pub enum Seipd {
    SED,
    SEIPD1,
    SEIPD2,
}

/// Preferred symmetric-key algorithms (in descending order of preference)
pub const PREFERRED_SEIPD_MECHANISMS: &[Seipd] = &[Seipd::SEIPD2, Seipd::SEIPD1];

/// FIXME: what's a good default?
pub const AEAD_CHUNK_SIZE: u8 = 8; // Chunk size: 16 KiB

/// Preferred symmetric-key algorithms (in descending order of preference)
pub const PREFERRED_SYMMETRIC_KEY_ALGORITHMS: &[SymmetricKeyAlgorithm] = &[
    SymmetricKeyAlgorithm::AES256,
    SymmetricKeyAlgorithm::AES192,
    SymmetricKeyAlgorithm::AES128,
    SymmetricKeyAlgorithm::Twofish,
    SymmetricKeyAlgorithm::Camellia256,
    SymmetricKeyAlgorithm::Camellia192,
    SymmetricKeyAlgorithm::Camellia128,
];

pub const PREFERRED_AEAD_ALGORITHMS: &[(SymmetricKeyAlgorithm, AeadAlgorithm)] = &[
    (SymmetricKeyAlgorithm::AES256, AeadAlgorithm::Ocb),
    (SymmetricKeyAlgorithm::AES192, AeadAlgorithm::Ocb),
    (SymmetricKeyAlgorithm::AES128, AeadAlgorithm::Ocb),
    (SymmetricKeyAlgorithm::AES256, AeadAlgorithm::Eax),
    (SymmetricKeyAlgorithm::AES192, AeadAlgorithm::Eax),
    (SymmetricKeyAlgorithm::AES128, AeadAlgorithm::Eax),
    (SymmetricKeyAlgorithm::AES256, AeadAlgorithm::Gcm),
    (SymmetricKeyAlgorithm::AES192, AeadAlgorithm::Gcm),
    (SymmetricKeyAlgorithm::AES128, AeadAlgorithm::Gcm),
];

/// Preferred hash algorithms (in descending order of preference)
pub const PREFERRED_HASH_ALGORITHMS: &[HashAlgorithm] = &[
    HashAlgorithm::SHA2_512,
    HashAlgorithm::SHA2_384,
    HashAlgorithm::SHA2_256,
    HashAlgorithm::SHA2_224,
];

/// Preferred compression algorithms (in descending order of preference)
pub const PREFERRED_COMPRESSION_ALGORITHMS: &[CompressionAlgorithm] = &[CompressionAlgorithm::ZLIB];

/// How many layers deep are we willing to unpack into a message?
pub const MAX_RECURSION: usize = 10;

/// Cutoff time for MD5 hash acceptance: January 1, 2010 12:00:00 AM (GMT)
const MD5_REJECT_AFTER: i64 = 1262304000;

/// Cutoff time for SHA1 hash acceptance in data signatures: January 1, 2014 12:00:00 AM (GMT)
/// (See https://csrc.nist.gov/projects/hash-functions)
const SHA1_FOR_DATA_REJECT_AFTER: i64 = 1388534400;

/// Cutoff time for SHA1 hash acceptance: February 1, 2023 12:00:00 AM (GMT)
const SHA1_REJECT_AFTER: i64 = 1675209600;

/// Cutoff time for RSA <2k bits acceptance: January 1, 2014 12:00:00 AM (GMT)
/// (e.g. NIST guidance was to drop RSA 1k after 2013).
const RSA_UNDER_2K_REJECT_AFTER: i64 = 1388534400;

const RSA_UNDER_2K_CUTOFF_BIT_SIZE: usize = 2048;

/// Cutoff time for DSA acceptance: February 3, 2023 12:00:00 AM (GMT)
/// See https://nvlpubs.nist.gov/nistpubs/FIPS/NIST.FIPS.186-5.pdf
const DSA_REJECT_AFTER: i64 = 1675382400;

/// Returns our policy about a `PublicParams` at a reference time.
pub(crate) fn acceptable_pk_algorithm(pp: &PublicParams, reference: &DateTime<Utc>) -> bool {
    if let PublicParams::RSA { n, e: _ } = pp {
        let mut rsa_bits = n.len() * 8;
        // There may be leading zero bits in this count, compensate
        if let Some(first) = n.first() {
            rsa_bits -= first.leading_zeros() as usize;
        }

        // Reject RSA with key size under 2048 bit, after 31.12.2013
        if rsa_bits < RSA_UNDER_2K_CUTOFF_BIT_SIZE
            && reference.timestamp() > RSA_UNDER_2K_REJECT_AFTER
        {
            return false;
        }
    }

    if let PublicParams::DSA { .. } = pp {
        // Reject DSA after 03.02.2023
        if reference.timestamp() > DSA_REJECT_AFTER {
            return false;
        }
    }

    true
}

/// Returns our policy about a `HashAlgorithm` at a reference time.
///
/// Note that we reject SHA-1 at different dates, depending on whether it is used for data signatures or other types of signatures.
pub(crate) fn acceptable_hash_algorithm(
    hash_algo: &HashAlgorithm,
    reference: &DateTime<Utc>,
    data_signature: bool,
) -> bool {
    // #![allow(clippy::match_like_matches_macro)]
    match hash_algo {
        // Consider MD5 signatures invalid if they claim to have been made after the cutoff timestamp.
        HashAlgorithm::MD5 => {
            // only return "true" for legacy signatures, false for new ones
            MD5_REJECT_AFTER > reference.timestamp()
        }

        // Consider SHA1 signatures invalid if they claim to have been made after the cutoff timestamp.
        HashAlgorithm::SHA1 => {
            // only return "true" for legacy signatures, false for new ones

            match data_signature {
                true => SHA1_FOR_DATA_REJECT_AFTER > reference.timestamp(),
                false => SHA1_REJECT_AFTER > reference.timestamp(),
            }
        }

        HashAlgorithm::RIPEMD160 => {
            // FIXME: reject RIPEMD160 (RFC 9580) [starting when?]
            true
        }

        _ => true,
    }
}

/// Return our policy about acceptable encryption mechanisms.
///
/// Note that encryption happens "now", by definition, so we're not allowing historical algorithms
/// (which we may still allow for decryption of historical data).
pub(crate) fn accept_for_encryption(pp: &PublicParams) -> bool {
    match pp {
        PublicParams::ECDH(EcdhPublicParams::Known { hash, alg_sym, .. }) => {
            if !PREFERRED_HASH_ALGORITHMS.contains(hash) {
                return false;
            }
            if !PREFERRED_SYMMETRIC_KEY_ALGORITHMS.contains(alg_sym) {
                return false;
            }

            true
        }
        _ => true,
    }
}
