// SPDX-FileCopyrightText: Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

//! # rpgpie 🦀️🔐🥧
//!
//! rpgpie is an experimental higher level [OpenPGP](https://www.rfc-editor.org/rfc/rfc4880)
//! library based on [rPGP](https://github.com/rpgp/rpgp).
//! It exposes a convenient API for some OpenPGP operations, and applies some amount of policy
//! (to limit reliance on obsolete algorithms).
//!
//! The main goals of rpgpie include simplicity, collaboration and fun 🥳.
//!
//! ## "OpenPGP for application developers"
//!
//! rpgpie applies the terminology outlined in the
//! ["OpenPGP for application developers"](https://openpgp.dev/) documentation.
//! We explore designing an API that build on its terminology and concepts.
//!
//! ## Stateless OpenPGP (SOP)
//!
//! See [rsop](https://crates.io/crates/rsop) for a
//! [stateless OpenPGP (SOP)](https://datatracker.ietf.org/doc/draft-dkg-openpgp-stateless-cli/)
//! implementation based on rpgpie.

pub mod card;
pub mod key;
pub mod msg;
pub mod policy;
pub mod sig;
pub(crate) mod util;

/// rpgpie version, via Cargo.toml
pub const VERSION: &str = env!("CARGO_PKG_VERSION");

/// version of the rpgp dependency we built against
pub const RPGP_VERSION: &str = pgp::VERSION;

/// Enum wrapper for the error types of this crate
#[derive(thiserror::Error, Debug)]
#[non_exhaustive]
pub enum Error {
    #[error("rPGP error: {0}")]
    Rpgp(pgp::errors::Error),

    #[error("OpenPGP card error: {0}")]
    Ocard(openpgp_card::Error),

    #[error("IO error: {0}")]
    Io(std::io::Error),

    #[error("Internal error: {0}")]
    Message(String),

    #[error("No binding signature for the primary key")]
    NoPrimaryBinding,
}

impl From<pgp::errors::Error> for Error {
    fn from(value: pgp::errors::Error) -> Self {
        Error::Rpgp(value)
    }
}

impl From<openpgp_card::Error> for Error {
    fn from(value: openpgp_card::Error) -> Self {
        Error::Ocard(value)
    }
}

impl From<std::io::Error> for Error {
    fn from(value: std::io::Error) -> Self {
        Error::Io(value)
    }
}

impl From<pgp::SubkeyParamsBuilderError> for Error {
    fn from(value: pgp::SubkeyParamsBuilderError) -> Self {
        Error::Rpgp(pgp::errors::Error::Message(format!(
            "SubkeyParamsBuilderError: {}",
            value
        )))
    }
}

impl From<pgp::SecretKeyParamsBuilderError> for Error {
    fn from(value: pgp::SecretKeyParamsBuilderError) -> Self {
        Error::Rpgp(pgp::errors::Error::Message(format!(
            "SecretKeyParamsBuilderError: {}",
            value
        )))
    }
}
