use std::{convert::TryFrom, ops::Deref, rc::Rc, str::FromStr};

use html5ever::{
    parse_fragment,
    tendril::TendrilSink,
    tree_builder::{QuirksMode, TreeBuilderOpts},
    ParseOpts, QualName,
};
use log::{debug, error, trace};
use url::Url;

use crate::types::{
    dom::{Node, RcDom},
    Class, Document, Fragment, Image, Item, ParentRelationship, PropertyValue, Relation,
    RelationshipKind, ValueKind,
};
use regex::Regex;

lazy_static::lazy_static! {
    static ref RE_CLASS_NAME: Regex = Regex::new(r#"^(?P<prefix>(h|p|u|dt|e))-(?P<name>([a-z0-9]+-)?[a-z]+(-[a-z]+)*)$"#).unwrap();
}

pub mod element;

use element::*;

/// Represents a Microformats2 parser.
pub struct Parser {
    dom: RcDom,
    pub(crate) document: Document,
    url: Url,
    base_url: Url,
}

fn ignored_elements() -> Vec<String> {
    vec![
        "script".to_owned(),
        "style".to_owned(),
        "template".to_owned(),
    ]
}

#[derive(thiserror::Error, Debug)]
pub enum RelationFromElementError {
    #[error("The provided element {0:?} is not a valid for relationship mapping.")]
    NotAValidElement(String),

    #[error("The provided element is missing a URL to reference.")]
    MissingURL,

    #[error("The provided element does not have an attribute with a 'rel' value.")]
    NotLinkRelation,
}

struct UrlRelation((Url, Relation));

impl UrlRelation {
    fn try_from_element(value: Rc<Element>, base_url: &Url) -> Result<Self, crate::Error> {
        if !["a", "link"].contains(&value.name.as_str()) {
            return Err(crate::Error::Relation(
                RelationFromElementError::NotAValidElement(value.name.clone()),
            ));
        }

        if !value.has_attribute("rel") {
            return Err(crate::Error::Relation(
                RelationFromElementError::NotLinkRelation,
            ));
        }

        let url_str = if let Some(v) = value.maybe_attribute("href") {
            v
        } else {
            return Err(crate::Error::Relation(RelationFromElementError::MissingURL));
        };

        url_str
            .parse()
            .or_else(|_| {
                if url_str.is_empty() {
                    Ok(base_url.clone())
                } else {
                    base_url.join(&url_str)
                }
            })
            .map_err(crate::Error::Url)
            .and_then(|url| {
                Ok(Self((
                    url,
                    Relation {
                        rels: value
                            .attribute("rel")
                            .map(|v| {
                                v.split_ascii_whitespace()
                                    .map(|s| s.to_string())
                                    .collect::<Vec<_>>()
                            })
                            .map(|mut l| {
                                l.dedup();
                                l
                            })
                            .unwrap_or_default(),
                        hreflang: value.attribute("hreflang").map(|s| s.trim().to_string()),
                        media: value.attribute("media").map(|s| s.trim().to_string()),
                        title: value.attribute("title").map(|s| s.trim().to_string()),
                        r#type: value.attribute("type").map(|s| s.trim().to_string()),
                        text: Some(extract_text(Rc::clone(&value.node), base_url.clone())?)
                            .filter(|v| !v.is_empty()),
                    },
                )))
            })
    }
}

/// Resolves an HTML fragment as a DOM using expected options.
pub(crate) fn from_html(html: &str) -> Result<RcDom, crate::Error> {
    parse_fragment(
        RcDom::default(),
        ParseOpts {
            tree_builder: TreeBuilderOpts {
                drop_doctype: true,
                scripting_enabled: true,
                quirks_mode: QuirksMode::Quirks,
                exact_errors: true,
                ..Default::default()
            },
            ..Default::default()
        },
        QualName {
            prefix: None,
            ns: "".into(),
            local: "html".into(),
        },
        vec![],
    )
    .from_utf8()
    .read_from(&mut html.as_bytes())
    .map_err(crate::Error::IO)
}

impl Parser {
    /// Merges a path component into a URL (or resolves it as a URL if it's independent)
    pub fn resolve_url(&self, url_str: &str) -> Result<Url, crate::Error> {
        trace!("Merging {:?} to {:?}", url_str, self.url.to_string());
        if url_str.is_empty() {
            Ok(self.base_url.clone())
        } else {
            self.base_url
                .join(url_str)
                .or_else(|_| url_str.parse())
                .map_err(crate::Error::Url)
        }
    }

    /// Creates a new parser with the provided document URL and HTML
    pub fn new(html: &str, url: Url) -> Result<Self, crate::Error> {
        from_html(html).map(|dom| Self {
            base_url: url.clone(),
            url,
            dom,
            document: Default::default(),
        })
    }

    /// Begins the operation of parsing the document tree.
    pub fn parse(&mut self) -> Result<Document, crate::Error> {
        self.walk_over_children(Rc::clone(&self.dom.document), ParentRelationship::Root)
            .map(|_| self.document.to_owned())
    }

    fn parse_rel(&mut self, rel_element: Rc<Element>) {
        if let Ok(UrlRelation((url, rel))) =
            UrlRelation::try_from_element(Rc::clone(&rel_element), &self.base_url)
        {
            self.document.add_relation(url, rel);
        }
    }

    fn walk_over_children(
        &mut self,
        node: Rc<Node>,
        relationship: ParentRelationship,
    ) -> Result<(), crate::Error> {
        let child_nodes = node.children.borrow();
        trace!(
            "Walking over the {node_count} children of the current node of {node:?}.",
            node_count = child_nodes.len()
        );
        for node in child_nodes.deref().iter() {
            if let Ok(element) = Element::try_from(Rc::clone(node)).map(Rc::new) {
                self.scan_element(element, relationship.to_owned())?
            }
        }

        Ok(())
    }

    fn scan_element(
        &mut self,
        element: Rc<Element>,
        relationship: ParentRelationship,
    ) -> Result<(), crate::Error> {
        if element.name == "base" {
            if let Some(url) = element.attribute("href").and_then(|u| u.parse().ok()) {
                trace!("Updated the base URL of this document to be {url:?}");
                self.base_url = url;
                Ok(())
            } else {
                Ok(())
            }
        } else if !ignored_elements().contains(&element.name) {
            if ["a", "link"].contains(&element.name.as_str()) && element.has_attribute("rel") {
                self.parse_rel(Rc::clone(&element))
            }
            self.parse_element_for_mf2(Rc::clone(&element), relationship)
        } else {
            trace!(
                "Ignoring element for relationship of {:?} of {:?}",
                relationship,
                element
            );
            Ok(())
        }
    }

    fn parse_element_for_mf2(
        &mut self,
        element: Rc<Element>,
        mut parent_item: ParentRelationship,
    ) -> Result<(), crate::Error> {
        trace!("Parsing element of {element:?} with {parent_item:?}");
        if let Some(property_classes) = element.property_classes() {
            trace!(
                "This element has {:?} as property classes.",
                property_classes
            );

            let non_root_property_classes = property_classes
                .iter()
                .filter(|p| !p.is_root())
                .cloned()
                .collect::<Vec<_>>();

            if element.is_microformat_item() {
                // We're processing a new item.
                let types = element.microformat_item_types();
                trace!(
                    "New root item encountered; has the root classes {:?}",
                    types
                );

                let mut current_item = parent_item.create_child_item(&mut self.document, &types);

                current_item.set_id(element.attribute("id"));

                self.walk_over_children(
                    Rc::clone(&element.node),
                    ParentRelationship::Child(current_item.clone()),
                )?;

                self.resolve_implied_properties(Rc::clone(&element), &mut current_item);

                if let Some(parent_item) = parent_item.resolve_item() {
                    trace!("Assocating this {current_item:?} to the parent item {parent_item:?}");
                    self.associate_item_to_parent_as_property(
                        parent_item.to_owned(),
                        current_item.to_owned(),
                        non_root_property_classes,
                    );
                }

                self.resolve_value_for_property_item(&mut current_item, Rc::clone(&element.node));

                trace!(
                    "Completed base processing of the new item {:#?}",
                    current_item
                );
            } else {
                if let Some(item) = parent_item.resolve_item_mut() {
                    trace!(
                        "No root item classes detected; working as a property to the current item {:#?}.", item
                    );
                    self.resolve_explicit_properties(
                        Rc::clone(&element),
                        item.to_owned(),
                        non_root_property_classes,
                    )?;
                    self.resolve_value_for_property_item(item, Rc::clone(&element.node));
                } else {
                    error!("No parent item DETECTED");
                }
                self.walk_over_children(Rc::clone(&element.node), parent_item)?;
            }

            Ok(())
        } else {
            trace!("No property classes detected on this element {element:?}.");
            self.walk_over_children(Rc::clone(&element.node), parent_item)
        }
    }

    fn resolve_implied_properties(&self, element: Rc<Element>, item: &mut Item) {
        if item.has_nested_microformats() {
            trace!("This item {item:#?} has nested microformats; skipping");
            return;
        } else {
            trace!("Getting value for {item:#?}");
        }

        if let Ok(Some(implied_name_value)) = self.resolve_implied_name(Rc::clone(&element), item) {
            let name = PropertyValue::Plain(implied_name_value);
            trace!("Implictly resolved the name to be {name:?}");
            item.append_property("name", name);
        }

        if let Some(url) = resolve_implied_url(Rc::clone(&element), item)
            .and_then(|u| self.resolve_url(&u).ok())
            .map(PropertyValue::Url)
        {
            trace!("Implictly resolved the url to be {url:?}");
            item.append_property("url", url);
        }

        if let Some(photo) = self.resolve_implied_photo(Rc::clone(&element), item) {
            trace!("Implictly resolved the photo to be {photo:?}");
            item.append_property("photo", photo);
        }
    }

    fn resolve_value_for_property_item(&self, item: &mut Item, node: Rc<Node>) {
        trace!("Attempting to resolve the value of the item {:#?}", item);
        let value = if let ParentRelationship::Property(relationship_kind, _) = item.parent() {
            trace!("Determined this item to be of a property with the relationship {:#?} with the properties {:#?}", relationship_kind, item.properties);

            match relationship_kind {
                RelationshipKind::Plain(property_name) => item
                    .get_property("name")
                    .or_else(|| item.get_property(&property_name))
                    .or_else(|| {
                        element::extract_only_text(node, self.base_url.clone())
                            .map(|t| t.trim().to_string())
                            .map(PropertyValue::Plain)
                            .ok()
                            .map(|v| vec![v])
                    })
                    .and_then(|v| {
                        v.iter().cloned().find_map(|v| {
                            if let PropertyValue::Plain(pv) = v {
                                Some(pv)
                            } else {
                                None
                            }
                        })
                    })
                    .map(|sv| ValueKind::Plain(sv.to_owned())),
                RelationshipKind::URL(property_name) => item
                    .get_property("url")
                    .or_else(|| item.get_property(&property_name))
                    .and_then(|v| {
                        v.iter().find_map(|v| {
                            if let PropertyValue::Url(u) = v {
                                Some(u.to_owned())
                            } else {
                                None
                            }
                        })
                    })
                    .map(ValueKind::Url),
                RelationshipKind::Datetime(property_name) => item
                    .get_property(&property_name)
                    .and_then(|values| {
                        values.into_iter().find_map(|v| {
                            if let PropertyValue::Temporal(dt) = v {
                                Some(dt)
                            } else {
                                None
                            }
                        })
                    })
                    .map(|sv| sv.to_string())
                    .map(ValueKind::Plain),
                RelationshipKind::HTML(property_name) => item
                    .get_property(&property_name)
                    .and_then(|v| {
                        v.iter().find_map(|v| {
                            if let PropertyValue::Fragment(Fragment { ref value, .. }) = v {
                                Some(value.to_owned())
                            } else {
                                None
                            }
                        })
                    })
                    .map(|sv| ValueKind::Plain(sv.to_owned())),
            }
        } else {
            trace!("Not associated as a property, moving on.");
            None
        };

        item.set_value(value);
    }

    // This will make it go from being a child to a property
    // This will also set `value` based on the kind of property it is.
    // FIXME: Remove logic for setting `value` into separate method.
    fn associate_item_to_parent_as_property(
        &self,
        mut parent_item: Item,
        mut current_item: Item,
        property_classes: Vec<PropertyClass>,
    ) {
        if property_classes.is_empty() {
            trace!("This is not a property element due to the lack of provided classes; skipping.");
        } else {
            debug!(
                "Associating {:#?} with the classes {:#?} to {:#?}",
                current_item, property_classes, parent_item
            );

            parent_item.remove_child(&current_item);

            for property in property_classes {
                trace!("Processing the associative property {:?}", property);

                let (property_name, property_relationship): (String, ParentRelationship) =
                    if let PropertyClass::Plain(property_name) = property {
                        (
                            property_name.to_owned(),
                            ParentRelationship::Property(
                                RelationshipKind::Plain(property_name),
                                parent_item.clone(),
                            ),
                        )
                    } else if let PropertyClass::Linked(property_name) = property {
                        (
                            property_name.to_owned(),
                            ParentRelationship::Property(
                                RelationshipKind::URL(property_name),
                                parent_item.clone(),
                            ),
                        )
                    } else if let PropertyClass::Timestamp(property_name) = property {
                        (
                            property_name.to_owned(),
                            ParentRelationship::Property(
                                RelationshipKind::Datetime(property_name),
                                parent_item.clone(),
                            ),
                        )
                    } else if let PropertyClass::Hypertext(property_name) = property {
                        (
                            property_name.to_owned(),
                            ParentRelationship::Property(
                                RelationshipKind::HTML(property_name),
                                parent_item.clone(),
                            ),
                        )
                    } else {
                        panic!("invalid")
                    };
                current_item.set_parent(property_relationship);

                parent_item
                    .append_property(&property_name, PropertyValue::Item(current_item.to_owned()));
            }
        }
    }

    // Resolves the properties directly associated to this item (that haven't been associated as its parent).
    fn resolve_explicit_properties(
        &mut self,
        element: Rc<Element>,
        mut item: Item,
        classes: Vec<PropertyClass>,
    ) -> Result<(), crate::Error> {
        if classes.is_empty() {
            return Ok(());
        }

        let groomed_classes = classes
            .iter()
            .filter(|class| match &class {
                PropertyClass::Root(_) => false,
                PropertyClass::Plain(p)
                | PropertyClass::Linked(p)
                | PropertyClass::Timestamp(p)
                | PropertyClass::Hypertext(p) => {
                    item.get_property(p)
                        .map(|values| values.contains(&PropertyValue::Item(item.to_owned())))
                        != Some(true)
                }
            })
            .collect::<Vec<_>>();
        trace!(
            "Resolving properties to add of {:#?} (only {:#?}) to {:#?}",
            classes,
            groomed_classes,
            item
        );

        let additional_property_values = groomed_classes
            .into_iter()
            .try_fold::<_, _, Result<_, crate::Error>>(Vec::default(), |mut acc, class| {
                trace!("Parsing {:?} as the property.", class);
                let parsed_value = if let PropertyClass::Plain(ref property_name) = class {
                    Some((
                        property_name.to_owned(),
                        self.parse_plain_property(Rc::clone(&element))?,
                    ))
                } else if let PropertyClass::Linked(ref property_name) = class {
                    Some((
                        property_name.to_owned(),
                        self.parse_linked_property(Rc::clone(&element))?,
                    ))
                } else if let PropertyClass::Timestamp(ref property_name) = class {
                    Some((
                        property_name.to_owned(),
                        self.parse_temporal_property(Rc::clone(&element)),
                    ))
                } else if let PropertyClass::Hypertext(ref property_name) = class {
                    Some((
                        property_name.to_owned(),
                        self.parse_html_property(Rc::clone(&element))?,
                    ))
                } else {
                    None
                };

                if let Some((name, values)) = parsed_value.filter(|(_, values)| !values.is_empty())
                {
                    acc.extend_from_slice(
                        &values
                            .into_iter()
                            .map(|value| (name.clone(), value))
                            .collect::<Vec<_>>(),
                    );
                    Ok(acc)
                } else {
                    Ok(acc)
                }
            })?;

        for (property_name, property_value) in additional_property_values {
            trace!(
                "Adding the value {:?} to the property {:?} in the item {:#?}",
                property_value,
                property_name,
                item
            );

            item.append_property(&property_name, property_value.clone());

            if let PropertyValue::Fragment { .. } = property_value {
                self.walk_over_children(
                    Rc::clone(&element.node),
                    ParentRelationship::Child(item.to_owned()),
                )?
            }
        }

        item.concatenate_times();
        Ok(())
    }

    fn parse_plain_property(
        &self,
        element: Rc<Element>,
    ) -> Result<Vec<PropertyValue>, crate::Error> {
        let value = if let Some(body) = value_class::parse(
            Rc::clone(&element.node),
            self.base_url.clone(),
            value_class::TypeHint::Plain,
        ) {
            return Ok(vec![body]);
        } else if let Some(v) = resolve_attribute_value_from_expected_property_element(
            Rc::clone(&element),
            &["abbr", "link"],
            "title",
        ) {
            vec![PropertyValue::Plain(v)]
        } else if let Some(v) = resolve_attribute_value_from_expected_property_element(
            Rc::clone(&element),
            &["data", "input"],
            "value",
        ) {
            vec![PropertyValue::Plain(v)]
        } else if let Some(v) = resolve_attribute_value_from_expected_property_element(
            Rc::clone(&element),
            &["img", "area"],
            "alt",
        ) {
            vec![PropertyValue::Plain(v)]
        } else {
            vec![PropertyValue::Plain(
                extract_only_text(Rc::clone(&element.node), self.base_url.clone())
                    .map(|v| v.trim().to_string())?,
            )]
        };

        Ok(value)
    }

    fn parse_html_property(
        &self,
        element: Rc<Element>,
    ) -> Result<Vec<PropertyValue>, crate::Error> {
        let html =
            element::extract_html_from_children(Rc::clone(&element.node), self.base_url.clone())?
                .trim()
                .to_string();
        let value = extract_text(Rc::clone(&element.node), self.base_url.clone())?
            .trim()
            .to_string();
        // FIXME: Resolve this from either <html lang=> or elsewhere.
        let lang = self
            .document
            .lang
            .clone()
            .or_else(|| element.attribute("lang"));

        Ok(vec![PropertyValue::Fragment(Fragment {
            html,
            value,
            lang,
        })])
    }

    fn parse_linked_property(
        &self,
        element: Rc<Element>,
    ) -> Result<Vec<PropertyValue>, crate::Error> {
        let value = if let Some(v) = resolve_maybe_attribute_value_from_expected_element(
            Rc::clone(&element),
            &["a", "area", "link"],
            "href",
        )
        .or_else(|| {
            resolve_maybe_attribute_value_from_expected_element(
                Rc::clone(&element),
                &["audio", "video", "source", "iframe"],
                "src",
            )
        })
        .or_else(|| {
            resolve_maybe_attribute_value_from_expected_element(
                Rc::clone(&element),
                &["video"],
                "poster",
            )
        })
        .or_else(|| {
            resolve_maybe_attribute_value_from_expected_element(
                Rc::clone(&element),
                &["object"],
                "data",
            )
        }) {
            Some(v)
        } else if let Some(value) = value_class::parse(
            Rc::clone(&element.node),
            self.base_url.clone(),
            value_class::TypeHint::Plain,
        )
        .filter(|s| !s.is_empty())
        .and_then(|value| {
            if let PropertyValue::Plain(s) = value {
                Some(s)
            } else {
                None
            }
        }) {
            Some(value)
        } else if let Some(v) = resolve_maybe_attribute_value_from_expected_element(
            Rc::clone(&element),
            &["data", "input"],
            "value",
        )
        .or_else(|| {
            resolve_maybe_attribute_value_from_expected_element(
                Rc::clone(&element),
                &["abbr"],
                "title",
            )
        }) {
            Some(v)
        } else if let Some(img) = Some(Rc::clone(&element))
            .filter(|element| &element.name == "img")
            .and_then(|elem| self.parse_into_image(elem, "src"))
        {
            return Ok(vec![img]);
        } else {
            Some(extract_text(
                Rc::clone(&element.node),
                self.base_url.clone(),
            )?)
        };

        if let Some(url) = value.and_then(|v: String| self.resolve_url(&v).ok()) {
            Ok(vec![PropertyValue::Url(url)])
        } else {
            Ok(Vec::default())
        }
    }

    fn resolve_implied_photo(
        &self,
        element: Rc<Element>,
        mf2_item: &Item,
    ) -> Option<PropertyValue> {
        let existing_photo_property = mf2_item.get_property("photo").filter(|p| !p.is_empty());

        if existing_photo_property.is_some() {
            trace!("This item already has the 'photo' property as {:#?}, no need to resolve an implied one.", existing_photo_property);
            return None;
        }

        if mf2_item.properties().values().flatten().any(|value| {
            matches!(
                value,
                PropertyValue::Url(_) | PropertyValue::Fragment { .. }
            )
        }) {
            return None;
        }

        // if img.h-x[src], then use the result of "parse an img element for src and alt" (see Sec.1.5) for photo
        if &element.name == "img" && element.has_attribute("src") {
            self.parse_into_image(Rc::clone(&element), "src")
        }
        // if object.h-x[data] then use data for photo
        else if &element.name == "object" && element.has_attribute("data") {
            self.parse_into_image(Rc::clone(&element), "data")
        }
        // if .h-x>img[src]:only-of-type:not[.h-*] then use the result of "parse an img element for src and alt" (see Sec.1.5) for photo
        // if .h-x>object[data]:only-of-type:not[.h-*] then use that object’s data for photo
        // if .h-x>:only-child:not[.h-*]>img[src]:only-of-type:not[.h-*], then use the result of "parse an img element for src and alt" (see Sec.1.5) for photo
        // if .h-x>:only-child:not[.h-*]>object[data]:only-of-type:not[.h-*], then use that object’s data for photo
        else if let Some((element, attribute)) =
            resolve_solo_of_type_of_solo_child(Rc::clone(&element), "img")
                .or_else(|| resolve_solo_of_type_of_solo_child(Rc::clone(&element), "object"))
                .or_else(|| resolve_solo_non_item_element(Rc::clone(&element), "img"))
                .or_else(|| resolve_solo_non_item_element(Rc::clone(&element), "object"))
                .or_else(|| resolve_only_child(Rc::clone(&element), &["img", "object"]))
                .and_then(|element| {
                    if &element.name == "img" && element.has_attribute("src") {
                        Some((element, "src"))
                    } else if &element.name == "object" && element.has_attribute("data") {
                        Some((element, "data"))
                    } else {
                        None
                    }
                })
        {
            self.parse_into_image(element, attribute)
        } else {
            None
        }
    }

    fn parse_into_image(
        &self,
        element: Rc<Element>,
        attribute_name: &str,
    ) -> Option<PropertyValue> {
        element
            .attribute(attribute_name)
            .and_then(|u| self.resolve_url(&u).ok())
            .map(|src| {
                if let Some(alt) = element.attribute("alt") {
                    PropertyValue::Image(Image { src, alt })
                } else {
                    PropertyValue::Url(src)
                }
            })
    }

    fn parse_temporal_property(&self, element: Rc<Element>) -> Vec<PropertyValue> {
        if let Some(timestamp) = value_class::parse(
            Rc::clone(&element.node),
            self.base_url.clone(),
            value_class::TypeHint::Temporal,
        ) {
            return vec![timestamp];
        }

        let maybe_text = if ["time", "ins", "del"].contains(&element.name.as_str())
            && element.has_attribute("datetime")
        {
            element.attribute("datetime")
        } else if ["abbr", "input"].contains(&element.name.as_str())
            && element.has_attribute("title")
        {
            element.attribute("title")
        } else if ["data", "input"].contains(&element.name.as_str())
            && element.has_attribute("value")
        {
            element.attribute("value")
        } else {
            extract_only_text(Rc::clone(&element.node), self.base_url.clone())
                .ok()
                .filter(|v| !v.is_empty())
                .map(|v| v.trim().to_string())
        };

        if let Some(text) = maybe_text {
            trace!("Resolved temporal information: {:?}.", text);
            match value_class::parse_temporal_value(&text) {
                Ok(tmp) => vec![PropertyValue::Temporal(tmp)],
                Err(e) => {
                    error!("Failed to parse {:?} as temporal info {:#?}", text, e);
                    vec![]
                }
            }
        } else {
            trace!("No temporal information could be resolved.");
            vec![]
        }
    }
    fn resolve_implied_name(
        &self,
        element: Rc<Element>,
        item: &Item,
    ) -> Result<Option<String>, crate::Error> {
        let properties = item.properties();

        trace!("Resolving the implied name of {item:?} for {element:?}");

        let has_existing_name = properties.contains_key("name");
        let has_other_plain_or_html_properties = properties.values().flatten().any(|value| {
            matches!(
                value,
                PropertyValue::Plain(_)
                    | PropertyValue::Fragment { .. }
                    | PropertyValue::Temporal(_)
            )
        });

        if has_existing_name {
            trace!(
            "The property 'name' is already defined on this item as {:#?}; not attempting to imply 'name'.",
            properties.get("name")
        );
            return Ok(None);
        } else if has_other_plain_or_html_properties {
            trace!(
            "The item {:#?} with {:#?} has existing textual properties associated; not attempting to imply 'name'.",
             item, element
        );
            return Ok(None);
        } else {
            trace!(
                "Attempting to resolve the implied name property from {:#?}",
                element
            );
        }

        if let Some(value) = resolve_attribute_value_from_expected_property_element(
            Rc::clone(&element),
            &["img", "area"],
            "alt",
        ) {
            // if img.h-x or area.h-x, then use its alt attribute for name
            Ok(Some(value))
        } else if let Some(value) = resolve_attribute_value_from_expected_property_element(
            Rc::clone(&element),
            &["abbr"],
            "title",
        ) {
            // if abbr.h-x[title] then use its title attribute for name
            Ok(Some(value))
        } else if let Some(value) = resolve_solo_non_item_element(Rc::clone(&element), "img")
            .or_else(|| resolve_solo_non_item_element(Rc::clone(&element), "area"))
            .and_then(|element| {
                resolve_attribute_value_from_expected_element(element, &["img", "area"], "alt")
            })
            .or_else(|| {
                resolve_solo_non_item_element(Rc::clone(&element), "abbr").and_then(|elem| {
                    resolve_attribute_value_from_expected_element(elem, &["abbr"], "title")
                })
            })
            .filter(|v| !v.is_empty())
        {
            // else if .h-x>img:only-child[alt]:not([alt=""]):not[.h-*] then use that img’s alt for name
            // else if .h-x>area:only-child[alt]:not([alt=""]):not[.h-*] then use that area’s alt for name
            // else if .h-x>abbr:only-child[title]:not([title=""]):not[.h-*] then use that abbr title for name
            Ok(Some(value))
        } else if let Some(value) =
            resolve_solo_child_element_of_solo_child(Rc::clone(&element), "img")
                .or_else(|| resolve_solo_child_element_of_solo_child(Rc::clone(&element), "area"))
                .or_else(|| resolve_solo_child_element_of_solo_child(Rc::clone(&element), "abbr"))
                .and_then(|element| {
                    if ["img", "area"].contains(&element.name.as_str())
                        && element.has_attribute("alt")
                    {
                        element.attribute("alt")
                    } else if ["abbr"].contains(&element.name.as_str())
                        && element.has_attribute("title")
                    {
                        element.attribute("title")
                    } else {
                        None
                    }
                })
                .filter(|v| !v.is_empty())
        {
            // .h-x>:only-child:not[.h-*]>img:only-child[alt]:not([alt=""]):not[.h-*] then use that img’s alt for name
            // if .h-x>:only-child:not[.h-*]>area:only-child[alt]:not([alt=""]):not[.h-*] then use that area’s alt for name
            // if .h-x>:only-child:not[.h-*]>abbr:only-child[title]:not([title=""]):not[.h-*] use that abbr’s title for name
            Ok(Some(value))
        } else {
            // else use the textContent of the .h-x for name after:
            // dropping any nested <script> & <style> elements;
            // replacing any nested <img> elements with their alt attribute, if present;
            let value = Some(extract_only_text(
                Rc::clone(&element.node),
                self.base_url.clone(),
            )?)
            .map(|v| v.trim().to_string())
            .filter(|v| !v.is_empty());
            Ok(value)
        }
    }
}

fn resolve_only_child(element: Rc<Element>, tag_names: &[&str]) -> Option<Rc<Element>> {
    tag_names.iter().find_map(|tag_name| {
        let elems = element
            .deref()
            .node
            .children
            .borrow()
            .iter()
            .filter_map(|node| Element::try_from(Rc::clone(node)).ok().map(Rc::new))
            .filter(|element| {
                !["script", "style", "template"].contains(&element.name.as_str())
                    && &element.name == tag_name
            })
            .collect::<Vec<_>>();

        elems.first().filter(|_| elems.len() == 1).map(Rc::clone)
    })
}

fn resolve_solo_child_element_of_solo_child(
    element: Rc<Element>,
    expected_tag_name: &str,
) -> Option<Rc<Element>> {
    resolve_single_child_element(Rc::clone(&element)).and_then(|child| {
        resolve_single_child_element(Rc::clone(&child))
            .filter(|element| element.name == expected_tag_name)
    })
}

fn resolve_solo_of_type_of_solo_child(
    element: Rc<Element>,
    expected_tag_name: &str,
) -> Option<Rc<Element>> {
    resolve_single_child_element(Rc::clone(&element)).and_then(|child| {
        child
            .node
            .children
            .borrow()
            .deref()
            .iter()
            .filter_map(|n| Element::try_from(Rc::clone(n)).ok().map(Rc::new))
            .filter(|element| element.name == expected_tag_name)
            .collect::<Vec<_>>()
            .first()
            .cloned()
    })
}

fn resolve_single_child_element(element: Rc<Element>) -> Option<Rc<Element>> {
    let items = element
        .deref()
        .node
        .children
        .borrow()
        .iter()
        .filter_map(|node| Element::try_from(Rc::clone(node)).ok().map(Rc::new))
        .filter(|element| !["script", "style", "template"].contains(&element.name.as_str()))
        .collect::<Vec<_>>();

    if items.len() == 1 {
        items.first().cloned()
    } else {
        None
    }
}

fn resolve_solo_non_item_element(
    element: Rc<Element>,
    expected_tag_name: &str,
) -> Option<Rc<Element>> {
    resolve_single_child_element(Rc::clone(&element))
        .filter(|element| element.name == expected_tag_name)
        .filter(|element| !element.is_microformat_item())
}

fn resolve_implied_url(element: Rc<Element>, item: &Item) -> Option<String> {
    trace!("Parsing the implied value of 'url' for this element.");

    if let Some(u) = item.get_property("url").filter(|u| !u.is_empty()) {
        trace!("Not implying a URL since 'url' is set to {:#?} already.", u);
        return None;
    }

    if item.properties().values().flatten().any(|value| {
        matches!(
            value,
            PropertyValue::Url(_) | PropertyValue::Item(_) | PropertyValue::Fragment { .. }
        )
    }) {
        trace!("This item contains a URL-based property, fragment or a nested item; bailng out.");
        return None;
    }
    if let Some(value) = resolve_maybe_attribute_value_from_expected_element(
        Rc::clone(&element),
        &["a", "area"],
        "href",
    )
    .filter(|_| element.is_microformat_item())
    {
        Some(value)
    } else {
        resolve_solo_non_item_element(Rc::clone(&element), "a")
            .or_else(|| resolve_solo_non_item_element(Rc::clone(&element), "area"))
            .or_else(|| resolve_solo_child_element_of_solo_child(Rc::clone(&element), "a"))
            .or_else(|| resolve_solo_child_element_of_solo_child(Rc::clone(&element), "area"))
            .or_else(|| resolve_only_child(Rc::clone(&element), &["a", "area"]))
            .and_then(|elem| elem.maybe_attribute("href"))
    }
}

fn expand_attribute_value(attribute_name: &str, attribute_value: &str, base_url: &Url) -> String {
    if attribute_name == "src" || attribute_name == "href" {
        if attribute_value.is_empty() {
            base_url.to_string()
        } else {
            base_url
                .join(attribute_value.clone())
                .map(|u| u.to_string())
                .unwrap_or(attribute_value.to_string())
        }
    } else {
        attribute_value.to_string()
    }
}

fn resolve_maybe_attribute_value_from_expected_element(
    element: Rc<Element>,
    tag_names: &[&str],
    attribute_name: &str,
) -> Option<String> {
    if !tag_names.contains(&element.name.as_str()) {
        return None;
    }

    element.maybe_attribute(attribute_name)
}

fn resolve_attribute_value_from_expected_element(
    element: Rc<Element>,
    tag_names: &[&str],
    attribute_name: &str,
) -> Option<String> {
    if !tag_names.contains(&element.name.as_str()) {
        return None;
    }

    element.attribute(attribute_name)
}

fn resolve_attribute_value_from_expected_property_element(
    element: Rc<Element>,
    tag_names: &[&str],
    attribute_name: &str,
) -> Option<String> {
    if !tag_names.contains(&element.name.as_str()) {
        return None;
    }

    if !element.is_property_element() {
        return None;
    }

    element.attribute(attribute_name)
}

#[test]
fn parser_parser_node() -> Result<(), serde_json::Error> {
    crate::test::enable_logging();
    let parser_html = r#"
    <html>
    <body>
        <div class="h-feed" id="main">
            <p class="p-summary">This feels pretty simple.</p>
            <div class="e-content">It's showtime, for <strong>real.</strong></div>
            <p>Last published <time class="dt-published" datetime="2022-02-01T17:38:31-05:00">last week</time></p>
            Written by <div class="u-author h-card"><a href="/u-author" class="u-url">Jacky</a></div> and <span class="p-author">PAuthor</span>
            <div class="h-cite"><a class="u-url" href="/post/1">Post</a></div>
            <div class="h-cite"><a class="u-url" href="/post/2">Post</a></div>
            <div class="h-cite"><a class="u-url" href="/post/3">Post</a></div>
            <div class="h-cite"><a class="u-url" href="/post/4">Post</a></div>
            <div class="h-cite"><a class="u-url" href="/post/5">Post</a></div>
            <div class="u-source h-cite"><a class="u-url" href="/post/6">Source Post</a></div>
        </div>
    </body>
    </html>
    "#;

    let parser_result = Parser::new(parser_html, "https://indieweb.org".parse().unwrap());
    assert_eq!(None, parser_result.as_ref().err());
    let mut parser = parser_result.unwrap();

    let parser_parsed_result = parser.parse();
    assert_eq!(None, parser_parsed_result.as_ref().err());
    let result = parser_parsed_result.unwrap();

    similar_asserts::assert_eq!(
        serde_json::from_str::<Document>(
            r#"{
            "items": [
                {
                    "children": [
                        {
                            "type": ["h-cite"],
                            "properties": {
                                "url": ["https://indieweb.org/post/1"],
                                "name": ["Post"]
                            }
                        },
                        {
                            "type": ["h-cite"],
                            "properties": {
                                "url": ["https://indieweb.org/post/2"],
                                "name": ["Post"]
                            }
                        },
                        {
                            "type": ["h-cite"],
                            "properties": {
                                "url": ["https://indieweb.org/post/3"],
                                "name": ["Post"]
                            }
                        },
                        {
                            "type": ["h-cite"],
                            "properties": {
                                "url": ["https://indieweb.org/post/4"],
                                "name": ["Post"]
                            }
                        },
                        {
                            "type": ["h-cite"],
                            "properties": {
                                "url": ["https://indieweb.org/post/5"],
                                "name": ["Post"]
                            }
                        }
                    ],
                    "properties": {
                        "author": [
                            {
                                "type": ["h-card"],
                                "properties": {
                                    "url": ["https://indieweb.org/u-author"],
                                    "name": ["Jacky"]
                                },
                                "value": "https://indieweb.org/u-author"
                            },
                            "PAuthor"
                        ],
                        "content": [
                            {
                                "html": "It's showtime, for <strong>real.</strong>",
                                "value": "It's showtime, for real."
                            }
                        ],
                        "summary": ["This feels pretty simple."],
                        "published": ["2022-02-01T17:38:31-0500"],
                        "source": [
                            {
                                "type": ["h-cite"],
                                "properties": {
                                    "name": ["Source Post"],
                                    "url": ["https://indieweb.org/post/6"]
                                },
                                "value": "https://indieweb.org/post/6"
                            }
                        ]
                    },
                    "id": "main",
                    "type": ["h-feed"]
                }
            ],
            "rels": {},
            "rel-urls": {}
        }"#
        )?,
        result
    );

    Ok(())
}

#[derive(Debug, PartialEq, Eq, PartialOrd, Ord, Clone)]
pub(crate) enum PropertyClass {
    /// Represents `h-`
    Root(String),
    /// Represents `p-`
    Plain(String),
    /// Represents `u-`
    Linked(String),
    /// Represents `dt-`
    Timestamp(String),
    /// Represents `e-`
    Hypertext(String),
}

impl PropertyClass {
    /// Extract property classes associated to the definition of a `crate::types::Class`.
    pub fn extract_root_classes(properties: Vec<Self>) -> Vec<Class> {
        properties
            .into_iter()
            .filter_map(|property| {
                if let Self::Root(class) = property {
                    Class::from_str(format!("h-{}", class).as_str()).ok()
                } else {
                    None
                }
            })
            .collect()
    }

    /// Determines if this class is a 'h-'
    pub fn is_root(&self) -> bool {
        matches!(self, Self::Root(_))
    }

    /// Convert string into a list of PropertyClass.
    ///
    /// Converts a string representing a list of class values into a list of `PropertyClass` items
    /// to their matching property names.
    pub fn list_from_string(property_class_string: String) -> Vec<Self> {
        let mut classes = property_class_string
            .split_ascii_whitespace()
            .filter_map(|class_name| {
                RE_CLASS_NAME.captures(class_name).and_then(|cp| {
                    let prefix = cp.name("prefix").map(|s| s.as_str());
                    let name = cp.name("name").map(|s| s.as_str());

                    prefix.and_then(|prefix| name.map(|p| (prefix, p)))
                })
            })
            .map(|(prefix, name)| Self::from_prefix_and_name(prefix, name))
            .collect::<Vec<Self>>();

        classes.sort();
        classes.dedup();
        classes
    }

    pub fn from_prefix_and_name(prefix: &str, name: &str) -> Self {
        match prefix {
            "u" => Self::Linked(name.to_string()),
            "dt" => Self::Timestamp(name.to_string()),
            "e" => Self::Hypertext(name.to_string()),
            "h" => Self::Root(name.to_string()),
            _ => Self::Plain(name.to_string()),
        }
    }
}

pub mod value_class;

#[test]
fn property_class_list_from_string() {
    crate::test::enable_logging();
    assert_eq!(
        PropertyClass::list_from_string("p-name".to_owned()),
        vec![PropertyClass::Plain("name".to_owned())]
    );

    assert_eq!(
        PropertyClass::list_from_string("p-name zoom x-foo e-content".to_owned()),
        vec![
            PropertyClass::Plain("name".to_owned()),
            PropertyClass::Hypertext("content".to_owned())
        ]
    );
}
