% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PAVranking.R
\name{PAVranking}
\alias{PAVranking}
\title{Parcel-Allocation Variability in Model Ranking}
\usage{
PAVranking(model0, model1, data, parcel.names, item.syntax, nAlloc = 100,
  fun = "sem", alpha = 0.05, bic.crit = 10,
  fit.measures = c("chisq", "df", "cfi", "tli", "rmsea", "srmr", "logl",
  "aic", "bic", "bic2"), ..., show.progress = FALSE, iseed = 12345,
  warn = FALSE)
}
\arguments{
\item{model0, model1}{\code{\link[lavaan]{lavaan}} model syntax specifying
nested models (\code{model0} within \code{model1}) to be fitted
to the same parceled data.  Note that there can be a mixture of
items and parcels (even within the same factor), in case certain items
should never be parceled. Can be a character string or parameter table.
Also see \code{\link[lavaan]{lavaanify}} for more details.}

\item{data}{A \code{data.frame} containing all observed variables appearing
in the \code{model}, as well as those in the \code{item.syntax} used to
create parcels. If the data have missing values, multiple imputation
before parceling is recommended: submit a stacked data set (with a variable
for the imputation number, so they can be separateed later) and set
\code{do.fit = FALSE} to return the list of \code{data.frame}s (one per
allocation), each of which is a stacked, imputed data set with parcels.}

\item{parcel.names}{\code{character} vector containing names of all parcels
appearing as indicators in \code{model}.}

\item{item.syntax}{\link[lavaan]{lavaan} model syntax specifying the model
that would be fit to all of the unparceled items, including items that
should be randomly allocated to parcels appearing in \code{model}.}

\item{nAlloc}{The number of random items-to-parcels allocations to generate.}

\item{fun}{\code{character} string indicating the name of the
\code{\link[lavaan]{lavaan}} function used to fit \code{model} to
\code{data}. Can only take the values \code{"lavaan"}, \code{"sem"},
\code{"cfa"}, or \code{"growth"}.}

\item{alpha}{Alpha level used as criterion for significance.}

\item{bic.crit}{Criterion for assessing evidence in favor of one model
over another.  See Rafferty (1995) for guidelines (default is "very
strong evidence" in favor of the model with lower BIC).}

\item{fit.measures}{\code{character} vector containing names of fit measures
to request from each fitted \code{\link[lavaan]{lavaan}} model.  See the
output of \code{\link[lavaan]{fitMeasures}} for a list of available measures.}

\item{\dots}{Additional arguments to be passed to
\code{\link[lavaan]{lavaanList}}. See also \code{\link[lavaan]{lavOptions}}}

\item{show.progress}{If \code{TRUE}, show a \code{\link[utils]{txtProgressBar}}
indicating how fast each model-fitting iterates over allocations.}

\item{iseed}{(Optional) Random seed used for parceling items. When the same
random seed is specified and the program is re-run, the same allocations
will be generated. The seed argument can be used to assess parcel-allocation
variability in model ranking when considering more than two models. For each
pair of models under comparison, the program should be rerun using the same
random seed. Doing so ensures that multiple model comparisons will employ
the same set of parcel datasets. \emph{Note}: When using \pkg{parallel}
options, you must first type \code{RNGkind("L'Ecuyer-CMRG")} into the R
Console, so that the seed will be controlled across cores.}

\item{warn}{Whether to print warnings when fitting models to each allocation}
}
\value{
\item{model0.results}{Results returned by \code{\link{parcelAllocation}}
  for \code{model0} (see the \bold{Value} section).}
\item{model1.results}{Results returned by \code{\link{parcelAllocation}}
  for \code{model1} (see the \bold{Value} section).}
\item{model0.v.model1}{A \code{list} of model-comparison results, including
  the following: \itemize{
  \item{\code{LRT_Summary:}}{ The average likelihood ratio test across
    allocations, as well as the \emph{SD}, minimum, maximum, range, and the
    proportion of allocations for which the test was significant.}
  \item{\code{Fit_Index_Differences:}}{ Differences in fit indices, organized
    by what proportion favored each model and among those, what the average
    difference was.}
  \item{\code{Favored_by_BIC:}}{ The proportion of allocations in which each
    model met the criterion (\code{bic.crit}) for a substantial difference
    in fit.}
  \item{\code{Convergence_Summary:}}{ The proportion of allocations in which
    each model (and both models) converged on a solution.}
  } Histograms are also printed to the current plot-output device.}
}
\description{
This function quantifies and assesses the consequences of parcel-allocation
variability for model ranking of structural equation models (SEMs) that
differ in their structural specification but share the same parcel-level
measurement specification (see Sterba & Rights, 2016). This function calls
\code{\link{parcelAllocation}}---which can be used with only one SEM in
isolation---to fit two (assumed) nested models to each of a specified number
of random item-to-parcel allocations.  Output includes summary information
about the distribution of model selection results (including plots) and the
distribution of results for each model individually, across allocations
within-sample. Note that this function can be used when selecting among more
than two competing structural models as well (see instructions below
involving the \code{seed} argument).
}
\details{
This is based on a SAS macro \code{ParcelAlloc} (Sterba & MacCallum, 2010).
The \code{PAVranking} function produces results discussed in Sterba and
Rights (2016) relevant to the assessment of parcel-allocation variability in
model selection and model ranking. Specifically, the \code{PAVranking}
function first calls \code{\link{parcelAllocation}} to generate a given
number (\code{nAlloc}) of item-to-parcel allocations, fitting both specified
models to each allocation, and providing summaryies of PAV for each model.
Additionally, \code{PAVranking} provides the following new summaries:

\itemize{
  \item{PAV in model selection index values and model ranking between
     Models \code{model0} and \code{model1}.}
  \item{The proportion of allocations that converged and the proportion of
    proper solutions (results are summarized for allocations with both
    converged and proper  allocations only).}
}

For further details on the benefits of the random allocation of items to
parcels, see Sterba (2011) and Sterba and MacCallum (2010).

To test whether nested models have equivalent fit, results can be pooled
across allocations using the same methods available for pooling results
across multiple imputations of missing data (see \bold{Examples}).

\emph{Note}: This function requires the \code{lavaan} package. Missing data
 must be coded as \code{NA}. If the function returns \code{"Error in
 plot.new() : figure margins too large"}, the user may need to increase
 size of the plot window (e.g., in RStudio) and rerun the function.
}
\examples{

## Specify the item-level model (if NO parcels were created)
## This must apply to BOTH competing models

item.syntax <- c(paste0("f1 =~ f1item", 1:9),
                 paste0("f2 =~ f2item", 1:9))
cat(item.syntax, sep = "\\n")
## Below, we reduce the size of this same model by
## applying different parceling schemes

## Specify a 2-factor CFA with correlated factors, using 3-indicator parcels
mod1 <- '
f1 =~ par1 + par2 + par3
f2 =~ par4 + par5 + par6
'
## Specify a more restricted model with orthogonal factors
mod0 <- '
f1 =~ par1 + par2 + par3
f2 =~ par4 + par5 + par6
f1 ~~ 0*f2
'
## names of parcels (must apply to BOTH models)
(parcel.names <- paste0("par", 1:6))

\dontrun{
## override default random-number generator to use parallel options
RNGkind("L'Ecuyer-CMRG")

PAVranking(model0 = mod0, model1 = mod1, data = simParcel, nAlloc = 100,
           parcel.names = parcel.names, item.syntax = item.syntax,
           std.lv = TRUE,       # any addition lavaan arguments
           parallel = "snow")   # parallel options



## POOL RESULTS by treating parcel allocations as multiple imputations.
## Details provided in Sterba & Rights (2016); see ?poolMAlloc.

## save list of data sets instead of fitting model yet
dataList <- parcelAllocation(mod.parcels, data = simParcel, nAlloc = 100,
                             parcel.names = parcel.names,
                             item.syntax = item.syntax,
                             do.fit = FALSE)
## now fit each model to each data set
fit0 <- cfa.mi(mod0, data = dataList, std.lv = TRUE)
fit1 <- cfa.mi(mod1, data = dataList, std.lv = TRUE)
anova(fit0, fit1)   # pooled test statistic comparing models
class?lavaan.mi     # find more methods for pooling results
}

}
\references{
Raftery, A. E. (1995). Bayesian model selection in social
research. \emph{Sociological Methodology, 25}, 111--163. doi:10.2307/271063

Sterba, S. K. (2011). Implications of parcel-allocation variability for
comparing fit of item-solutions and parcel-solutions. \emph{Structural
Equation Modeling: A Multidisciplinary Journal, 18}(4), 554--577.
doi:10.1080/10705511.2011.607073

Sterba, S. K., & MacCallum, R. C. (2010). Variability in parameter estimates
and model fit across repeated allocations of items to parcels.
\emph{Multivariate Behavioral Research, 45}(2), 322--358.
doi:10.1080/00273171003680302

Sterba, S. K., & Rights, J. D. (2016). Accounting for parcel-allocation
variability in practice: Combining sources of uncertainty and choosing the
number of allocations. \emph{Multivariate Behavioral Research, 51}(2--3),
296--313. doi:10.1080/00273171.2016.1144502

Sterba, S. K., & Rights, J. D. (2017). Effects of parceling on model
selection: Parcel-allocation variability in model ranking.
\emph{Psychological Methods, 22}(1), 47--68. doi:10.1037/met0000067
}
\seealso{
\code{\link{parcelAllocation}} for fitting a single model,
  \code{\link{poolMAlloc}} for choosing the number of allocations
}
\author{
Terrence D. Jorgensen (University of Amsterdam; \email{TJorgensen314@gmail.com})
}
