// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/extensions/api/passwords_private/passwords_private_api.h"

#include <memory>

#include "base/functional/bind.h"
#include "base/functional/callback_helpers.h"
#include "base/location.h"
#include "base/metrics/histogram_macros.h"
#include "base/strings/stringprintf.h"
#include "base/strings/utf_string_conversions.h"
#include "base/task/single_thread_task_runner.h"
#include "base/values.h"
#include "chrome/browser/extensions/api/passwords_private/passwords_private_delegate_factory.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/common/extensions/api/passwords_private.h"
#include "components/password_manager/core/browser/manage_passwords_referrer.h"
#include "components/password_manager/core/browser/password_manager_util.h"
#include "components/sync/driver/sync_service.h"
#include "content/public/browser/web_contents.h"
#include "extensions/browser/extension_function_registry.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace extensions {

namespace {

using ResponseAction = ExtensionFunction::ResponseAction;

constexpr char kNoDelegateError[] =
    "Operation failed because PasswordsPrivateDelegate wasn't created.";

scoped_refptr<PasswordsPrivateDelegate> GetDelegate(
    content::BrowserContext* browser_context) {
  return PasswordsPrivateDelegateFactory::GetForBrowserContext(
      browser_context,
      /*create=*/false);
}

}  // namespace

// PasswordsPrivateRecordPasswordsPageAccessInSettingsFunction
ResponseAction
PasswordsPrivateRecordPasswordsPageAccessInSettingsFunction::Run() {
  UMA_HISTOGRAM_ENUMERATION(
      "PasswordManager.ManagePasswordsReferrer",
      password_manager::ManagePasswordsReferrer::kChromeSettings);
  return RespondNow(NoArguments());
}

// PasswordsPrivateChangeSavedPasswordFunction
ResponseAction PasswordsPrivateChangeSavedPasswordFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::ChangeSavedPassword::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  auto new_id = GetDelegate(browser_context())
                    ->ChangeSavedPassword(parameters->id, parameters->params);
  if (new_id.has_value()) {
    return RespondNow(ArgumentList(
        api::passwords_private::ChangeSavedPassword::Results::Create(
            new_id.value())));
  }
  return RespondNow(Error(
      "Could not change the password. Either the password is empty, the user "
      "is not authenticated or no matching password could be found for the "
      "id."));
}

// PasswordsPrivateRemoveSavedPasswordFunction
ResponseAction PasswordsPrivateRemoveSavedPasswordFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::RemoveSavedPassword::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);
  GetDelegate(browser_context())
      ->RemoveSavedPassword(parameters->id, parameters->from_stores);
  return RespondNow(NoArguments());
}

// PasswordsPrivateRemovePasswordExceptionFunction
ResponseAction PasswordsPrivateRemovePasswordExceptionFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::RemovePasswordException::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);
  GetDelegate(browser_context())->RemovePasswordException(parameters->id);
  return RespondNow(NoArguments());
}

// PasswordsPrivateUndoRemoveSavedPasswordOrExceptionFunction
ResponseAction
PasswordsPrivateUndoRemoveSavedPasswordOrExceptionFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())->UndoRemoveSavedPasswordOrException();
  return RespondNow(NoArguments());
}

// PasswordsPrivateRequestPlaintextPasswordFunction
ResponseAction PasswordsPrivateRequestPlaintextPasswordFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::RequestPlaintextPassword::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  GetDelegate(browser_context())
      ->RequestPlaintextPassword(
          parameters->id, parameters->reason,
          base::BindOnce(
              &PasswordsPrivateRequestPlaintextPasswordFunction::GotPassword,
              this),
          GetSenderWebContents());

  // GotPassword() might respond before we reach this point.
  return did_respond() ? AlreadyResponded() : RespondLater();
}

void PasswordsPrivateRequestPlaintextPasswordFunction::GotPassword(
    absl::optional<std::u16string> password) {
  if (password) {
    Respond(OneArgument(base::Value(std::move(*password))));
    return;
  }

  Respond(Error(base::StringPrintf(
      "Could not obtain plaintext password. Either the user is not "
      "authenticated or no password with id = %d could be found.",
      api::passwords_private::RequestPlaintextPassword::Params::Create(args())
          ->id)));
}

// PasswordsPrivateRequestCredentialDetailsFunction
ResponseAction PasswordsPrivateRequestCredentialsDetailsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::RequestCredentialsDetails::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  GetDelegate(browser_context())
      ->RequestCredentialsDetails(
          parameters->ids,
          base::BindOnce(
              &PasswordsPrivateRequestCredentialsDetailsFunction::GotPasswords,
              this),
          GetSenderWebContents());

  // GotPasswords() might have responded before we reach this point.
  return did_respond() ? AlreadyResponded() : RespondLater();
}

void PasswordsPrivateRequestCredentialsDetailsFunction::GotPasswords(
    const PasswordsPrivateDelegate::UiEntries& entries) {
  if (!entries.empty()) {
    Respond(ArgumentList(
        api::passwords_private::RequestCredentialsDetails::Results::Create(
            entries)));
    return;
  }

  Respond(Error(
      "Could not obtain password entry. Either the user is not "
      "authenticated or no credential with matching ids could be found."));
}

// PasswordsPrivateGetSavedPasswordListFunction
ResponseAction PasswordsPrivateGetSavedPasswordListFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  // GetList() can immediately call GotList() (which would Respond() before
  // RespondLater()). So we post a task to preserve order.
  base::SingleThreadTaskRunner::GetCurrentDefault()->PostTask(
      FROM_HERE,
      base::BindOnce(&PasswordsPrivateGetSavedPasswordListFunction::GetList,
                     this));
  return RespondLater();
}

void PasswordsPrivateGetSavedPasswordListFunction::GetList() {
  GetDelegate(browser_context())
      ->GetSavedPasswordsList(base::BindOnce(
          &PasswordsPrivateGetSavedPasswordListFunction::GotList, this));
}

void PasswordsPrivateGetSavedPasswordListFunction::GotList(
    const PasswordsPrivateDelegate::UiEntries& list) {
  Respond(ArgumentList(
      api::passwords_private::GetSavedPasswordList::Results::Create(list)));
}

// PasswordsPrivateGetCredentialGroupsFunction
ResponseAction PasswordsPrivateGetCredentialGroupsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(
      ArgumentList(api::passwords_private::GetCredentialGroups::Results::Create(
          GetDelegate(browser_context())->GetCredentialGroups())));
}

// PasswordsPrivateGetPasswordExceptionListFunction
ResponseAction PasswordsPrivateGetPasswordExceptionListFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  // GetList() can immediately call GotList() (which would Respond() before
  // RespondLater()). So we post a task to preserve order.
  base::SingleThreadTaskRunner::GetCurrentDefault()->PostTask(
      FROM_HERE,
      base::BindOnce(&PasswordsPrivateGetPasswordExceptionListFunction::GetList,
                     this));
  return RespondLater();
}

void PasswordsPrivateGetPasswordExceptionListFunction::GetList() {
  GetDelegate(browser_context())
      ->GetPasswordExceptionsList(base::BindOnce(
          &PasswordsPrivateGetPasswordExceptionListFunction::GotList, this));
}

void PasswordsPrivateGetPasswordExceptionListFunction::GotList(
    const PasswordsPrivateDelegate::ExceptionEntries& entries) {
  Respond(ArgumentList(
      api::passwords_private::GetPasswordExceptionList::Results::Create(
          entries)));
}

// PasswordsPrivateMovePasswordToAccountFunction
ResponseAction PasswordsPrivateMovePasswordsToAccountFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::MovePasswordsToAccount::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);
  GetDelegate(browser_context())
      ->MovePasswordsToAccount(parameters->ids, GetSenderWebContents());
  return RespondNow(NoArguments());
}

// PasswordsPrivateImportPasswordsFunction
ResponseAction PasswordsPrivateImportPasswordsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::ImportPasswords::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);
  GetDelegate(browser_context())
      ->ImportPasswords(
          parameters->to_store,
          base::BindOnce(
              &PasswordsPrivateImportPasswordsFunction::ImportRequestCompleted,
              this),
          GetSenderWebContents());

  // `ImportRequestCompleted()` might respond before we reach this point.
  return did_respond() ? AlreadyResponded() : RespondLater();
}

void PasswordsPrivateImportPasswordsFunction::ImportRequestCompleted(
    const api::passwords_private::ImportResults& result) {
  Respond(ArgumentList(
      api::passwords_private::ImportPasswords::Results::Create(result)));
}

// PasswordsPrivateExportPasswordsFunction
ResponseAction PasswordsPrivateExportPasswordsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())
      ->ExportPasswords(
          base::BindOnce(
              &PasswordsPrivateExportPasswordsFunction::ExportRequestCompleted,
              this),
          GetSenderWebContents());
  return RespondLater();
}

void PasswordsPrivateExportPasswordsFunction::ExportRequestCompleted(
    const std::string& error) {
  if (error.empty())
    Respond(NoArguments());
  else
    Respond(Error(error));
}

// PasswordsPrivateCancelExportPasswordsFunction
ResponseAction PasswordsPrivateCancelExportPasswordsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())->CancelExportPasswords();
  return RespondNow(NoArguments());
}

// PasswordsPrivateRequestExportProgressStatusFunction
ResponseAction PasswordsPrivateRequestExportProgressStatusFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(ArgumentList(
      api::passwords_private::RequestExportProgressStatus::Results::Create(
          GetDelegate(browser_context())->GetExportProgressStatus())));
}

// PasswordsPrivateIsOptedInForAccountStorageFunction
ResponseAction PasswordsPrivateIsOptedInForAccountStorageFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(OneArgument(base::Value(
      GetDelegate(browser_context())->IsOptedInForAccountStorage())));
}

// PasswordsPrivateOptInForAccountStorageFunction
ResponseAction PasswordsPrivateOptInForAccountStorageFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::OptInForAccountStorage::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters.get());

  GetDelegate(browser_context())
      ->SetAccountStorageOptIn(parameters->opt_in, GetSenderWebContents());
  return RespondNow(NoArguments());
}

// PasswordsPrivateGetInsecureCredentialsFunction:
PasswordsPrivateGetInsecureCredentialsFunction::
    ~PasswordsPrivateGetInsecureCredentialsFunction() = default;

ResponseAction PasswordsPrivateGetInsecureCredentialsFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(ArgumentList(
      api::passwords_private::GetInsecureCredentials::Results::Create(
          GetDelegate(browser_context())->GetInsecureCredentials())));
}

// PasswordsPrivateGetCredentialsWithReusedPasswordFunction:
PasswordsPrivateGetCredentialsWithReusedPasswordFunction::
    ~PasswordsPrivateGetCredentialsWithReusedPasswordFunction() = default;

ResponseAction PasswordsPrivateGetCredentialsWithReusedPasswordFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(ArgumentList(
      api::passwords_private::GetCredentialsWithReusedPassword::Results::Create(
          GetDelegate(browser_context())->GetCredentialsWithReusedPassword())));
}

// PasswordsPrivateMuteInsecureCredentialFunction:
PasswordsPrivateMuteInsecureCredentialFunction::
    ~PasswordsPrivateMuteInsecureCredentialFunction() = default;

ResponseAction PasswordsPrivateMuteInsecureCredentialFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::MuteInsecureCredential::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  if (!GetDelegate(browser_context())
           ->MuteInsecureCredential(parameters->credential)) {
    return RespondNow(
        Error("Could not mute the insecure credential. Probably no matching "
              "password could be found."));
  }

  return RespondNow(NoArguments());
}

// PasswordsPrivateUnmuteInsecureCredentialFunction:
PasswordsPrivateUnmuteInsecureCredentialFunction::
    ~PasswordsPrivateUnmuteInsecureCredentialFunction() = default;

ResponseAction PasswordsPrivateUnmuteInsecureCredentialFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::UnmuteInsecureCredential::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  if (!GetDelegate(browser_context())
           ->UnmuteInsecureCredential(parameters->credential)) {
    return RespondNow(
        Error("Could not unmute the insecure credential. Probably no matching "
              "password could be found."));
  }

  return RespondNow(NoArguments());
}

// PasswordsPrivateRecordChangePasswordFlowStartedFunction:
PasswordsPrivateRecordChangePasswordFlowStartedFunction::
    ~PasswordsPrivateRecordChangePasswordFlowStartedFunction() = default;

ResponseAction PasswordsPrivateRecordChangePasswordFlowStartedFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::RecordChangePasswordFlowStarted::Params::Create(
          args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  GetDelegate(browser_context())
      ->RecordChangePasswordFlowStarted(parameters->credential);
  return RespondNow(NoArguments());
}

// PasswordsPrivateStartPasswordCheckFunction:
PasswordsPrivateStartPasswordCheckFunction::
    ~PasswordsPrivateStartPasswordCheckFunction() = default;

ResponseAction PasswordsPrivateStartPasswordCheckFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())
      ->StartPasswordCheck(base::BindOnce(
          &PasswordsPrivateStartPasswordCheckFunction::OnStarted, this));

  // OnStarted() might respond before we reach this point.
  return did_respond() ? AlreadyResponded() : RespondLater();
}

void PasswordsPrivateStartPasswordCheckFunction::OnStarted(
    password_manager::BulkLeakCheckService::State state) {
  const bool is_running =
      state == password_manager::BulkLeakCheckService::State::kRunning;
  Respond(is_running ? NoArguments()
                     : Error("Starting password check failed."));
}

// PasswordsPrivateStopPasswordCheckFunction:
PasswordsPrivateStopPasswordCheckFunction::
    ~PasswordsPrivateStopPasswordCheckFunction() = default;

ResponseAction PasswordsPrivateStopPasswordCheckFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())->StopPasswordCheck();
  return RespondNow(NoArguments());
}

// PasswordsPrivateGetPasswordCheckStatusFunction:
PasswordsPrivateGetPasswordCheckStatusFunction::
    ~PasswordsPrivateGetPasswordCheckStatusFunction() = default;

ResponseAction PasswordsPrivateGetPasswordCheckStatusFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(ArgumentList(
      api::passwords_private::GetPasswordCheckStatus::Results::Create(
          GetDelegate(browser_context())->GetPasswordCheckStatus())));
}

// PasswordsPrivateIsAccountStoreDefaultFunction
ResponseAction PasswordsPrivateIsAccountStoreDefaultFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  return RespondNow(OneArgument(
      base::Value(GetDelegate(browser_context())
                      ->IsAccountStoreDefault(GetSenderWebContents()))));
}

// PasswordsPrivateGetUrlCollectionFunction:
ResponseAction PasswordsPrivateGetUrlCollectionFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::GetUrlCollection::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  const absl::optional<api::passwords_private::UrlCollection> url_collection =
      GetDelegate(browser_context())->GetUrlCollection(parameters->url);
  if (!url_collection) {
    return RespondNow(
        Error("Provided string doesn't meet password URL requirements. Either "
              "the format is invalid or the scheme is not unsupported."));
  }

  return RespondNow(
      ArgumentList(api::passwords_private::GetUrlCollection::Results::Create(
          url_collection.value())));
}

// PasswordsPrivateAddPasswordFunction
ResponseAction PasswordsPrivateAddPasswordFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters = api::passwords_private::AddPassword::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  if (!GetDelegate(browser_context())
           ->AddPassword(parameters->options.url,
                         base::UTF8ToUTF16(parameters->options.username),
                         base::UTF8ToUTF16(parameters->options.password),
                         base::UTF8ToUTF16(parameters->options.note),
                         parameters->options.use_account_store,
                         GetSenderWebContents())) {
    return RespondNow(Error(
        "Could not add the password. Either the url is invalid, the password "
        "is empty or an entry with such origin and username already exists."));
  }

  return RespondNow(NoArguments());
}

// PasswordsPrivateExtendAuthValidityFunction
ResponseAction PasswordsPrivateExtendAuthValidityFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())->ExtendAuthValidity();
  return RespondNow(NoArguments());
}

// PasswordsPrivateSwitchBiometricAuthBeforeFillingStateFunction
ResponseAction
PasswordsPrivateSwitchBiometricAuthBeforeFillingStateFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())
      ->SwitchBiometricAuthBeforeFillingState(GetSenderWebContents());
  return RespondNow(NoArguments());
}

// PasswordsPrivateShowExportedFileInShellFunction
ResponseAction PasswordsPrivateShowExportedFileInShellFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  auto parameters =
      api::passwords_private::ShowExportedFileInShell::Params::Create(args());
  EXTENSION_FUNCTION_VALIDATE(parameters);

  GetDelegate(browser_context())
      ->ShowExportedFileInShell(GetSenderWebContents(), parameters->file_path);
  return RespondNow(NoArguments());
}

// PasswordsPrivateShowAddShortcutDialogFunction
ResponseAction PasswordsPrivateShowAddShortcutDialogFunction::Run() {
  if (!GetDelegate(browser_context())) {
    return RespondNow(Error(kNoDelegateError));
  }

  GetDelegate(browser_context())->ShowAddShortcutDialog(GetSenderWebContents());
  return RespondNow(NoArguments());
}

}  // namespace extensions
