// Copyright 2022 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/signin/primary_account_policy_manager.h"

#include "base/strings/string_util.h"
#include "base/strings/utf_string_conversions.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/profiles/profiles_state.h"
#include "chrome/browser/signin/identity_manager_factory.h"
#include "chrome/browser/signin/signin_util.h"
#include "chrome/grit/generated_resources.h"
#include "components/prefs/pref_service.h"
#include "components/signin/public/base/signin_metrics.h"
#include "components/signin/public/base/signin_pref_names.h"
#include "components/signin/public/identity_manager/account_info.h"
#include "components/signin/public/identity_manager/identity_manager.h"
#include "components/signin/public/identity_manager/identity_utils.h"
#include "components/signin/public/identity_manager/primary_account_mutator.h"
#include "google_apis/gaia/gaia_auth_util.h"
#include "ui/base/l10n/l10n_util.h"

#if BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/browser/ui/browser_finder.h"
#include "chrome/browser/ui/browser_list.h"
#include "chrome/browser/ui/browser_list_observer.h"
#include "chrome/browser/ui/browser_window.h"
#include "chrome/browser/ui/simple_message_box.h"
#include "chrome/browser/ui/startup/startup_types.h"
#include "chrome/browser/ui/webui/profile_helper.h"
#endif

#if BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
// Manager that presents the profile will be deleted dialog on the first active
// browser window.
class PrimaryAccountPolicyManager::DeleteProfileDialogManager
    : public BrowserListObserver {
 public:
  DeleteProfileDialogManager(std::string primary_account_email,
                             PrimaryAccountPolicyManager* delegate)
      : primary_account_email_(primary_account_email), delegate_(delegate) {}
  ~DeleteProfileDialogManager() override { BrowserList::RemoveObserver(this); }

  DeleteProfileDialogManager(const DeleteProfileDialogManager&) = delete;
  DeleteProfileDialogManager& operator=(const DeleteProfileDialogManager&) =
      delete;

  void PresentDialogOnAllBrowserWindows(
      Profile* profile,
      bool auto_confirm_profile_deletion_for_testing) {
    DCHECK(profile);
    DCHECK(profile_path_.empty());
    profile_path_ = profile->GetPath();

    if (auto_confirm_profile_deletion_for_testing) {
      base::ThreadTaskRunnerHandle::Get()->PostTask(
          FROM_HERE,
          base::BindOnce(&DeleteProfileDialogManager::
                             HandleUserConfirmedProfileDeletionAndDie,
                         weak_factory_.GetWeakPtr()));

      return;
    }

    BrowserList::AddObserver(this);
    Browser* active_browser = chrome::FindLastActiveWithProfile(profile);
    if (active_browser)
      OnBrowserSetLastActive(active_browser);
  }

  void OnBrowserSetLastActive(Browser* browser) override {
    DCHECK(!profile_path_.empty());

    if (profile_path_ != browser->profile()->GetPath())
      return;

    active_browser_ = browser;

    // Display the dialog on the next run loop as otherwise the dialog can block
    // browser from displaying because the dialog creates a nested run loop.
    //
    // This happens because the browser window is not fully created yet when
    // OnBrowserSetLastActive() is called. To finish the creation, the code
    // needs to return from OnBrowserSetLastActive().
    //
    // However, if we open a warning dialog from OnBrowserSetLastActive()
    // synchronously, it will create a nested run loop that will not return
    // from OnBrowserSetLastActive() until the dialog is dismissed. But the user
    // cannot dismiss the dialog because the browser is not even shown!
    base::ThreadTaskRunnerHandle::Get()->PostTask(
        FROM_HERE,
        base::BindOnce(&DeleteProfileDialogManager::ShowDeleteProfileDialog,
                       weak_factory_.GetWeakPtr(), browser));
  }

  // Called immediately after a browser becomes not active.
  void OnBrowserNoLongerActive(Browser* browser) override {
    if (active_browser_ == browser)
      active_browser_ = nullptr;
  }

  void OnBrowserRemoved(Browser* browser) override {
    if (active_browser_ == browser)
      active_browser_ = nullptr;
  }

 private:
  void ShowDeleteProfileDialog(Browser* browser) {
    // Block opening dialog from nested task.
    static bool is_dialog_shown = false;
    if (is_dialog_shown)
      return;
    base::AutoReset<bool> auto_reset(&is_dialog_shown, true);

    // Check that |browser| is still active.
    if (!active_browser_ || active_browser_ != browser)
      return;

    // Show the dialog.
    DCHECK(browser->window()->GetNativeWindow());
    chrome::MessageBoxResult result = chrome::ShowWarningMessageBox(
        browser->window()->GetNativeWindow(),
        l10n_util::GetStringUTF16(IDS_PROFILE_WILL_BE_DELETED_DIALOG_TITLE),
        l10n_util::GetStringFUTF16(
            IDS_PROFILE_WILL_BE_DELETED_DIALOG_DESCRIPTION,
            base::ASCIIToUTF16(primary_account_email_),
            base::ASCIIToUTF16(
                gaia::ExtractDomainName(primary_account_email_))));

    switch (result) {
      case chrome::MessageBoxResult::MESSAGE_BOX_RESULT_NO: {
        // If the warning dialog is automatically dismissed or the user closed
        // the dialog by clicking on the close "X" button, then re-present the
        // dialog (the user should not be able to interact with the browser
        // window as the profile must be deleted).
        base::ThreadTaskRunnerHandle::Get()->PostTask(
            FROM_HERE,
            base::BindOnce(&DeleteProfileDialogManager::ShowDeleteProfileDialog,
                           weak_factory_.GetWeakPtr(), browser));
        break;
      }
      case chrome::MessageBoxResult::MESSAGE_BOX_RESULT_YES:
        HandleUserConfirmedProfileDeletionAndDie();
        break;
      case chrome::MessageBoxResult::MESSAGE_BOX_RESULT_DEFERRED:
        NOTREACHED() << "Message box must not return deferred result when run "
                        "synchronously";
        break;
    }
  }

  void HandleUserConfirmedProfileDeletionAndDie() {
    delegate_->OnUserConfirmedProfileDeletion(this, profile_path_);
    // |this| may be destroyed at this point. Avoid using it.
  }

  std::string primary_account_email_;
  raw_ptr<PrimaryAccountPolicyManager> delegate_;
  base::FilePath profile_path_;
  raw_ptr<Browser> active_browser_;
  base::WeakPtrFactory<DeleteProfileDialogManager> weak_factory_{this};
};
#endif  // BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)

PrimaryAccountPolicyManager::PrimaryAccountPolicyManager(Profile* profile)
    : profile_(profile) {
  DCHECK(profile_);
  DCHECK(!profile_->IsOffTheRecord());
}

PrimaryAccountPolicyManager::~PrimaryAccountPolicyManager() = default;

void PrimaryAccountPolicyManager::Initialize() {
  EnsurePrimaryAccountAllowedForProfile(
      profile_, signin_metrics::SIGNIN_NOT_ALLOWED_ON_PROFILE_INIT);

  signin_allowed_.Init(
      prefs::kSigninAllowed, profile_->GetPrefs(),
      base::BindRepeating(
          &PrimaryAccountPolicyManager::OnSigninAllowedPrefChanged,
          weak_pointer_factory_.GetWeakPtr()));

  local_state_pref_registrar_.Init(g_browser_process->local_state());
  local_state_pref_registrar_.Add(
      prefs::kGoogleServicesUsernamePattern,
      base::BindRepeating(
          &PrimaryAccountPolicyManager::OnGoogleServicesUsernamePatternChanged,
          weak_pointer_factory_.GetWeakPtr()));
}

void PrimaryAccountPolicyManager::Shutdown() {
  local_state_pref_registrar_.RemoveAll();
  signin_allowed_.Destroy();
}

void PrimaryAccountPolicyManager::OnGoogleServicesUsernamePatternChanged() {
  EnsurePrimaryAccountAllowedForProfile(
      profile_, signin_metrics::GOOGLE_SERVICE_NAME_PATTERN_CHANGED);
}

void PrimaryAccountPolicyManager::OnSigninAllowedPrefChanged() {
  EnsurePrimaryAccountAllowedForProfile(profile_,
                                        signin_metrics::SIGNOUT_PREF_CHANGED);
}

void PrimaryAccountPolicyManager::EnsurePrimaryAccountAllowedForProfile(
    Profile* profile,
    signin_metrics::ProfileSignout clear_primary_account_source) {
// All primary accounts are allowed on ChromeOS, so this method is a no-op on
// ChromeOS.
#if !BUILDFLAG(IS_CHROMEOS_ASH)
  auto* identity_manager = IdentityManagerFactory::GetForProfile(profile);
  if (!identity_manager->HasPrimaryAccount(signin::ConsentLevel::kSync))
    return;

  CoreAccountInfo primary_account =
      identity_manager->GetPrimaryAccountInfo(signin::ConsentLevel::kSync);
  if (profile->GetPrefs()->GetBoolean(prefs::kSigninAllowed) &&
      signin::IsUsernameAllowedByPatternFromPrefs(
          g_browser_process->local_state(), primary_account.email)) {
    return;
  }

#if BUILDFLAG(IS_CHROMEOS_LACROS)
  // Disabling signin in chrome and 'RestrictSigninToPattern' policy
  // are not supported on Lacros. This code should be unreachable. The main
  // profile should never be deleted.
  DCHECK(false)
      << "Disabling signin in chrome and 'RestrictSigninToPattern' policy "
         "are not supported on Lacros.";
#else
  switch (signin_util::UserSignoutSetting::GetForProfile(profile)
              ->signout_allowed()) {
    case signin::Tribool::kUnknown:
      NOTREACHED();
      break;
    case signin::Tribool::kTrue: {
      // Force clear the primary account if it is no longer allowed and if sign
      // out is allowed.
      auto* primary_account_mutator =
          identity_manager->GetPrimaryAccountMutator();
      primary_account_mutator->ClearPrimaryAccount(
          clear_primary_account_source,
          signin_metrics::SignoutDelete::kIgnoreMetric);
      break;
    }
    case signin::Tribool::kFalse:
#if BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
      // Force remove the profile if sign out is not allowed and if the
      // primary account is no longer allowed.
      // This may be called while the profile is initializing, so it must be
      // scheduled for later to allow the profile initialization to complete.
      CHECK(profiles::IsMultipleProfilesEnabled());
      base::ThreadTaskRunnerHandle::Get()->PostTask(
          FROM_HERE,
          base::BindOnce(&PrimaryAccountPolicyManager::ShowDeleteProfileDialog,
                         weak_pointer_factory_.GetWeakPtr(), profile,
                         primary_account.email));
#elif BUILDFLAG(IS_ANDROID)
      // The CHECK below was disabled on Android as test
      // HistoryActivityTest#testSupervisedUser signs out a supervised account.
      // We believe this state is not expected on Android as supervised users
      // are not allowed to sign out.
      // See https://crbug.com/1285271#c7 for more info.
      //
      // TODO(crbug/1312416): Understand if this test covers a valid usecase
      // and see how this should be handled on Android.
      LOG(WARNING) << "Unexpected state: User is signed in, signin is not "
                      "allowed, sign out is not allowed. Do nothing.";
#else
      CHECK(false) << "Deleting profiles is not supported.";
#endif  // BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
      break;
  }
#endif  // BUILDFLAG(IS_CHROMEOS_LACROS)
#endif  // !BUILDFLAG(IS_CHROMEOS_ASH)
}

#if BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
// Shows the delete profile dialog on the first browser active window.
void PrimaryAccountPolicyManager::ShowDeleteProfileDialog(
    Profile* profile,
    const std::string& email) {
  if (delete_profile_dialog_manager_)
    return;

  delete_profile_dialog_manager_ =
      std::make_unique<DeleteProfileDialogManager>(email, this);
  delete_profile_dialog_manager_->PresentDialogOnAllBrowserWindows(
      profile, hide_ui_for_testing_);
}

void PrimaryAccountPolicyManager::OnUserConfirmedProfileDeletion(
    DeleteProfileDialogManager* dialog_manager,
    base::FilePath profile_path) {
  DCHECK_EQ(delete_profile_dialog_manager_.get(), dialog_manager);
  delete_profile_dialog_manager_.reset();

  DCHECK(profiles::IsMultipleProfilesEnabled());

  g_browser_process->profile_manager()->MaybeScheduleProfileForDeletion(
      profile_path,
      hide_ui_for_testing_ ? base::DoNothing()
                           : base::BindOnce(&webui::OpenNewWindowForProfile),
      ProfileMetrics::DELETE_PROFILE_PRIMARY_ACCOUNT_NOT_ALLOWED);
}
#endif  // BUILDFLAG(IS_WIN) || BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_MAC)
