macro(odb_add_schema)

    find_package(PythonInterp REQUIRED)

    set(options NOINSTALL)
    set(single_value_args TARGET TYPE)
    set(multi_value_args SOURCES VIEWS DEFINITIONS CONDITION)

    cmake_parse_arguments(_PAR "${options}" "${single_value_args}" "${multi_value_args}" ${ARGN})

    if(NOT _PAR_TARGET)
        message(FATAL_ERROR "odb_add_schema: TARGET argument not specified")
    endif()

    if(NOT _PAR_SOURCES)
        message(FATAL_ERROR "odb_add_schema: SOURCES argument not specified")
    endif()

    if(_PAR_NOINSTALL)
        set(_PAR_UNPARSED_ARGUMENTS "NOINSTALL" ${_PAR_UNPARSED_ARGUMENTS})
    endif()

    if(_PAR_TYPE)
        set(_PAR_UNPARSED_ARGUMENTS TYPE ${_PAR_TYPE} ${_PAR_UNPARSED_ARGUMENTS})
    endif()

    set(TARGET ${_PAR_TARGET})
    set(SOURCES ${_PAR_SOURCES})

    set(VIEWS)
    foreach(view_sql ${_PAR_VIEWS})
        if(NOT IS_ABSOLUTE ${view_sql})
            set(view_sql ${CMAKE_CURRENT_SOURCE_DIR}/${view_sql})
        endif()
        list(APPEND VIEWS ${view_sql})
    endforeach()

    # Declare DDL and SQL files as project files

    ecbuild_declare_project_files(${_PAR_SOURCES} ${_PAR_VIEWS})

    # Allow conditional build

    if(DEFINED _PAR_CONDITION)
        set(_target_condition_file "${CMAKE_CURRENT_BINARY_DIR}/set_${_PAR_TARGET}_condition.cmake")
        file(WRITE ${_target_condition_file} "  if( ")
        foreach(term ${_PAR_CONDITION})
            file(APPEND ${_target_condition_file} " ${term}")
        endforeach()
        file(APPEND ${_target_condition_file} " )\n    set(_${_PAR_TARGET}_condition TRUE)\n  else()\n    set(_${_PAR_TARGET}_condition FALSE)\n  endif()\n")
        include(${_target_condition_file})
    else()
        set(_${_PAR_TARGET}_condition TRUE)
    endif()

    # Do not continue pass this point if condition is false

    if(NOT _${_PAR_TARGET}_condition)
        return()
    endif()

    # Create a list of ODB-SQL compiler flags

    set(_flags "-I.")

    foreach(dir ${ODB_INCLUDE_DIRS})
        list(APPEND _flags "-I${dir}")
    endforeach()

    foreach(def ${_PAR_DEFINITIONS})
        list(APPEND _flags "-D${def}")
    endforeach()

    # Find target's DDL file (FIXME: assuming it's the first file in the list of sources)

    list(GET _PAR_SOURCES 0 target_ddl)

    # Parse DDL file and generate a list of all table names in the schema

    if(IS_ABSOLUTE ${target_ddl})
        if(CMAKE_VERSION VERSION_LESS 3.0)
          get_filename_component(work_dir ${target_ddl} PATH)
        else()
          get_filename_component(work_dir ${target_ddl} DIRECTORY)
        endif()
    else()
        set(work_dir ${CMAKE_CURRENT_SOURCE_DIR})
        set(target_ddl ${CMAKE_CURRENT_SOURCE_DIR}/${target_ddl})
    endif()

    execute_process(COMMAND ${PYTHON_EXECUTABLE} ${ODB_CMAKE_DIR}/odb_generate_table_names.py ${_flags} ${target_ddl}
        WORKING_DIRECTORY ${work_dir}
        OUTPUT_VARIABLE table_names)

    # Create a list of all table source files that will be generated by ODB-SQL compiler

    set(table_sources)
    foreach(name ${table_names})
        list(APPEND table_sources ${CMAKE_CURRENT_BINARY_DIR}/${TARGET}_T_${name}.c)
    endforeach()

    # Use ODB/SQL compiler to generate the actual table sources

    add_custom_command(OUTPUT ${TARGET}.ddl_ ${TARGET}.h ${TARGET}.c ${table_sources}
        COMMAND odb98.x -C ${_flags} -o ${CMAKE_CURRENT_BINARY_DIR} -l ${TARGET} -c ${target_ddl}
        DEPENDS ${SOURCES} odb98.x)

    # Generate a static stub file for this schema

    add_custom_command(OUTPUT ${TARGET}_Sstatic.c
        COMMAND ${CMAKE_COMMAND} -D TARGET=${TARGET} -D VIEWS="${VIEWS}"
            -P ${ODB_CMAKE_DIR}/odb_create_static_stub.cmake
        DEPENDS ${TARGET}.h ${VIEWS})

    # Use ODB/SQL compiler to generate the actual view sources

    set(view_sources)

    foreach(view_sql ${VIEWS})

        get_filename_component(view_name ${view_sql} NAME_WE)

        add_custom_command(OUTPUT ${TARGET}_${view_name}.c
            COMMAND odb98.x -C ${_flags} -c -l ${TARGET} -w -Q ${view_sql}
            DEPENDS ${view_sql} ${TARGET}.h odb98.x)

        list(APPEND view_sources ${TARGET}_${view_name}.c)

    endforeach()

    # Add the actual schema library

    ecbuild_add_library(TARGET ${TARGET}
        DEFINITIONS ${_PAR_DEFINITIONS}
        SOURCES ${TARGET}.c ${TARGET}.h ${TARGET}_Sstatic.c ${table_sources} ${view_sources}
        PRIVATE_INCLUDES ${CMAKE_CURRENT_BINARY_DIR} ${ODB_INCLUDE_DIRS}
        LIBS ${ODB_LIBRARIES}
        CFLAGS -O0
        ${_PAR_UNPARSED_ARGUMENTS})

endmacro()
