/* parser.y
   
   Bison grammar for a parser of PDU definitions. The job of this parser
   is to transform input like:
  
   "ip(dst = cisco.com)/icmp-echo(id = 1)/data(data = '1234')"

   into an internal data representation.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

%{
#include <stdarg.h>
#include <ctype.h>
#include <glib.h>
#include <math.h>

#include "pbuild-priv.h"

struct parser_vars {
    GNode *prev_pdu;
    GPtrArray *fields;
    GPtrArray *numspecs;
};

%}

%pure-parser

%error-verbose

%parse-param {GNode **pdu}
%parse-param {char *errbuf}
%parse-param {struct parser_vars *vars}
%parse-param {void *scanner}
%lex-param   {void *scanner}

%start pdu_list

%union {
    int i;
    char *string;
    ip_addr_t ipaddr;
    GNode *node;
    double d;
    pdu_t *pdu;
}

%token NO

%token <i> INTEGER
%token <string> STRING
%token <string> NAME
%token <ipaddr> IPADDR
%token <d> DOUBLE
%token END 0 "end of PDU definition"

%type <node> parm parm_list pdu
%type <pdu> pdu_name
%type <string> braced_parms

%left '/' /* Token used to join PDUs in the same chain */
%left ',' /* Token used to separate PDU option fields */

%{

/* Lex interface */
extern void *pb_scan_string(const char *, void *);
extern int pb_delete_buffer(void *, void *);
extern int pblex_init(void **);
extern int pblex_destroy(void *);
extern int pblex(YYSTYPE *, void *);

static void pberror(GNode **, char *, struct parser_vars *, void *,
		    char const *, ...);

/* Helper functions used from within the parser */

static field_t *get_field(field_t *, const char *);
static GNode *new_parm_node(struct parser_vars *, char *, enum yytokentype,
			    void *, char *);

/* Local variables */

/* Fields allowed in any PDU */
static field_t global_fields[] = {
    {.name = "only-first", .type = PDU_FTYPE_UINT32},
    {.name = NULL}
};

%}

%%

/*

What This Grammar Does
----------------------

For the following input:

pdu1(parm1, braced-args1(parm2, parm3), parm4) /
pdu2(parm5, parm6, braced-args2(parm7, parm8) )

this grammar creates a parse tree like this:

                     o pdu1
                     | 
      +-----+--------+------------+
     /     /          \            \
    /     /            \            \
   /     /              \            \
  o     o braced-args1   o            \ pdu2
parm1  / \             parm4   +-------o----+
      /   \                   /       /      \
     o     o                 o       o        o braced-args2
   parm2  parm3            parm5   parm6     / \
                                            /   \
                                           o     o
                                         parm7  parm8
Notes:

- A PDU definition is converted into a parse tree via the grammar.

- A PDU is stored as a parent node, and the parameters that define
  a PDU are stored as child nodes.

- Subsequent PDUs (as "pdu2" in "pdu1()/pdu2()" are stored as children
  nodes too. In this case the new PDU's node is the last child of the
  parent PDU, as shown above for "pdu2".

- In the diagram above, an 'o' represnts a node in the tree. As
  can be seen, even leaves are nodes.

- In the sample input "ip(dst = cisco.com)/icmp-echo(id = 1)", "ip"
  and "icmp-echo" are PDUs, and "dst" and "id" are PDU fields, also
  called PDU parameters.

*/

pdu_list	: pdu
    {
	/*
	 * Save the latest PDU (represented by the root node of the PDU)
	 * in a temp. variable since we want the latest PDU to be appended
	 * to the previous PDU (see tree representation above.)
	 */
	vars->prev_pdu = $1;

	*pdu = $1;
    }
		| pdu_list '/' pdu
    {
	/* Add the new PDU to the tree */
	g_node_append(vars->prev_pdu, $3);

	vars->prev_pdu = $3;
    }
;

pdu		: pdu_name '(' parm_list ')'
    {
	struct node_data *node_data;

	/* Create data structure for the new PDU's node */
	node_data = xmalloc(sizeof *node_data);
	memset(node_data, 0, sizeof *node_data);
	node_data->type = PDU_NODE_PDU;
	node_data->name = xstrdup($1->name);
	node_data->_data_pdu.template = $1;
	node_data->_data_pdu.opts_data = NULL;

	$3->data = node_data;

	$$ = $3;

	/* "Pop" the field array of the braced arguments list from a stack */
	g_ptr_array_remove_index(vars->fields, vars->fields->len - 1);
    }
		| STRING /* Assume that a simple string is a data (raw) PDU */
    {
	struct node_data *node_data;
	GNode *parm_list, *pdu_node;
	pdu_t *pdu_template;

	pdu_template = _pb_getpdutemplate("data");

	/*
	 * "Push" the field array of the braced arguments list into a stack.
	 * We need this because new_parm_node() (called below) relies on this.
	 */
	g_ptr_array_add(vars->fields, pdu_template->fields);

	/* Create data structure for the new PDU's node */
	node_data = xmalloc(sizeof *node_data);
	memset(node_data, 0, sizeof *node_data);
	node_data->type = PDU_NODE_PDU;
	node_data->name = xstrdup("data");
	node_data->_data_pdu.template = pdu_template;
	node_data->_data_pdu.opts_data = NULL;

	parm_list = new_parm_node(vars, xstrdup("data"), STRING, $1, errbuf);
	if (parm_list == NULL)
	    YYABORT;

	pdu_node = g_node_new(NULL);
	g_node_append(pdu_node, parm_list);

	pdu_node->data = node_data;

	$$ = pdu_node;

	/* "Pop" the field array of the braced arguments list from a stack */
	g_ptr_array_remove_index(vars->fields, vars->fields->len - 1);
    }
;

pdu_name	: NAME
    {
	/* Check PDU name and get a pointer to the PDU definition template
	 * at the same time. This pointer is the semantic value of the
	 * "pdu_name" non-terminal symbol.
	 */
	if ( ($$ = _pb_getpdutemplate($1) ) == NULL) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "\"%s\" is not a known PDU", $1);
	    YYABORT;
	}

	/* "Push" the field array of the braced arguments list into a stack */
	g_ptr_array_add(vars->fields, $$->fields);

	free($1); /* String representing the PDU name is not needed anymore
		     since the PDU definition template contains the same
		     name. */
    }
;

parm_list	: /* Empty list; allows for a PDU with default options */
    {
	$$ = g_node_new(NULL); /* Create root node for the current PDU
				  or braced parameter list */
    }
		| parm
    {
	$$ = g_node_new(NULL); /* Create root node for the current PDU
				  or braced options */
	/*
	 * Append the parameter (which is a node) to the root node.
	 */
	g_node_append($$, $1);
    }
		| parm_list ',' parm
    {
	g_node_append($1, $3); /* Append new parameter to current PDU */
    }
;

parm		: NAME '=' STRING
    {
	$$ = new_parm_node(vars, $1, STRING, $3, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| NAME '=' INTEGER
    {
	$$ = new_parm_node(vars, $1, INTEGER, &$3, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| NAME '=' DOUBLE
    {
	$$ = new_parm_node(vars, $1, DOUBLE, &$3, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| NAME '=' IPADDR
    {
	$$ = new_parm_node(vars, $1, IPADDR, &$3, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| NAME /* Strictly interpreted as a boolean */
    {
	$$ = new_parm_node(vars, $1, 0, (int []) {-1}, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| NO NAME /* Strictly interpreted as a boolean */
    {
	$$ = new_parm_node(vars, $2, 0, NULL, errbuf);
	if ($$ == NULL)
	    YYABORT;
    }
		| braced_parms '(' parm_list ')'
    {
	struct node_data *node;

	node = xmalloc(sizeof(struct node_data) );
	node->type = PDU_NODE_BRACED_PARMS;
	node->name = $1;
	node->_data_braced.fields = g_ptr_array_index(vars->fields,
						      vars->fields->len - 1);

	$3->data = node;

	$$ = $3;

	/* "Pop" the field array of the braced arguments list from a stack */
	g_ptr_array_remove_index(vars->fields, vars->fields->len - 1);
    }
;

braced_parms	: NAME
    {
	field_t *field, *fields;

	fields = g_ptr_array_index(vars->fields, vars->fields->len - 1);

	/* Check name of this braced arguments list */
	if ( (field = get_field(fields, $1) ) == NULL) {
	    pberror(pdu, errbuf, vars, scanner,
		     "\"%s\" is not a known braced arguments list", $1);
	    YYABORT;
	}

	if (field->type != PDU_FTYPE_BRACED_ARGS) {
	    pberror(pdu, errbuf, vars, scanner,
		     "\"%s\" is not a braced arguments list", $1);
	    YYABORT;
	}

	/* "Push" the field array of the braced arguments list into a stack */
	g_ptr_array_add(vars->fields, field->data);
    }
;

%%

static void
pberror(GNode **pdu _U_, char *errbuf, struct parser_vars *vars _U_,
	 void *scanner _U_, char const *fmt, ...)
{
    va_list ap;

    if (errbuf) {
	va_start(ap, fmt);
	vsnprintf(errbuf, PDU_ERRBUF_SIZE, fmt, ap);
    }
}

/*
 * Retrieves a field_t structure (which describes a field) from
 * the latest field list in an array of field lists.
 */
static field_t *
get_field(field_t *fields, const char *fname)
{
    field_t *f;

    /*
     * It is possible for a PDU's pdu_t to not have a "fields"
     * structure. This could happen if someone forgot to create
     * this field when the pdu_t was defined, but I guess there
     * could be a PDU that does not have any fields (although I
     * don't know of any). To avoid crashing in these cases we
     * need to check that we actually have a valid pointer.
     */
    if (fields == NULL)
	return NULL;

    for (f = fields; f->name; f++)
	if (!strcmp(f->name, fname) )
	    return f;

    return NULL;
}

/*
 * Creates a new node that contains a parameter. Parameter names are
 * validated. If the parameter is not present in the current PDU's
 * list of valid parameters then it is searched in a global list
 * of valid parameters.
 *
 * Remember: if "token_type" is STRING then "value" is a pointer
 * to a string. This string was dynamically allocated by the
 * lexer via strdup(), so it needs to be free()'d as soon as we're
 * done using it.
 */
static GNode *
new_parm_node(struct parser_vars *vars, char *name,
	      enum yytokentype token_type, void *value, char *errbuf)
{
    field_t *field, *fields;
    struct node_data *node;
    void *data;
    char payload_errbuf[PAYLOAD_ERRBUF_SIZE];
    struct addr a;
    struct pdu_dict *dict_entry;
    uint32_t int_part;

    fields = g_ptr_array_index(vars->fields, vars->fields->len - 1);

    if ( (field = get_field(fields, name) ) == NULL) {
	/*
	 * Field not found in the current PDU/braced-arguments list.
	 * Look in the global fields list.
	 */
	if ( (field = get_field(global_fields, name) ) == NULL) {
	    /* Not found in the global fields list either. Abort. */
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "\"%s\" is not a known field for the current PDU", name);
	    return NULL;
	}
    };

    switch (field->type) {
    case PDU_FTYPE_BRACED_ARGS:
	data = NULL;
	break;
    case PDU_FTYPE_NUMTYPE:
    case PDU_FTYPE_BITFIELD:
    case PDU_FTYPE_UINT8:
    case PDU_FTYPE_UINT16:
    case PDU_FTYPE_UINT32:
	switch (token_type) {
	case INTEGER:
	    /*
	     * Small optimization: if the lexer found out this was an integer
	     * then we can just call num_initn() and then avoid an additional
	     * lexical analysis in libnumbers. If it is not an integer then
	     * num_init() will have to perform the additional lexical analysis.
	     */
	    data = num_initn(*(uint32_t *) value);
	    /*
	     * No check for data == NULL here since num_initn()
	     * always succeeds.
	     */
	    break;
	case STRING:
	    if (field->data) {
		/*
		 * There's an enumeration for this field; try to get the actual
		 * value from the dictionary that field->data points to.
		 */
		for (dict_entry = field->data; dict_entry->key; dict_entry++)
		    if (!strcasecmp(value, dict_entry->key) ) {
			data = num_initn(*(uint32_t *) dict_entry->data);
			break;
		    }

		if (dict_entry->key)
		    /* Found an enum */
		    break;
	    }
	   
	    /*
	     * The string (value) was not in the hash table; assume a regular
	     * numspec.
	     */

	    data = num_init(value);
	    if (data == NULL) {
		snprintf(errbuf, PDU_ERRBUF_SIZE,
			 "invalid numeric format \"%s\"", (char *) value);
		free(value);
		return NULL;
	    }

	    if (num_nvalues(data) > 1)
		g_ptr_array_add(vars->numspecs, data);

	    free(value);
	    break;
	default:
	    /* Should not happen */
	    snprintf(errbuf, PDU_ERRBUF_SIZE, "incompatible parameter type");
	    return NULL;
	}
	break;
    case PDU_FTYPE_UINT:
	data = xmalloc(sizeof(unsigned long int) );

	switch (token_type) {
	case INTEGER:
	    *(long int *) data =  *(long int *) value;
	    break;
	case STRING:
	    *(long int *) data = strtol(value, NULL, 0);
	    break;
	default:
	    /* Should not happen */
	    snprintf(errbuf, PDU_ERRBUF_SIZE, "incompatible parameter type");
	    return NULL;
	}
	break;
    case PDU_FTYPE_IP:
	/*
	 * Small optimization -- similar to the one above but for IP
	 * addresses instead of integers.
	 */
	switch (token_type) {
	case IPADDR:
	    data = ip_initn(*(ip_addr_t *) value);
	    /*
	     * No check for data == NULL here since ip_initn()
	     * always succeeds.
	     */
	    break;
	case STRING:
	    data = ip_init(value);
	    if (data == NULL) {
		snprintf(errbuf, PDU_ERRBUF_SIZE,
			 "invalid IP address specification \"%s\"",
			 (char *) value);
		free(value);
		return NULL;
	    }

	    if (num_nvalues(data) > 1)
		g_ptr_array_add(vars->numspecs, data);

	    free(value);
	    break;
	default:
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting an IP address");
	    return NULL;
	}
	break;
    case PDU_FTYPE_BIT:
    case PDU_FTYPE_BOOL:
	data = value;
	break;
    case PDU_FTYPE_MACADDR:
	if (token_type != STRING) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting a string");
	    return NULL;
	}

	data = xmalloc(sizeof(eth_addr_t) );
	if (_pb_format_ethernet_addr(value, data) == -1) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "invalid MAC address \"%s\"", (char *) value);
	    free(value);
	    return NULL;
	}

	free(value);
	break;
    case PDU_FTYPE_DATA:
	if (token_type != STRING) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting a string");
	    return NULL;
	}

	data = xmalloc(sizeof(struct payload) );
	if (create_payload(value, data, payload_errbuf) == -1) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE, "error creating payload: %s",
		     payload_errbuf);
	    free(value);
	    return NULL;
	}

	free(value);
	break;
    case PDU_FTYPE_IP6ADDR:
	if (token_type != STRING) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting a string");
	    return NULL;
	}

	if (addr_pton(value, &a) == -1) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "can't resolve \"%s\"", (char *) value);
	    free(value);
	    return NULL;
	} 

	data = xmalloc(sizeof(ip6_addr_t) );
	*(ip6_addr_t *) data = a.addr_ip6;

	free(value);
	break;
    case PDU_FTYPE_FIXEDP32:
	if (token_type != DOUBLE) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting a double");
	    return NULL;
	}

	data = xmalloc(sizeof(struct fixedp_num) );
	int_part = *(double *) value;
	( (struct fixedp_num *) data)->int_part = int_part;
	( (struct fixedp_num *) data)->frac_part
	    = (fabs(*(double *) value) - abs(int_part) ) * pow(2, 16);
	break;
    case PDU_FTYPE_FIXEDP64:
	if (token_type != DOUBLE) {
	    snprintf(errbuf, PDU_ERRBUF_SIZE,
		     "incompatible parameter type; expecting a double");
	    return NULL;
	}

	data = xmalloc(sizeof(struct fixedp_num) );
	int_part = *(double *) value;
	( (struct fixedp_num *) data)->int_part = int_part;
	( (struct fixedp_num *) data)->frac_part
	    = (fabs(*(double *) value) - abs(int_part) ) * pow(2, 32);
	break;
    }

    node = xmalloc(sizeof *node);

    node->type = PDU_NODE_PARM;
    node->name = name;
    node->_data_parm.field = field;
    node->_data_parm.data = data;

    return g_node_new(node);
}

static size_t
calc_opt_lengths(const GNode *options, const char *opts_class)
{
    GNode *option, *pdu;
    pdu_option_t *o;
    size_t opts_len, len;
    GPtrArray *opts_data;
    struct option_data *option_data;

    opts_data = g_ptr_array_sized_new(10);

    pdu = (GNode *) _pb_parent_pdu(options);
    ( (struct node_data *) pdu->data)->_data_pdu.opts_data = opts_data;

    for (option = g_node_first_child(options), opts_len = 0;
	 option;
	 option = g_node_next_sibling(option) ) {
	/*
	 * Obtain a pointer to the PDU option template based on the
	 * name of the option, i.e. the name of the node.
	 */
	o = _pb_getopttemplate(opts_class,
			       ( (struct node_data *) option->data)->name);

	len = o->len != 0 ? o->len : o->get_len(option);

	option_data = xmalloc(sizeof *option_data);
	option_data->len = len;
	option_data->template = o;
	option_data->option = option;

	g_ptr_array_add(opts_data, option_data);

	opts_len += len;
    }

    return opts_len;
}

/*
 * Calculates the length of the header, options, and payload of a
 * PDU. The lengths are stored in the PDU node's data.
 */
static size_t
calc_lengths(const GNode *pdu)
{
    const pdu_t *template;
    struct node_data *node_data;
    const GNode *next_pdu, *options;

    node_data = pdu->data;

    template = node_data->_data_pdu.template;

    /* Header length */
    node_data->_data_pdu.hdr_len = template->len != 0
				   ? template->len : template->get_len(pdu);

    /* Payload length */
    if ( (next_pdu = pb_nextpdu(pdu) ) )
	node_data->_data_pdu.payload_len = calc_lengths(next_pdu);
    else
	node_data->_data_pdu.payload_len = 0;

    /* Options length */
    if ( (options = _pb_findnode(pdu, "options") ) )
	node_data->_data_pdu.opts_len
	    = calc_opt_lengths(options, template->options_class);
    else
	node_data->_data_pdu.opts_len = 0;

    return   node_data->_data_pdu.hdr_len
	   + node_data->_data_pdu.opts_len
	   + node_data->_data_pdu.payload_len;
}

/*
 * Calculates the change frequencies of each numspec in a PDU tree.
 */
static void
calc_change_frequencies(GPtrArray *numspecs)
{
    guint nspecs;
    int change_freq;

    nspecs = numspecs->len;

    /* numspecs[last element]->change_freq = 1 */
    num_setfreq(g_ptr_array_index(numspecs, nspecs - 1), 1);

    while (nspecs-- > 1) {
	/* change_freq = numspecs[n]->change_freq*numspecs[n]->nvalues */
	change_freq = num_getfreq(g_ptr_array_index(numspecs, nspecs) )
		      *num_nvalues(g_ptr_array_index(numspecs, nspecs) );

	/* numspecs[n - 1]->change_freq = change_freq */
	num_setfreq(g_ptr_array_index(numspecs, nspecs - 1), change_freq);
    }
}

/* Used for debugging only. Don't wrap with #ifdef DEBUG to make sure
 * it always compiles. */
void
_pb_dump_numspecs(GPtrArray *numspecs)
{
    size_t i;

    printf("Number of numspecs: %d\n", numspecs->len);

    for (i = 0; i < numspecs->len; i++) {
	printf("    numspec #%zu: sequence=%d, nvalues=%d, change_freq=%d\n"
	       "	info=%s\n",
	       i,
	       num_seq(g_ptr_array_index(numspecs, i) ),
	       num_nvalues(g_ptr_array_index(numspecs, i) ),
	       num_getfreq(g_ptr_array_index(numspecs, i) ),
	       num_info(g_ptr_array_index(numspecs, i) ) );
    }
}

/*
 * Parses a PDU definition, converting it to a parse tree and populating
 * the data in each node of the tree.
 *
 * Inputs:
 *   - pdudef: PDU definition.
 *   - errbuf: pointer to a buffer where error messages will be left.
 *
 * Output: a parse tree representing the PDU.
 *
 */
const GNode *
pb_parsedef(const char *pdudef, char *errbuf)
{
    void *yybuf;
    void *scanner;
    int retval;
    struct parser_vars vars;
    GNode *pdu;

    pblex_init(&scanner);

    pdu = NULL;

    /* vars.fields is used as a stack */
    vars.fields = g_ptr_array_sized_new(10); /* 10 is arbitrary */

    vars.numspecs = g_ptr_array_sized_new(10); /* 10 is arbitrary */

    yybuf = pb_scan_string(pdudef, scanner);
    retval = pbparse(&pdu, errbuf, &vars, scanner);
    pb_delete_buffer(yybuf, scanner);

    g_ptr_array_free(vars.fields, TRUE);

    if (retval > 0) {
	/*
	 * There was some parsing error (retval is 1) or the parser ran out
	 * of memory (retval is 2); attempt to clean up. Note that not being
	 * able to resolve a host name (if all interfaces are down, for
	 * example), will be handled as a parsing error, i.e. the parser
	 * will return 1.
	 *
	 * XXX - we leak memory when the parser encounters a parsing error.
	 * This is because we don't have a way to get to the root node,
	 * and therefore we can't properly clean things up. We should find
	 * a way to fix this.
	 */

	if (pdu)
	    pb_destroy(pdu);

	g_ptr_array_free(vars.numspecs, TRUE);

	return NULL;
    }

    /*
     * Calculate the lengths of each PDU.
     */
    calc_lengths(pdu);

    if (vars.numspecs->len > 1) {
	/*
	 * Initialize the change frequency of numeric specifications inside
	 * a PDU.
	 */
	calc_change_frequencies(vars.numspecs);
#ifdef DEBUG
	_pb_dump_numspecs(vars.numspecs);
#endif
    }

    g_ptr_array_free(vars.numspecs, TRUE);

    pblex_destroy(scanner);

    return pdu;
}
