/* pdu-ospf.c
 
   PDU builder for OSPF packets

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-ospf.h"

/*
 * Most OSPF builders only need to calculate the OSPF checksum and the length
 * of the OSPF message in the second pass. This generic second pass builder can
 * be used for all these OSPF builders.
 */
static void
pdu_ospf_generic_postbuild(const GNode *pdu, void *dest,
			   void *prev_pdu_hdr _U_)
{
    struct ospf_hdr *ospf;
    numspec_t *parm;
    uint16_t u16;
    size_t hdr_len, opts_len, payload_len;

    ospf = dest;

    hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    if ( (parm = _pb_pdata(pdu, "length") ) )
	u16 = htons(num_next(parm) );
    else
	u16 = htons(hdr_len + opts_len + payload_len);
    SSVAL(ospf, offsetof(struct ospf_hdr, length), u16);

    if ( (parm = _pb_pdata(pdu, "cksum") ) )
	u16 = htons(num_next(parm) );
    else {
	SSVAL(ospf, offsetof(struct ospf_hdr, cksum), 0);
	u16 = _pb_cksum(dest, hdr_len + opts_len + payload_len);
    }
    SSVAL(ospf, offsetof(struct ospf_hdr, cksum), u16);
}

/*********************************************************************/
/************************* Raw OSPF message **************************/
/*********************************************************************/

static void
pdu_ospfrawhdr_builder(const GNode *pdu, void *dest)
{
    struct ospf_hdr *ospf;
    void *field;
    uint16_t u16;
    uint32_t u32;

    ospf = dest;

    ospf->version = (field = _pb_pdata(pdu, "version") )
		    ? num_next(field) : OSPF_VERSION_2;

    ospf->type = num_next(_pb_pdata(pdu, "type") );

    u32 = num_next(_pb_pdata(pdu, "routerid") );
    SIVAL(ospf, offsetof(struct ospf_hdr, routerid), u32);

    u32 = num_next(_pb_pdata(pdu, "areaid") );
    SIVAL(ospf, offsetof(struct ospf_hdr, areaid), u32);

    u16 = num_next(_pb_pdata(pdu, "authtype") );
    SSVAL(ospf, offsetof(struct ospf_hdr, autype), u16);
}

#if 0
static void
pdu_ospfrawhdr_dumper(pdu_t *p, const char *prefix)
{
    struct ospfrawhdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Version: %s\n", prefix, num_info(hdr_data->version) );
    printf("%s    Type: %s\n", prefix, num_info(hdr_data->type) );
    printf("%s    Length: %s\n", prefix,
	   hdr_data->length ? num_info(hdr_data->length) : "automatic");
    printf("%s    Router ID: %s\n", prefix, num_info(hdr_data->routerid) );
    printf("%s    Area ID: %s\n", prefix, num_info(hdr_data->areaid) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
    printf("%s    Auth. Type: %s\n", prefix, num_info(hdr_data->autype) );
}
#endif

/*********************************************************************/
/************************* OSPF hello message ************************/
/*********************************************************************/

static void
pdu_ospfhello_builder(const GNode *pdu, void *dest)
{
    struct ospfhello_hdr *ospf;
    uint16_t u16;
    uint32_t u32;
    ip_addr_t ipaddr, *neighbors;
    GNode *child;
    struct node_data *node_data;
    void *field;

    ospf = dest;

    ospf->version = (field = _pb_pdata(pdu, "version") )
		    ? num_next(field) : OSPF_VERSION_2;

    ospf->type = OSPF_HELLO;

    u32 = num_next(_pb_pdata(pdu, "routerid") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, routerid), u32);

    u32 = num_next(_pb_pdata(pdu, "areaid") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, areaid), u32);

    u16 = num_next(_pb_pdata(pdu, "authtype") );
    SSVAL(ospf, offsetof(struct ospfhello_hdr, autype), u16);

    ipaddr = ip_next(_pb_pdata(pdu, "netmask") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, netmask), ipaddr);

    u16 = num_next(_pb_pdata(pdu, "hellointerval") );
    SSVAL(ospf, offsetof(struct ospfhello_hdr, hellointerval), u16);

    ospf->options = num_next(_pb_pdata(pdu, "options") );

    ospf->rtrpri = num_next(_pb_pdata(pdu, "prio") );

    u32 = num_next(_pb_pdata(pdu, "deadinterval") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, deadinterval), u32);

    ipaddr = ip_next(_pb_pdata(pdu, "designatedrouter") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, designatedrouter), ipaddr);

    ipaddr = ip_next(_pb_pdata(pdu, "backuprouter") );
    SIVAL(ospf, offsetof(struct ospfhello_hdr, backuprouter), ipaddr);

    neighbors = dest + sizeof(struct ospfhello_hdr);
    
    for (neighbors = dest + sizeof(struct ospfhello_hdr),
	 child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child), neighbors++) {
	node_data = child->data;

	/* We only care about "neighbor"; everything else is handled above. */
	if (!strcmp(node_data->name, "neighbor") ) {
	    ipaddr = ip_next(node_data->_data_parm.data);
	    SIVAL(neighbors, 0, ipaddr);
	}
    }
}

static size_t
pdu_ospfhello_len(const GNode *pdu)
{
    GNode *child;
    struct node_data *node_data;
    size_t hdr_len;

    hdr_len = sizeof(struct ospfhello_hdr);
    
    for (child = g_node_first_child(pdu);
	 child;
	 child = g_node_next_sibling(child)) {
	node_data = child->data;

	/* We only care about "neighbor"; everything else is handled above. */
	if (!strcmp(node_data->name, "neighbor") )
	    hdr_len += sizeof(ip_addr_t);
    }

    return hdr_len;
}

#if 0
static void
pdu_ospfhellohdr_dumper(pdu_t *p, const char *prefix)
{
    struct ospfhellohdr_options *hdr_data;
    char addr[INET_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Version: %s\n", prefix, num_info(hdr_data->version) );
    printf("%s    Length: %s\n", prefix,
	   hdr_data->length ? num_info(hdr_data->length) : "automatic");
    printf("%s    Router ID: %s\n", prefix, num_info(hdr_data->routerid) );
    printf("%s    Area ID: %s\n", prefix, num_info(hdr_data->areaid) );
    printf("%s    Checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
    printf("%s    Auth. Type: %s\n", prefix, num_info(hdr_data->autype) );
    printf("%s    Network mask: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->netmask, addr, INET_ADDRSTRLEN) );
    printf("%s    Hello interval: %s\n", prefix,
	   num_info(hdr_data->hellointerval) );
    printf("%s    Options: %s\n", prefix, num_info(hdr_data->options) );
    printf("%s    Router priority: %s\n", prefix,
	   num_info(hdr_data->rtrpri) );
    printf("%s    Router dead interval: %s\n", prefix,
	   num_info(hdr_data->deadinterval) );
    printf("%s    Designated router: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->designatedrouter, addr, INET_ADDRSTRLEN) );
    printf("%s    Backup designated router: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->backuprouter, addr, INET_ADDRSTRLEN) );
    printf("%s    Number of neighbors: %d\n", prefix, hdr_data->nneighbors);
}
#endif

static pdu_t pdu_ospfraw = {
    .name = "ospf",
    .description = "Raw OSPF message",
    .documented_in = "RFC 2328",
    .len = sizeof(struct ospf_hdr),
    .fields = (field_t []) {
	{.name = "version", .type = PDU_FTYPE_NUMTYPE},
	{.name = "routerid", .type = PDU_FTYPE_NUMTYPE},
	{.name = "areaid", .type = PDU_FTYPE_NUMTYPE},
	{.name = "authtype", .type = PDU_FTYPE_NUMTYPE},
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "length", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_ospfrawhdr_builder,
    .postbuild = &pdu_ospf_generic_postbuild
};

static pdu_t pdu_ospfhello = {
    .name = "ospf-hello",
    .description = "OSPF HELLO packet",
    .documented_in = "RFC 2328",
    .fields = (field_t []) {
	{.name = "version", .type = PDU_FTYPE_NUMTYPE},
	{.name = "routerid", .type = PDU_FTYPE_NUMTYPE},
	{.name = "areaid", .type = PDU_FTYPE_NUMTYPE},
	{.name = "authtype", .type = PDU_FTYPE_NUMTYPE},
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "length", .type = PDU_FTYPE_NUMTYPE},
	{.name = "netmask", .type = PDU_FTYPE_IP},
	{.name = "interval", .type = PDU_FTYPE_NUMTYPE},
	{.name = "options", .type = PDU_FTYPE_NUMTYPE},
	{.name = "prio", .type = PDU_FTYPE_NUMTYPE},
	{.name = "dead-interval", .type = PDU_FTYPE_NUMTYPE},
	{.name = "designated-rtr", .type = PDU_FTYPE_IP},
	{.name = "backup-rtr", .type = PDU_FTYPE_IP},
	{.name = "neighbor", .type = PDU_FTYPE_IP},
	{.name = NULL}
    },
    .build = &pdu_ospfhello_builder,
    .postbuild = &pdu_ospf_generic_postbuild,
    .get_len = &pdu_ospfhello_len
};

void
_pb_register_ospf(void)
{
    _pb_register_protocol(&pdu_ospfraw);
    _pb_register_protocol(&pdu_ospfhello);
}
