/* util.c
  
   Utility functions

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

void
dump(const unsigned char *data, unsigned len)
{
    unsigned i, j;

    for (i = 0; i <= len/16; i++) {
	printf("%08x  ", i*16);

	for (j = 0; j < 16; j++) {
	    if (i*16 + j < len)
		printf("%02x", data[i*16 + j]);
	    else
		printf("  ");

	    if (j & 1)
		printf(" ");
	}

	for (j = 0; j < 16; j++)
	    if (i*16 + j < len)
		printf("%c", isprint( (int ) data[i*16 + j])
			     ? data[i*16 + j] : '.');

	printf("\n");
    }

    printf("\n");
}

void
error(const char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "Error: ");
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    if (*fmt) {
	fmt += strlen(fmt);
	if (fmt[-1] != '\n')
	    fputc('\n', stderr);
    }

    exit(EXIT_FAILURE);
}

void
warn(const char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "Warning: ");
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    if (*fmt) {
	fmt += strlen(fmt);
	if (fmt[-1] != '\n')
	    fputc('\n', stderr);
    }
}

void
debug(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    if (*fmt) {
	fmt += strlen(fmt);
	if (fmt[-1] != '\n')
	    fputc('\n', stderr);
    }
}

unsigned short
in_checksum(void *data, int len)
{
    int counter = len;
    unsigned short *w = data;
    int sum = 0;
    unsigned short tmp = 0;

    while (counter > 1) {
	sum += *w++;
	counter -= 2;
    }

    if (counter == 1) {
	/* Length is odd */
	*(unsigned char *) &tmp = *(unsigned char *) w;
	sum += tmp;
    }

    /* We want a 1's complement sum, so we must add whatever was
       carried out from the low 16 bits to the low 16 bits */
    sum = (sum >> 16) + (unsigned short) sum;

    /* Add the carry from the previous sum */
    sum += sum >> 16;

    /* Now we take the 1's complement of the 1's complement sum */
    tmp = ~sum;

    return tmp;
}

/*
 * Used by dump_pdu(). Creates a prefix of whitespace that is used to
 * indent PDU dumps.
 */
char *
make_prefix(int level)
{
    static char prefix[512];
    int nspaces;

    nspaces = level*2;
    if (nspaces > (int) sizeof prefix)
	/* To prevent writing past the end of the prefix[] array should
	 * the integer "level" gets out of control.
	 */
	nspaces = sizeof(prefix) - 1;
    memset(prefix, ' ', nspaces);
    prefix[nspaces] = '\0';
    
    return prefix;
}

/********************************************************************
 *			 Address Manipulation                       *
 ********************************************************************/

/*
 * Takes as input a MAC address and returns its ASCII representation.
 * The returned string lives in a static buffer that is overwritten
 * with each call.
 */
char *
mac_to_ascii(eth_addr_t *mac)
{
    static char mac_ascii[sizeof("xx:xx:xx:xx:xx:xx")];

    snprintf(mac_ascii, sizeof(mac_ascii), "%02x:%02x:%02x:%02x:%02x:%02x",
	     mac->data[0], mac->data[1],
	     mac->data[2], mac->data[3],
	     mac->data[4], mac->data[5]);

    return mac_ascii;
}

/*
 * Converts the ASCII representation of an Ethernet MAC address
 * to its binary representation.
 *
 * Stolen from packit, with some enhancements for netexpect
 *
 * Returns 0 if ASCII representation of a MAC address is valid,
 * or -1 otherwise.
 *
 * Accepts a MAC address as xx:xx:xx:xx:xx:xx or as
 * xxxx.xxxx.xxxx (as used in Cisco IOS.)
 */
int
format_ethernet_addr(const char *ethstr, uint8_t u_eaddr[ETH_ADDR_LEN])
{
    int i;
    unsigned long base16;
    char *eptr;
    char o_ethstr[sizeof("xx:xx:xx:xx:xx:xx")]; /* will hold temp. copy of
						   input string since we
						   don't want to modify the
						   input string */

    if (ethstr) {
	if (   !strcasecmp("broadcast", ethstr)
	    || !strcasecmp("bcast", ethstr) ) {
	    /*
	     * MAC address is ff:ff:ff:ff:ff:ff.
	     */
	    memset(u_eaddr, 0xff, ETH_ADDR_LEN);
	} else if (!strcasecmp("random", ethstr) ) {
	    /*
	     * MAC address is random.
	     *
	     * We use IANA Ethernet address block for unicast use. See
	     * http://www.iana.org/assignments/ethernet-numbers.
	     */
	    u_eaddr[0] = 0x00;
	    u_eaddr[1] = 0x00;
	    u_eaddr[2] = 0x5e;
	    random_block(u_eaddr + 3, ETH_ADDR_LEN - 3);
	} else {
	    /*
	     * Not a special MAC address (broadcast or random) so we need
	     * to parse the ASCII representation of the MAC address.
	     */

	    /*
	     * We don't want to modify the input string ethstr (and
	     * that's why we declared it as "const char *") so we need
	     * to make a local copy of it because strtok() does modify
	     * the input string.
	     */
	    strlcpy(o_ethstr, ethstr, sizeof(o_ethstr) );

	    if (strchr(o_ethstr, '.') ) {
		/* Assume ffff.ffff.ffff format */

		for (i = 0, eptr = strtok( (char *) o_ethstr, ".");
		     eptr && i <= ETH_ADDR_LEN;
		     eptr = strtok(NULL, ":"), i += 2) {
		    if ( (base16 = strtoul(eptr, 0, 16) ) > 0xffff)
			return -1;

		    u_eaddr[i] = base16;
		}

		if (i != ETH_ADDR_LEN || eptr)
		    /*
		     * Not enough address bytes or there's still some input left.
		     * This is an invalid MAC address.
		     */
		    return -1;
	    } else {
		/* Assume ff:ff:ff:ff:ff:ff format */

		for (i = 0, eptr = strtok( (char *) o_ethstr, ":");
		     eptr && i <= ETH_ADDR_LEN;
		     eptr = strtok(NULL, ":"), i++) {
		    if ( (base16 = strtoul(eptr, 0, 16) ) > 0xff)
			return -1;

		    u_eaddr[i] = base16;
		}

		if (i != ETH_ADDR_LEN || eptr)
		    /*
		     * Not enough address bytes or there's still some input left.
		     * This is an invalid MAC address.
		     */
		    return -1;
	    }
	}
    } else
	/*
	 * We received NULL for the MAC address string. Assume a MAC
	 * address of 00:00:00:00:00:00.
	 */
	memset(u_eaddr, 0x00, ETH_ADDR_LEN);

    return 0;
}

/*
 * Basically a wrapper around format_ethernet_addr() that takes as input
 * a pointer to a libdnet "addr" structure instead of an array of
 * bytes. If we are going to standardize on the use of libdnet "addr"
 * structures then we need a wrapper like this.
 */
int
parse_mac_addr(const char *macaddr, struct addr *a)
{
    memset(a, 0, sizeof(*a) );
    a->addr_type = ADDR_TYPE_ETH;
    a->addr_bits = ETH_ADDR_BITS;

    if (format_ethernet_addr(macaddr, a->addr_eth.data) == -1)
	return -1;

    return 0;
}

char *
ipaddr2str(ip_addr_t ip)
{
    struct addr a;
    static char addr[20];

    memset(&a, 0, sizeof(struct addr) );
    a.addr_type = ADDR_TYPE_IP;
    a.addr_bits = IP_ADDR_BITS;
    a.addr_ip = ip;

    return addr_ntop(&a, addr, sizeof(addr) );
}

char *
ip6addr2str(ip6_addr_t ip6)
{
    struct addr a;
    static char addr[46];

    memset(&a, 0, sizeof(struct addr) );
    a.addr_type = ADDR_TYPE_IP6;
    a.addr_bits = IP6_ADDR_BITS;
    a.addr_ip6 = ip6;

    return addr_ntop(&a, addr, sizeof(addr) );
}

/********************************************************************/

/*
 * Copy arg vector into a new buffer, concatenating arguments with spaces.
 * (stolen from tcpdump)
 */
char *
copy_argv(char **argv)
{
    char **p;
    size_t len;
    char *buf, *src, *dst;

    p = argv;
    if (!p || *p == NULL)
	return NULL;

    for (len = 0; *p; p++)
	len += strlen(*p) + 1;

    buf = dst = xmalloc(len);

    for (p = argv; (src = *p++) != NULL;) {
	while ( (*dst++ = *src++) != '\0')
	    ;
	dst[-1] = ' ';
    }
    dst[-1] = '\0';

    return buf;
}


/********************************************************************
 *				Random                              *
 ********************************************************************/

/*
 * Returns a random IP address that falls within a given range. The IP
 * address is returned in network byte order.
 */
ip_addr_t
random_ip(ip_addr_t a, ip_addr_t b)
{
    return htonl(ntohl(a) + (rand() % (ntohl(b) - ntohl(a) + 1) ) );
}

/*
 * Returns a random integer between a and b.
 */
int
random_int(int a, int b)
{
    return (rand() % (b - a + 1) ) + a;
}

/*
 * Fills a block of memory with random data.
 */
void
random_block(uint8_t *dest, size_t len)
{
    while (len--)
	*dest++ = rand();
}

/********************************************************************
 *				Time                                *
 ********************************************************************/

/*
 * Convert a timestamp (in struct timeval format) to its string representation.
 * Stolen from tcpdump's ts_print().
 */
const char *
ts2str(register const struct timeval *tvp)
{
    register int s;
    struct tm *tm;
    time_t Time;
    static unsigned b_sec;
    static unsigned b_usec;
    static char buf[256];
    int32_t thiszone;

    *buf = '\0';

    /*
     * To be more efficient we can make thiszone a global and update it
     * less frequently, i.e. just when tflag changes.
     */
    thiszone = tflag > 0 ? gmt2local(0) : 0;

    switch(tflag) {
    case 1: /* Default */
	s = (tvp->tv_sec + thiszone) % 86400;
	snprintf(buf, sizeof(buf), "%02d:%02d:%02d.%06u ",
		 s / 3600, (s % 3600) / 60, s % 60, (unsigned) tvp->tv_usec);
	break;
    case -1: /* Unix timeval style */
	snprintf(buf, sizeof(buf), "%u.%06u ",
		 (unsigned) tvp->tv_sec, (unsigned) tvp->tv_usec);
	break;
    case -2:
	if (b_sec == 0)
	    snprintf(buf, sizeof(buf), "000000 ");
	else {
	    int d_usec = tvp->tv_usec - b_usec;
	    int d_sec = tvp->tv_sec - b_sec;

	    while (d_usec < 0) {
		d_usec += 1000000;
		d_sec--;
	    }
	    if (d_sec)
		snprintf(buf, sizeof(buf), "%d. ", d_sec);
	    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf),
		     "%06d ", d_usec);
	}
	b_sec = tvp->tv_sec;
	b_usec = tvp->tv_usec;
	break;
    case -3: /* Default + Date*/
	s = (tvp->tv_sec + thiszone) % 86400;
	Time = (tvp->tv_sec + thiszone) - s;
	tm = gmtime(&Time);
	if (!tm)
	    snprintf(buf, sizeof(buf), "Date fail  ");
	else
	    snprintf(buf, sizeof(buf), "%04d-%02d-%02d ",
		     tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday);
	snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf),
		 "%02d:%02d:%02d.%06u ",
		 s / 3600, (s % 3600) / 60, s % 60, (unsigned) tvp->tv_usec);
	break;
    }

    return buf;
}

/*
 * Returns the difference between gmt and local time in seconds.
 * Use gmtime() and localtime() to keep things simple.
 *
 * Stolen from tcpdump.
 */
int32_t
gmt2local(time_t t)
{
    register int dt, dir;
    register struct tm *gmt, *loc;
    struct tm sgmt;

    if (t == 0)
	t = time(NULL);
    gmt = &sgmt;
    *gmt = *gmtime(&t);
    loc = localtime(&t);
    dt = (loc->tm_hour - gmt->tm_hour) * 60 * 60
	 + (loc->tm_min - gmt->tm_min) * 60;

    /*
     * If the year or julian day is different, we span 00:00 GMT
     * and must add or subtract a day. Check the year first to
     * avoid problems when the julian day wraps.
     */
    dir = loc->tm_year - gmt->tm_year;
    if (dir == 0)
	dir = loc->tm_yday - gmt->tm_yday;
    dt += dir * 24 * 60 * 60;

    return (dt);
}

/********************************************************************/

/*
 * Used by the binary search function bsearch() and the quick sort
 * function qsort() to compare two integers.
 */
int
compare_integers(const void *a, const void *b)
{
    return *(int *) a - *(int *) b;
}
