#ifndef GUI_H
#define GUI_H

#include "commandline.h"

//Seamless conversion between contour::XY and ImGui's ImVec2
#define IM_VEC2_CLASS_EXTRA                                                              \
    explicit constexpr ImVec2(const XY &xy) noexcept : x(float(xy.x)), y(float(xy.y)) {} \
    explicit operator XY() const noexcept { return XY{x,y}; }                            \
    bool operator==(const ImVec2 &) const noexcept = default;

#include "imgui.h"

enum MscGenKey_ {
    MscGenKey_F2,   // F2 compiles
    MscGenKey_F3,   // F3 searches
    MscGenKey_F4,   // F4 highlights
    MscGenKey_F5,   // F5 alternative compilation
    MscGenKey_F8,   // F8 moves around in Errors
    MscGenKey_S,    // Ctrl+S saves, Ctrl+Shift+S Save as
    MscGenKey_N,    // Ctrl+N FileNew
    MscGenKey_O,    // Ctrl+O FileOpen
    MscGenKey_R,    // Ctrl+R Rename Entity
    MscGenKey_F,    // Ctrl+F Search
    MscGenKey_H,    // Ctrl+H Replace
    MscGenKey_COUNT
};

extern std::array<ImFont *, 8> MscGenFonts;//defined in gui.cpp, filled in backend
extern float MscGenEditorFontSize;         //defined in gui.cpp, filled in backend
extern float MscGenWindowFontSize;         //defined in gui.cpp, filled in backend
extern int MscGenKeyMap[MscGenKey_COUNT];  //defined and filled in backend

std::optional<ImTextureID> TextureFromSurface(cairo_surface_t *);
void DestroyTexture(ImTextureID);
void LoadFonts();
std::map<std::string, std::string> RegisterLibrariesGUI();
void GUIBeep();
std::string_view FileNameFromPath(std::string_view full_name) noexcept;

struct GUIReturn {
    bool exit = false;
    std::string window_title; // Do not update when empty
    bool full_screen = false; // Set the window to full screen (or keep it there); or switch back when false.
};

/** What events the host OS have collected before this frame. */
struct GUIEvents {
    struct Drop {
        std::vector<std::string> full_names; ///<The list of files dropped on us. We reject multi-file drops.
        ImVec2 pos;                          ///<The position of the drop - not used now.
        explicit operator bool() const noexcept { return !full_names.empty(); }
        void clear() noexcept { full_names.clear(); }
    };
    bool request_exit = false;                          ///<If the user requested exit via a window close action
    std::optional<std::chrono::microseconds> total_cpu; ///<What is the total CPU time (user+kernel) used by this process until now. Empty if cannot be determined.
    Drop dropped;                                       ///<Non-empty if files have been dropped to us. Includes the coordinates.
};

/** Initialize the GUI. Call this before a call to DoGui()
 * @param [in] init The parameters parsed from command-line.*/
void InitGUI(const GUIInit &init);

/** Called before each ImGUI Frame.
 * It allows recompiling the font atlas on a DPI change.
 * @param [in] dpi_mul Multiplier of the monitor/display of the main window.
 *             On Linux the value of 1. shall correspond to 72 DPI, on Windows 96 DPI.
 *             Providing this value for each frame allows dynamic change.
 * @returns true if the DPI has changed.*/
bool PreFrameGUI(float dpi_mul);

/** Draw the entire GUI. 
 * @param [in] events Information on what happened to the computer since the last frame.*/
 GUIReturn DoGUI(const GUIEvents &events);

/** Shut down the GUI */
void ShutdownGUI();

#endif //GUI_H
