/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QFile>
#include <QDebug>
#include <QRegularExpression>
#include <QRegularExpressionMatch>

/////////////////////// Local includes
#include "MonomerSpec.hpp"


namespace MsXpS
{

namespace libXpertMass
{

/*!
\class MsXpS::libXpertMass::MonomerSpec
\inmodule libXpertMass
\ingroup PolChemDefBuildingdBlocks
\inheaderfile MonomerSpec.hpp

\brief The MonomerSpec class provides the specification about how \l Monomer
objects are represented.

The MonomerSpec class specifies how a \l Monomer object is represented
graphically, mainly by connecting its monomer code to a graphics SVG file that
is located in the Polymer chemistry definition directory. That connection is
performed in the "monomer_dictionary" dictionary file itself also located in
the polymer chemistry definition directory. Its contents look like this:

\code
A%alanine.svg
C%cysteine.svg
D%aspartate.svg
E%glutamate.svg
F%phenylalanine.svg
G%glycine.svg
H%histidine.svg"
\endcode

The \c{A%alanine.svg} line indicates that, when a Monomer object by code 'A' is
to be rendered graphically, the corresponding vignette to be used is in the
file named "alanine.svg" in the polymer chemistry definition directory.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_name

\brief The name of the monomer.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_code

\brief The code of the monomer.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_raster

\brief The file name of the raster representation of the monomer.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_vector

\brief The filename of the vector representation of the monomer.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_nameSound

\brief The file name of the sound for the name of the monomer.
*/

/*!
\variable int MsXpS::libXpertMass::MonomerSpec::m_codeSound

\brief The file name of the sound for the code of the monomer.
*/

/*!
\brief Constructs a MonomerSpec instance.
*/
MonomerSpec::MonomerSpec()
{
}

/*!
\brief Destructs this MonomerSpec instance.
*/
MonomerSpec::~MonomerSpec()
{
}

/*!
\brief Sets the monomer \a name.
*/
void
MonomerSpec::setName(const QString &name)
{
  m_name = name;
}

/*!
\brief Returns the monomer name.
*/
const QString &
MonomerSpec::name()
{
  return m_name;
}


/*!
\brief Sets the monomer \a code.
*/
void
MonomerSpec::setCode(const QString &code)
{
  m_code = code;
}


/*!
\brief Returns the monomer code.
*/
const QString &
MonomerSpec::code()
{
  return m_code;
}

/*!
\brief Sets the raster image file name to \a raster.
*/
void
MonomerSpec::setRaster(const QString &raster)
{
  m_raster = raster;
}

/*!
\brief Returns the raster image file name.
*/
const QString &
MonomerSpec::raster()
{
  return m_raster;
}

/*!
\brief Sets the vector image file name to \a vector.
*/
void
MonomerSpec::setVector(const QString &vector)
{
  m_vector = vector;
}

/*!
\brief Returns the vector image file name.
*/
const QString &
MonomerSpec::vector()
{
  return m_vector;
}


/*!
\brief Sets the file name of the Monomer's name \a sound file.
*/
void
MonomerSpec::setNameSound(const QString &sound)
{
  m_nameSound = sound;
}


/*!
\brief Returns the file name of the Monomer's name sound file.
*/
const QString &
MonomerSpec::nameSound()
{
  return m_nameSound;
}


/*!
\brief Sets the file name of the Monomer's code \a sound file.
*/
void
MonomerSpec::setCodeSound(const QString &sound)
{
  m_codeSound = sound;
}


/*!
\brief Returns the file name of the Monomer's code sound file.
*/
const QString &
MonomerSpec::codeSound()
{
  return m_codeSound;
}


/*!
\brief Parses the \a file_path dictionary containing the Monomer
specifications.

At the moment the file has this format:

\code
A%alanine.svg
C%cysteine.svg
D%aspartate.svg
E%glutamate.svg
\endcode

Upon parsing, the \a monomer_spec_list of MonomerSpec instances will
be filled with instances created on the basis of each parsed line in the file.

Returns true if the parsing was successful, false otherwise.
*/
bool
MonomerSpec::parseFile(QString &file_path, QList<MonomerSpec *>
*monomer_spec_list)
{
  MonomerSpec *monomerSpec = 0;

  qint64 lineLength;

  QString line;
  QString temp;

  char buffer[1024];

  Q_ASSERT(monomer_spec_list != 0);

  if(file_path.isEmpty())
    return false;

  QFile file(file_path);

  if(!file.open(QFile::ReadOnly))
    return false;

  // The lines we have to parse are of the following type:
  // A%alanine.svg|alanine.png
  // Any line starting with '#' are not parsed.

  // Get the first line of the file. Next we enter in to a
  // while loop.

  lineLength = file.readLine(buffer, sizeof(buffer));

  while(lineLength != -1)
    {
      // The line is now in buffer, and we want to convert
      // it to Unicode by setting it in a QString.
      line = buffer;

      // The line that is in line should contain something like:
      // A%alanine.svg

      // Remove all the spaces from the borders: whitespace means any
      // character for which QChar::isSpace() returns true. This
      // includes the ASCII characters '\t', '\n', '\v', '\f', '\r',
      // and ' '.

      line = line.trimmed();

      if(line.isEmpty() || line.startsWith('#', Qt::CaseInsensitive))
        {
          lineLength = file.readLine(buffer, sizeof(buffer));
          continue;
        }

      // Now some other checks. Remember the format of the line:
      // A%alanine.svg
      QString code;
      QString file_name;
      QRegularExpression reg_exp(QString("^([A-Z][a-z]*)%(.*)$"));

      QRegularExpressionMatch match = reg_exp.match(line);

      if(match.hasMatch())
        {
          code      = match.captured(1);
          file_name = match.captured(2);
        }

      // qDebug() << "code:" << code << "file_name:" << file_name;

      // OK, we finally can allocate a new MonomerSpec *.
      monomerSpec = new MonomerSpec();

      monomerSpec->m_code   = code;
      monomerSpec->m_vector = file_name;

      monomer_spec_list->append(monomerSpec);
      lineLength = file.readLine(buffer, sizeof(buffer));
    }

  return true;
}

} // namespace libXpertMass

} // namespace MsXpS
