#include "osl/record/opening/openingBook.h"
#include "osl/record/csaRecord.h"
#include "osl/record/compactBoard.h"
#include "osl/oslConfig.h"
#include <boost/foreach.hpp>

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>

class OpeningBookTest : public CppUnit::TestFixture
{
  CPPUNIT_TEST_SUITE(OpeningBookTest);
  CPPUNIT_TEST(testOpen);
  CPPUNIT_TEST(testGetMoves);
  CPPUNIT_TEST(testMove);
  CPPUNIT_TEST(testCompactBoard);
  CPPUNIT_TEST(testCompactBoardOrder);
  CPPUNIT_TEST_SUITE_END();
public:
  void testOpen();
  void testGetMoves();
  void testMove();
  void testCompactBoard();
  void testCompactBoardOrder();
};

CPPUNIT_TEST_SUITE_REGISTRATION(OpeningBookTest);
using osl::OslConfig;

void OpeningBookTest::testOpen()
{
  osl::record::opening::WeightedBook book(osl::OslConfig::openingBook());
}

void OpeningBookTest::testGetMoves()
{
  osl::record::opening::WeightedBook book(osl::OslConfig::openingBook());
  osl::vector<osl::record::opening::WMove> moves = book.getMoves(book.getStartState());
  osl::vector<osl::record::opening::WMove> moves_nonzero = book.getMoves(book.getStartState(), false);
  BOOST_FOREACH(const osl::record::opening::WMove& move, moves_nonzero)
  {
    CPPUNIT_ASSERT(move.getWeight() > 0);
  }
  // moves usually include a (or more) zero weighted move. 
  CPPUNIT_ASSERT(moves_nonzero.size() < moves.size());
}

void OpeningBookTest::testMove()
{
  std::ifstream ifs(osl::OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  const int max=OslConfig::inUnitTestShort ? 10 : 100;
  std::string fileName;
  int i=0;
  while((ifs >> fileName) && (++i<max)) 
  {
    std::string fullFileName = osl::OslConfig::testCsaFile(fileName);

    osl::Record rec=osl::CsaFile(fullFileName).getRecord();
    const osl::vector<osl::Move> moves=rec.getMoves();
    for (size_t i=0; i<moves.size(); ++i)
    {
      const osl::Move& m = moves[i];
      osl::record::opening::OMove om(m);
      CPPUNIT_ASSERT_EQUAL(m.from(), om.getFrom());
      CPPUNIT_ASSERT_EQUAL(m.to(), om.getTo());
      CPPUNIT_ASSERT_EQUAL(m.isPromotion(), om.isPromotion());
      CPPUNIT_ASSERT_EQUAL(m.capturePtype(), om.getCapturePtype());
      CPPUNIT_ASSERT_EQUAL(m.ptype(), om.getPtype());
      CPPUNIT_ASSERT_EQUAL(m.player(), om.getPlayer());
      CPPUNIT_ASSERT_EQUAL(m, om.operator osl::Move());
    }
  }
}

void OpeningBookTest::testCompactBoard()
{
  std::ifstream ifs(osl::OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  const int max=OslConfig::inUnitTestShort ? 10 :100;
  std::string fileName;
  int i=0;
  while((ifs >> fileName) && (++i<max)) 
  {
    std::string fullFileName = osl::OslConfig::testCsaFile(fileName);

    osl::Record rec=osl::CsaFile(fullFileName).getRecord();
    const osl::vector<osl::Move> moves=rec.getMoves();
    osl::NumEffectState state(rec.getInitialState());
    // This doesn't test initial state itself but oh well...
    for (size_t i=0; i<moves.size(); ++i)
    {
      const osl::Move& m = moves[i];
      state.makeMove(m);
      for (int p_index = 0; p_index < osl::Piece::SIZE; p_index++)
      {
	osl::Piece piece = state.pieceOf(p_index);
	osl::record::OPiece opiece(piece);
	CPPUNIT_ASSERT_EQUAL(piece.square(), opiece.getSquare());
	CPPUNIT_ASSERT_EQUAL(piece.ptype(), opiece.getPtype());
	CPPUNIT_ASSERT_EQUAL(piece.owner(), opiece.getOwner());
      }
      osl::record::CompactBoard board(state);
      CPPUNIT_ASSERT_EQUAL(static_cast<const osl::SimpleState&>(state), board.getState());

      std::ostringstream oss(std::ostringstream::out);
      osl::record::CompactBoard reconstructed_board;
      oss << board;
      std::istringstream iss(oss.str());
      iss >> reconstructed_board;
      CPPUNIT_ASSERT_EQUAL(board.getState(), reconstructed_board.getState());
    }
  }
}

void OpeningBookTest::testCompactBoardOrder()
{
  std::ifstream ifs(osl::OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  const int max=OslConfig::inUnitTestShort ? 10 : 100;
  std::string fileName;
  int i=0;
  while((ifs >> fileName) && (++i<max)) 
  {
    std::string fullFileName = osl::OslConfig::testCsaFile(fileName);

    osl::Record rec=osl::CsaFile(fullFileName).getRecord();
    const osl::vector<osl::Move> moves=rec.getMoves();
    osl::NumEffectState state(rec.getInitialState());
    // This doesn't test initial state itself but oh well...
    for (size_t i=0; i<moves.size(); ++i)
    {
      const osl::Move& m = moves[i];
      state.makeMove(m);
      osl::record::CompactBoard board(state);

      std::ostringstream oss(std::ostringstream::out);
      oss << board;
      // test uniquness of CompactBoard representationg for given state.
      const std::string s = oss.str();
      char str[s.length() + 1];
      s.copy(str, s.length());
      str[s.length()] = '\0';
      // Swap pieces
      for (int i = 0; i < 1; i++)
      {
	for (int j = 0; j < 4; j++)
	{
	  char tmp = str[i * 4 + j];
	  str[i * 4 + j] = str[(40 - 1 - i) * 4 + j];
	  str[(40 - 1 - i) * 4 + j] = tmp;
	}
       }
      std::string board_string(str, s.length());
      std::istringstream iss(board_string);
      osl::record::CompactBoard tmp_board;
      iss >> tmp_board;
      // We need to do SimpleState -> CompactBoard convesion because
      // that's where sorting happens.
      osl::record::CompactBoard reconstructed_board(tmp_board.getState());
      std::ostringstream oss2(std::ostringstream::out);
      oss2 << reconstructed_board;
      CPPUNIT_ASSERT_EQUAL(oss.str(), oss2.str());
    }
  }
}
// ;;; local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
