/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.loader;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.Modification;
import org.jboss.cache.config.CacheLoaderConfig.IndividualCacheLoaderConfig;
import org.jboss.cache.loader.tcp.TcpCacheOperations;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.Socket;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * DelegatingCacheLoader implementation which delegates to a remote (not in the same VM)
 * CacheImpl using TCP/IP for communication. Example configuration for connecting to a TcpCacheServer
 * running at myHost:12345:<pre>
 * <attribute name="CacheLoaderClass">org.jboss.cache.loader.TcpDelegatingCacheLoader</attribute>
 * <attribute name="CacheLoaderConfig">
 * host=localhost
 * port=2099
 * </attribute>
 * </pre>
 *
 * @author Bela Ban
 * @version $Id: TcpDelegatingCacheLoader.java 6120 2008-06-30 11:58:59Z manik.surtani@jboss.com $
 */
public class TcpDelegatingCacheLoader extends AbstractCacheLoader
{
   private Socket sock;
   private TcpDelegatingCacheLoaderConfig config;
   ObjectInputStream in;
   ObjectOutputStream out;
   private static final Log log = LogFactory.getLog(TcpDelegatingCacheLoader.class);
   private static Method GET_CHILDREN_METHOD, GET_METHOD, PUT_KEY_METHOD, PUT_DATA_METHOD, REMOVE_KEY_METHOD, REMOVE_METHOD, PUT_MODS_METHOD, EXISTS_METHOD, REMOVE_DATA_METHOD;

   static
   {
      try
      {
         GET_CHILDREN_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_getChildrenNames", Fqn.class);
         GET_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_get", Fqn.class);
         EXISTS_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_exists", Fqn.class);
         PUT_KEY_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_put", Fqn.class, Object.class, Object.class);
         PUT_DATA_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_put", Fqn.class, Map.class);
         REMOVE_KEY_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_remove", Fqn.class, Object.class);
         REMOVE_DATA_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_removeData", Fqn.class);
         REMOVE_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_remove", Fqn.class);
         PUT_MODS_METHOD = TcpDelegatingCacheLoader.class.getDeclaredMethod("_put", List.class);

      }
      catch (Exception e)
      {
         log.fatal("Unable to initialise reflection methods", e);
      }
   }


   /**
    * Allows configuration via XML config file.
    */
   public void setConfig(IndividualCacheLoaderConfig base)
   {
      if (base instanceof TcpDelegatingCacheLoaderConfig)
      {
         this.config = (TcpDelegatingCacheLoaderConfig) base;
      }
      else
      {
         config = new TcpDelegatingCacheLoaderConfig(base);
      }
   }

   public IndividualCacheLoaderConfig getConfig()
   {
      return config;
   }

   /**
    * Invokes the specified Method with the specified parameters, catching SocketExceptions and attempting to reconnect
    * to the TcpCacheServer if necessary.
    *
    * @param m      method to invoke
    * @param params parameters
    * @return method return value
    */
   protected Object invokeWithRetries(Method m, Object... params)
   {
      long endTime = System.currentTimeMillis() + config.getTimeout();
      do
      {
         try
         {
            return m.invoke(this, params);
         }
         catch (IllegalAccessException e)
         {
            log.error("Should never get here!", e);
         }
         catch (InvocationTargetException e)
         {
            if (e.getCause() instanceof IOException)
            {
               try
               {
                  // sleep 250 ms
                  if (log.isDebugEnabled()) log.debug("Caught IOException.  Retrying.", e);
                  Thread.sleep(config.getReconnectWaitTime());
                  restart();
               }
               catch (IOException e1)
               {
                  // IOException starting; sleep a bit and retry
               }
               catch (InterruptedException e1)
               {
                  // do nothing
               }
            }
            else
            {
               throw new CacheException("Problems invoking method call!", e);
            }
         }
      }
      while (System.currentTimeMillis() < endTime);
      throw new CacheException("Unable to communicate with TCPCacheServer(" + config.getHost() + ":" + config.getPort() + ") after " + config.getTimeout() + " millis, with reconnects every " + config.getReconnectWaitTime() + " millis.");
   }

   // ------------------ CacheLoader interface methods, which delegate to retry-aware methods

   public Set<?> getChildrenNames(Fqn fqn) throws Exception
   {

      return (Set<?>) invokeWithRetries(GET_CHILDREN_METHOD, fqn);
   }

   public Map<Object, Object> get(Fqn name) throws Exception
   {
      return (Map<Object, Object>) invokeWithRetries(GET_METHOD, name);
   }

   public boolean exists(Fqn name) throws Exception
   {
      return (Boolean) invokeWithRetries(EXISTS_METHOD, name);
   }

   public Object put(Fqn name, Object key, Object value) throws Exception
   {
      return invokeWithRetries(PUT_KEY_METHOD, name, key, value);
   }

   public void put(Fqn name, Map<Object, Object> attributes) throws Exception
   {
      invokeWithRetries(PUT_DATA_METHOD, name, attributes);
   }

   @Override
   public void put(List<Modification> modifications) throws Exception
   {
      invokeWithRetries(PUT_MODS_METHOD, modifications);
   }

   public Object remove(Fqn fqn, Object key) throws Exception
   {
      return invokeWithRetries(REMOVE_KEY_METHOD, fqn, key);
   }

   public void remove(Fqn fqn) throws Exception
   {
      invokeWithRetries(REMOVE_METHOD, fqn);
   }

   public void removeData(Fqn fqn) throws Exception
   {
      invokeWithRetries(REMOVE_DATA_METHOD, fqn);
   }

   // ------------------ Retry-aware CacheLoader interface method counterparts

   protected Set<?> _getChildrenNames(Fqn fqn) throws Exception
   {
      Set cn;
      synchronized (out)
      {
         out.reset();
         out.writeByte(TcpCacheOperations.GET_CHILDREN_NAMES);
         out.writeObject(fqn);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
         cn = (Set) retval;
      }

      // the cache loader contract is a bit different from the cache when it comes to dealing with childrenNames
      if (cn.isEmpty()) return null;
      return cn;
   }

   protected Map<Object, Object> _get(Fqn name) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.GET);
         out.writeObject(name);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
         return (Map) retval;
      }
   }

   protected boolean _exists(Fqn name) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.EXISTS);
         out.writeObject(name);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
         return (Boolean) retval;
      }
   }

   protected Object _put(Fqn name, Object key, Object value) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.PUT_KEY_VAL);
         out.writeObject(name);
         out.writeObject(key);
         out.writeObject(value);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
         return retval;
      }
   }

   protected void _put(Fqn name, Map<Object, Object> attributes) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.PUT);
         out.writeObject(name);
         out.writeObject(attributes);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
      }
   }

   protected void _put(List<Modification> modifications) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.PUT_LIST);
         int length = modifications.size();
         out.writeInt(length);
         for (Modification m : modifications)
         {
            m.writeExternal(out);
         }
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
      }
   }

   protected Object _remove(Fqn fqn, Object key) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.REMOVE_KEY);
         out.writeObject(fqn);
         out.writeObject(key);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
         return retval;
      }
   }

   protected void _remove(Fqn fqn) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.REMOVE);
         out.writeObject(fqn);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
      }
   }

   protected void _removeData(Fqn fqn) throws Exception
   {
      synchronized (out)
      {
         out.reset();

         out.writeByte(TcpCacheOperations.REMOVE_DATA);
         out.writeObject(fqn);
         out.flush();
         Object retval = in.readObject();
         if (retval instanceof Exception)
         {
            throw (Exception) retval;
         }
      }
   }

   // ----------------- Lifecycle and no-op methods


   @Override
   public void start() throws IOException
   {
      sock = new Socket(config.getHost(), config.getPort());
      out = new ObjectOutputStream(new BufferedOutputStream(sock.getOutputStream()));
      out.flush();
      in = new ObjectInputStream(new BufferedInputStream(sock.getInputStream()));
   }

   @Override
   public void stop()
   {
      try
      {
         if (in != null) in.close();
      }
      catch (IOException e)
      {
         if (log.isDebugEnabled()) log.debug("Unable to close TCP stream.", e);
      }
      try
      {
         if (out != null) out.close();
      }
      catch (IOException e)
      {
         if (log.isDebugEnabled()) log.debug("Unable to close TCP stream.", e);
      }
      try
      {
         if (sock != null) sock.close();
      }
      catch (IOException e)
      {
         if (log.isDebugEnabled()) log.debug("Unable to close socket.", e);
      }
   }

   protected void restart() throws IOException
   {
      stop();
      start();
   }

   @Override
   public void loadEntireState(ObjectOutputStream os) throws Exception
   {
      throw new UnsupportedOperationException("operation is not currently supported - need to define semantics first");
   }

   @Override
   public void loadState(Fqn subtree, ObjectOutputStream os) throws Exception
   {
      throw new UnsupportedOperationException("operation is not currently supported - need to define semantics first");
   }

   @Override
   public void storeEntireState(ObjectInputStream is) throws Exception
   {
      throw new UnsupportedOperationException("operation is not currently supported - need to define semantics first");
   }

   @Override
   public void storeState(Fqn subtree, ObjectInputStream is) throws Exception
   {
      throw new UnsupportedOperationException("operation is not currently supported - need to define semantics first");
   }
}
