/*
 * @(#)FileInputStreamGenerator.java
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.io.v1;

import java.io.InputStream;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;


/**
 * A type of class which can generate a new InputStream based on just a
 * partial name.  Useful if a resource is loaded from a specific location
 * (classpath resource, file system, URL, etc), and it references other
 * resources which must be loaded relative to it.
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    0.9.1d Alpha (sometime in 2001)
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class FileInputStreamGenerator implements IInputStreamGenerator
{
    private String root;
    private String orig;
    
    /**
     * Accepts the name of the 1st file loaded.  All files generated from
     * this generator will be relative to its location.  If the name is
     * a directory, then all relative names will be in this directory.
     */
    public FileInputStreamGenerator( String originalName )
            throws IOException
    {
        this.orig = originalName;
        File f = new File( originalName );
        if (!f.isDirectory())
        {
            originalName = f.getParent();
            if (originalName == null)
            {
                originalName = "";
            }
        }
        if (originalName.endsWith( File.separator ))
        {
            originalName = originalName.substring( 0,
                originalName.length() - File.separator.length() );
        }
        this.root = originalName;
        if (this.root.length() <= 0)
        {
            this.root = ".";
        }
    }
    

    public File getFile( String relativeName )
    {
        if (!this.orig.equals( relativeName ))
        {
            if (!relativeName.startsWith( "/" ) &&
                !relativeName.startsWith( File.separator ))
            {
                // only adjust the relative name if it is relative.
                relativeName = this.root + File.separator + relativeName;
            }
        }
        return new File( relativeName );
    }
    
    
    public String getFullName( String relativeName )
    {
        File f = getFile( relativeName );
        try
        {
            return f.getCanonicalPath();
        }
        catch (IOException ioe)
        {
            return f.getAbsolutePath();
        }
    }
    
    
    public InputStream createInputStream( String relativeName )
            throws IOException
    {
        return new FileInputStream( getFile( relativeName ) );
    }
}
