/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file:     submesh.c                                                      */
/*                                                                          */
/*                                                                          */
/* description: Support for master/slave meshes                             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by D. Koester (2004)                                                */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

#include "submesh_1d.c"
#if DIM_OF_WORLD > 1
#include "submesh_2d.c"
#if DIM_OF_WORLD > 2
#include "submesh_3d.c"
#endif
#endif

/****************************************************************************/
/* read_submesh_gen(read_xdr, master, slave_filename, binding_method,       */
/*                  init_boundary):                                         */
/* Read a slave mesh from file "slave_name" and bind it to "master". Assumes*/
/* that master and slave were written at the same time. Available for native*/
/* and XDR formats. Return a pointer to the slave mesh.                     */
/****************************************************************************/

static MESH *read_submesh_gen(const int read_xdr, MESH *master,
			      const char *slave_filename,
			      int (*binding_method)(MESH *master, MACRO_EL *el,
						    int face, void *data),
		       NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int),
			      void *data)
{
  FUNCNAME("read_submesh_gen");
  MESH            *slave = nil;
  MESH_MEM_INFO   *s_info, *m_info;
  MACRO_EL        *m_mel, *s_mel;
  const DOF_ADMIN *admin = NULL;
  const FE_SPACE  *m_space, *s_space;
  DOF_PTR_VEC     *m_dpv, *s_dpv;
  int              n_dof[N_NODE_TYPES] = {};
  int              i, j, n, m_dim;

/****************************************************************************/
/* Do the checks for obvious user errors.                                   */
/****************************************************************************/

  TEST_EXIT(master,"No master mesh given!\n");
  TEST_EXIT(master->dim > 0,"Master mesh has dim == 0!\n");
  m_dim = master->dim;
  TEST_EXIT(slave_filename,"No filename for the slave mesh given!\n");
  TEST_EXIT(binding_method,"No binding method given!\n");

/****************************************************************************/
/* Read the mesh from the file. Do not use the time value pointer.          */
/****************************************************************************/
  if(read_xdr)
    slave = read_mesh_xdr(slave_filename, nil, n_proj);
  else
    slave = read_mesh(slave_filename, nil, n_proj);

/****************************************************************************/
/* Set the mem_info components in master and slave.                         */
/****************************************************************************/

  s_info = (MESH_MEM_INFO *)slave->mem_info;
  m_info = (MESH_MEM_INFO *)master->mem_info;

  m_info->slaves = MEM_REALLOC(m_info->slaves,
			       m_info->n_slaves,
			       m_info->n_slaves + 1,
			       MESH *);

  m_info->slaves[m_info->n_slaves] = slave;

  m_info->n_slaves++;

  s_info->master         = master;
  s_info->binding_method = binding_method;

/* Check for the correct DOF_ADMINs for slave and master.                    */

  n_dof[CENTER] = 1;

  for (i = 0; i < slave->n_dof_admin; i++) {
    admin = slave->dof_admin[i];
    for (j = 0; j < N_NODE_TYPES; j++) {
      if (admin->n_dof[j] != n_dof[j]) goto bad_admin;
    }
    if(!admin->preserve_coarse_dofs) goto bad_admin;
    break;
  bad_admin:
    admin = nil;
  }
  TEST_EXIT(admin,"Slave mesh does not seem to have had a master!\n");
  s_space = get_fe_space(slave, "Center dof fe_space", n_dof, nil, 1);

  n_dof[CENTER] = 0;
  switch(m_dim) {
  case 1:
    n_dof[VERTEX] = 1;
    break;
  case 2:
    n_dof[EDGE] = 1;
    break;
  case 3:
    n_dof[FACE] = 1;
    break;
  }

  for (i = 0; i < master->n_dof_admin; i++) {
    admin = master->dof_admin[i];
    for (j = 0; j < N_NODE_TYPES; j++) {
      if (admin->n_dof[j] != n_dof[j]) goto bad_admin2;
    }
    if(!admin->preserve_coarse_dofs) goto bad_admin2;
    break;
  bad_admin2:
    admin = nil;
  }
  TEST_EXIT(admin,"Given master mesh does not seem to have had slaves!\n");
  m_space = get_fe_space(master, "Center dof fe_space", n_dof, nil, 1);

/* Allocate element pointer vectors.                                         */

  s_dpv = get_dof_ptr_vec("Slave - master pointers",
			  s_space);
  s_info->master_binding  = s_dpv;

  m_dpv = get_dof_ptr_vec("Master - slave pointers",
			  m_space);
  s_info->slave_binding   = m_dpv;

  switch(m_dim) {
  case 1:
    m_dpv->refine_interpol = master_interpol_1d;
    m_dpv->coarse_restrict = master_restrict_1d;
    break;
#if DIM_OF_WORLD > 1
  case 2:
    m_dpv->refine_interpol = master_interpol_2d;
    m_dpv->coarse_restrict = master_restrict_2d;
    break;
#if DIM_OF_WORLD == 3
  case 3:
    m_dpv->refine_interpol = master_interpol_3d;
    m_dpv->coarse_restrict = master_restrict_3d;
#endif
#endif
  }

/****************************************************************************/
/* Set the element pointer vec entries to the correct values.               */
/* This assumes that slave macro elements were allocated in the order given */
/* by the loop below.                                                       */
/****************************************************************************/

  FOR_ALL_DOFS(s_dpv->fe_space->admin, s_dpv->vec[dof] = nil);
  FOR_ALL_DOFS(m_dpv->fe_space->admin, m_dpv->vec[dof] = nil);

  s_mel = slave->macro_els;
  for(n = 0; n < master->n_macro_el; n++) {
    m_mel = master->macro_els + n;

    for(i = 0; i < N_NEIGH(m_dim); i++)
      if(binding_method(master, m_mel, i, data)) {
	TEST_EXIT(s_mel,"Ran out of slave macro elements... Wrong meshes?\n");

	/* Here we take care of node projection function transfer. */
	if(m_dim > 1) {
	  if(m_mel->projection[i+1])
	    s_mel->projection[0] = m_mel->projection[i+1];
	  else
	    s_mel->projection[0] = m_mel->projection[0];
	}

	if(m_dim == 1)
	  join_elements_recursive_1d(master, slave,
				     m_dpv->fe_space->admin,
				     s_dpv->fe_space->admin,
				     m_dpv, s_dpv, i, m_mel->el, s_mel->el);
#if DIM_OF_WORLD > 1
	else if(m_dim == 2)
	  join_elements_recursive_2d(master, slave,
				     m_dpv->fe_space->admin,
				     s_dpv->fe_space->admin,
				     m_dpv, s_dpv, i, m_mel->el, s_mel->el);
#if DIM_OF_WORLD == 3
	else if(m_dim == 3)
	  join_elements_recursive_3d(master, slave,
				     m_dpv->fe_space->admin,
				     s_dpv->fe_space->admin,
				     m_dpv, s_dpv, i, m_mel->el, s_mel->el,
				     m_mel->orientation, m_mel->el_type);
#endif
#endif

	s_mel++;
      }
  }

  return slave;
}


/****************************************************************************/
/* set_lambdas(m_lambda, s_lambda): Translate slave barycentric coordinates */
/* to master barycentric coordinates in the canonical fashion.              */
/****************************************************************************/

static const EL_INFO    *m_el_info; 
static int               m_subsimplex;

static void set_lambdas(REAL m_lambda[N_LAMBDA], const REAL s_lambda[N_LAMBDA])
{
  if(m_el_info->mesh->dim == 1)
    switch(m_subsimplex) {
    case 0:
      m_lambda[0] = 1.0;
      m_lambda[1] = 0.0;
      break;
    case 1:
      m_lambda[0] = 0.0;
      m_lambda[1] = 1.0;
      break;
    }
#if DIM_OF_WORLD > 1
  else if(m_el_info->mesh->dim == 2)
    switch(m_subsimplex) {
    case 0:
      m_lambda[1] = s_lambda[0];
      m_lambda[2] = s_lambda[1];
      break;
    case 1:
      m_lambda[0] = s_lambda[1];
      m_lambda[2] = s_lambda[0];
      break;
    case 2:
      m_lambda[0] = s_lambda[0];
      m_lambda[1] = s_lambda[1];
      break;
    }
#if DIM_OF_WORLD == 3
  else {
    int m_orient = (m_el_info->orientation > 0) ? 0 : 1;
    int m_type = (int)m_el_info->el_type;

    switch(m_subsimplex) {
    case 0:
      m_lambda[1] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][1]];
      m_lambda[2] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][2]];
      m_lambda[3] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][3]];
      break;
    case 1:
      m_lambda[0] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][0]];
      m_lambda[2] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][2]];
      m_lambda[3] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][3]];
      break;
    case 2:
      m_lambda[0] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][0]];
      m_lambda[1] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][1]];
      m_lambda[3] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][3]];
      break;
    case 3:
      m_lambda[0] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][0]];
      m_lambda[1] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][1]];
      m_lambda[2] =
	s_lambda[slave_numbering_3d[m_type][m_orient][m_subsimplex][2]];
      break;
    }
  }
#endif
#endif

  return;
}


/****************************************************************************/
/* trace_loc[_d](s_el_info, s_lambda): These routines perform the local     */
/* trace operation on each element. Since eval_uh() is used it is possible  */
/* to have different FE_SPACES on master and slave element while performing */
/* a trace operation.                                                       */
/****************************************************************************/

static DOF_REAL_VEC     *m_vec;
static const REAL     *(*get_local_m_vec)(const EL *, const DOF_REAL_VEC *,
					  REAL *);

static REAL trace_loc(const EL_INFO *s_el_info, 
		      const REAL s_lambda[N_LAMBDA])
{
  const REAL  *m_local_vec = get_local_m_vec(m_el_info->el, m_vec, nil);
  REAL         m_lambda[N_LAMBDA] = {};

  set_lambdas(m_lambda, s_lambda);

  return eval_uh(m_lambda, m_local_vec, m_vec->fe_space->bas_fcts);
}

static DOF_REAL_D_VEC   *m_vec_d;
static const REAL_D   *(*get_local_m_vec_d)(const EL *, const DOF_REAL_D_VEC *,
					    REAL_D *);

static const REAL *trace_loc_d(const EL_INFO *s_el_info, 
			       const REAL s_lambda[N_LAMBDA], REAL_D val)
{
  static REAL_D  buffer;
  REAL          *res = val ? val : buffer; 
  const REAL_D  *m_local_vec_d = get_local_m_vec_d(m_el_info->el, m_vec_d,nil);
  REAL           m_lambda[N_LAMBDA] = {};

  set_lambdas(m_lambda, s_lambda);

  eval_uh_d(m_lambda, m_local_vec_d, m_vec_d->fe_space->bas_fcts, res);

  return res;
}

/****************************************************************************/
/* sm_dof_connection_?d[degree-1][type in 3D][orientation in 3D]            */
/*                     [m_subsimplex][s_dof]:                               */
/* assign a master DOF to each slave DOF at all subsimplices (Lagrange      */
/* degree between 1 and 4). NOTE: Type 1 and 2 have equal mapping!          */
/****************************************************************************/

#if DIM_OF_WORLD > 1
static const int sm_dof_connection_2d[4][3][5] =
  {/* p=1 */ {{1,2},{2,0},{0,1}},
   /* p=2 */ {{1,2,3},{2,0,4},{0,1,5}},
   /* p=3 */ {{1,2,3,4},{2,0,5,6},{0,1,7,8}},
   /* p=4 */ {{1,2,3,4,5},{2,0,6,7,8},{0,1,9,10,11}}};

#if DIM_OF_WORLD > 2
static const int sm_dof_connection_3d[4][2][2][4][15] =
  {/*p=1*/{/*t=0*/{/*o=+*/{{3,1,2},
			   {2,0,3},
			   {0,1,3},
			   {1,0,2}},
		   /*o=-*/{{1,3,2},
			   {0,2,3},
			   {1,0,3},
			   {0,1,2}}},
	   /*t=1*/{/*o=+*/{{1,2,3},
			   {2,0,3},
			   {0,1,3},
			   {1,0,2}},
		   /*o=-*/{{2,1,3},
			   {0,2,3},
			   {1,0,3},
			   {0,1,2}}}},
   /*p=2*/{/*t=0*/{/*o=+*/{{3,1,2,7,9,8},
			   {2,0,3,6,9,5},
			   {0,1,3,8,6,4},
			   {1,0,2,5,7,4}},
		   /*o=-*/{{1,3,2,9,7,8},
			   {0,2,3,9,6,5},
			   {1,0,3,6,8,4},
			   {0,1,2,7,5,4}}},
	   /*t=1*/{/*o=+*/{{1,2,3,9,8,7},
			   {2,0,3,6,9,5},
			   {0,1,3,8,6,4},
			   {1,0,2,5,7,4}},
		   /*o=-*/{{2,1,3,8,9,7},
			   {0,2,3,9,6,5},
			   {1,0,3,6,8,4},
			   {0,1,2,7,5,4}}}},
   /*p=3*/{/*t=0*/{/*o=+*/{{3,1,2,10,11,14,15,13,12,16},
			   {2,0,3,8,9,15,14,7,6,17},
			   {0,1,3,12,13,9,8,4,5,18},
			   {1,0,2,6,7,11,10,5,4,19}},
		   /*o=-*/{{1,3,2,15,14,11,10,12,13,16},
			   {0,2,3,14,15,9,8,6,7,17},
			   {1,0,3,8,9,13,12,5,4,18},
			   {0,1,2,10,11,7,6,4,5,19}}},
	   /*t=1*/{/*o=+*/{{1,2,3,14,15,13,12,10,11,16},
			   {2,0,3,8,9,15,14,7,6,17},
			   {0,1,3,12,13,9,8,4,5,18},
			   {1,0,2,6,7,11,10,5,4,19}},
		   /*o=-*/{{2,1,3,12,13,15,14,11,10,16},
			   {0,2,3,14,15,9,8,6,7,17},
			   {1,0,3,8,9,13,12,5,4,18},
			   {0,1,2,10,11,7,6,4,5,19}}}},
   /*p=4*/{/*t=0*/{/*o=+*/{{3,1,2,13,14,15,19,20,21,18,17,16,24,22,23},
			   {2,0,3,10,11,12,21,20,19,9,8,7,26,25,27},
			   {0,1,3,16,17,18,12,11,10,4,5,6,28,29,30},
			   {1,0,2,7,8,9,15,14,13,6,5,4,32,31,33}},
		   /*o=-*/{{1,3,2,21,20,19,15,14,13,16,17,18,22,24,23},
			   {0,2,3,19,20,21,12,11,10,7,8,9,25,26,27},
			   {1,0,3,10,11,12,18,17,16,6,5,4,29,28,30},
			   {0,1,2,13,14,15,9,8,7,4,5,6,31,32,33}}},
	   /*t=1*/{/*o=+*/{{1,2,3,19,20,21,18,17,16,13,14,15,22,23,24},
			   {2,0,3,10,11,12,21,20,19,9,8,7,26,25,27},
			   {0,1,3,16,17,18,12,11,10,4,5,6,28,29,30},
			   {1,0,2,7,8,9,15,14,13,6,5,4,32,31,33}},
		   /*o=-*/{{2,1,3,16,17,18,21,20,19,15,14,13,23,22,24},
			   {0,2,3,19,20,21,12,11,10,7,8,9,25,26,27},
			   {1,0,3,10,11,12,18,17,16,6,5,4,29,28,30},
			   {0,1,2,13,14,15,9,8,7,4,5,6,31,32,33}}}}};
#endif
#endif



/****************************************************************************/
/* Interface to the maintainers                                             */
/****************************************************************************/

/****************************************************************************/
/* AI_check_slavery(master): Do a consistency check for all submeshes.      */
/****************************************************************************/

void AI_check_slavery(MESH *master)
{
  FUNCNAME("AI_check_slavery");
  MESH_MEM_INFO   *m_mem_info, *s_mem_info;
  int              i, k, n_slaves, slave_el_count;
  MESH            *slave;
  DOF_PTR_VEC     *m_dpv, *s_dpv;
  TRAVERSE_STACK  *stack;
  const DOF_ADMIN *m_admin, *s_admin;
  const EL_INFO   *m_el_info, *s_el_info;
  const EL        *m_el, *s_el;

  if(!master) {
    MSG("No mesh provided!\n");
    return;
  }
  TEST_EXIT(m_mem_info = (MESH_MEM_INFO *)(master->mem_info),
    "No memory management present for \"%s\"!\n", master->name);
  n_slaves = m_mem_info->n_slaves;

  if(!n_slaves) {
    INFO(4,4,"Mesh \"%d\" has no slaves.\n", master->name);
    return;
  }

  stack = get_traverse_stack();

/****************************************************************************/
/* Run over all slave meshes.                                               */
/****************************************************************************/
  for(k = 0; k < n_slaves; k++) {
    slave = m_mem_info->slaves[k];
    TEST_EXIT(slave,"Slave mesh no. %d not found!\n", k);
    INFO(6,6,"Analysing slave \"%s\"...\n", slave->name);

    TEST_EXIT(slave->dim + 1 == master->dim,
      "Bad dimension of slave!\n");

    TEST_EXIT(s_mem_info = (MESH_MEM_INFO *)(slave->mem_info),
      "No memory management present for slave!\n");

    TEST_EXIT(s_mem_info->master == master,
      "Wrong mem_info->master pointer on slave!\n");
    TEST_EXIT(s_mem_info->binding_method,
      "No binding method present!\n");
    TEST_EXIT(m_dpv = s_mem_info->slave_binding,
      "No binding vector to slave present!\n");
    TEST_EXIT(s_dpv = s_mem_info->master_binding,
      "No binding vector to master present!\n");

    INFO(8,8,"Slave mesh has %d subslaves.\n", s_mem_info->n_slaves);
    
    m_admin = m_dpv->fe_space->admin;
    s_admin = s_dpv->fe_space->admin;

    INFO(10,10,"Current master leaf elements:\n");
    m_el_info = traverse_first(stack, master, -1,
			       CALL_LEAF_EL);
    while(m_el_info) {
      INFO(10,10,"%d\n", INDEX(m_el_info->el));
      m_el_info = traverse_next(stack, m_el_info);
    }

    INFO(10,10,"Current slave leaf elements:\n");
    s_el_info = traverse_first(stack, slave, -1,
			       CALL_LEAF_EL);
    while(s_el_info) {
      INFO(10,10,"%d\n", INDEX(s_el_info->el));
      s_el_info = traverse_next(stack, s_el_info);
    }

/****************************************************************************/
/* Run over the slave mesh and check correspondance to master.              */
/****************************************************************************/
    slave_el_count = 0;
    s_el_info = traverse_first(stack, slave, -1,
			       CALL_EVERY_EL_PREORDER);
    while(s_el_info) {
      slave_el_count++;
      s_el = s_el_info->el;
      INFO(10,10,"Analysing slave el %d...\n", INDEX(s_el));
      if(!IS_LEAF_EL(s_el))
	INFO(10,10,"(Child elements: %d, %d)\n", INDEX(s_el->child[0]),
		    INDEX(s_el->child[1]));
      TEST_EXIT(m_el = (EL *) s_dpv->vec[s_el->dof[slave->node[CENTER]]
					  [s_admin->n0_dof[CENTER]]],
"Slave element %d does not point to a master element!\n", INDEX(s_el));
      INFO(10,10,"slave el %d points to master el %d\n",
		  INDEX(s_el), INDEX(m_el));

      for(i = 0; i < N_NEIGH(master->dim); i++) {
	if(master->dim == 2) {
	 if(s_el == (EL *)m_dpv->vec[m_el->dof[master->node[EDGE]+i]
				     [m_admin->n0_dof[EDGE]]])
	   break;
	}
	else
	  if(s_el == (EL *)m_dpv->vec[m_el->dof[master->node[FACE]+i]
				      [m_admin->n0_dof[FACE]]])
	    break;
      }
      TEST_EXIT(i < N_NEIGH(master->dim),
	"Master element %d does not point back to slave element %d!\n",
	 INDEX(m_el), INDEX(s_el));
      
      s_el_info = traverse_next(stack, s_el_info);
    }      
    if(slave_el_count < slave->n_hier_elements)
      ERROR_EXIT("slave element count == %d < %d == slave->n_elements!\n",
		 slave_el_count, slave->n_elements);
    if(slave_el_count > slave->n_hier_elements)
      ERROR_EXIT("slave element count == %d > %d == slave->n_elements!\n",
		 slave_el_count, slave->n_elements);

/****************************************************************************/
/* Run over the current master mesh and check correspondance to slave.      */
/****************************************************************************/
    m_el_info = traverse_first(stack, master, -1,
			       CALL_EVERY_EL_PREORDER);
    while(m_el_info) {
      m_el = m_el_info->el;
      INFO(10,10,"Analysing master el %d...\n", INDEX(m_el));
      if(!IS_LEAF_EL(m_el))  
	INFO(10,10,"(Child elements: %d, %d)\n", INDEX(m_el->child[0]),
		    INDEX(m_el->child[1]));

      for(i = 0; i < N_NEIGH(master->dim); i++) {
	if(master->dim == 2)
	  s_el = (EL *)m_dpv->vec[m_el->dof[master->node[EDGE]+i]
				  [m_admin->n0_dof[EDGE]]];
	else
	  s_el = (EL *)m_dpv->vec[m_el->dof[master->node[FACE]+i]
				  [m_admin->n0_dof[FACE]]];
	if(s_el) {
	  INFO(10,10,"master el %d, subsimplex %d, points to slave el %d\n",
		      INDEX(m_el), i, INDEX(s_el));

	  if(IS_LEAF_EL(m_el)) 
	    TEST_EXIT(m_el == (EL *) s_dpv->vec[s_el->dof[slave->node[CENTER]]
					      [s_admin->n0_dof[CENTER]]],
	      "Slave element %d does not point back to master element %d!\n",
	       INDEX(s_el), INDEX(m_el));
	}
      }

      m_el_info = traverse_next(stack, m_el_info);
    }      
  }

  INFO(4,4,"No errors found.\n");
  free_traverse_stack(stack);

  return;
}

/****************************************************************************/
/* Interface to the user                                                    */
/****************************************************************************/


/****************************************************************************/
/* get_submesh(master, name, init_leaf_data, binding_method):               */
/* The main allocation routine for getting slave meshes.                    */
/****************************************************************************/

extern MESH *get_submesh(MESH *master, const char *name,
			 int (*binding_method)(MESH *master, 
					       MACRO_EL *el, int face,
					       void *data),
			 void *data)
{
  FUNCNAME("get_submesh");
  MESH *slave = NULL;

/****************************************************************************/
/* Do the checks for obvious user errors.                                   */
/****************************************************************************/

  TEST_EXIT(master,"No master mesh specified!\n");

  TEST_EXIT(master->dim > 0,
    "Does not make sense for dim 0 master meshes!\n");

  TEST_EXIT(binding_method,
    "Parameter 'binding_method' must be nonzero!\n");

/****************************************************************************/
/* Do the dimension dependent stuff for the slave.                          */
/****************************************************************************/

  if(master->dim == 1)
    slave = get_submesh_1d(master, name, binding_method, data);
#if DIM_OF_WORLD > 1
  else if (master->dim == 2)
    slave = get_submesh_2d(master, name, binding_method, data);
#if DIM_OF_WORLD == 3
  else
    slave = get_submesh_3d(master, name, binding_method, data);
#endif
#endif

/****************************************************************************/
/* Turn the submesh into a parametric mesh if necessary.                    */
/****************************************************************************/
  if(master->parametric)
    master->parametric->inherit_parametric(slave);

  return slave;
}  


/****************************************************************************/
/* unchain_submesh(slave): This routine destroys the ties between a slave   */
/* mesh and its master mesh. It DOES NOT destroy the slave mesh! That has to*/
/* be done by another call to free_mesh(). After this call, refinement and  */
/* coarsening of master and slave are independent, and there is no further  */
/* tie between the two meshes.                                              */
/****************************************************************************/

extern void unchain_submesh(MESH *slave)
{
  FUNCNAME("unchain_submesh");
  MESH           *master;
  const FE_SPACE *fe_space;
  MESH_MEM_INFO  *master_info, *slave_info;
  int             i;

/****************************************************************************/
/* Do the checks for obvious user errors.                                   */
/****************************************************************************/

  if(!slave) {
    ERROR("No slave mesh specified!\n");
    return;
  }

  slave_info = (MESH_MEM_INFO *)slave->mem_info;

  if(!(master = slave_info->master)) {
    ERROR("This mesh is not a slave mesh!\n");
    return;
  }

/****************************************************************************/
/* Look for the slave and the master mesh->mem_info.                        */
/****************************************************************************/

  master_info = (MESH_MEM_INFO *)master->mem_info;

  for(i = 0; i < master_info->n_slaves; i++)
    if(master_info->slaves[i] == slave)
      break;
  TEST_EXIT(i < master_info->n_slaves,
    "Could not find the slave mesh in slave vector!\n");

/****************************************************************************/
/* Resize the slaves vector on the master mesh->mem_info                    */
/****************************************************************************/

  for(; i < master_info->n_slaves - 1; i++)
    master_info->slaves[i] = master_info->slaves[i+1];

  if(master_info->n_slaves > 1)
    master_info->slaves = MEM_REALLOC(master_info->slaves,
				      master_info->n_slaves,
				      master_info->n_slaves - 1,
				      MESH *);
  else {
    MEM_FREE(master_info->slaves, 1, MESH *);

    master_info->slaves = nil;
  }
  master_info->n_slaves--;

/****************************************************************************/
/* Free the DOF_PTR_VECs and fe_spaces                                      */
/****************************************************************************/

  fe_space = slave_info->master_binding->fe_space;
  free_dof_ptr_vec(slave_info->master_binding);
  free_fe_space((FE_SPACE *)fe_space);

  fe_space = slave_info->slave_binding->fe_space;
  free_dof_ptr_vec(slave_info->slave_binding);
  free_fe_space((FE_SPACE *)fe_space);

/****************************************************************************/
/* Set the necessary pointers to nil, so that this mesh is now free.        */
/****************************************************************************/

  slave_info->master         = nil;
  slave_info->master_binding = nil;
  slave_info->slave_binding  = nil;
  slave_info->binding_method = nil;

  return;
}


extern MESH *read_submesh(MESH *master,
			  const char *slave_filename,
			  int (*binding_method)(MESH *master, 
						MACRO_EL *el, int face,
						void *data),
			  NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int),
			  void *data)
{
  return read_submesh_gen(false, master, slave_filename, binding_method,
			  n_proj, data);
}

extern MESH *read_submesh_xdr(MESH *master,
			      const char *slave_filename,
			      int (*binding_method)(MESH *master, 
						    MACRO_EL *el, int face,
						    void *data),
		       NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int),
			      void *data)
{
  return read_submesh_gen(true, master, slave_filename, binding_method,
			  n_proj, data);
}

/****************************************************************************/
/* trace_dof_real_vec[_d](slave_vec, master_vec):                           */
/* A routine for transporting the values of "master_vec" at DOFs along the  */
/* interface to a slave mesh to "slave_vec". This could be described as a   */
/* discrete trace operation.                                                */
/****************************************************************************/

void trace_dof_real_vec(DOF_REAL_VEC *slave_vec, DOF_REAL_VEC *master_vec)
{
  FUNCNAME("trace_dof_real_vec");

  MESH             *slave = nil, *master = nil;
  TRAVERSE_STACK   *m_stack = get_traverse_stack();
  EL_INFO           s_el_info = {};
  DOF_PTR_VEC      *s_dpv, *m_dpv;
  const DOF_ADMIN  *s_admin, *s_ptr_admin, *m_ptr_admin;
  const DOF      *(*s_get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const REAL     *(*s_interpol)(const EL_INFO *el_info, int n,
				const int *indices, 
				REAL (*f)(const REAL_D),
				REAL (*f_loc)(const EL_INFO *el_info,
					      const REAL lambda[N_LAMBDA]),
				REAL *coeff);
  int               s_n_bas_fcts, m_dim;

/****************************************************************************/
/* Check for user errors.                                                   */
/****************************************************************************/
  TEST_EXIT(slave_vec,"No slave DOF_REAL_VEC given!\n");
  TEST_EXIT(m_vec = master_vec,"No master DOF_REAL_VEC given!\n");

/****************************************************************************/
/* Set the necessary pointers and allocate two TRAVERSE_STACKS.             */
/****************************************************************************/
  slave   = slave_vec->fe_space->mesh;
  master  = master_vec->fe_space->mesh;

  TEST_EXIT(((MESH_MEM_INFO *)slave->mem_info)->master == master,
    "Master and slave vectors do not seem to fit together!\n");

  m_dim   = master->dim;

  s_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->master_binding;
  m_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->slave_binding;

  s_admin     = slave_vec->fe_space->admin;
  s_ptr_admin = s_dpv->fe_space->admin;
  m_ptr_admin = m_dpv->fe_space->admin;

  s_n_bas_fcts = slave_vec->fe_space->bas_fcts->n_bas_fcts;

  TEST_EXIT(s_interpol = slave_vec->fe_space->bas_fcts->interpol,
    "Interpolation routine not present for slave vector!\n");

  TEST_EXIT(s_get_dof_indices = slave_vec->fe_space->bas_fcts->get_dof_indices,
    "'get_dof_indices' routine not present for slave vector!\n");

  TEST_EXIT(get_local_m_vec = master_vec->fe_space->bas_fcts->get_real_vec,
    "'get_real_vec' routine not present for master vector!\n");
    
  m_el_info      = traverse_first(m_stack, master, -1,
				  CALL_LEAF_EL|FILL_ORIENTATION|FILL_EL_TYPE);
  s_el_info.mesh = slave;

  {
    int i;
    REAL s_local_vec[s_n_bas_fcts];
    DOF  s_local_dofs[s_n_bas_fcts];

    while(m_el_info) {
      for(m_subsimplex = 0; m_subsimplex < N_NEIGH(m_dim); m_subsimplex++) {
/****************************************************************************/
/* Look for a slave element on edge/face m_subsimplex.                     */
/****************************************************************************/
	switch(m_dim) {
	case 1:
	  if((s_el_info.el = (EL *) 
	     m_dpv->vec[m_el_info->el->dof[master->node[VERTEX] + m_subsimplex]
			[m_ptr_admin->n0_dof[VERTEX]]]))
	  break;
	case 2:
	  if((s_el_info.el = (EL *) 
	     m_dpv->vec[m_el_info->el->dof[master->node[EDGE] + m_subsimplex]
			[m_ptr_admin->n0_dof[EDGE]]]))
	    goto found_slave_element;
	  break;
	case 3:
	  if((s_el_info.el = (EL *)
	     m_dpv->vec[m_el_info->el->dof[master->node[FACE] + m_subsimplex]
			[m_ptr_admin->n0_dof[FACE]]]))
	    goto found_slave_element;
	  break;
	}
	continue;

      found_slave_element:
/****************************************************************************/
/* Interpolate the master vector into s_local_vec and copy these values     */
/* to the correct position in slave_vec->vec.                               */
/****************************************************************************/
	s_get_dof_indices(s_el_info.el, s_admin, s_local_dofs);
	s_interpol(&s_el_info, 0, nil, nil, trace_loc, s_local_vec);
	
	for(i = 0; i < s_n_bas_fcts; i++)
	  slave_vec->vec[s_local_dofs[i]] = s_local_vec[i];
      }

/****************************************************************************/
/* Advance the slave stack.                                                 */
/****************************************************************************/
      m_el_info = traverse_next(m_stack, m_el_info);
    }
  }


/****************************************************************************/
/* Clean up.                                                                */
/****************************************************************************/
  free_traverse_stack(m_stack);
  
  return;
}


void trace_dof_real_d_vec(DOF_REAL_D_VEC *slave_vec,
			  DOF_REAL_D_VEC *master_vec)
{
  FUNCNAME("trace_dof_real_d_vec");

  MESH             *slave = nil, *master = nil;
  TRAVERSE_STACK   *m_stack = get_traverse_stack();
  EL_INFO           s_el_info = {};
  DOF_PTR_VEC      *s_dpv, *m_dpv;
  const DOF_ADMIN  *s_admin, *s_ptr_admin, *m_ptr_admin;
  const DOF      *(*s_get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const REAL_D   *(*s_interpol_d)(const EL_INFO *el_info, int n,
				  const int *indices,
				  const REAL *(*f)(const REAL_D, REAL_D),
			 const REAL *(*f_loc)(const EL_INFO *el_info,
					      const REAL lambda[N_LAMBDA],
					      REAL_D val),
				  REAL_D *coeff);
  int               s_n_bas_fcts, m_dim;

/****************************************************************************/
/* Check for user errors.                                                   */
/****************************************************************************/
  TEST_EXIT(slave_vec,"No slave DOF_REAL_D_VEC given!\n");
  TEST_EXIT(m_vec_d = master_vec,"No master DOF_REAL_D_VEC given!\n");

/****************************************************************************/
/* Set the necessary pointers and allocate two TRAVERSE_STACKS.             */
/****************************************************************************/
  slave   = slave_vec->fe_space->mesh;
  master  = master_vec->fe_space->mesh;

  TEST_EXIT(((MESH_MEM_INFO *)slave->mem_info)->master == master,
    "Master and slave vectors do not seem to fit together!\n");

  m_dim   = master->dim;

  s_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->master_binding;
  m_dpv   = ((MESH_MEM_INFO *)slave->mem_info)->slave_binding;

  s_admin     = slave_vec->fe_space->admin;
  s_ptr_admin = s_dpv->fe_space->admin;
  m_ptr_admin = m_dpv->fe_space->admin;

  s_n_bas_fcts = slave_vec->fe_space->bas_fcts->n_bas_fcts;

  TEST_EXIT(s_interpol_d = slave_vec->fe_space->bas_fcts->interpol_d,
    "Interpolation routine not present for slave vector!\n");

  TEST_EXIT(s_get_dof_indices = slave_vec->fe_space->bas_fcts->get_dof_indices,
    "'get_dof_indices' routine not present for slave vector!\n");

  TEST_EXIT(get_local_m_vec_d = master_vec->fe_space->bas_fcts->get_real_d_vec,
    "'get_real_d_vec' routine not present for master vector!\n");
    
  m_el_info      = traverse_first(m_stack, master, -1,
				  CALL_LEAF_EL|FILL_ORIENTATION|FILL_EL_TYPE);
  s_el_info.mesh = slave;

  {
    int i, j;
    REAL_D       s_local_vec_d[s_n_bas_fcts];
    DOF          s_local_dofs[s_n_bas_fcts];

    while(m_el_info) {
      for(m_subsimplex = 0; m_subsimplex < N_NEIGH(m_dim); m_subsimplex++) {
/****************************************************************************/
/* Look for a slave element on edge/face m_subsimplex.                      */
/****************************************************************************/
	switch(m_dim) {
	case 1:
	  if((s_el_info.el = (EL *) 
	     m_dpv->vec[m_el_info->el->dof[master->node[VERTEX] + m_subsimplex]
			[m_ptr_admin->n0_dof[VERTEX]]]))
	  break;
	case 2:
	  if((s_el_info.el = (EL *) 
	     m_dpv->vec[m_el_info->el->dof[master->node[EDGE] + m_subsimplex]
			[m_ptr_admin->n0_dof[EDGE]]]))
	    goto found_slave_element;
	  break;
	case 3:
	  if((s_el_info.el = (EL *)
	     m_dpv->vec[m_el_info->el->dof[master->node[FACE] + m_subsimplex]
			[m_ptr_admin->n0_dof[FACE]]]))
	    goto found_slave_element;
	  break;
	}
	continue;
	
      found_slave_element:
/****************************************************************************/
/* Interpolate the master vector into s_local_vec and copy these values     */
/* to the correct position in slave_vec->vec.                               */
/****************************************************************************/
	s_get_dof_indices(s_el_info.el, s_admin, s_local_dofs);
	s_interpol_d(&s_el_info, 0, nil, nil, trace_loc_d, s_local_vec_d);

	for(i = 0; i < s_n_bas_fcts; i++)
	  for(j = 0; j < DIM_OF_WORLD; j++)
	    slave_vec->vec[s_local_dofs[i]][j] = s_local_vec_d[i][j];
      }

/****************************************************************************/
/* Advance the slave stack.                                                 */
/****************************************************************************/
      m_el_info = traverse_next(m_stack, m_el_info);
    }
  }

/****************************************************************************/
/* Clean up.                                                                */
/****************************************************************************/
  free_traverse_stack(m_stack);
  
  return;
}



/****************************************************************************/
/* get_slave_dof_mapping(m_fe_space, s_map): Fill s_map on the slave mesh   */
/* with the corresponding DOFs of m_fe_space. This is only implemented for  */
/* Lagrange FE spaces. Master and slave fe_spaces must have the same degree.*/
/*                                                                          */
/* s_map is not updated during mesh or DOF (dof_compress(master)) changes!! */
/****************************************************************************/

void get_slave_dof_mapping(const FE_SPACE *m_fe_space,
			   DOF_INT_VEC *s_map)
{
  FUNCNAME("get_slave_dof_mapping");
  MESH             *master;
  MESH             *slave;
  const FE_SPACE   *s_fe_space;
  const DOF_ADMIN  *m_admin, *s_admin;
  int               degree;
  DOF_PTR_VEC      *m_dpv, *s_dpv;
  TRAVERSE_STACK   *m_stack = get_traverse_stack();
  const EL_INFO    *m_el_info;
  const DOF      *(*m_get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF      *(*s_get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  DOF              *m_local_dofs, *s_local_dofs;
  int               m_dim, m_n_dofs, s_n_dofs, m_subsimplex;
  int               m_n0, s_n0, m_n, s_n;
  const EL         *s_el, *m_el;
  FLAGS             which_elements;

/****************************************************************************/
/* Check for user errors.                                                   */
/****************************************************************************/
  TEST_EXIT(m_fe_space,"No master FE_SPACE given!\n");
  TEST_EXIT(s_map,"No DOF_INT_VEC s_map given!\n");

  s_fe_space = s_map->fe_space;

  TEST_EXIT(s_fe_space,"No slave FE_SPACE found!\n");

  m_admin = m_fe_space->admin;
  s_admin = s_fe_space->admin;

  master = m_fe_space->mesh;
  slave  = s_fe_space->mesh;
  m_dim  = master->dim;

  TEST_EXIT(((MESH_MEM_INFO *)slave->mem_info)->master == master,
    "Master and slave meshes do not seem to belong together!\n");
  TEST_EXIT(strstr(m_fe_space->bas_fcts->name, "lagrange")
	    && strstr(s_fe_space->bas_fcts->name, "lagrange"),
    "Sorry, only implemented for Lagrange Finite Elements!\n");

  degree = m_fe_space->bas_fcts->degree;
  if(m_dim > 1) {
    TEST_EXIT(degree == s_fe_space->bas_fcts->degree,
      "Degree of slave elements == %d != %d == degree of master elements!\n",
       s_fe_space->bas_fcts->degree, degree);
  }
  TEST_EXIT(degree > 0,
    "Degree must be at least 1!\n");

  TEST_EXIT(m_admin->preserve_coarse_dofs == s_admin->preserve_coarse_dofs,
    "preserve_coarse_dofs entries are not equal!\n");
  if(s_admin->preserve_coarse_dofs)
    which_elements = CALL_EVERY_EL_PREORDER;
  else
    which_elements = CALL_LEAF_EL;

/****************************************************************************/
/* Initialize values.                                                       */
/****************************************************************************/

  /* Mark everything as unused. */
  FOR_ALL_DOFS(s_admin, s_map->vec[dof] = -1);

  m_dpv             = ((MESH_MEM_INFO *)slave->mem_info)->slave_binding;
  s_dpv             = ((MESH_MEM_INFO *)slave->mem_info)->master_binding;
  m_get_dof_indices = m_fe_space->bas_fcts->get_dof_indices;
  s_get_dof_indices = s_fe_space->bas_fcts->get_dof_indices;

  s_n0              = s_dpv->fe_space->admin->n0_dof[CENTER];
  s_n               = slave->node[CENTER];

  m_n_dofs = m_fe_space->bas_fcts->n_bas_fcts;
  s_n_dofs = s_fe_space->bas_fcts->n_bas_fcts;

  m_local_dofs = MEM_ALLOC(m_n_dofs, DOF);
  s_local_dofs = MEM_ALLOC(s_n_dofs, DOF);

  switch(m_dim) {
  case 1:
    m_n0 = m_dpv->fe_space->admin->n0_dof[VERTEX];
    m_n  = master->node[VERTEX];

    for(m_el_info = traverse_first(m_stack, master, -1, which_elements);
	m_el_info;
	m_el_info = traverse_next(m_stack, m_el_info)) {

      m_el = m_el_info->el;
      m_get_dof_indices(m_el, m_admin, m_local_dofs);
      
      for(m_subsimplex = 0; m_subsimplex < N_NEIGH_1D; m_subsimplex++) {
	s_el = (EL *)m_dpv->vec[m_el->dof[m_n + m_subsimplex][m_n0]];
	if(s_el && m_el==(EL *)s_dpv->vec[s_el->dof[s_n][s_n0]]) {
	  s_get_dof_indices(s_el, s_admin, s_local_dofs);
	  s_map->vec[s_local_dofs[0]] = m_local_dofs[m_subsimplex];
	}
      }
    }
    break;
#if DIM_OF_WORLD > 1
  case 2:
    m_n0 = m_dpv->fe_space->admin->n0_dof[EDGE];
    m_n = master->node[EDGE];

    for(m_el_info = traverse_first(m_stack, master, -1, which_elements);
	m_el_info;
	m_el_info = traverse_next(m_stack, m_el_info)) {
      
      m_el = m_el_info->el;
      m_get_dof_indices(m_el, m_admin, m_local_dofs);
    
      for(m_subsimplex = 0; m_subsimplex < N_NEIGH_2D; m_subsimplex++) {
	int i;

	s_el = (EL *)m_dpv->vec[m_el->dof[m_n + m_subsimplex][m_n0]];
	if(s_el && m_el==(EL *)s_dpv->vec[s_el->dof[s_n][s_n0]]) {
	  s_get_dof_indices(s_el, s_admin, s_local_dofs);

	  for(i = 0; i < s_n_dofs; i++)
	    s_map->vec[s_local_dofs[i]] =  
	      m_local_dofs[sm_dof_connection_2d[degree-1][m_subsimplex][i]];
	}
      }
    }
    break;
#if DIM_OF_WORLD > 2
  case 3:
    m_n0 = m_dpv->fe_space->admin->n0_dof[FACE];
    m_n = master->node[FACE];

    for(m_el_info = traverse_first(m_stack, master, -1,
			   which_elements|FILL_ORIENTATION|FILL_EL_TYPE);
	m_el_info;
	m_el_info = traverse_next(m_stack, m_el_info)) {
      int type, i;
      
      m_el = m_el_info->el;
      m_get_dof_indices(m_el, m_admin, m_local_dofs);

      if(m_el_info->el_type == 0)
	type = 0;
      else
	type = 1;
      
      for(m_subsimplex = 0; m_subsimplex < N_NEIGH_3D; m_subsimplex++) {
	s_el = (EL *)m_dpv->vec[m_el->dof[m_n + m_subsimplex][m_n0]];
	if(s_el && m_el==(EL *)s_dpv->vec[s_el->dof[s_n][s_n0]]) {
	  s_get_dof_indices(s_el, s_admin, s_local_dofs);

	  for(i = 0; i < s_n_dofs; i++) {
	    if(m_el_info->orientation > 0)
	      s_map->vec[s_local_dofs[i]] =  
	m_local_dofs[sm_dof_connection_3d[degree-1][type][0][m_subsimplex][i]];
	    else
	      s_map->vec[s_local_dofs[i]] =  
	m_local_dofs[sm_dof_connection_3d[degree-1][type][1][m_subsimplex][i]];
	  }
	}
      }
    }
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal dimension!\n");
  }

/****************************************************************************/
/* Clean up and return.                                                     */
/****************************************************************************/
  free_traverse_stack(m_stack);
  MEM_FREE(m_local_dofs, m_n_dofs, DOF);
  MEM_FREE(s_local_dofs, s_n_dofs, DOF);
 
  return;
}


/****************************************************************************/
/* get_master(slave): Return the master of slave if present.                */
/****************************************************************************/

extern MESH *get_master(MESH *slave)
{
  return ((MESH_MEM_INFO *)slave->mem_info)->master;
}


/****************************************************************************/
/* get_master_binding(slave): Return the vector giving pointers to master   */
/*                            elements.                                     */
/****************************************************************************/

extern DOF_PTR_VEC *get_master_binding(MESH *slave)
{
  return ((MESH_MEM_INFO *)slave->mem_info)->master_binding;
}


/****************************************************************************/
/* get_master_binding(slave): Return the vector giving pointers to slave    */
/*                            elements.                                     */
/****************************************************************************/

extern DOF_PTR_VEC *get_slave_binding(MESH *slave)
{
  return ((MESH_MEM_INFO *)slave->mem_info)->slave_binding;
}
