/*
    SPDX-FileCopyrightText: 2003-2006 Cies Breijs <cies AT kde DOT nl>

    SPDX-License-Identifier: GPL-2.0-or-later
*/

#include "translator.h"

#include <KLocalizedString>
#include <QRegularExpression>
#include "token.h"


Translator* Translator::m_instance = nullptr;  // initialize pointer

Translator* Translator::instance()
{
	if (m_instance == nullptr) m_instance = new Translator;  // create sole instance if its the first call
	return m_instance;  // address of sole instance
}

Translator::Translator()
    : localizer(QStringList() << QLatin1String(DEFAULT_LANGUAGE_CODE))
{
}

Translator::~Translator()
{
	delete m_instance;
	m_instance = nullptr;
}


int Translator::look2type(QString& look)
{
	if (look2typeMap.contains(look))
		return look2typeMap[look];
	return Token::Unknown;
}

int Translator::look2type(QChar& look)
{
	if (look2typeMap.contains(static_cast<QString>(look)))
		return look2typeMap[static_cast<QString>(look)];
	return Token::Unknown;
}

QList<QString> Translator::type2look(int type)
{
	return look2typeMap.keys(type);
}

QHash<int, QList<QString> > Translator::token2stringsMap()
{
	QHash<int, QList<QString> > resultMap;
    const QList<int> tokenList = look2typeMap.values();
    for (int token : tokenList) resultMap.insert(token, look2typeMap.keys(token));
	return resultMap;
}


bool Translator::setLanguage(const QString &lang_code)
{
    // FIXME default to GUI language? return false when language not available?
    localizer = QStringList() << lang_code << QLatin1String(DEFAULT_LANGUAGE_CODE);

	setDictionary();
	setExamples();

	return true;
}


void Translator::setDictionary()
{
	look2typeMap.clear();
	default2localizedMap.clear();

	QString localizedCommandLook;


//BEGIN GENERATED translator_cpp CODE

/* The code between the line that start with "//BEGIN GENERATED" and "//END GENERATED"
 * is generated by "generate.rb" according to the definitions specified in
 * "definitions.rb". Please make all changes in the "definitions.rb" file, since all
 * all change you make here will be overwritten the next time "generate.rb" is run.
 * Thanks for looking at the code!
 */

	look2typeMap[QStringLiteral("$")] = Token::VariablePrefix;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'True' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"true").toString(localizer);
	default2localizedMap[QStringLiteral("true")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::True;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'False' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"false").toString(localizer);
	default2localizedMap[QStringLiteral("false")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::False;

	look2typeMap[QStringLiteral("#")] = Token::Comment;

	look2typeMap[QStringLiteral("\"")] = Token::StringDelimiter;

	look2typeMap[QStringLiteral("{")] = Token::ScopeOpen;

	look2typeMap[QStringLiteral("}")] = Token::ScopeClose;

	look2typeMap[QStringLiteral("(")] = Token::ParenthesisOpen;

	look2typeMap[QStringLiteral(")")] = Token::ParenthesisClose;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'ArgumentSeparator' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		",").toString(localizer);
	default2localizedMap[QStringLiteral(",")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::ArgumentSeparator;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'DecimalSeparator' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		".").toString(localizer);
	default2localizedMap[QStringLiteral(".")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::DecimalSeparator;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Exit' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"exit").toString(localizer);
	default2localizedMap[QStringLiteral("exit")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Exit;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'If' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"if").toString(localizer);
	default2localizedMap[QStringLiteral("if")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::If;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Else' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"else").toString(localizer);
	default2localizedMap[QStringLiteral("else")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Else;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Repeat' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"repeat").toString(localizer);
	default2localizedMap[QStringLiteral("repeat")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Repeat;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'While' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"while").toString(localizer);
	default2localizedMap[QStringLiteral("while")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::While;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'For' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"for").toString(localizer);
	default2localizedMap[QStringLiteral("for")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::For;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'To' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"to").toString(localizer);
	default2localizedMap[QStringLiteral("to")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::To;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Step' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"step").toString(localizer);
	default2localizedMap[QStringLiteral("step")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Step;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Break' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"break").toString(localizer);
	default2localizedMap[QStringLiteral("break")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Break;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Return' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"return").toString(localizer);
	default2localizedMap[QStringLiteral("return")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Return;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Wait' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"wait").toString(localizer);
	default2localizedMap[QStringLiteral("wait")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Wait;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Assert' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"assert").toString(localizer);
	default2localizedMap[QStringLiteral("assert")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Assert;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'And' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"and").toString(localizer);
	default2localizedMap[QStringLiteral("and")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::And;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Or' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"or").toString(localizer);
	default2localizedMap[QStringLiteral("or")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Or;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Not' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"not").toString(localizer);
	default2localizedMap[QStringLiteral("not")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Not;

	look2typeMap[QStringLiteral("==")] = Token::Equals;

	look2typeMap[QStringLiteral("!=")] = Token::NotEquals;

	look2typeMap[QStringLiteral(">")] = Token::GreaterThan;

	look2typeMap[QStringLiteral("<")] = Token::LessThan;

	look2typeMap[QStringLiteral(">=")] = Token::GreaterOrEquals;

	look2typeMap[QStringLiteral("<=")] = Token::LessOrEquals;

	look2typeMap[QStringLiteral("+")] = Token::Addition;

	look2typeMap[QStringLiteral("-")] = Token::Subtraction;

	look2typeMap[QStringLiteral("*")] = Token::Multiplication;

	look2typeMap[QStringLiteral("/")] = Token::Division;

	look2typeMap[QStringLiteral("^")] = Token::Power;

	look2typeMap[QStringLiteral("=")] = Token::Assign;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Learn' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"learn").toString(localizer);
	default2localizedMap[QStringLiteral("learn")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Learn;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Reset' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"reset").toString(localizer);
	default2localizedMap[QStringLiteral("reset")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Reset;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Clear' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"clear").toString(localizer);
	default2localizedMap[QStringLiteral("clear")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Clear;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Clear' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"ccl").toString(localizer);
	default2localizedMap[QStringLiteral("ccl")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Clear;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Center' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"center").toString(localizer);
	default2localizedMap[QStringLiteral("center")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Center;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Go' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"go").toString(localizer);
	default2localizedMap[QStringLiteral("go")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Go;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GoX' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"gox").toString(localizer);
	default2localizedMap[QStringLiteral("gox")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GoX;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GoX' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"gx").toString(localizer);
	default2localizedMap[QStringLiteral("gx")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GoX;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GoY' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"goy").toString(localizer);
	default2localizedMap[QStringLiteral("goy")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GoY;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GoY' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"gy").toString(localizer);
	default2localizedMap[QStringLiteral("gy")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GoY;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Forward' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"forward").toString(localizer);
	default2localizedMap[QStringLiteral("forward")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Forward;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Forward' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"fw").toString(localizer);
	default2localizedMap[QStringLiteral("fw")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Forward;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Backward' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"backward").toString(localizer);
	default2localizedMap[QStringLiteral("backward")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Backward;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Backward' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"bw").toString(localizer);
	default2localizedMap[QStringLiteral("bw")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Backward;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Direction' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"direction").toString(localizer);
	default2localizedMap[QStringLiteral("direction")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Direction;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Direction' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"dir").toString(localizer);
	default2localizedMap[QStringLiteral("dir")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Direction;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'TurnLeft' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"turnleft").toString(localizer);
	default2localizedMap[QStringLiteral("turnleft")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::TurnLeft;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'TurnLeft' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"tl").toString(localizer);
	default2localizedMap[QStringLiteral("tl")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::TurnLeft;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'TurnRight' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"turnright").toString(localizer);
	default2localizedMap[QStringLiteral("turnright")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::TurnRight;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'TurnRight' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"tr").toString(localizer);
	default2localizedMap[QStringLiteral("tr")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::TurnRight;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenWidth' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"penwidth").toString(localizer);
	default2localizedMap[QStringLiteral("penwidth")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenWidth;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenWidth' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pw").toString(localizer);
	default2localizedMap[QStringLiteral("pw")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenWidth;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenUp' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"penup").toString(localizer);
	default2localizedMap[QStringLiteral("penup")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenUp;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenUp' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pu").toString(localizer);
	default2localizedMap[QStringLiteral("pu")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenUp;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenDown' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pendown").toString(localizer);
	default2localizedMap[QStringLiteral("pendown")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenDown;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenDown' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pd").toString(localizer);
	default2localizedMap[QStringLiteral("pd")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenDown;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenColor' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pencolor").toString(localizer);
	default2localizedMap[QStringLiteral("pencolor")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenColor;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'PenColor' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pc").toString(localizer);
	default2localizedMap[QStringLiteral("pc")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::PenColor;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'CanvasColor' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"canvascolor").toString(localizer);
	default2localizedMap[QStringLiteral("canvascolor")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::CanvasColor;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'CanvasColor' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"cc").toString(localizer);
	default2localizedMap[QStringLiteral("cc")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::CanvasColor;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'CanvasSize' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"canvassize").toString(localizer);
	default2localizedMap[QStringLiteral("canvassize")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::CanvasSize;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'CanvasSize' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"cs").toString(localizer);
	default2localizedMap[QStringLiteral("cs")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::CanvasSize;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'SpriteShow' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"spriteshow").toString(localizer);
	default2localizedMap[QStringLiteral("spriteshow")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::SpriteShow;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'SpriteShow' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"ss").toString(localizer);
	default2localizedMap[QStringLiteral("ss")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::SpriteShow;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'SpriteHide' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"spritehide").toString(localizer);
	default2localizedMap[QStringLiteral("spritehide")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::SpriteHide;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'SpriteHide' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"sh").toString(localizer);
	default2localizedMap[QStringLiteral("sh")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::SpriteHide;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Print' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"print").toString(localizer);
	default2localizedMap[QStringLiteral("print")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Print;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'FontSize' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"fontsize").toString(localizer);
	default2localizedMap[QStringLiteral("fontsize")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::FontSize;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Random' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"random").toString(localizer);
	default2localizedMap[QStringLiteral("random")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Random;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Random' COMMAND ALIAS, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"rnd").toString(localizer);
	default2localizedMap[QStringLiteral("rnd")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Random;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GetX' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"getx").toString(localizer);
	default2localizedMap[QStringLiteral("getx")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GetX;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GetY' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"gety").toString(localizer);
	default2localizedMap[QStringLiteral("gety")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GetY;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Message' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"message").toString(localizer);
	default2localizedMap[QStringLiteral("message")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Message;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Ask' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"ask").toString(localizer);
	default2localizedMap[QStringLiteral("ask")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Ask;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Pi' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"pi").toString(localizer);
	default2localizedMap[QStringLiteral("pi")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Pi;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Tan' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"tan").toString(localizer);
	default2localizedMap[QStringLiteral("tan")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Tan;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Sin' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"sin").toString(localizer);
	default2localizedMap[QStringLiteral("sin")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Sin;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Cos' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"cos").toString(localizer);
	default2localizedMap[QStringLiteral("cos")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Cos;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'ArcTan' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"arctan").toString(localizer);
	default2localizedMap[QStringLiteral("arctan")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::ArcTan;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'ArcSin' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"arcsin").toString(localizer);
	default2localizedMap[QStringLiteral("arcsin")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::ArcSin;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'ArcCos' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"arccos").toString(localizer);
	default2localizedMap[QStringLiteral("arccos")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::ArcCos;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Sqrt' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"sqrt").toString(localizer);
	default2localizedMap[QStringLiteral("sqrt")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Sqrt;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Round' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"round").toString(localizer);
	default2localizedMap[QStringLiteral("round")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Round;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'GetDirection' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"getdirection").toString(localizer);
	default2localizedMap[QStringLiteral("getdirection")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::GetDirection;

	localizedCommandLook = ki18nc(
		"You are about to translate the 'Mod' COMMAND, there are some rules on how to translate it."
		"Please see https://edu.kde.org/kturtle/translator.php to learn how to properly translate it.",
		"mod").toString(localizer);
	default2localizedMap[QStringLiteral("mod")] = localizedCommandLook;
	look2typeMap[localizedCommandLook] = Token::Mod;


//END GENERATED translator_cpp CODE

}



void Translator::setExamples()
{
	examples.clear();
	QString exampleName;

	exampleName = ki18nc(
		"This is an EXAMPLE NAME in KTurtle."
		"Please see https://edu.kde.org/kturtle/translator.php to learn know how to properly translate it.",
		"triangle").toString(localizer);
    examples[exampleName] = localizeScript(QStringLiteral(
				"@(reset)\n"
				"@(repeat) 3 {\n"
				"  @(forward) 100\n"
				"  @(turnleft) 120\n"
				"}\n"
		));

	exampleName = ki18nc(
		"This is an EXAMPLE NAME in KTurtle."
		"Please see https://edu.kde.org/kturtle/translator.php to learn know how to properly translate it.",
		"curly").toString(localizer);
    examples[exampleName] = localizeScript(QStringLiteral(
				"@(reset)\n"
				"@(penup)\n"
				"@(forward) 50\n"
				"@(pendown)\n"
				"\n"
				"@(repeat) 4 {\n"
				"  @(for) $x = 1 @(to) 100 {\n"
				"    @(forward) 10\n"
				"    @(turnright) 100 - $x\n"
				"  }\n"
				"}\n"
		));

	exampleName = ki18nc(
		"This is an EXAMPLE NAME in KTurtle."
		"Please see https://edu.kde.org/kturtle/translator.php to learn know how to properly translate it.",
		"arrow").toString(localizer);
    examples[exampleName] = localizeScript(QStringLiteral(
				"@(reset)\n"
				"\n"
				"@(canvassize) 200@(,) 200\n"
				"@(canvascolor) 0@(,) 0@(,) 0\n"
				"@(pencolor) 255@(,) 0@(,) 0\n"
				"@(penwidth) 5\n"
				"\n"
				"@(go) 20@(,)20\n"
				"@(direction) 135\n"
				"\n"
				"@(forward) 200\n"
				"@(turnleft) 135\n"
				"@(forward) 100\n"
				"@(turnleft) 135\n"
				"@(forward) 141\n"
				"@(turnleft) 135\n"
				"@(forward) 100\n"
				"@(turnleft) 45\n"
				"\n"
				"@(go) 40@(,) 100"
		));

	exampleName = ki18nc(
		"This is an EXAMPLE NAME in KTurtle."
		"Please see https://edu.kde.org/kturtle/translator.php to learn know how to properly translate it.",
		  "flower").toString(localizer);
    examples[exampleName] = localizeScript(QStringLiteral(
				"@(reset)\n"
				"@(canvascolor) 255@(,) 55@(,) 140\n"
				"@(pencolor) 160@(,) 0@(,) 255\n"
				"@(penwidth) 3\n"
				"\n"
				"@(repeat) 8 {\n"
				"  @(repeat) 4 {\n"
				"    @(forward) 20\n"
				"    @(turnright) 30\n"
				"  }\n"
				"  @(repeat) 7 {\n"
				"    @(forward) 10\n"
				"    @(turnright) 15\n"
				"  }\n"
				"  @(repeat) 9 {\n"
				"    @(forward) 3\n"
				"    @(turnright) 10\n"
				"  }\n"
				"}\n"
				"\n"
				"@(go) 145@(,) 145\n"
				"@(direction) 0"
		));

}

QString Translator::localizeScript(const QString& untranslatedScript)
{
	QString result = untranslatedScript;
    Translator* translator = Translator::instance();
    QRegularExpression rx(QStringLiteral("@\\((.*?)\\)"));

	int pos = 0;
	while (true) {
		const auto match = rx.match(result, pos);
		if (!match.hasMatch()) {
			break;
		}
		pos = match.capturedStart();
		result.replace(pos, match.capturedLength(), translator->default2localized(match.captured(1)));
	}

	return result;
}
