/*
 *  NASPRO - The NASPRO Architecture for Sound Processing
 *  Common headers for all ino implementations
 *
 *  Copyright (C) 2011, 2012 ino development team
 *
 *  See the COPYING file for license conditions.
 */

/*
   Title: ino

   *Version*: 0.4.0, *API*: 0.0.0.

   About:

     <ino at http://naspro.atheme.org/libraries/ino/> is a minimalist C API to
     execute JavaScript code and to expose native methods to JavaScript
     execution contexts.

     The code is released under an ISC-style license (see the COPYING file).

   API conventions:

     * It is not thread-safe to concurrently operate on the same <ino_context>,
       unless otherwise specified;
     * All strings are null-terminated, UTF-8 or UTF-16 encoded according to the
       Unicode 6.0 standard and without BOM (byte-order-mark) characters, unless
       otherwise specified;
     * <ino_variant> objects are automatically garbage collected or otherwise
       destroyed when <ino_context_free()> is called for the execution context
       they belong to;
     * No function does input validation, hence, in case the API is misused in
       this sense, the results are undefined.

   Build-time dependencies:

     * An environment capable of running Autotools-based build systems;
     * (optional) <GNU Autoconf at http://www.gnu.org/software/autoconf/> >=
       2.68 and <GNU Automake at http://www.gnu.org/software/automake/> to
       regenerate the build system;
     * (optional) <Natural Docs at http://www.naturaldocs.org/> >= 1.5 to
       regenerate the build system and/or the documentation.

   Usage:

     Both ino implementations and applications/libraries using them are meant to
     be including <ino/lib.h> in their source files.

     Furthermore, ino implementations also need to #define the
     INO_IMPLEMENTATION flag before such inclusion and should provide suitable
     <pkg-config at http://pkg-config.freedesktop.org/> files for the build
     systems of applications/libraries to retrieve compiler and linker flags.

     This package itself supplies a pkg-config file
     (package name: ino-headers-0) specifying compiler flags for the inclusion
     of headers.
 */

#ifndef _INO_LIB_H
#define _INO_LIB_H

#include <stddef.h>
#include <stdbool.h>
#include <stdint.h>

/*
   Group: Global Declaration Macros

   These macros are meant to be used to delimit declarations in public header
   files.

   Example:

     myheader.h

     > #ifndef MY_HEADER_H
     > #define MY_HEADER_H
     >
     > INO_BEGIN_C_DECLS
     >
     > ... blah blah blah ...
     >
     > INO_END_C_DECLS
     >
     > #endif
 */

/*
   Macro: INO_BEGIN_C_DECLS

   Delimits the beginning of public declarations.

   So that C++ compilers don't mangle their names.
 */

/*
   Macro: INO_END_C_DECLS

   Delimits the end of public declarations.

   So that C++ compilers don't mangle their names.
 */

/*
   Group: Symbol Declaration Macros

   These macros are meant to be used for *SINGLE* symbol declarations (variables
   or functions) by prepending them to such declarations.

   They can be combined together, unless otherwise told.

   Example:

     myfunc.h

     > INO_PUBLIC void myfunc();

     myfunc.c

     > INO_PUBLIC void
     > myfunc()
     > {
     >         ... blah blah blah ...
     > }
 */

/*
   Macro: INO_IMPORT

   Specifies that a symbol is imported from a library.

   Cannot be combined with <INO_EXPORT>.
 */

/*
   Macro: INO_EXPORT

   Specifies that a symbol is to be exported.

   Cannot be combined with <INO_IMPORT>.
 */

/*
   Macro: INO_PUBLIC

   Specifies that a symbol is publicly visible.

   Cannot be combined with <INO_PRIVATE>.
 */

/*
   Macro: INO_PRIVATE

   Specifies that a symbol has hidden visibility.

   Cannot be combined with <INO_PUBLIC>.
 */

/*
   Macro: INO_DEF

   Equivalent to a combination of <INO_PUBLIC> and <INO_EXPORT> if the
   INO_IMPLEMENTATION flag is defined or to a combination of <INO_PRIVATE> and
   <INO_IMPORT> otherwise.
 */

#ifdef __cplusplus
# define INO_BEGIN_C_DECLS	extern "C" {
# define INO_END_C_DECLS	}
#else
# define INO_BEGIN_C_DECLS
# define INO_END_C_DECLS
#endif

#ifdef __GNUC__		/* assuming gcc >= 2.95 */

# ifdef _WIN32
#  define INO_IMPORT __attribute__((dllimport))
#  define INO_EXPORT __attribute__((dllexport))
# else
#  define INO_IMPORT
#  define INO_EXPORT
# endif

# if (__GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ >= 3)) \
     && !defined(_WIN32)
#  define INO_PUBLIC		__attribute__((visibility("default")))
#  define INO_PRIVATE		__attribute__((visibility("hidden")))
# else
#  define INO_PUBLIC
#  define INO_PRIVATE
# endif

#else

# define INO_IMPORT
# define INO_EXPORT
# define INO_PUBLIC
# define INO_PRIVATE

#endif

#ifdef INO_IMPLEMENTATION
# define INO_DEF	INO_PUBLIC INO_EXPORT
#else
# define INO_DEF	INO_PUBLIC INO_IMPORT
#endif

INO_BEGIN_C_DECLS

/*
   Group: Enumerations
 */

/*
   Enum: ino_variant_type

   Variant type.

   INO_VARINAT_TYPE_UNDEFINED	- undefined value.
   INO_VARIANT_TYPE_NULL	- null value.
   INO_VARIANT_TYPE_BOOLEAN	- Boolean value.
   INO_VARIANT_TYPE_STRING	- String.
   INO_VARIANT_TYPE_NUMBER	- Numeric value.
   INO_VARIANT_TYPE_OBJECT	- Object.
 */
typedef enum
  {
	INO_VARIANT_TYPE_UNDEFINED,
	INO_VARIANT_TYPE_NULL,
	INO_VARIANT_TYPE_BOOLEAN,
	INO_VARIANT_TYPE_STRING,
	INO_VARIANT_TYPE_NUMBER,
	INO_VARIANT_TYPE_OBJECT
  } ino_variant_type;

/*
   Group: Types
 */

/*
   Type: ino_context

   Execution context.
 */
typedef void*	ino_context;

/*
   Type: ino_variant

   Variant.
 */
typedef void*	ino_variant;

/*
   Type: ino_method_cb

   Native method callback.

   Parameters:

     ctx	- Execution context.
     args	- Array of arguments passed to the method.
     n_args	- Number of elements in args.

   Returns:

     New variant holding the result value.
 */
typedef ino_variant
(*ino_method_cb)(ino_context ctx, const ino_variant *args, size_t n_args);

/*
   Group: Functions
 */

/*
   Function: ino_context_new()

   Creates a new execution context.

   Returns:

     New execution context.
 */
INO_DEF ino_context
ino_context_new();

/*
   Function: ino_context_free()

   Destroys an execution context.
 */
INO_DEF void
ino_context_free(ino_context ctx);

/*
   Function: ino_context_get_global_object()

   Gets the global object in ctx.

   Parameters:

     ctx	- Execution context.

   Returns:

     Global object in ctx.
 */
INO_DEF ino_variant
ino_context_get_global_object(ino_context ctx);

/*
   Function: ino_variant_new_undefined()

   Creates a new variant holding undefined.

   Paramters:

     ctx	- Execution context.

   Returns:

     Variant holding undefined.
 */
INO_DEF ino_variant
ino_variant_new_undefined(ino_context ctx);

/*
   Function: ino_variant_new_null()

   Creates a new variant holding null.

   Paramters:

     ctx	- Execution context.

   Returns:

     Variant holding null.
 */
INO_DEF ino_variant
ino_variant_new_null(ino_context ctx);

/*
   Function: ino_variant_new_boolean()

   Creates a new variant holding the specified boolean value.

   Paramters:

     ctx	- Execution context.
     value	- The value.

   Returns:

     Variant holding the specified boolean value.
 */
INO_DEF ino_variant
ino_variant_new_boolean(ino_context ctx, bool value);

/*
   Function: ino_variant_new_string()

   Creates a new variant holding the specified string.

   Paramters:

     ctx	- Execution context.
     value	- UTF-16 encoded string.

   Returns:

     Variant holding the specified string.
 */
INO_DEF ino_variant
ino_variant_new_string(ino_context ctx, const uint16_t *value);

/*
   Function: ino_variant_new_string_utf8()

   Creates a new variant holding the specified string.

   Paramters:

     ctx	- Execution context.
     value	- UTF-8 encoded string.

   Returns:

     Variant holding the specified string.
 */
INO_DEF ino_variant
ino_variant_new_string_utf8(ino_context ctx, const char *value);

/*
   Function: ino_variant_new_number()

   Creates a new variant holding the specified numeric value.

   Paramters:

     ctx	- Execution context.
     value	- The numeric value.

   Returns:

     Variant holding the specified numeric value.
 */
INO_DEF ino_variant
ino_variant_new_number(ino_context ctx, double value);

/*
   Function: ino_variant_new_object()

   Creates a new variant holding a new object.

   Paramters:

     ctx	- Execution context.

   Returns:

     Variant holding a new object.
 */
INO_DEF ino_variant
ino_variant_new_object(ino_context ctx);

/*
   Function: ino_variant_get_type()

   Gets the type of value held by a variant.

   Parameters:

      ctx	- Execution context.
      variant	- Variant.

   Returns:

     Variant type.
 */
INO_DEF ino_variant_type
ino_variant_get_type(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_to_boolean()

   Converts a variant to boolean and returns its value.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.

   Returns:

     Boolean value.
 */
INO_DEF bool
ino_variant_to_boolean(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_to_string()

   Converts a variant to string and returns a UTF-16 encoding malloc()-allocated
   copy.

   The returned value should be free()d whe it is no longer needed.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.

   Returns:

     UTF-16 encoded malloc()-allocated string copy.
 */
INO_DEF uint16_t *
ino_variant_to_string(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_to_string_utf8()

   Converts a variant to string and returns a UTF-8 encoding malloc()-allocated
   copy.

   The returned value should be free()d whe it is no longer needed.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.

   Returns:

     UTF-8 encoded malloc()-allocated string copy.
 */
INO_DEF char *
ino_variant_to_string_utf8(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_to_number()

   Converts a variant to number and returns its value.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.

   Returns:

     Numeric value.
 */
INO_DEF double
ino_variant_to_number(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_object_get_property()

   Gets a property from an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-16 encoded property name.

   Returns:

     Variant containing the property value or the undefined value if the object
     does not have the specified property.
 */
INO_DEF ino_variant
ino_variant_object_get_property(ino_context ctx, ino_variant object,
				const uint16_t *name);

/*
   Function: ino_variant_object_get_property_utf8()

   Gets a property from an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-8 encoded property name.

   Returns:

     Variant containing the property value or the undefined value if the object
     does not have the specified property.
 */
INO_DEF ino_variant
ino_variant_object_get_property_utf8(ino_context ctx, ino_variant object,
				     const char *name);

/*
   Function: ino_variant_object_set_property()

   Sets a property on an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-16 encoded property name.
     value	- Variant containing the property value.
 */
INO_DEF void
ino_variant_object_set_property(ino_context ctx, ino_variant object,
				const uint16_t *name, ino_variant value);

/*
   Function: ino_variant_object_set_property_utf8()

   Sets a property on an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-8 encoded property name.
     value	- Variant containing the property value.
 */
INO_DEF void
ino_variant_object_set_property_utf8(ino_context ctx, ino_variant object,
				     const char *name, ino_variant value);

/*
   Function: ino_variant_object_add_method()

   Adds a natively implemented method to an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-16 encoded method name.
     cb		- Callback implementing the method.

   Returns:

     Object variant representing the method.
 */
INO_DEF ino_variant
ino_variant_object_add_method(ino_context ctx, ino_variant object,
			      const uint16_t *name, ino_method_cb cb);

/*
   Function: ino_variant_object_add_method_utf8()

   Adds a natively implemented method to an object.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     name	- UTF-8 encoded method name.
     cb		- Callback implementing the method.

   Returns:

     Object variant representing the method.
 */
INO_DEF ino_variant
ino_variant_object_add_method_utf8(ino_context ctx, ino_variant object,
				   const char *name, ino_method_cb cb);

/*
   Function: ino_variant_object_call()

   Call an object as a function.

   Parameters:

     ctx	- Execution context.
     object	- Object variant.
     args	- Array of arguments.
     n_args	- Number of elements in args.

   Returns:

     Variant holding the result value.
 */
INO_DEF ino_variant
ino_variant_object_call(ino_context ctx, ino_variant object,
			const ino_variant *args, size_t n_args);

/*
   Function: ino_variant_protect()

   Protects a variant from garbage collection.

   A variant may be protected multiple times and must be unprotected an equal
   number of times before becoming eligible for garbage collection.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.
 */
INO_DEF void
ino_variant_protect(ino_context ctx, ino_variant variant);

/*
   Function: ino_variant_unprotect()

   Unprotects a variant from garbage collection.

   Parameters:

     ctx	- Execution context.
     variant	- Variant.
 */
INO_DEF void
ino_variant_unprotect(ino_context ctx, ino_variant variant);

/*
   Function: ino_eval()

   Evaluates a string of JavaScript code.

   Parameters:

     ctx	- Execution context.
     script	- UTF-16 encoded string of JavaScript code.

   Returns:

     Variant holding the result value.
 */
INO_DEF ino_variant
ino_eval(ino_context ctx, const uint16_t *script);

/*
   Function: ino_eval_utf8()

   Evaluates a string of JavaScript code.

   Parameters:

     ctx	- Execution context.
     script	- UTF-8 encoded string of JavaScript code.

   Returns:

     Variant holding the result value.
 */
INO_DEF ino_variant
ino_eval_utf8(ino_context ctx, const char *script);

INO_END_C_DECLS

#endif
