#!/bin/csh -f
#-----------------------------------------------------------------------------
# file:         rundrivers.csh
#
# description:
#       Shell script to run all the HDFEOS5 Drivers within a directory.
#
# notes:
#       1) Any existing driver output files (except the samples) should be 
#	   deleted.  The difference check will fail if they aren't.  Its 
#	   probably a good idea to delete all the output files generated 
#	   after each run.  The difference files can hang around without
#	   causing a problem (they will be overwritten if the script is run
#	   again.
#	2) It is up to the user to set the FORTRAN90 flags and have the
#	   Toolkit compiled using FORTRAN90 prior to the running this script.
#	3) The script will take care of calling the proper makefile.
#	   (FORTRAN90 uses makefile.f90)
#
# usage:
#       rundrivers.csh LANGUAGE [LANGUAGE2] (ex. rundrivers.csh c f90)
#
#       At least one parameter(language) needs to be specified on the command
#	line.  If none are specified, no drivers will be run.  For each language
#	all the drivers within the directory will be run using the input file 
#	and the resulting output file will be checked against the sample output
#	file.
#
# author:
#       Abe Taaheri
#
# history:
#       07-Jul-2014 AT  Used TOOLKIT testdrivers originally written by
#                       Jason R. Watts / Hughes Applied Information Systems
#                       to create this and other scripts that this script runs
#----------------------------------------------------------------------------
 
# check the number of arguments coming in

if ($#argv < 1) then
    echo "";echo "No language(s) specified, exiting ..."; echo ""
    exit
endif

# set environment to build binary in current directory

setenv HDFEOS5_TST_BIN .
setenv DIFF $HDFEOS5_TST_SRC/Common/diff.csh
setenv HDFDIFF $HDFEOS5_TST_SRC/Common/hediff.csh


# loop through each argument on the command line
# (c ==> 'C', f ==> 'FORTRAN', f90 ==> 'FORTRAN90')

foreach LANG ($argv)

    if ($LANG == "c" || $LANG == "f" || $LANG == "f90") then

        # setup variables for which makefile to use
        # and how to name the output file
        # The filenames the 'C' drivers generate have '_c.out.(date)'
        # appended to the end, the FORTRAN drivers append '_f77.out'
        # (or f90) only (no date).  This makes it necessary to have
        # a SUFFIX variable.
 
        if ($LANG == "c") then
            set OUT_LANG = "c"
            set CODE_EXT = "c"
            set F_type = ""
            set SUFFIX = "c.out"
            set MAKE_FILE = "$HDFEOS5_TST_SRC/makefile_tst"
        else if ($LANG == "f") then
            set OUT_LANG = "f"
            set CODE_EXT = "f"
            set F_type = "f"
            set SUFFIX = "f.out"
            set MAKE_FILE = "$HDFEOS5_TST_SRC/makefile_tst"
        else if ($LANG == "f90") then
            set LANG = "f"
            set CODE_EXT = "f"
            set F_type = "f"
            set OUT_LANG = "f"
            set SUFFIX = "f.out"
            set MAKE_FILE = "$HDFEOS5_TST_SRC/makefile_tst"
        endif

	echo ""
	echo "Processing ${OUT_LANG} drivers ..."
	
	# List all the drivers into 'DRIVERS'

	if ($LANG == "c") then
	    set SRCDRIVERS = `ls *.${CODE_EXT}`
	    set DRIVERS = ""
	    foreach DRIVER ($SRCDRIVERS)
		if(`echo $DRIVER | grep Alias | wc -l` > 0) then
		    if(`echo $SRCDRIVERS | grep Grid | wc -l` > 0) then
			set ALIASSUFIX = "grid"
		    else if(`echo $SRCDRIVERS | grep Swath | wc -l` > 0) then
			set ALIASSUFIX = "swath"
		    else if(`echo $SRCDRIVERS | grep Za | wc -l` > 0) then
			set ALIASSUFIX = "za"
		    endif
		else
		    set ALIASSUFIX = ""
		endif

		set DRIVERS = ( $DRIVERS `( echo  $DRIVER| \
		awk -F\.${CODE_EXT} '{len=length($1); \
		printf("%s\n", substr($1,1,len))}' )`)
		set DRIVERS = (${DRIVERS}${ALIASSUFIX})
	    end
        else
	    set SRCDRIVERS = `ls *.${CODE_EXT}|grep ${F_type}`

	    set DRIVERS = ""
	    foreach DRIVER ($SRCDRIVERS)
		if(`echo $BRAND | grep 64 | wc -l` > 0) then
		    if ( ${DRIVER} == testgrid64.f || ${DRIVER} == testswath64.f || ${DRIVER} == testpoint64.f || ${DRIVER} == testza64.f || ${DRIVER} == testgrid_szip64.f) then
			set DRIVERS = ( $DRIVERS `( echo  $DRIVER| \
			awk -F\.${CODE_EXT} '{len=length($1); \
			printf("%s\n", substr($1,1,len))}' )`)
		    endif
		else 
		    if ( ${DRIVER} == testgrid32.f || ${DRIVER} == testswath32.f || ${DRIVER} == testpoint32.f || ${DRIVER} == testza32.f || ${DRIVER} == testgrid_szip32.f) then

			set DRIVERS = ( $DRIVERS `( echo  $DRIVER| \
			awk -F\.${CODE_EXT} '{len=length($1); \
			printf("%s\n", substr($1,1,len))}' )`)
		    endif
		endif
	    end
	endif

	# Go through list of drivers and do make, run driver, log difference
	
    	foreach DRIVER ($DRIVERS)
	
		echo ""; echo "Making ${DRIVER}_${LANG} ..."; echo ""
	
		# remove the existing driver

		\rm -f ${DRIVER}_${LANG}
	
		# make the driver

		make -f ${MAKE_FILE} ${DRIVER}_${LANG}
	
		# if make has a problem report an error 
                # and don't even try and run the driver

		if ($status != 0) then
	
	    	    echo ""; echo "ERROR::Could not compile ${DRIVER}_${LANG}."
	
		else
	
		    # make was successful, run driver 
                    # and check difference against sample

	    	    echo ""; echo "Running ${DRIVER}  DRIVER ..."
	    	    ./${DRIVER}_${LANG} -v h >&! ${DRIVER}_${SUFFIX}
		    if ( $LANG == "f") then
			if ( -f *_${CODE_EXT}.txtout) then
			    /bin/mv *_${CODE_EXT}.txtout ${DRIVER}_${LANG}.out
			endif
		    endif
                echo ""; echo "Running diff ${DRIVER}_${LANG}.out_sample ${DRIVER}_${SUFFIX} >&! ${DRIVER}_${OUT_LANG}.diff ..."
                ${DIFF} ${DRIVER}_${LANG}.out_sample ${DRIVER}_${SUFFIX} >&! ${DRIVER}_${OUT_LANG}.diff
		endif
    	end

# we are done with drivers. Lets diff output he5 files
	set HDFFILES = `ls *.he5`
    	foreach HEFILE ($HDFFILES)
	    if ( `echo $HEFILE | grep sample | wc -l` == 0 ) then
		echo ""; echo "Comparing ${HEFILE} with sample using ncdump and diff ..."; echo ""
                ${HDFDIFF} ${HEFILE} sample_{${HEFILE}} >&! ${HEFILE}.diff
		echo ""; echo "Done comparing ${HEFILE} with sample."
	    endif
    	end
    else

	echo "";echo "ERROR::$LANG is an invalid language...";echo ""

    endif

end

echo "Processing complete, check the difference files (*.diff) ...";echo ""
