/**
 * @file fg_string.h
 * @brief Functions to manipulate strings used by Flowgrind
 */

/*
 * Copyright (C) 2014 Alexander Zimmermann <alexander.zimmermann@netapp.com>
 *
 * This file is part of Flowgrind.
 *
 * Flowgrind is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Flowgrind is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flowgrind.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef _FG_STRING_H_
#define _FG_STRING_H_

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

/**
 * Append the duplication of string @p a to the given string @p s.
 *
 * @param[in] s destination string to append to
 * @param[in] a source string to be append
 * @return concatenated strings, NULL on error
 */
char *strdup_append(char *s, const char *a);

/**
 * Append at most n characters of the duplication of string @p a to the given
 * string @p s.
 *
 * @param[in] s destination string to append to
 * @param[in] a source string to be append
 * @param[in] n number of characters to be append from the string
 * @return concatenated strings, NULL on error
 */
char *strndup_append(char *s, const char *a, size_t n);

/**
 * Determine the number of characters that would be generated by a printf
 * with format string @p fmt and arguments @p ap.
 *
 * @param[in] fmt format string
 * @param[in] ap parameters used to fill fmt
 * @return number of number of characters
 */
size_t fmtlen(char const *fmt, va_list ap);

/**
 * Realloc @p resultp to append the formatted result of @p fmt and return a
 * pointer to it via the first argument.
 *
 * Good for gradually accumulating output into a string buffer. Appends at the
 * end of the string
 *
 * @param[in,out] resultp destination string to append to
 * @param[in] fmt format string
 * @param[in] ... parameters used to fill fmt
 * @return return the number of bytes printed for success, or -1 for failure
 */
int asprintf_append(char **resultp, const char *fmt, ...)
	__attribute__((format(printf, 2, 3)));

/**
 * Realloc @p resultp to append the formatted result of @p fmt and @p ap and
 * return a pointer to it via the first argument.
 *
 * Good for gradually accumulating output into a string buffer. Appends at the
 * end of the string.
 *
 * @param[in,out] resultp destination string to append to
 * @param[in] fmt format string
 * @param[in] ap parameters used to fill fmt
 * @return return the number of bytes printed for success, or -1 for failure
 */
int vasprintf_append(char **resultp, const char *fmt, va_list ap)
	__attribute__((format(printf, 2, 0)));

#endif /* _FG_STRING_H_ */
