/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#ifndef DIN

#define DIN

#include "curve_editor.h"
#include "listeners.h"
#include "range.h"
#include "ui.h"
#include "audio.h"
#include "play.h"
#include "oscilloscope.h"
#include "curve_library.h"
#include "beat2value.h"
#include "help.h"
#include "phrasor.h"
#include "input.h"
#include "command.h"
#include "globals.h"
#include "random.h"
#include "drone.h"

struct din_info {

  struct scroll_t {
    double rept; // key repeat time
    int rate; // scrolls per second
    int dx, dy; // x & y amount per scroll
    scroll_t () : rate (80), dx(15), dy(1) {}
    void calc_repeat_time () {
      if (rate) rept = 1./ rate;
    }
  };

  scroll_t scroll;

  int height; // din board height

  int gater; // gater enabled?

  int delay; // delay enabled?

  int compress; // compressor enabled?

  int voice; // lead voice?

  int anchor; // draw drone anchor?

  din_info ();
  void save ();

};

struct octave_shift_data {

  float tonic; // current tonic
  int dir; // shift direction; 0 = up, 1 = down
  int active; // active?
  octave_shift_data () : tonic (0), dir (-1), active (0) {}

};

struct din : ui {

  multi_curve wave; // waveform
  curve_editor waved; // waveform shape editor
  wave_listener wavlis;

  solver wavsol;
  play wavplay;

  // tone ranges
  //

  std::vector<range> ranges;
  int num_ranges, last_range, current_range;
  range *firstr, *lastr;

  float step; // represents pitch (see note.h)
  float delta; // 0 - left of range, 1 - right of range
  float octave_position; // octave position of pitch (any value between 0 to 2 ie 3 octaves)

  int show_cursor_info; // display frequency / volume under mouse cursor?
  std::string cursor_info;

  int visl, visr; // visible ranges
  void find_visible_ranges (int dir = 0);

  bool load_ranges ();
  void save_ranges ();
  void create_ranges (int n);
  void setup_ranges (int load = 1);
  void setup_range_notes ();
  void update_range_notes ();
  void calc_range_label ();
  void range_right_changed ();
  void range_left_changed ();
  void set_range_size (int r, int sz);
  void set_range_size (int s, int t, int sz);
  int find_range (int x, int r);
  void find_current_range ();
  bool find_tone_and_volume ();
  void retune_note (const std::string& nn, float v);
  void retune_note ();
  void restore_note ();
  void restore_all_notes ();
  float get_note_value (const std::string& s);


  //
  // mouse board
  //
  box<int> win;
  void scroll (int dx, int dy);
  void calc_win_mouse ();
  int win_mousex, win_mousey; // mouse pos in win
  int delta_mousex, delta_mousey;
  int tonex, toney;

  int dv; // driver volume ie mouse height on board

  inline void find_volume () {

    extern int BOTTOM;
    extern double VOLUME;
    extern float DELTA_VOLUME;
    dv = toney - BOTTOM;
    VOLUME = dv * DELTA_VOLUME;

  }

  void set_tonic (float s);
  float get_tonic ();
  void mouse2tonic ();


  //
  // drones
  //

  box_selector<int> selector;

  std::vector<drone> drones;
  std::vector<int> selected_drones;

  float drone_master_volume;
  float last_drone_master_volume;

  multi_curve drone_wave; // drone waveform
  curve_editor droneed;
  wave_listener dronelis;

  int snap_drones; // snap to range ends ie notes?
  int editing_drone;


  // drone selection
  void find_selected_drones ();
  int select_all_drones ();
  void pick_drone ();
  std::string get_selected_drones ();
  void clear_selected_drones ();

  void load_drones ();
  void save_drones ();

  void add_drone ();
  void set_drone (int d);

  void update_drone_tone ();
  void update_drone_x (int i, int j);
  void update_drone_master_volume (float delta);
  void update_drone_players ();
  void update_drone_solvers ();
  void update_drone_ranges ();
  void update_drone_anchors ();

  std::string on_delete;
  void delete_selected_drones ();
  int delete_all_drones ();

  int anchor;
  void draw_drones ();
  void set_drone_volume (int i, float v);

  int do_drone_modulation;
  void change_drone_bpm (int what, float delta);
  void change_drone_depth (int what, float delta);
  void change_drone_trail_points (int n);
  void change_drone_handle_size (int n);

  void modulate_drones ();

  int rising, falling;
  std::vector<int> dead_drones;
  void rise_drones ();
  void fall_drones ();

  // am, fm & gaters
  //
  float am_depth, fm_depth;
  void change_depth (int i, float d);
  void change_bpm (beat2value& which, float amt);

  float am_vol;
  float fm_step;
  beat2value fm, am, gatr;
  void calc_am_fm_gater ();


  // gater editor
  curve_editor gated;
  curve_library gatlib;
  beat2value_listener gatrlis;


  // am & fm modulation editors
  curve_editor moded;
  beat2value_listener fmlis, amlis;

  // musical key
  int key_id; // C = 0, C# = 1, D = 2, D# = 3, E = 4, F = 5, F# = 6, G = 7, G# = 8, A = 9, A# = 10, B = 11
  void set_key_id (int i);
  int get_key_id () { return key_id;}

  // phrasor - record/play/scratch a phrase
  phrasor phrasor0;
  int jogged;

  oscilloscope osc;

  din (cmdlist& cmdlst);
  ~din ();

  bool handle_input ();

  void load_scale ();
  void save_scale ();

  void enter ();
  void bg (); // background execution
  void draw ();

  // din saved info
  din_info dinfo;
  void save_info ();

  // help
  help helptext;

  // range display notation
  std::string notation;
  void set_notation (const std::string& s);
  void notate_ranges ();

  void render_audio (float* out0, float* out1);

  void roll_console ();

};

#endif
