/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#include<X11/X.h>
#include<X11/Xlib.h>
#include <X11/Xatom.h>
#include <string>
#include <map>
#include <vector>
#include <iostream>
#include <fstream>
#include <list>
#include <algorithm>
#include <tcl.h>
#include "dingl.h"
#include "main.h"
#include "box.h"
#include "utils.h"
#include "font.h"
#include "console.h"
#include "input.h"
#include "viewwin.h"
#include "curve.h"
#include "multi_curve.h"
#include "solver.h"
#include "chrono.h"
#include "ui_list.h"
#include "curve_editor.h"
#include "font_editor.h"
#include "din.h"
#include "curve_library.h"
#include "console_iterator.h"
#include "delay.h"
#include "random.h"
#include "globals.h"
#include "command.h"
#include "scalelist.h"
#include "morse_code.h"
#include "sine_mixer.h"
#include "checkdotdin.h"
#include "osc.h"
#include "bot.h"
#include "tcl_interp.h"
#include "key_consts.h"
#include "ansi_color_codes.h"
#include "compressor.h"
#include "oscilloscope.h"
#include "scale_info.h"
#include "keyboard_keyboard.h"
#include "fft.h"
#include "authors_note.h"

#include <iostream>
using namespace std;

// declare / define all global vars
//

struct flob {
  flob () {
    std::cout << PASS << "+++ initialised random number generator +++" << ENDL;
    seed_rand_gen (clock()); // seed random number generator
  }

  ~flob () {
    std::cout << FAIL << "!!! din shutdown !!!" << ENDL;
  }
} _flob; // first and last object

widget* widget::focus = 0; // widget that has focus
widget* ui_list::mm_last = 0; // last button clicked on the label field sliders (see min_max_clicked below)
const color checkbutton::on_color (0.25, 1, 0.25), checkbutton::off_color (1, 0.25, 0.25); // colors in checkbutton

std::string APP_NAME; // in format din-x.y.[z] where x.y.[z] is version
std::string WIN_TITLE; // title of din window (includes APP_NAME and jack connection name)
int SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH; // initial screen width, height & color depth

int SAMPLE_RATE; // sample rate for the session
float SAMPLE_DURATION; // duration of an audio sample in seconds

std::string SCALE; // current scale
int NUM_OCTAVES; // number of octaves in din

std::string TUNING; // current tuning
int NUM_INTERVALS; // number of intervals in tuning
std::string INTERVALS_FILE; // file name that contains interval ratios
std::map <std::string, float> INTERVALS; // maps interval name with ratio
std::vector<float> INTERVALS_SEQ; // sequential intervals (eg., value of 1 2b 3 3b ... 7b 7 2)
std::map <std::string, int> NOTE_POS; // maps interval name to note number
std::string FIRST_INTERVAL_NAME, LAST_INTERVAL_NAME;

// for microtonal keyboard
//
int LEFT, BOTTOM, RIGHT, TOP, HEIGHT; // extents
int NUM_MICROTONES; // number of microtones in a range
int NUM_VOLUMES, LAST_VOLUME; // number of volume levels
float DELTA_VOLUME; // change in volume every pixel of board height
int TRAIL_LENGTH = 1; // drone trail length (== number of trail points)
int DRONE_HANDLE_SIZE; // for the session

// for CPU loading (see http://www.dinisnoise.org/faq/#load)
//
int FPS, USLEEP;
double FPS_T;

float DELTA_BPM; // change in bpm when using key shorcuts (see http://www.dinisnoise.org/din_help/)

float PITCH_BEND; // in Hz; used for pitch bending notes on keyboard-keyboard

float HZ = 440; // frequency that determines number of samples displayed on waveform editors

// western notation
//
int NUM_NOTES = 13;
const char* WESTERN_SHARP [] = {"C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B", "C"};
const char* WESTERN_FLAT [] = {"C", "Db", "D", "Eb", "E", "F", "Gb", "G", "Ab", "A", "Bb", "B", "C"};
float WIKIPEDIA_KEY_FREQUENCIES [] = { // octave of Middle-C; from wikipedia - piano key frequencies page.
  261.626, 277.183, 293.665, 311.127, 329.628, 349.228, 369.994, 391.995, 415.305, 440, 466.164, 493.883, 523.251
};

// jack port names
//
std::string name ("din"), dinl, dinr;
const char* colon = ":", *L = "L", * R = "R", *MIDI_IN = "midi";

// screens - 1 is instrument; 2 - 8 are editors
//
const int ui_list::key [] = {k_2, k_3, k_4, k_5, k_6, k_7, k_8}; // keys 2 - 8
ui* ui_list::ed [] = {0};
char basic_editor::buf [] = {0};

// solver xhandlers; see solver.cc
atmin _atmin;
atmax _atmax;
tomin _tomin;
loopmin _loopmin;
loopmax _loopmax;
pongmin _pongmin;
pongmax _pongmax;

// see multi_curve.cc
multi_curve curve_editor::copy;

// 0 init correct for oscilloscope; see oscilloscope.cc
float oscilloscope::sample_t::lmin = 0, oscilloscope::sample_t::lmax = 0;
float oscilloscope::sample_t::rmin = 0, oscilloscope::sample_t::rmax = 0;

// see phrasor.cc
int phrasor::JOG = 3; // jog amount

// see range.cc
int range::char_height, range::ybot1, range::ybot2, range::ytop1, range::ytop2;

// see sine_mixer.cc
const float sine_mixer::pi = 3.14159;
int sine_mixer::NUM_SINE_SAMPLES = 100;

// see viewwin.cc
int viewport::handle_radius = 50;
float viewport::handle_factor = 0;
int window::PAN_RATE = 100, window::ZOOM_RATE = 100;
double window::PAN_REPEAT = 1. / PAN_RATE, window::ZOOM_REPEAT = 1. / ZOOM_RATE;
float window::PAN_AMOUNT = 0.1, window::ZOOM_AMOUNT = 0.1;

// ~/.din contains defaults, user savings and preferences
std::string dotdin;
checkdotdin cdd;

// load initial globals
//
load_globals lg;

// custom std::vector font
//
font fnt ("jag.fnt");

// din clocks
//
chrono clk; // audio clock
double TIME_NOW = 0; // time now in seconds on the audio clock (ie clk) stored in variable timenow in Tcl interpreter
nano_chrono ui_clk; // ui clock

// audio output
//
int audio_out::AUTO_CONNECT_OUTPUTS = 1; // to JACK
audio_out aout;

// console
//

// possible text colors of console
const float cc = 0.6;
const color console::yellow (1, 1, cc);
const color console::green (cc, 1, cc);
const color console::red (1, cc, cc);
const color console::cyan (cc, 1, 1);

console cons;

char xkey; // X window lookedup char for console / field typings (see console.cc / field.cc)

// display - using OpenGL
viewport view;

// list of scales
scalelist scalelst;

// keyboard and mouse state
key_state keys;
int mousex = 0, mousey = 0, mouseyy = 0; // absolute mouse x, y & main viewport y
int lmb = 0, rmb = 0, mmb = 0; // mouse buttons

//
// L and R delays
//
delay left_delay (1000, "feedback-l.crv", "volume-l.crv"), right_delay (1000, "feedback-r.crv", "volume-r.crv");
curve_editor delayed ("delay.ed");

// vars in Tcl interpreter
//

// midi bpm from external midi clock
double MIDI_BPM = 0;

// tap bpm from computer keyboard
double TAP_BPM = 0;

// din board height (0 to 1)
double VOLUME = 0;

sine_mixer sinemixer; // sine mixer for waveform edit

fft fft0; // FFT of waveform in waveform editors

cmdlist cmdlst; // list of din commands

// 1 oscillator / waveform for lead
//

float WAVE_VOLUME = 0.15;
curve_library wavlib ("waveforms.lib"); // waveform library

// compressor
//
compressor coml ("coml.crv"), comr ("comr.crv");
curve_editor comed ("compressor.ed");

scale_info scaleinfo;

// octave shift
//
octave_shift_data osd;
beat2value octave_shift ("Octave Shift", "octave-shift.crv");
curve_editor octed ("octave-shift.ed");
curve_library octlib ("octave-shift-patterns.lib");
beat2value_listener octlis (&octave_shift);

// drone modulation
//
beat2value drone_modulation::am_crv ("drone AM", "dam.crv");
beat2value drone_modulation::fm_crv ("drone FM", "dfm.crv");
curve_editor dmod_ed ("drone-modulation.ed");
beat2value_listener damlis (&drone_modulation::am_crv), dfmlis (&drone_modulation::fm_crv);

// microtonal-keyboard see din.cc/.h
//
din din0 (cmdlst);

// keyboard-keyboard
//
float NOTE_VOLUME = 0.75 * WAVE_VOLUME;

// in secs
float ATTACK_TIME = 0.05;
float DECAY_TIME = 5.0;
float DELTA_TIME = 0.025;

const color keyboard_keyboard::note_color [] = {{0.9, 0.9, 0.9}, {0.2, 0.2, 0.2}}; // midi keybd note color
const int keyboard_keyboard::color_index [] = {0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0}; // 0 - white, 1 - black; 1 octave
keyboard_keyboard keybd2;
gotog _gotog (1, &keybd2.attacked); // default sustain (see keyboard-keyboard.cc)

std::string INSTRUMENT = "keyboard_keyboard";

void modulate_up () {
  if (!osd.active) {
    static const std::string down = "down";
    start_octave_shift (0, down);
  }
}

void modulate_down () {
  if (!osd.active) {
    static const std::string up = "up";
    start_octave_shift (1, up);
  }
}

void start_octave_shift (int idir, const std::string& sdir) {
  osd.tonic = scaleinfo.tonic;
  osd.dir = idir;
  osd.active = 1;
  octave_shift.now = octave_shift.sol.firstx;
  cons << console::yellow  << "modulating " << sdir << eol;
}

void set_tonic (float f) {
  scaleinfo.set_tonic (f);
  din0.update_range_notes ();
  din0.update_drone_tone ();
  keybd2.setup_notes (0);
}

void do_octave_shift () {

  extern audio_out aout;
  octave_shift.now += (octave_shift.delta * aout.samples_per_channel);

  if (octave_shift.now > octave_shift.sol.lastx) {
    octave_shift.now = octave_shift.sol.lastx;
    osd.active = 0;
  }

  float shift = octave_shift.sol (octave_shift.now);
  if (shift != 0) {
    if (osd.dir) set_tonic (osd.tonic * shift);
    else set_tonic (osd.tonic / shift);
  }

  if (!osd.active) {
    cons << console::yellow << "modulation complete" << eol;
    cons ("key");
  }

}

void setup_editors () {

  // in delay editor load feedback & volume curves
  delayed.add (&left_delay.fbk_crv, &left_delay.fbk_lis);
  delayed.add (&left_delay.vol_crv, &left_delay.vol_lis);
  delayed.add (&right_delay.fbk_crv, &right_delay.fbk_lis);
  delayed.add (&right_delay.vol_crv, &right_delay.vol_lis);

  // +listener to compressor editor
  comed.add (&coml.crv, &coml.lis);
  comed.add (&comr.crv, &comr.lis);

  // octave shift
  octave_shift.xmin = &_atmin;
  octave_shift.xmax = &_atmax;
  octed.add (&octave_shift.crv, &octlis);
  octed.attach_library (&octlib);
  octed.bv.push_back (&octave_shift);

  // drone modulation
  dmod_ed.add (&drone_modulation::am_crv.crv, &damlis);
  dmod_ed.add (&drone_modulation::fm_crv.crv, &dfmlis);
  dmod_ed.attach_library (&wavlib);

}

std::vector<multi_curve*> curve_list;
void setup_curve_list () {
  multi_curve* lst [] = {
    &din0.wave,
    &keybd2.wave,
    &din0.fm.crv, &din0.am.crv, &din0.gatr.crv,
    &octave_shift.crv, &din0.drone_wave, &left_delay.fbk_crv, &left_delay.vol_crv,
    &right_delay.fbk_crv, &right_delay.vol_crv
  };
  for (int i = 0; i < 11; ++i) curve_list.push_back (lst[i]);
}

const char* bpm_com::str [bpm_com::NUM] = {"gr", "am", "fm", "os"};
beat2value* bpm_com::bv [] = {&din0.gatr, &din0.am, &din0.fm, &octave_shift};

#define DEFINE_TCL_CMD(X) inline int (X) (ClientData cd, Tcl_Interp* ti, int objc, Tcl_Obj* CONST objv[]) { return tcl_run (cd, ti, objc, objv);}

DEFINE_TCL_CMD(tcl_key)
DEFINE_TCL_CMD(tcl_setv)
DEFINE_TCL_CMD(tcl_getv)
DEFINE_TCL_CMD(tcl_set_delay)
DEFINE_TCL_CMD(tcl_get_delay)
DEFINE_TCL_CMD(tcl_ls_bpm);
DEFINE_TCL_CMD(tcl_set_bpm)
DEFINE_TCL_CMD(tcl_get_bpm)
DEFINE_TCL_CMD (tcl_set_beat);
DEFINE_TCL_CMD (tcl_get_beat);
DEFINE_TCL_CMD(tcl_set_style)
DEFINE_TCL_CMD(tcl_get_style)
DEFINE_TCL_CMD(tcl_gaters)
DEFINE_TCL_CMD(tcl_add_scale)
DEFINE_TCL_CMD(tcl_ls_scale)
DEFINE_TCL_CMD(tcl_remove_scale)
DEFINE_TCL_CMD(tcl_load_scale)
DEFINE_TCL_CMD(tcl_ls_notes)
DEFINE_TCL_CMD(tcl_set_kern)
DEFINE_TCL_CMD(tcl_get_kern)
DEFINE_TCL_CMD(tcl_set_font_size)
DEFINE_TCL_CMD(tcl_get_font_size)
DEFINE_TCL_CMD(tcl_note_distance)
DEFINE_TCL_CMD(tcl_chord)
DEFINE_TCL_CMD(tcl_notation)
DEFINE_TCL_CMD(tcl_echo)
DEFINE_TCL_CMD(tcl_curve_value)
DEFINE_TCL_CMD(tcl_curve_name)
DEFINE_TCL_CMD(tcl_curve_library)
DEFINE_TCL_CMD(tcl_morse_code)
DEFINE_TCL_CMD(tcl_bot)
DEFINE_TCL_CMD (tcl_set_editor);
DEFINE_TCL_CMD (tcl_get_editor);
DEFINE_TCL_CMD (tcl_set_scope);
DEFINE_TCL_CMD (tcl_get_scope);
DEFINE_TCL_CMD (tcl_get_drone);
DEFINE_TCL_CMD (tcl_set_drone);
DEFINE_TCL_CMD (tcl_load_tuning);
DEFINE_TCL_CMD (tcl_text_color);
DEFINE_TCL_CMD (tcl_paste_gater);
DEFINE_TCL_CMD (tcl_get_selection);
DEFINE_TCL_CMD (tcl_get_intervals);
DEFINE_TCL_CMD (tcl_osc_server);
DEFINE_TCL_CMD (tcl_osc_connect);
DEFINE_TCL_CMD (tcl_osc_send);
DEFINE_TCL_CMD (tcl_num_octaves);
DEFINE_TCL_CMD (tcl_quit);

// din commands
//
// format: long name, short name
//

// for L and R delays
set_delay sd (&left_delay, &right_delay, "set-delay", "sd");
get_delay gd (&left_delay, &right_delay, "get-delay", "gd");

// for adding/listing/removing scales
add_scale as ("add-scale", "as");
remove_scale rs ("remove-scale", "rs");
load_scale los (&din0, "load-scale", "los");
ls_scales ls ("list-scales", "ls");
ls_notes lsn ("list-notes", "ln");

// set and get din variables
set_var sv (&din0, "set-var", "sv");
get_var gv (&din0, "get-var", "gv");

// bpm commands
//
ls_bpm lb ("list-bpms", "lb");
set_bpm sb ("set-bpm", "sb");
get_bpm gb ("get-bpm", "gb");
set_beat sn ("set-beat", "sbt");
get_beat gn ("get-beat", "gbt");
set_style ss ("set-style", "ss");
get_style gs ("get-style", "gs");

// set key/tonic
key ky (&din0, "key", "key");

// set tuning
load_tuning ltu (&din0, "load-tuning", "ltu");

// display notation on the keyboard
notation no (&din0, "notation", "no");

// music utils
note_distance nd ("note-distance", "nd");
chord ch ("chord", "ch");

// font cmds
set_font_size sfs ("set-font-size", "sfs");
get_font_size gfs ("get-font-size", "gfs");
set_kern sk ("set-kern", "sk");
get_kern gk ("get-kern", "gk");

// curve cmds
curve_name cn ("curve-name", "cn");
curve_value cv ("curve-value", "cv");
curve__library cl ("curve-library", "cl");
set_curve_editor sced ("set-curve-editor", "sced");
paste_gater pasg ("paste-gater", "paste-gater");

// morse code
morse_code mc ("morse-code", "mc");

// unix like echo
echo ech ("echo", "eo");

// OSC server/client commands
lo_address client  = 0;
oserver srv ("server", "server");
oconnect con (client, "connect", "connect");
osend snd (client, "send", "send");

// IRC bot
bot_t bot ("bot", ":");

// oscilloscope
set_scope ssco ("set-scope", "ssco", &din0);
get_scope gsco ("get-scope", "gsco", &din0);

// drone
get_drone gdro ("get-drone", "gdro", din0);
set_drone sdro ("set-drone", "sdro", din0);

// console
set_text_color stc ("set-text-color", "stc");

get_selection gsel ("get-selection", "gsel");

get_intervals gint ("get-intervals", "gint");

num_octaves noct ("num-octaves", "noct", din0);

quit qt ("quit", "exit");

void add_commands (Tcl_Interp* interp) { // add din commands to Tcl interpreter

  int ncmds = 45;
  tclcmd cmd_funcs [] = {
    tcl_key, tcl_setv, tcl_getv, tcl_set_delay, tcl_get_delay, tcl_ls_bpm, tcl_set_bpm, tcl_get_bpm, tcl_set_beat, tcl_get_beat,
    tcl_set_style, tcl_get_style, tcl_add_scale, tcl_ls_scale, tcl_remove_scale, tcl_load_scale, tcl_ls_notes, tcl_set_kern, tcl_get_kern,
    tcl_set_font_size, tcl_get_font_size, tcl_note_distance, tcl_chord, tcl_notation, tcl_echo, tcl_curve_value, tcl_curve_name,
    tcl_curve_library, tcl_morse_code, tcl_bot, tcl_set_editor, tcl_get_editor, tcl_set_scope, tcl_get_scope, tcl_get_drone, tcl_set_drone,
    tcl_load_tuning, tcl_text_color, tcl_paste_gater, tcl_get_selection, tcl_get_intervals, tcl_osc_server, tcl_osc_connect, tcl_osc_send,
    tcl_num_octaves, tcl_quit

  };

  command* cmds [] = {
    &ky, &sv, &gv, &sd, &gd, &lb, &sb, &gb, &sn, &gn,
    &ss, &gs, &as, &ls, &rs, &los, &lsn, &sk, &gk,
    &sfs, &gfs, &nd, &ch, &no, &ech, &cv, &cn,
    &cl, &mc, &bot, &sced, &ssco, &gsco, &gdro,
    &sdro, &ltu, &stc, &pasg, &gsel, &gint, &srv, &con, &snd, &noct, &qt
  };

  extern cmdlist cmdlst;
  for (int i = 0; i < ncmds; ++i) {
    command* cmdp = cmds[i];
    cmdlst.add (cmdp);
    Tcl_CreateObjCommand (interp, cmdp->longname.c_str(), cmd_funcs[i], (void *) i, 0);
    Tcl_CreateObjCommand (interp, cmdp->shortname.c_str(), cmd_funcs[i], (void *) i, 0);
  }

  cout << PASS << "+++ added " << ncmds << " din commands to the Tcl interpreter +++" << ENDL;


}

// bring up the Tcl interpreter; all din commands are available in the interpreter.
//
tcl_interp interpreter;

authors_note anote;
ui_list uis;


void update_window (int wnow, int hnow, int wprev, int hprev) { // called when main window resized

  glViewport (0, 0, wnow, hnow); // update OpenGL viewport

  view (wnow, hnow); // update viewport data

  int w = wnow - 1, h = hnow - 1;

  sinemixer.sine_levels.chkpos ();

  // update console window co-ords
  cons.set_window (box<int>(0, -h, w, 0));

  // update din window co-ords
  din0.win (din0.win.left, din0.win.bottom, din0.win.left + w, din0.win.bottom + h);

  uis.update_widgets ();

  keybd2.calc_visual_params ();

  // update current curve editor co-ords
  if (uis.crved) uis.crved->enter ();

  if (uis.current == &din0) {
    // find range, tone & volume
    din0.find_current_range ();
    din0.find_tone_and_volume ();
  }

}

void save_window () {
  extern std::string dotdin;
  std::string fname = dotdin + "window";
  ofstream file (fname.c_str(), ios::out);
  extern viewport view;
  if (file) {
    file << "view_width " << view.width << eol;
    file << "view_height " << view.height << eol;
    file << "win_left " << din0.win.left << eol;
    file << "win_bottom " << din0.win.bottom << eol;
  } else {
    cout << FAIL << "!!! couldnt save window in " << fname << " !!!" << ENDL;
    return;
  }

  cout << PASS << "+++ saved window in " << fname << " +++" << ENDL;


}

Display *display; // X display

// din window
Window win;
XSetWindowAttributes swa;
XWindowAttributes gwa;

// OpenGL settings for din window
GLint glattribs [] = {GLX_RGBA, GLX_DOUBLEBUFFER, None};
XVisualInfo *visual;
Colormap colormap;
GLXContext  glc;

// X input
XEvent event;

void load_instrument () {
  extern ui_list uis;
  if (INSTRUMENT == "keyboard_keyboard") {
    uis.set_current (&keybd2, 0);
    cons ("setup-editors");
  } else {
    uis.set_current (&din0, 0);
    cons ("setup-editors");
  }
}

void accept_keypress (int accept) {
  swa.event_mask = ExposureMask | StructureNotifyMask | FocusChangeMask;
  if (accept) {
    swa.event_mask |= KeyPressMask;
    xkey = 0;
  }
  XChangeWindowAttributes (display, win, CWColormap | CWEventMask, &swa);
}

void warp_pointer_abs (int x, int y) {
  XWarpPointer (display, None, win, 0, 0, view.width, view.height, x, y);
}

void warp_pointer (int dx, int dy) {
  XWarpPointer (display, win, None, 0, 0, view.width, view.height, dx, dy);
}

void restore_last_window () {

  extern int SCREEN_WIDTH, SCREEN_HEIGHT;
  int width = SCREEN_WIDTH, height = SCREEN_HEIGHT; // defaults from ~/.din/globals

  extern std::string dotdin;
  std::string fname (dotdin + "window");
  ifstream file (fname.c_str(), ios::in); // ~/.din/window

  if (file) {

    std::string ignore;

    // last window size
    file >> ignore >> width;
    file >> ignore >> height;

    // last din board position
    file >> ignore >> din0.win.left;
    file >> ignore >> din0.win.bottom;

  } else cout << FAIL << "<<< couldnt load last window from: " << fname << ", will use defaults >>>" << ENDL;

  view (width, height);
  din0.prev_mousey = view.ymax;

}

int focus = 0; // din has focus?

int quit_now = 0; // quit now?

int main (int argc, char** argv) {

  int ignore = system (". ~/.din/m00"); // xset m 0 0 for clean mouse data, without any acceleration applied by X

  // bring up Tcl interpreter
  std::string source = "source " + dotdin;

  std::string load_init_tcl (source + "init.tcl"); // init scripts
  interpreter (load_init_tcl);

  std::string load_settings_tcl (source + "settings.tcl"); // init din settings
  interpreter (load_settings_tcl);

  setup_editors ();
  setup_curve_list ();

  // bring up OpenGL window
  //

  display = XOpenDisplay (0);
  if (display == 0) {
    cout << FAIL << "!!! couldnt connect to X server !!!" << ENDL;
    exit (1);
  }

  visual = glXChooseVisual(display, DefaultScreen (display), glattribs);

  if(visual == 0) {
    cout << FAIL << "!!! couldnt setup GLX visual !!!" << ENDL;
    exit (1);
  } else {
    cout << PASS << "+++ opened GLX visual " << (void *) visual->visualid << " for rendering +++" << ENDL;
  }

  colormap = XCreateColormap (display, RootWindow (display, visual->screen), visual->visual, AllocNone);

  swa.colormap = colormap;
  swa.event_mask = ExposureMask | StructureNotifyMask | FocusChangeMask;

  restore_last_window ();
  win = XCreateWindow (display, RootWindow (display, visual->screen), 0, 0, view.width, view.height, 0, visual->depth, InputOutput, visual->visual, CWColormap | CWEventMask, &swa);

  Atom delete_window_message = XInternAtom (display, "WM_DELETE_WINDOW", False);
  XSetWMProtocols (display, win, &delete_window_message, 1);

  glc = glXCreateContext(display, visual, 0, GL_TRUE);
  glXMakeCurrent(display, win, glc);

  XMapWindow(display, win); // show window
  XMoveWindow (display, win, 0, 0);

  extern std::string WIN_TITLE;
  XStoreName (display, win, WIN_TITLE.c_str());

  glClearColor(0, 0, 0, 0); // black bg

  uis.set_current (&anote, 0); // open with author's note

  cons.home (); // display all text in console

  aout.set_callbacks ();
  aout.start (); // start audio loop in a separate thread. see audio_wanted (..)

  // for mouse
  Window rw, cw;
  unsigned int buttons;
  int rx, ry;

  extern bot_t bot;

  // ui loop
  double futuret = 0;
  extern double FPS_T;
  extern int USLEEP;
  ui_clk.reset ();

  while (1) {

    if (quit_now) {
      if ((din0.drones.size () == 0) && (uis.fdr_voice.on == 0) && (uis.fdr_delay.on == 0) && (keybd2.triggered_notes.size () == 0)) // make sure all sound making things are faded out/off
        break;
    }

    if (focus) { // get mouse & keyboard state if din window has focus

      XQueryKeymap (display, keys.now); // get keyboard state
      XQueryPointer (display, win, &rw, &cw, &rx, &ry, &mousex, &mousey, &buttons); // get mouse state
      mouseyy = view.ymax - mousey;

      // mouse buttons
      lmb = buttons & Button1Mask;
      rmb = buttons & Button3Mask;

    }

    // handle input
    //
    osc (); // OSC
    midi (); // MIDI via JACK
    bot.handle_input (); // IRC mesgs/cmds
    uis.handle_input (); // ui mesgs & background processing

    memcpy (keys.last, keys.now, key_state::NUM_BYTES); // save keyboard state

    // write audio
    //
    if (aout.can_write ()) {

      float* out0 = aout.writep; // left
      float* out1 = out0 + aout.samples_per_channel; // right

      // apply octave shift
      //
      if (osd.active) do_octave_shift ();

      memset (aout.writep, 0, aout.samples_buffer_size); // silence everything

      interpreter ("loop");

      din0.render_audio (out0, out1);
      keybd2.render_audio (out0, out1);
      applyfx (out0, out1, din0.dinfo.delay, din0.dinfo.compress);

      // add audio to oscilloscope
      if (!din0.osc.paused && din0.osc.visible) din0.osc.add_samples (out0, out1, aout.samples_per_channel);

      update_audio ();

    }


    // draw ui
    //
    if (ui_clk.secs >= futuret) { // fps & ui sleep limiting to manage processor usage
      uis.draw ();
      glXSwapBuffers (display, win);
      futuret = ui_clk.secs + FPS_T;
    }

    // handle window event
    if ( XPending (display) ) {

      XNextEvent(display, &event);

      if (focus) {
        if (event.type == KeyPress) { // for console typings
          XLookupString (&event.xkey, &xkey, 1, 0, 0);
        } else if (event.type == FocusOut) focus = 0;
      } else if (event.type == FocusIn) focus = 1;

      if(event.type == Expose) { // window resized
        XGetWindowAttributes (display, win, &gwa);
        update_window (gwa.width, gwa.height, view.width, view.height);
      } else if ((event.type == ClientMessage) && (event.xclient.data.l[0] == (unsigned int) delete_window_message)) { // window deleted
        // can only quit thru ESC, ESC again (see ui.cc)
      }

    }

    if (USLEEP) usleep (USLEEP);

    ui_clk.tick();

  }

  aout.close ();

  // saves
  save_window ();
  std::string save_settings_tcl ("source " + dotdin + "save_settings.tcl");
  interpreter (save_settings_tcl);
  ignore = system ("xset m $ACCEL $THRESHOLD 2>err"); // restore mouse settings

  // glx quit
  glXMakeCurrent(display, None, 0);
  glXDestroyContext(display, glc);
  XDestroyWindow(display, win);
  XCloseDisplay(display);

  return 0;

}
