/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "fft.h"
#include "multi_curve.h"
#include "solver.h"
#include "math.h"
#include "ansi_color_codes.h"

#include <algorithm>
#include <iostream>
using namespace std;

fft::fft () : lev ("fft_levels") {

  n = 20; // 2 x number of harmonics
  halfn = n / 2;
  dx = 1. / n;
  x = 0;

  // use fftw library for fft
  in = (float *) fftwf_malloc (sizeof(float) * n);
  out = (fftwf_complex *) fftwf_malloc (sizeof (fftwf_complex) * (halfn + 1));

  extern string dotdin;
  string fname = dotdin + "fftwf_wisdom";
  FILE* wis = fopen (fname.c_str(), "r");
  if (wis) {
    cout << DOING << "+++ using saved FFT wisdom +++" << ENDL;
    fftwf_import_wisdom_from_file (wis);
    fclose (wis);
  }

  plan = fftwf_plan_dft_r2c_1d (n, in, out, FFTW_MEASURE);

  cout << PASS << "+++ FFT setup complete +++ " << ENDL;
}

fft::~fft () {
    extern string dotdin;
    string fname = dotdin + "fftwf_wisdom";
    FILE* wis = fopen (fname.c_str(), "w");
    if (wis) {
      fftwf_export_wisdom_to_file (wis);
      fclose (wis);
    }
  fftwf_destroy_plan (plan);
  fftwf_free (in);
  fftwf_free (out);
}

void fft::go (multi_curve* crv) {

  // do fft on a bezier multi curve

  solver s (crv);

  x = 0;
  s (x, dx, n, in);

  fftwf_execute (plan);

  harms.clear ();
  for (int i = 1, j = halfn +1; i < j; ++i) {
    float x = out[i][0];
    float y = out[i][1];
    float x2 = x*x;
    float y2 = y*y;
    harms.push_back (sqrt (x2 + y2));
    // ignore phase (atan2 (y2/x2)) for now, but have to add in later revision.
  }

  float m0 = *min_element (harms.begin(), harms.end());
  float m1 = *max_element (harms.begin(), harms.end());
  float mm = m1 - m0;
  if (m1 == m0) return;

  for (int i = 0, j = harms.size(); i < j; ++i) {
    float dm = harms[i] - m0;
    float y = dm / mm;
    lev.set (i, y);
  }

}
