(ns leiningen.debian.dh-clojure-lein
  "A leiningen middleware to help adjust upstream projects for debian.

  See dh-clojure-lein(7) for more information."
  (:require
   [clojure.pprint :refer [pprint]]
   [clojure.string :as str]
   [leiningen.core.main :as main]
   [leiningen.core.project :as proj])
  (:import
   [java.io FileNotFoundException]))

(defn abort [& msg] (apply main/abort "dh-clojure-lein:" msg))
(defn info [& msg] (apply main/info "dh-clojure-lein:" msg))
(defn debug [& msg] (apply main/debug "dh-clojure-lein:" msg))

(defn normalize-dep [dep]
  ;; leiningen can handle a missing version, but always expects a vec
  (if (symbol? dep) [dep] dep))

(defn dep-map-key [m]
  ;; See proj/dep-key
  (select-keys m [:group-id :artifact-id :classifier :extension]))

(defn dep-vecs->map
  "Returns a leiningen project style map for a project.clj vector of
  dependencies like [[org.clojure/clojure \"42.x\"]].  The result maps
  DEP-KEYs to the corresponding dependency maps, e.g. {{:group-id
  \"org.clojure\", :artifact-id \"clojure\"} {:artifact-id
  \"clojure\", :group-id \"org.clojure\", :version \"42.x\"}}."
  [deps]
  (reduce (fn add-dep-map [m dep]
            (let [dm (-> dep normalize-dep proj/dependency-map)]
              (assoc m (dep-map-key dm) dm)))
          {}
          deps))

(defn- map->dep-vecs [m]
  (mapv proj/dependency-vec (vals m)))

(defn- set-default-dep-map-versions [dep-map]
  (reduce-kv (fn set-default-version [result dep-k info]
               (assoc result dep-k
                      (assoc info :version
                             (case [(:group-id dep-k) (:artifact-id dep-k)]
                               ["org.clojure" "clojure"] "1.x"
                               "debian"))))
             {} dep-map))

(defn- exec-config
  "This leiningen middleware loads the config into a temporary namespace
  which (currently) must define dhclj-adjust-project.  That function
  is invoked with one argument, a function that will return the given
  project, after adjusting it to have leiningen project style
  dependency maps (see dep-vecs->map), and after defaulting all the
  version via set-default-dep-map-versions.  The dhclj-adjust-project
  invocation must return a possibly modified version of the project,
  which will be returned from this function after being converted back
  to leiningen's project.clj format (e.g. with dependency vectors, not
  maps).  Although the config will typically be a
  path (e.g. debian/ch-clojure-lein.clj), it may also be a vector of
  forms or nil, the latter handled just as if a path had been given
  and the file was missing."
  [config project]
  (let [n (create-ns (gensym "dh-clojure-lein-config-"))
        found? (binding [*ns* n]
                 (refer-clojure)
                 (cond
                   (nil? config) false
                   (sequential? config)
                   (do
                     (doseq [form config] (eval form))
                     true)
                   :else
                   (try
                     (load-file config)
                     true
                     (catch FileNotFoundException _
                       false))))
        adjust (when found? (ns-resolve n 'dhclj-adjust-project))]
    (if-not found?
      (if (nil? config)
        (info "no config (applying defaults)")
        (info "no" (pr-str config) "(applying defaults)"))
      (when-not adjust ;; for now, required
        (abort "dhclj-adjust-project was not defined in" (pr-str config))))
    (let [prep-deps #(-> % dep-vecs->map set-default-dep-map-versions)
          prepared (-> project
                       (update :dependencies prep-deps)
                       (update :managed-dependencies prep-deps)
                       (update :plugins prep-deps))]
      ;; Pass a function instead of the project itself, so we'll have
      ;; room to maneuver if needed, e.g. if we need support for
      ;; something like (get-project :unmodified true)
      (-> (if adjust
            (binding [*ns* n] (adjust (fn get-project [] prepared)))
            prepared)
          (update :dependencies map->dep-vecs)
          (update :managed-dependencies map->dep-vecs)
          (update :plugins map->dep-vecs)))))

(defn middleware
  [project]
  (let [project (exec-config "debian/dh-clojure-lein.clj" project)]
    (when main/*debug*
      (debug "adjusted project")
      (main/debug (str/trim-newline (with-out-str (pprint project)))))
    project))
