/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* PATH_MAX */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
// #include <dirent.h> /* PATH_MAX */

#include "cdw_ncurses.h"
#include "cdw_config.h"
#include "cdw_colors.h"
#include "cdw_string.h"
#include "cdw_fs.h"
#include "cdw_debug.h"
#include "cdw_utils.h"


/**
   \file cdw_colors.c

   This file implements:
   \li reading color scheme (pairs of color attributes for fore- and
       background color) from cdw colors configuration file
   \li passing those pairs to ncurses function init_pair() to initialize
       'curses pair' (pair in strict, ncurses-related sense, see
       'man init_pair' for reference)
   \li a mechanism of referring to these 'curses pairs' by symbolic names
       in form of CDW_COLORS_MAIN or CDW_COLORS_MENU etc.

   Conventions used in this file:
   \li 'color attribute' is variable with name of form CONF_COLOR_MAIN_FORE
       or CONF_COLOR_DIALOG_BACK - an int value describing color of foreground
       or background of some class of UI elements (1 is class of elements
       used in main window, 2 is class of elements used to build dialog
       windows, see list below)
   \li 'pair' is used to call two color attributes describing fore- and
       background of the same class of UI elements. Values of pair of the
       attributes are used as 2nd and 3rd arguments to init_pair() function
       call. First argument of this function call is a 'ncurses pair' - pair
       in strict sense. Such pairs are referred to in code in other files by
       symbolic names such as CDW_COLORS_WARNING
   \li 'line' is one line read from cdw color configuration file

   CDW uses following pairs of foreground-background colors (values of type
   cdw_colors_t):
   1  -> CDW_COLORS_MAIN      - main window
   2  -> CDW_COLORS_DIALOG    - dialog windows of various kinds
   3  -> CDW_COLORS_TITLE     - application title in main window
   4  -> CDW_COLORS_MENU      - menus (including main application menu)
   5  -> CDW_COLORS_PROGRESS  - progress bar
   6  -> CDW_COLORS_INPUT     - input fields
   7  -> CDW_COLORS_WARNING   - warning, may be used in dialogs
   8  -> CDW_COLORS_SIZE_INFO - (unused, kept for compatibility)
   9  -> CDW_COLORS_DISABLED  - disabled items
   10 -> CDW_COLORS_UNUSED    - unused
   11 -> CDW_COLORS_TOOLTIPS  - tooltips at the bottom of main window
   12 -> CDW_COLORS_TEXTAREA  - text areas, like window displaying license
   13 -> CDW_COLORS_ERROR     - error - may be used in dialogs

   These values should be used when passing color argument to cdw widgets, e.g.
   cdw_dialogbox("Message",
                 "This is how you should use values of type cdw_colors_t",
		 DIALOG_OK, CDW_COLORS_DIALOG);

   When setting colors of bare ncurses widgets you should use COLORS_PAIR()
   macro, like this:
   wbkgd(my_window, COLORS_PAIR(CDW_COLORS_DIALOG));
*/


enum {
	CONF_EMPTY           = -3, /* empty line */
	CONF_INVALID         = -2, /* invalid entry */
	CONF_COMMENT         = -1, /* comment, line starting with '#' */
	CONF_COLOR_MAIN_FORE = 0,
	CONF_COLOR_MAIN_BG,
	CONF_COLOR_DIALOG_FORE,
	CONF_COLOR_DIALOG_BG,
	CONF_COLOR_TITLE_FORE,
	CONF_COLOR_TITLE_BG,
	CONF_COLOR_MENU_FORE,
	CONF_COLOR_MENU_BG,
	CONF_COLOR_PROGRESS_FORE,
	CONF_COLOR_PROGRESS_BG,
	CONF_COLOR_INPUT_FORE,
	CONF_COLOR_INPUT_BG,
	CONF_COLOR_WARNING_FORE,
	CONF_COLOR_WARNING_BG,
	CONF_COLOR_SIZE_INFO_FORE,
	CONF_COLOR_SIZE_INFO_BG,
	CONF_COLOR_DISABLED_FORE,
	CONF_COLOR_DISABLED_BG,
	CONF_COLOR_UNUSED_FORE,
	CONF_COLOR_UNUSED_BG,
	CONF_COLOR_TOOLTIPS_FORE,
	CONF_COLOR_TOOLTIPS_BG,
	CONF_COLOR_TEXTAREA_FORE,
 	CONF_COLOR_TEXTAREA_BG,
	CONF_COLOR_ERROR_FORE,
	CONF_COLOR_ERROR_BG
};


enum {
	CDW_COLOR_MAIN    =  0,
	CDW_COLOR_DIALOG,
	CDW_COLOR_TITLE,
	CDW_COLOR_MENU,
	CDW_COLOR_PROGRESS,
	CDW_COLOR_INPUT,
	CDW_COLOR_WARNING,
	CDW_COLOR_SIZE_INFO,
	CDW_COLOR_DISABLED,
	CDW_COLOR_UNUSED,
	CDW_COLOR_TOOLTIPS,
	CDW_COLOR_TEXTAREA,
	CDW_COLOR_ERROR,

	CDW_COLOR_INIT,    /* initialization value */
	CDW_COLOR_EMPTY,   /* empty line */
	CDW_COLOR_INVALID, /* invalid entry */
	CDW_COLOR_COMMENT, /* comment, line starting with '#' */
	CDW_COLOR_FINAL,   /* guard */
};



//#define CDW_COLORS_NEW

#ifdef CDW_COLORS_NEW

cdw_id_clabel_t cdw_color_schemes[] = {
	{ CDW_COLOR_MAIN,      "MAIN"        },
	{ CDW_COLOR_DIALOG,    "DIALOG"      },
	{ CDW_COLOR_TITLE,     "TITLE"       },
	{ CDW_COLOR_MENU,      "MENU"        },
	{ CDW_COLOR_PROGRESS,  "PROGRESS"    },
	{ CDW_COLOR_INPUT,     "INPUT"       },
	{ CDW_COLOR_WARNING,   "WARNING"     },
	{ CDW_COLOR_SIZE_INFO, "SIZE_INFO"   },
	{ CDW_COLOR_DISABLED,  "DISABLED"    },
	{ CDW_COLOR_UNUSED,    "UNUSED"      },
	{ CDW_COLOR_TOOLTIPS,  "TOOLTIPS"    },
	{ CDW_COLOR_TEXTAREA,  "TEXTAREA"    },
	{ CDW_COLOR_ERROR,     "ERROR"       },

	{ CDW_COLOR_INIT,      "INIT"        }, /* initialization value */
	{ CDW_COLOR_EMPTY,     "EMPTY"       }, /* empty line */
	{ CDW_COLOR_INVALID,   "INVALID"     }, /* invalid entry */
	{ CDW_COLOR_COMMENT,   "COMMENT"     }, /* comment, line starting with '#' */
	{ CDW_COLOR_FINAL,     "FINAL"       },

	{ -1,                  (char *) NULL }
};



cdw_id_clabel_t cdw_color_names[] = {
	{ COLOR_BLACK,   "BLACK"   },
	{ COLOR_RED,     "RED"     },
	{ COLOR_GREEN,   "GREEN"   },
	{ COLOR_YELLOW,  "YELLOW"  },
	{ COLOR_BLUE,    "BLUE"    },
	{ COLOR_MAGENTA, "MAGENTA" },
	{ COLOR_CYAN,    "CYAN"    },
	{ COLOR_WHITE,   "WHITE"   },

	{ COLOR_BLACK   | A_BOLD, "BOLD_BLACK"   },
	{ COLOR_RED     | A_BOLD, "BOLD_RED"     },
	{ COLOR_GREEN   | A_BOLD, "BOLD_GREEN"   },
	{ COLOR_YELLOW  | A_BOLD, "BOLD_YELLOW"  },
	{ COLOR_BLUE    | A_BOLD, "BOLD_BLUE"    },
	{ COLOR_MAGENTA | A_BOLD, "BOLD_MAGENTA" },
	{ COLOR_CYAN    | A_BOLD, "BOLD_CYAN"    },
	{ COLOR_WHITE   | A_BOLD, "BOLD_WHITE"   },
	{ -1,                     (char *) NULL  }
};

#endif

struct {
	cdw_id_t fore;
	cdw_id_t back;
} cdw_colors_table[CDW_COLOR_FINAL];



#define CDW_COLORS_BUFFER_SIZE 999

#ifdef CDW_COLORS_NEW
static int cdw_colors_new_get_and_init_color(const char *line, cdw_id_t scheme);
#endif

static short int cdw_colors_get_attribute_name(char *line);
static int  cdw_colors_get_attribute_value(const char *source);
static void cdw_colors_read_config_file(FILE *file);
static void cdw_colors_read_and_init_color(const char *line, int i);

static const char *config_file_name = "cdw.colors";


/**
   \brief Set up defaut (hardwired) color schemes

   \date Function's top-level comment reviewed on 2012-02-05
   \date Function's body reviewed on 2012-02-05

   Use this function right after setting up basic curses to
   provide colors for some widgets. It is required to have some
   colors defined when fs module is not set up yet, because you
   might have to display dialog or input widgets when configuring
   file system module.
*/
void cdw_colors_init_phase1(void)
{
	for (int i = 0; i < CDW_COLOR_FINAL; i++) {
		cdw_colors_table[i].fore = CDW_COLOR_INIT;
		cdw_colors_table[i].back = CDW_COLOR_INIT;
	}

	/* first set default values: foreground - background */
	init_pair(CDW_COLORS_MAIN, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_DIALOG, COLOR_WHITE, COLOR_BLUE);
	init_pair(CDW_COLORS_TITLE, COLOR_RED, COLOR_BLACK);
	init_pair(CDW_COLORS_MENU, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_PROGRESS, COLOR_YELLOW, COLOR_BLACK);
	init_pair(CDW_COLORS_INPUT, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_WARNING, COLOR_BLUE, COLOR_YELLOW);
	init_pair(CDW_COLORS_SIZE_INFO, COLOR_WHITE, COLOR_YELLOW);
	init_pair(CDW_COLORS_DISABLED, COLOR_BLACK, COLOR_WHITE);
	init_pair(CDW_COLORS_UNUSED, COLOR_WHITE, COLOR_RED);
	init_pair(CDW_COLORS_TOOLTIPS, COLOR_YELLOW, COLOR_BLACK);
	init_pair(CDW_COLORS_TEXTAREA, COLOR_WHITE, COLOR_BLACK);
	/* FIXME: black letters on red background aren't too readable,
	   fix this by using other color scheme for error dialogs */
	init_pair(CDW_COLORS_ERROR, COLOR_BLACK, COLOR_RED);

	return;
}





/**
   \brief Read custom colors from config file, use them to set up color schemes

   \date Function's top-level comment reviewed on 2012-01-12
   \date Function's body reviewed on 2012-01-12

   Call this function when file system module is set up.

   \return CDW_ERROR on errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_colors_init_phase2(void)
{
	const char *dir = cdw_config_get_config_dir();
	if (!dir) {
		cdw_vdm ("ERROR: can't get config dir\n");
		return CDW_ERROR;
	}

	char *path = cdw_string_concat(dir, config_file_name, (char *) NULL);
	if (!path) {
		cdw_vdm ("ERROR: concatenation failed\n");
		return CDW_ERROR;
	}

	size_t len = strlen(path);
	/* PATH_MAX includes ending null */
	if (len > PATH_MAX - 1) {
		free(path);
		path = (char *) NULL;
		return CDW_ERROR;
	}
	FILE *f = fopen(path, "r");
	if (!f) {
		cdw_vdm ("WARNING: can't open color file \"%s\"\n", path);
		free(path);
		path = (char *) NULL;
		return CDW_ERROR;
	} else {
		cdw_colors_read_config_file(f);
		fclose(f);
		free(path);
		path = (char *) NULL;
		cdw_vdm ("INFO: color file read properly\n");
		return CDW_OK;
	}
}





/**
   \brief Read cdw colors configuration file, redefine default colors

   \date Function's top-level comment reviewed on 2012-02-05
   \date Function's body reviewed on 2012-02-05

   Read configuration file with definitions of color schemes.
   If a color is defined in the config file, redefine the default color
   definition (using init_pair() calls) for this ncurses application.

   \param file - color configuration file, it has to be already open
*/
void cdw_colors_read_config_file(FILE *file)
{
	cdw_assert (file, "ERROR: you forgot to open file\n");
	/* this function is called when there is some color config file,
	   let's change default values */

	char buffer[CDW_COLORS_BUFFER_SIZE];
	int old_style = 0;
	//int new_style = 0;

	while (fgets(buffer, CDW_COLORS_BUFFER_SIZE, file)) {
		buffer[CDW_COLORS_BUFFER_SIZE - 1] = '\0';

		char *line = cdw_string_ltrim(buffer);
		cdw_string_rtrim(line);

		int i = 0;
		switch (i = cdw_colors_get_attribute_name(line)) {
		        case CONF_COMMENT:
		        case CONF_EMPTY:
				break;
			case CONF_INVALID:
				cdw_vdm ("ERROR: invalid line in colors config file:\n");
				cdw_vdm ("ERROR: \"%s\n", line);
				break;
			default:
				old_style++;
				i /= 2;
				cdw_colors_read_and_init_color(line, i);
				break;
		}
#ifdef CDW_COLORS_NEW
		cdw_option_t option;
		if (cdw_config_split_options_line_new(&option, line)) {
			cdw_id_t scheme = cdw_utils_id_label_table_get_id(cdw_color_schemes, option.name, CDW_COLOR_INVALID);
			switch (scheme) {
		                case CDW_COLOR_COMMENT:
		                case CDW_COLOR_EMPTY:
					break;
			        case CDW_COLOR_INVALID:
					cdw_vdm ("ERROR: invalid line in colors config file: \"%s\n", line);
					break;
			        default:
					new_style++;
					cdw_colors_new_get_and_init_color(option.value, scheme);
					break;
			}
		} else {
			cdw_vdm ("WARNING: failed to get option's name and value from line \"%s\"\n", line);
		}
		cdw_config_option_free_new(&option);
#endif
	}
#ifdef CDW_COLORS_NEW
	for (int i = 0; i <= CDW_COLOR_ERROR; i++) {
		cdw_vdm ("INFO: scheme = \"%10s\", fore = \"%10s\", back = \"%8s\", %lld, %lld\n",
			 cdw_utils_id_label_table_get_label(cdw_color_schemes, i),
			 cdw_utils_id_label_table_get_label(cdw_color_names, cdw_colors_table[i].fore),
			 cdw_utils_id_label_table_get_label(cdw_color_names, cdw_colors_table[i].back), cdw_colors_table[i].fore, cdw_colors_table[i].back);
	}
#endif
	return;
}





/**
   \brief Recognize color definition values and save them

   \date Function's top-level comment reviewed on 2012-02-05
   \date Function's body reviewed on 2012-02-05

   Function recognizes definition of color \i stored in \p line,
   saves it in temporary variable, and uses it to redefine color \p i

   \param line - line from color config file
   \param i - color index
*/
void cdw_colors_read_and_init_color(const char *line, int i)
{
	static int fb = 0; /* indicates which value was read in: fore- or background; even values are for fore */
	static int fore = CONF_INVALID; /* foreground color read from file */
	static int back = CONF_INVALID; /* background color read from file */

	/* code in 'default' branch checks if both
	   foreground and background colors were read
	   in properly, and if so, then places them
	   in color attributes table; if one or both
	   of values are invalid, they are discarded,
	   and default values are used (are not
	   overwritten) */

	if (fb % 2 == 0) { /* this is line with foreground */
		fore = cdw_colors_get_attribute_value(line);
		cdw_vdm ("INFO: fb = %d, fore = %d, line = '%s'\n", fb, fore, line);
		fb++;

		cdw_colors_table[i].fore = fore;

	} else { /* this is line with background */
		back = cdw_colors_get_attribute_value(line);
		cdw_vdm ("INFO: fb = %d, back = %d, line = '%s'\n", fb, back, line);
		fb++;

		cdw_colors_table[i].back = back;
	}

	if (fb % 2 == 0) {
		/* fb is even -> fore and back possibly store parameters
		   of a new color definition -> use them to redefine color i */

		if (fore != CONF_INVALID && back != CONF_INVALID) {
			/* init_pair(short pair, short fore, short back) */
			init_pair((short) (i + 1), (short) fore, (short) back);
		}
	}
}





/**
   \brief Check what variable is described by given line

   \date Function's top-level comment reviewed on 2012-01-12
   \date Function's body reviewed on 2012-01-12

   Check which one of all possible color scheme attributes is described by
   given line. The line should be read from cdw color configuration file.

   This function can recognize empty or invalid line (is not empty and is
   not a comment) and discard it (return some error value).

   Lines containing only white chars are treated as empty lines.
   Lines starting with comment char ('#') are also signalled.

   \param line - line from configuration file to be parsed

   \return CONF_COMMENT if given line has only comment (starts with comment char)
   \return CONF_EMPTY if given line is empty
   \return CONF_INVALID if line is invalid
   \return CONF_COLOR_* - value corresponding to particular color scheme attribute.
*/
short int cdw_colors_get_attribute_name(char *line)
{
	const char *tline = cdw_string_ltrim(line);

	if (tline[0] == '#') {
		return CONF_COMMENT;
	} else if ((strlen(tline) == 0) || tline[0] == '\n') {
		return CONF_EMPTY;
	} else {
		/* strings passed as second arguments of starts_with_ci()
		   are left unchanged for compatibility with existing
		   cdw color configuration files, although it would be
		   nice to have variable names in more readable form of
		   Conf_color_main_fore */
		if (cdw_string_starts_with_ci(tline, "Color1_fore"))  return CONF_COLOR_MAIN_FORE;
		if (cdw_string_starts_with_ci(tline, "Color1_bg"))    return CONF_COLOR_MAIN_BG;
		if (cdw_string_starts_with_ci(tline, "Color2_fore"))  return CONF_COLOR_DIALOG_FORE;
		if (cdw_string_starts_with_ci(tline, "Color2_bg"))    return CONF_COLOR_DIALOG_BG;
		if (cdw_string_starts_with_ci(tline, "Color3_fore"))  return CONF_COLOR_TITLE_FORE;
		if (cdw_string_starts_with_ci(tline, "Color3_bg"))    return CONF_COLOR_TITLE_BG;
		if (cdw_string_starts_with_ci(tline, "Color4_fore"))  return CONF_COLOR_MENU_FORE;
		if (cdw_string_starts_with_ci(tline, "Color4_bg"))    return CONF_COLOR_MENU_BG;
		if (cdw_string_starts_with_ci(tline, "Color5_fore"))  return CONF_COLOR_PROGRESS_FORE;
		if (cdw_string_starts_with_ci(tline, "Color5_bg"))    return CONF_COLOR_PROGRESS_BG;
		if (cdw_string_starts_with_ci(tline, "Color6_fore"))  return CONF_COLOR_INPUT_FORE;
		if (cdw_string_starts_with_ci(tline, "Color6_bg"))    return CONF_COLOR_INPUT_BG;
		if (cdw_string_starts_with_ci(tline, "Color7_fore"))  return CONF_COLOR_WARNING_FORE;
		if (cdw_string_starts_with_ci(tline, "Color7_bg"))    return CONF_COLOR_WARNING_BG;
		if (cdw_string_starts_with_ci(tline, "Color8_fore"))  return CONF_COLOR_SIZE_INFO_FORE;
		if (cdw_string_starts_with_ci(tline, "Color8_bg"))    return CONF_COLOR_SIZE_INFO_BG;
		if (cdw_string_starts_with_ci(tline, "Color9_fore"))  return CONF_COLOR_DISABLED_FORE;
		if (cdw_string_starts_with_ci(tline, "Color9_bg"))    return CONF_COLOR_DISABLED_BG;
		if (cdw_string_starts_with_ci(tline, "Color10_fore")) return CONF_COLOR_UNUSED_FORE;
		if (cdw_string_starts_with_ci(tline, "Color10_bg"))   return CONF_COLOR_UNUSED_BG;
		if (cdw_string_starts_with_ci(tline, "Color11_fore")) return CONF_COLOR_TOOLTIPS_FORE;
		if (cdw_string_starts_with_ci(tline, "Color11_bg"))   return CONF_COLOR_TOOLTIPS_BG;
		if (cdw_string_starts_with_ci(tline, "Color12_fore")) return CONF_COLOR_TEXTAREA_FORE;
		if (cdw_string_starts_with_ci(tline, "Color12_bg"))   return CONF_COLOR_TEXTAREA_BG;
		if (cdw_string_starts_with_ci(tline, "Color13_fore")) return CONF_COLOR_ERROR_FORE;
		if (cdw_string_starts_with_ci(tline, "Color13_bg"))   return CONF_COLOR_ERROR_BG;
	}

	return CONF_INVALID;
}





/**
   \brief Extract color value from line read from color config file

   \date Function's top-level comment reviewed on 2012-01-12
   \date Function's body reviewed on 2012-01-12

   Extract name of color (e.g. BLACK, BOLD_YELLOW etc.) from given string
   that comes from configuration file.

   Sample argument is like "Color10_bg = BLACK".
   We have to extract right-hand value after '=' char.

   Spurious white chars are stripped by this function.

   \param line - line read from color config file

   \return curses color attribute corresponding to value extracted from line
   \return -1 if line does not have any color attribute value
*/
int cdw_colors_get_attribute_value(const char *line)
{
	char *eq = strstr(line, "=");
	if (!eq) {
		return -1;
	}

	const char *value = cdw_string_ltrim(eq + 1);

	if (cdw_string_starts_with_ci(value, "BLACK"))   return COLOR_BLACK;
	if (cdw_string_starts_with_ci(value, "RED"))     return COLOR_RED;
	if (cdw_string_starts_with_ci(value, "GREEN"))   return COLOR_GREEN;
	if (cdw_string_starts_with_ci(value, "YELLOW"))  return COLOR_YELLOW;
	if (cdw_string_starts_with_ci(value, "BLUE"))    return COLOR_BLUE;
	if (cdw_string_starts_with_ci(value, "MAGENTA")) return COLOR_MAGENTA;
	if (cdw_string_starts_with_ci(value, "CYAN"))    return COLOR_CYAN;
	if (cdw_string_starts_with_ci(value, "WHITE"))   return COLOR_WHITE;

	if (cdw_string_starts_with_ci(value, "BOLD_BLACK"))   return COLOR_BLACK   | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_RED"))     return COLOR_RED     | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_GREEN"))   return COLOR_GREEN   | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_YELLOW"))  return COLOR_YELLOW  | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_BLUE"))    return COLOR_BLUE    | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_MAGENTA")) return COLOR_MAGENTA | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_CYAN"))    return COLOR_CYAN    | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_WHITE"))   return COLOR_WHITE   | A_BOLD;

	return -1;
}


#ifdef CDW_COLORS_NEW


int cdw_colors_new_get_and_init_color(const char *names, cdw_id_t scheme)
{
	cdw_vdm ("INFO: ooo: scheme = %lld, names = \"%s\"\n", scheme, names);

	/* we expect that 'names' should have this form:
	   " <fore color> / <back color> "; */

	char *slash = strstr(names, "/");
	if (!slash) {
		return -1;
	}
	*slash = '\0';

	char *fore_name = cdw_string_trim(names);
	char *back_name = cdw_string_trim(slash + 1);

	if (fore_name && back_name) {
		cdw_id_t fore = cdw_utils_id_label_table_get_id(cdw_color_names, fore_name, -1);
		cdw_id_t back = cdw_utils_id_label_table_get_id(cdw_color_names, back_name, -1);

		if (fore != -1 && back != -1) {
			/* init_pair(short pair, short fore, short back) */
			init_pair((short) (scheme + 1), (short) fore, (short) back);
			cdw_colors_table[scheme].fore = fore;
			cdw_colors_table[scheme].back = back;
		}
	}

	if (fore_name) {
		free(fore_name);
		fore_name = (char *) NULL;
	}

	if (back_name) {
		free(back_name);
		back_name = (char *) NULL;
	}

	return 0;
}
#endif
