/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "cdw_utils.h"
#include "cdw_config.h"
#include "gettext.h"
#include "cdw_widgets.h" /* cdw_buttons_dialog() */
#include "cdw_drive.h"
#include "cdw_main_window.h"
#include "cdw_disc.h"
#include "cdw_fs.h"
#include "cdw_debug.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"
#include "cdw_cdio_drives.h"
#include "cdw_string.h"



extern cdw_cdio_drive_t cdw_cdio_drives[];

/* main application configuration variable */
extern cdw_config_t global_config;

static int cdw_drive_toggle_tray2(const char *device_fullpath, bool with_processwin);
static int cdw_drive_execute_request(const char *device_fullpath, long unsigned int request);

static int cdw_drive_close_tray_internal(const char *device_fullpath);

static const char *cdw_drive_get_cds_label(int rv);

static struct {
	CDW_DROPDOWN *drives_dropdown;
	int n_drives;
	/* cdio provides table that is indexed from zero - hence here is "ind"="index" */
	int cdio_default_drive_ind;
	/* cdw uses CDW_DRIVE_ID_CUSTOM == -1 to designate "custom", and
	   0 to N as regular indexes; because of using "-1" the name of
	   variable is "id"; for almost all cases id is a valid index;
	   except for -1; */
	cdw_id_t current_drive_id;
} drives;


static cdw_id_clabel_t drive_requests[] = {
	{ CDROMRESET,         "CDROMRESET" },
	{ CDROMEJECT,         "CDROMEJECT" },
	{ CDROMCLOSETRAY,     "CDROMCLOSETRAY" },
	{ CDROM_DRIVE_STATUS, "CDROM_DRIVE_STATUS" },
	{ CDROM_LOCKDOOR,     "CDROM_LOCKDOOR" },
	{ 0,                  (char *) NULL}}; /* guard */




void cdw_drive_init(void)
{
	drives.n_drives = cdw_cdio_drives_get_n_drives();
	if (drives.n_drives == 0) {
		drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
		return;
	}

	drives.cdio_default_drive_ind = cdw_cdio_drives_get_cdio_default_drive_ind();

	/* global_config.selected_drive is a string describing currently
	   selected drive; it can have three values:
	    - "default": use drive that is indicated as "default" by cdio
	       module; in practice this value happens only when there is
	       no cdw config file, and cdw runs with default settings;
	       there is no "select default drive" option;
	    - "custom": this value of "selected_drive" indicates that
	      the path to "real" drive is in global_config.custom_drive
	    - '/dev/xyz': a path to device file in form of "/dev/xyz";
	      the value is selected from a dropdown, and since the dropdown
	      is built from values provided by cdio module, the value
	      should be present in cdw_cdio_drives[] */

	/* find correct index to table of drives, regardless of value
	   of global_config.selected_drive */
	if (! strcmp(global_config.selected_drive, "default")) {
		drives.current_drive_id = drives.cdio_default_drive_ind;
	} else if (! strcmp(global_config.selected_drive, "custom")) {
		drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
	} else {
		/* we need to search for device path from cdw_drive
		   on list of drives detected by cdio */
		drives.current_drive_id = -1;
		for (int i = 0; i < drives.n_drives; i++) {
			if (!strcmp(cdw_cdio_drives[i].fullpath, global_config.selected_drive)) {
				/* here index is a valid id */
				drives.current_drive_id = i;
				break;
			}
		}
		if (drives.current_drive_id == -1) {
			/* can't find a drive from configuration on list
			   of drives detected by cdio */
			cdw_vdm ("ERROR: can't find a drive from configuration on list of drives detected by cdio: \"%s\"\n", global_config.selected_drive);
			/* fall back to safe value */
			if (drives.n_drives == 0) {
				drives.current_drive_id = CDW_DRIVE_ID_CUSTOM;
			} else {
				/* let's hope that there is some default drive */
				drives.current_drive_id = drives.cdio_default_drive_ind;
			}
		}
	}

	return;
}





void cdw_drive_clean(void)
{
	return;
}





/**
   \brief Check for CD drive status

   Check for CD drive status using ioctl (with parameter CDROM_DRIVE_STATUS).
   Documentation on ioctl can be found in linux source code documentation in
   Documentation/ioctl/cdrom.txt.

   This function tries to open CD device described by \p device_fullpath.

   Return codes are mapped to new names in cdrom_ctrl.h, so there is no need
   to include linux/cdrom.h again in *c files.

   Function does not inform user about errors, caller function has to do it.

   \param device_fullpath - full path to device that you want to check

   \return -1 if cannot open() cdrom file, or ioctl() failed
   \return ioctl return code: CDS_NO_INFO (information not available)
   \return ioctl return code: CDS_NO_DISC
   \return ioctl return code: CDS_TRAY_OPEN
   \return ioctl return code: CDS_DRIVE_NOT_READY
   \return ioctl return code: CDS_DISC_OK
 */
int cdw_drive_status(const char *device_fullpath)
{
	if (!device_fullpath) {
		cdw_vdm ("ERROR: device fullpath is NULL\n");
		return -1;
	} else {
		return cdw_drive_execute_request(device_fullpath, CDROM_DRIVE_STATUS);
	}
}





/**
   \brief Eject cdrom drive tray

   \param device_fullpath - full path to '/dev' file corresponding to device

   \return CDW_OK on success
   \return CDW_SYS_ERROR if function cannot open() device or if ioctl call for this device fails
 */
cdw_rv_t cdw_drive_eject_tray(const char *device_fullpath)
{
	/* sometimes call to ioctl(..., CDROMEJECT) fails, not sure why,
	   especially for DVD discs (and for DVD-RW in particular);
	   perhaps some other process is trying to access freshly burned
	   disc as well?
	   let's give the drive some time to calm down, but it should
	   take much less time than waiting for drive after closing a tray;
	   experiments show that it may take 2-3 seconds before call to
	   cdw_drive_execute_request(..., CDROMEJECT) is successful */
	const int arbitrary_limit = 10;
	for (int i = 0; i < arbitrary_limit; i++) {
		cdw_vdm ("INFO: trying to eject drive, second %d\n", i);
		int rv = cdw_drive_execute_request(device_fullpath, CDROMEJECT);

		cdw_vdm ("INFO: CDROMEJECT returns %d / %s\n", rv, cdw_drive_get_cds_label(rv));

		if (rv == 0) {
			cdw_vdm ("INFO: ejected drive after %d seconds\n", i);
			return CDW_OK;
		} else {
#ifdef CDROM_LOCKDOOR
			usleep(200000);
			/* calls ioctl with 0 as a third argument -> unlock */
			cdw_drive_execute_request(device_fullpath, CDROM_LOCKDOOR);
#else
			/* some error message maybe? */
#endif
			sleep(1);
		}
	}
	cdw_vdm ("ERROR: failed to eject a drive\n");
	return CDW_ERROR;
}





/**
   \brief Close cdrom drive tray

   It is strongly advised to call sleep(x) (with x being e.g. 3) after
   calling this function. This gives the drive some time to recognize media.
   Otherwise any reads from disc in drive may yield incorrect results due to
   timeouts.

   \param device_fullpath - full path to '/dev' file corresponding to device

   \return CDW_OK on success
   \return CDW_ERROR if function cannot open() device or if ioctl call for this device fails
 */
cdw_rv_t cdw_drive_close_tray(const char *device_fullpath)
{
	int rv = cdw_drive_close_tray_internal(device_fullpath);
	if (rv == -1) {
		cdw_vdm ("ERROR: close_tray(%s) returns -1\n", device_fullpath);
		return CDW_ERROR;
	} else {
		if (rv == CDS_NO_DISC || rv == CDS_DISC_OK) {
			cdw_vdm ("INFO: status = %d / %s\n", rv, cdw_drive_get_cds_label(rv));
			return CDW_OK;
		} else {
			cdw_vdm ("ERROR: status = %d / %s\n", rv, cdw_drive_get_cds_label(rv));
			return CDW_ERROR;
		}
	}
}





/**
   \brief Close cdrom drive tray

   It is strongly advised to call sleep(x) (with x being e.g. 3) after
   calling this function. This gives the drive some time to recognize media.
   Otherwise any reads from disc in drive may yield incorrect results due to
   timeouts.

   TODO: make this function an internal function that returns int with
   ioctl return values (CDS_*). Let this module use

   \param device_fullpath - full path to '/dev' file corresponding to device

   \return -1 on error
   \return ioctl return code: CDS_NO_INFO (information not available)
   \return ioctl return code: CDS_NO_DISC
   \return ioctl return code: CDS_TRAY_OPEN
   \return ioctl return code: CDS_DRIVE_NOT_READY
   \return ioctl return code: CDS_DISC_OK
 */
int cdw_drive_close_tray_internal(const char *device_fullpath)
{
	int rv = cdw_drive_execute_request(device_fullpath, CDROMCLOSETRAY);
	if (rv == -1) {
		cdw_vdm ("ERROR: ioctl(%s, CDROMCLOSETRAY, ...) returns -1\n", device_fullpath);
		return -1;
	} else {
		/* Give your drive some time to scan disc after
		   closing tray. The drive really may need 40 seconds:
		   on my not-so-old machine a drive with DVD+RW in it
		   settles its state in 17 seconds, and for DVD-RW it
		   just took 24 seconds. */
		const int arbitrary_limit = 40;
		for (int i = 0; i < arbitrary_limit; i++) {
			rv = cdw_drive_status(device_fullpath);
			if (rv == -1) {
				cdw_vdm ("ERROR: can't get drive status\n");
				break;
			} else if (rv == CDS_DRIVE_NOT_READY) {
				cdw_vdm ("INFO: waiting for drive (CDS_DRIVE_NOT_READY), second %d\n", i);
				sleep(1);
			} else if (rv == CDS_NO_DISC || rv == CDS_DISC_OK) {
				cdw_vdm ("INFO: drive is ready after %d seconds (%d / %s)\n", i, rv, cdw_drive_get_cds_label(rv));
				break;
			} else {
				cdw_vdm ("WARNING: problematic return value %d / %s\n", rv, cdw_drive_get_cds_label(rv));
				break;
			}
		}
		return rv;
	}
}





/* TODO: make the function return errno, so that caller can
   act upon an error */
int cdw_drive_execute_request(const char *device_fullpath, long unsigned int request)
{
	/* yes, I know about sloppy casting of long unsigned int to
	   long signed int, but this is "only" debug code;
	   FIXME: fix this somehow; but how?
	   1. make int field in cdw_id_clabel_t of unsigned type, but
	      IDs can be negative
	   2. make two cdw_id_clabel_t types - for singed and unsigned int
	      field - maintenance nightmare */

	int cddev = open(device_fullpath, O_RDONLY | O_NONBLOCK);
	int e = errno;
	if (cddev == -1) {
		cdw_vdm ("ERROR: failed to open() device \"%s\", for request %s, error = \"%s\"\n",
			 device_fullpath,
			 cdw_utils_id_label_table_get_label(drive_requests, (long int) request),
			 strerror(e));
		return -1;
	}

	int rv = ioctl(cddev, request, 0);
	e = errno;
	close(cddev);
	if (rv == -1) {
		cdw_vdm ("ERROR: ioctl(%s, %s, ...) returns -1 / \"%s\"\n",
			 device_fullpath,
			 cdw_utils_id_label_table_get_label(drive_requests, (long int) request),
			 strerror(e));
	}

	return rv;
}





/**
   \brief Check availability of optical drive and disc

   This function provides information if there is a drive and a disc
   in the drive, and informs user (with dialog window messages) about
   potential problems with finding drive or disc.

   \return CDW_CANCEL if there is no optical drive available
   \return CDW_NO if there is a drive, but there is no disc in it
   \return CDW_OK if there is a drive, and there is a disc in it
 */
cdw_rv_t cdw_drive_disc_availability(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	if (!drive) {
		cdw_vdm ("WARNING: no drive\n");
		cdw_buttons_dialog(_("Warning"), _("No drive.\n"), CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_CANCEL;
	}

	/* 2TRANS: this is title of dialog window:
	   some problem occurred while trying to access disc */
	const char *title = _("CD drive error");
	char *message = (char *) NULL;
	int buttons = 0;
	for (int i = 0; i < 10; i++) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Getting drive status"));
		cdw_processwin_wrefresh();
		int rv = cdw_drive_status(drive);
		cdw_vdm ("INFO: disc status = %d / %s\n", rv, cdw_drive_get_cds_label(rv));
		switch (rv) {
			case CDS_DISC_OK:
				return CDW_OK;
			case CDS_NO_DISC:
				/* 2TRANS: this is message in dialog window: no optical disc
				   found in drive; user can select OK or CANCEL button */
				message =  _("No disc in CD/DVD drive. Please insert disc.");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_TRAY_OPEN:
				/* 2TRANS: this is message in dialog window; user
				   can select OK or CANCEL button */
				message = _("Please close CD/DVD drive tray.");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_DRIVE_NOT_READY:
				sleep(2);
				/* 2TRANS: this is message in dialog window;
				   user can select OK or CANCEL button */
				message = _( "CD/DVD drive doesn't respond (timeout). Try again?");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case CDS_NO_INFO:
				/* 2TRANS: this is message in dialog window;
				   cannot read some meta-data from cd; user
				   can select OK or CANCEL button */
				message = _("Can't get CD/DVD drive info. Continue?");
				buttons = CDW_BUTTONS_OK_CANCEL;
				break;
			case -1: /* cannot open device file or ioctl() failed */
				/* 2TRANS: this is message in dialog window;
				   for some reason program cannot open CD device.
				   User should check his settings */
				message = _("Can't open CD/DVD drive device. Please check your configuration.");
				buttons = CDW_BUTTONS_OK;
				break;
			default: /* all return values from ioctl() and open() covered, so this */
				/* 2TRANS: this is message in dialog window:
				   unknown error occurred */
				message = _("Can't open CD/DVD drive device. Unknown error.");
				buttons = CDW_BUTTONS_OK;
				break;
		} /* switch */

		cdw_rv_t crv = cdw_buttons_dialog(title, message,
						  buttons, CDW_COLORS_ERROR);
		cdw_main_window_wrefresh();
		if (buttons == CDW_BUTTONS_OK) {
			break; /* break loop, go to final error message */
		} else {
			if (crv == CDW_OK) {
				; /* continue loop */
			} else {
				return CDW_NO;
			}
		}

		sleep(5);
	} /* for () */

	cdw_buttons_dialog(title,
			   /* 2TRANS: this is message in dialog window: some errors
			      occurred while accessing drive */
			   _("Something went wrong.\nPlease check settings in Configuration -> Hardware and try once more."),
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	cdw_main_window_wrefresh();

	return CDW_NO; /* user couldn't decide what to do or hardware settings in cdw options are invalid */
}





/**
   \brief Reload or eject disc, update disc info view

   Eject tray of cd drive. If \p final_eject is true, check value
   of config.eject to see if tray should stay open. If \p final_eject is
   false, close tray.

   Values displayed in disc info view are reset after ejecting tray and
   updated with valid values after closing tray. Valid values are
   collected by cdw_disc_get().

   This code would be used two or three times in main(), so
   I decided to extract it to separate function.

   \param device_fullpath - path to device that you want to control
   \param final_eject - should the function look at config.eject to check if tray should be left open?
 */
void cdw_drive_reload_tray_with_ui_update(const char *device_fullpath, bool final_eject, bool config_eject)
{
	cdw_drive_eject_tray_with_ui_update(device_fullpath);
	if (final_eject) {
		if (config_eject) { /* leave tray open */
			return;
		}
	}

	/* close tray: either user don't want to keep it opened after
	   performing writing operation (this is when config.eject == false)
	   or cdw will be doing something more on the disc (this is when
	   final_eject == false) */
	cdw_drive_close_tray_with_ui_update(device_fullpath);

	return;
}





void cdw_drive_eject_tray_with_ui_update(const char *device_fullpath)
{
	/* 2TRANS: this is message displayed in process window */
	cdw_processwin_display_sub_info(_("Ejecting tray"));
	cdw_processwin_force_refresh();
	cdw_drive_eject_tray(device_fullpath);

	cdw_drive_update_ui_with_disc(false);

	return;
}





void cdw_drive_close_tray_with_ui_update(const char *device_fullpath)
{
	/* 2TRANS: this is message displayed in process window;
	   "tray" is an optical drive tray */
	cdw_processwin_display_sub_info(_("Closing tray, waiting for drive"));
	cdw_processwin_force_refresh();
	/* to avoid sending to drive "eject" and "close" commands
	   in very short time  */
	usleep(500000);
	cdw_drive_close_tray_internal(device_fullpath);
	sleep(2);

	cdw_drive_update_ui_with_disc(false);

	return;
}





int cdw_drive_toggle_tray2(const char *device_fullpath, bool with_processwin)
{
	cdw_rv_t m = cdw_fs_check_device_mounted(device_fullpath);
	if (m == CDW_ERROR) {
		; /* let's try anyway */
	} else if (m == CDW_OK) { /* device is mounted */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Message"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot eject tray. Please unmount your drive."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_ERROR;
	} else { /* crv == CDW_NO, not mounted */
		;
	}

	for (int i = 0; i < 10; ) {
		int rv = cdw_drive_status(device_fullpath);
		if (rv == CDS_DISC_OK
		    || rv == CDS_NO_DISC) {
			cdw_vdm ("INFO: tray is closed (%d / %s) - ejecting\n", rv, cdw_drive_get_cds_label(rv))
			if (with_processwin) {
				/* 2TRANS: this is message displayed in process window */
				cdw_processwin_display_sub_info(_("Ejecting tray"));
				cdw_processwin_force_refresh();
			}
			cdw_drive_eject_tray(device_fullpath);
			break;

		} else if (rv == CDS_TRAY_OPEN) {
			cdw_vdm ("INFO: tray is open (CDS_TRAY_OPEN) - closing\n");
			if (with_processwin) {
				/* 2TRANS: this is message displayed in process window */
				cdw_processwin_display_sub_info(_("Closing tray and waiting for drive"));
				cdw_processwin_force_refresh();
			}
			usleep(500000);
			cdw_drive_close_tray_internal(device_fullpath);
			sleep(2);
			break;

		} else if (rv == -1 || rv == CDS_NO_INFO) {
			cdw_vdm ("ERROR: -1 || CDS_NO_INFO\n");
			sleep(1);
			break;

		} else { /* CDS_DRIVE_NOT_READY */
			cdw_vdm ("WARNING: CDS_DRIVE_NOT_READY\n");
			sleep(1);
			i++;
		}
	}

	/* get final status: if there were no problems this will be a status
	   after toggling, and if there were problems, the error value will
	   be passed to caller */
	int retval = cdw_drive_status(device_fullpath);

	if (retval == -1 || retval == CDS_NO_INFO) {
		cdw_vdm ("ERROR: CDW_SYS_ERROR || CDS_NO_INFO\n");
		/* 2TRANS: this is title of dialog window: some error occurred
		   while trying to access disc */
		cdw_buttons_dialog(_("Drive error"),
				   /* 2TRANS: this is message in dialog window:
				      some error occurred while accessing cdrom */
				   _("Can't open device. Please check your configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else if (retval == CDS_DRIVE_NOT_READY) {
		cdw_vdm ("ERROR: CDS_DRIVE_NOT_READY\n");
		/* 2TRANS: this is title of dialog window: some error occurred
		   while trying to access disc */
		cdw_buttons_dialog(_("CD drive error"),
				   /* 2TRANS: this is message in dialog window */
				   _("CDROM doesn't respond (timeout)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	} else {
		/* some non-error value */
	}

	return retval;
}





void cdw_drive_toggle_tray_with_ui_update2(void)
{
	const char *drive = cdw_drive_get_drive_fullpath();
	if (drive == (const char *) NULL) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Can't find any drive. Please check your hardware configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return;
	}

	bool private_processwin = false;
	if (!cdw_processwin_is_active()) {

		/* 2TRANS: this is title of process window;
		   "drive" is "optical drive" - a burner */
		cdw_processwin_create(_("Drive tray"),
				      /* 2TRANS: this is message in process window,
					 "drive" is "optical drive" - a burner */
				      _("Opening/closing drive tray"), false);

		private_processwin = true;
	}

	cdw_drive_toggle_tray2(drive, true);

	/* We have some state of drive/disc. Update relevant
	   information in main window. */
	cdw_drive_update_ui_with_disc(false);

	if (private_processwin) {
		cdw_processwin_destroy("", false);
	}

	return;
}





cdw_disc_t *cdw_drive_get_disc(int rv)
{
	cdw_disc_t *disc = cdw_disc_new();
	if (!disc) {
		return NULL;
	}

	if (rv == CDS_DISC_OK) {
		cdw_rv_t crv = cdw_disc_get(disc);
		if (crv != CDW_OK) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot get media info."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		}
	} else if (rv == CDS_NO_DISC || rv == CDS_TRAY_OPEN) {
		;
	} else if (rv == CDS_DRIVE_NOT_READY) {
		cdw_vdm ("WARNING: drive still not ready\n");
	} else {
		cdw_vdm ("ERROR: failed to toggle drive tray\n");
	}

	return disc;
}





CDW_DROPDOWN *cdw_drive_make_drives_dropdown(WINDOW *window, int row, int col, int width)
{
#define LABEL_LEN 50

	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window, row, col, width,
						  drives.n_drives + 1, CDW_COLORS_DIALOG);


	char label[LABEL_LEN + 1];
	size_t max_len = width > LABEL_LEN ? LABEL_LEN : (size_t) width;
	/* 2TRANS: this is a label in a dropdown; "custom path" is
	   "path to a device file, specified by user" */
	strncpy (label, _("Use custom path to drive"), max_len);
	label[LABEL_LEN] = '\0';
	cdw_rv_t crv = cdw_dropdown_add_item(dropdown, CDW_DRIVE_ID_CUSTOM, label);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to add first label \"%s\" to dropdown\n", label);
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < drives.n_drives; i++) {
		snprintf(label, max_len + 1, "%s%s (%s / %s)",
			 i == drives.cdio_default_drive_ind ? "* " : "",
			 cdw_cdio_drives[i].fullpath,
			 cdw_cdio_drives[i].hw_info.psz_vendor,
			 cdw_cdio_drives[i].hw_info.psz_model);

		cdw_vdm ("INFO: adding dropdown label #%d = \"%s\"\n", i, label);
		crv = cdw_dropdown_add_item(dropdown, i, label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to add first label \"%s\" to dropdown\n", label);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}
	cdw_dropdown_finalize(dropdown);

	crv = cdw_dropdown_set_current_item_by_id(dropdown, drives.current_drive_id);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set current item\n");
		/* what now? */
	}
	cdw_dropdown_display_current_item(dropdown);

	return dropdown;
}





const char *cdw_drive_get_drive_fullpath(void)
{
	if (drives.n_drives == 0) {
		/* there are zero drives detected by cdio, but there may be
		   some custom path entered in configuration window;
		   if there is no custom path, then function will return
		   empty string, which won't be an error, but will signify
		   that there are surely no drives available */
		const char *path = cdw_config_get_custom_drive();
		if (!strlen(path)) {
			path = (const char *) NULL;
		}
		cdw_vdm ("INFO: n_drives = 0, selecting custom drive path = \"%s\"\n", path);
		return path;
	}

	if (drives.current_drive_id == CDW_DRIVE_ID_CUSTOM) { /* use custom drive path */
		const char *path = cdw_config_get_custom_drive();
		cdw_vdm ("INFO: selecting custom drive path = \"%s\"\n", path);
		return path;
	} else {
		const char *path = cdw_cdio_drives[drives.current_drive_id].fullpath;
		cdw_vdm ("INFO: selecting cdio drive path %lld = \"%s\"\n", drives.current_drive_id, path);
		return path;
	}
}





char *cdw_drive_get_help_message(void)
{
#if 1
	char *message = cdw_string_concat
		/* 2TRANS: this is a help message */
		(drives.n_drives == 0 ? _("cdw can't detect any drives, you may want to enter and use custom path to a device.\n") : "",
		 /* here -1 is only an initial value */
		 drives.n_drives != 0 && drives.cdio_default_drive_ind != -1 ?
		 /* 2TRANS: this is a help message, describing content
		    of a dropdown */
		 _("'*' in dropdown denotes default drive.\n") : "",
		 /* 2TRANS: this is a help message, keep \"/dev/xxx\" unchanged */
		 _("Custom path to device should have form \"/dev/xxx\", without ending slash.\n\"SCSI device\" field is for experts only, and if filled, cdw may not work correctly.\n"),
		 (char *) NULL);
#else
	/* test message */
	char *message = strdup("dddddddddddwwwwwwwwwwwwiiiiiiiiiissssssllllllllllllbbbbbbbbboooooooooooeeeeeejjjjjjjd");
#endif
	return message;
}





/**
   \brief Save changes made in configuration window into "cdw drive" data structures

   Function saves some options of current configuration into
   variables that are private to "cdw drive" module.

   There are also other options in configuration window, but their values
   should be saved to config variable - use cdw_drive_set_config_fields
   to do this.

   \p id may be either a "real" ID (CDW_DRIVE_ID_CUSTOM), or non-negative
   index of one of dropdown items. CDW_DRIVE_ID_CUSTOM is an ID of first
   item in the dropdown, following IDs are indexed starting from zero.

   \param id - id of current item in drive dropdown in config window;
*/
void cdw_drive_save_configuration(cdw_id_t id)
{
	drives.current_drive_id = id;
	return;

}





/**
   \brief Save values of selected configuration fields in config variable

   Function saves value of drive dropdown in appropriate field of config
   variable.

   \param config - config variable in which to save state of "drive" dropdown
   \param id - id of current item of "drive" dropdown
*/
void cdw_drive_set_config_fields(cdw_config_t *config, cdw_id_t id)
{
	if (id == CDW_DRIVE_ID_CUSTOM) {
		/* current selection is "use custom drive path" */
		strncpy(config->selected_drive, "custom", OPTION_FIELD_LEN_MAX);
		config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
		cdw_assert (!strcmp(config->selected_drive, "custom"), "ERROR: failed to correctly save \"custom\" value\n");
	} else {
		/* current selection is one of drives detected by cdio */
		strncpy(config->selected_drive, cdw_cdio_drives[id].fullpath, OPTION_FIELD_LEN_MAX);
		config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
		cdw_assert (!strcmp(config->selected_drive, cdw_cdio_drives[id].fullpath), "ERROR: failed to correctly save non-custom value \"%s\"\n", cdw_cdio_drives[id].fullpath);
	}

	return;
}





cdw_id_t cdw_drive_get_current_drive_id(void)
{
	return drives.current_drive_id;
}


static cdw_id_clabel_t cds_labels[] = {
	{ CDS_NO_INFO,         "CDS_NO_INFO"         },
	{ CDS_NO_DISC,         "CDS_NO_DISC"         },
	{ CDS_TRAY_OPEN,       "CDS_TRAY_OPEN"       },
	{ CDS_DRIVE_NOT_READY, "CDS_DRIVE_NOT_READY" },
	{ CDS_DISC_OK,         "CDS_DISC_OK"         },
	{ 0,                   (char *) NULL         } /* Guard. */
};


const char *cdw_drive_get_cds_label(int rv)
{
	return cdw_utils_id_label_table_get_label(cds_labels, rv);
}





int cdw_drive_update_ui_with_disc(bool rescan_selected_files)
{
	int rv = cdw_drive_status(cdw_drive_get_drive_fullpath());
	cdw_disc_t *disc = cdw_drive_get_disc(rv);

	cdw_main_window_disc_info_view_display_data(disc);
	cdw_main_window_volume_info_view_update(disc->capacity.sectors_used, disc->capacity.sectors_total, rescan_selected_files);

	cdw_disc_delete(&disc);

	if (rv == CDS_NO_DISC || rv == CDS_TRAY_OPEN) {
		return -1;
	} else {
		return 0;
	}
}
