/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

/* lstat() */
#include <sys/types.h>
#include <sys/stat.h>

#include <cdio/cdio.h>
#include <cdio/device.h>
#include <cdio/logging.h>

#include "cdw_utils.h"
#include "cdw_debug.h"
#include "cdw_cdio_drives.h"
#include "cdw_string.h"
#include "canonicalize.h"


cdw_cdio_drive_t cdw_cdio_drives[CDW_CDIO_DRIVES_N_DEVICES_MAX + 1];
static int cdw_cdio_drives_cdio_default_drive_ind = -1;
static int cdw_cdio_drives_config_drive_ind = -1;

static void     cdw_cdio_drives_init_internal(void);
static cdw_rv_t cdw_cdio_drives_collect(void);

static char *   cdw_cdio_drives_resolve_link(const char *fullpath);
static void     cdw_cdio_drives_mark_default_drive(void);
static cdw_rv_t cdw_cdio_drives_get_drive_capabilities(int i);
static int      cdw_cdio_drives_search(const char *fullpath);




void cdw_cdio_drives_init_internal(void)
{
	for (int i = 0; i < CDW_CDIO_DRIVES_N_DEVICES_MAX + 1; i++) {
		cdw_cdio_drives[i].fullpath = (char *) NULL;
		cdw_cdio_drives[i].read_cap = 0;
		cdw_cdio_drives[i].write_cap = 0;
		cdw_cdio_drives[i].misc_cap = 0;
	}
	cdw_cdio_drives_cdio_default_drive_ind = -1;
	cdw_cdio_drives_config_drive_ind = -1;

	return;
}



cdw_rv_t cdw_cdio_drives_init(void)
{
	/* cdw_cdio_drives_init() is called before cdw_cdio_init(),
	   so cdio_init() call is made here */
	cdio_init();

	cdw_cdio_drives_init_internal();

	cdw_rv_t crv = cdw_cdio_drives_collect();
	if (crv == CDW_OK) {
		return CDW_OK;
	} else if (crv == CDW_NO) {
		cdw_vdm ("WARNING: no drives detected\n");
		return CDW_NO;
	} else {
		cdw_vdm ("ERROR: failed at collecting information about available drives\n");
		return CDW_ERROR;
	}
}





void cdw_cdio_drives_clean(void)
{
	for (int i = 0; i < CDW_CDIO_DRIVES_N_DEVICES_MAX + 1; i++) {
		if (cdw_cdio_drives[i].fullpath == (char *) NULL) {
			break;
		}
		free(cdw_cdio_drives[i].fullpath);
		cdw_cdio_drives[i].fullpath = (char *) NULL;

		cdw_cdio_drives[i].read_cap = 0;
		cdw_cdio_drives[i].write_cap = 0;
		cdw_cdio_drives[i].misc_cap = 0;
	}

	return;
}





cdw_rv_t cdw_cdio_drives_collect(void)
{
	char **devices = cdio_get_devices(DRIVER_DEVICE);
	if (devices == (char **) NULL) {
		return CDW_NO;
	}
	int discovered = 0;
	int saved = 0;
	while (devices[discovered] != (char *) NULL) {
		char *resolved = cdw_cdio_drives_resolve_link(devices[discovered]);
		if (resolved == (char *) NULL) {
			cdw_vdm ("ERROR: failed to resolve path \"%s\"\n", devices[discovered]);
			return CDW_ERROR;
		}
		discovered++;

		/* check if given path is already on list of devices;
		   this may happen if some of discovered device files are
		   symlinks to real device files */
		int i = cdw_cdio_drives_search(resolved);
		if (i == -1) {
			cdw_cdio_drives[saved].fullpath = resolved;
		} else {
			free(resolved);
			resolved = (char *) NULL;
			continue;
		}

		cdw_rv_t crv = cdw_cdio_drives_get_drive_capabilities(saved);
		if (crv != CDW_OK) {
			/* May be e.g. a /dev/loop1 device for ISO
			   image mounted by root user.  Skip it,
			   forget it, move on. */
			cdw_vdm ("WARNING: can't get capabilities for drive #%d: \"%s\"\n",
				 saved, cdw_cdio_drives[saved].fullpath);
			free(cdw_cdio_drives[saved].fullpath);
			cdw_cdio_drives[saved].fullpath = (char *) NULL;
			continue;
		}
		saved++;
	}
	cdio_free_device_list(devices);

	cdw_cdio_drives_mark_default_drive();

	return CDW_OK;
}





char *cdw_cdio_drives_resolve_link(const char *fullpath)
{
	struct stat s;
	int r = lstat(fullpath, &s);
	if (r == -1) {
		cdw_vdm ("ERROR: can't stat() \"%s\"\n", fullpath);
		return (char *) NULL;
	}
	char *result = (char *) NULL;
	/* perhaps here we are too optimistic about possible file
	   types: there can (?) be more file types pointed to by
	   full path than link and regular file; but I don't think
	   that it matters that much here */
	if (S_ISLNK(s.st_mode)) {
		result = canonicalize_filename_mode(fullpath, CAN_EXISTING);
		cdw_vdm ("INFO: canonicalized device path = \"%s\"\n", result);
	} else {
		result = strdup(fullpath);
		cdw_vdm ("INFO: %s is not a link\n", fullpath);
	}
	return result;
}





void cdw_cdio_drives_mark_default_drive(void)
{
	char *default_device = cdio_get_default_device(NULL);
	char *resolved = cdw_cdio_drives_resolve_link(default_device);
	if (resolved == (char *) NULL) {
		cdw_vdm ("ERROR: failed to resolve path \"%s\"\n", default_device);
		return;
	} else {
		cdw_vdm ("INFO: according to cdio default device is \"%s\"\n", resolved);
	}

	int i = cdw_cdio_drives_search(resolved);
	if (i == -1) {
		/* can't find default device? */
		cdw_vdm ("WARNING: can't find default device \"%s\" on list of cdio devices\n", resolved);
		/* let's use first device on list of devices as default one */
		cdw_cdio_drives_cdio_default_drive_ind = 0;
	} else {
		cdw_cdio_drives_cdio_default_drive_ind = i;
	}

	free(default_device);
	default_device = (char *) NULL;
	free(resolved);
	resolved = (char *) NULL;

	return;
}





cdw_rv_t cdw_cdio_drives_get_drive_capabilities(int i)
{

	CdIo_t *p_cdio = cdio_open(cdw_cdio_drives[i].fullpath, DRIVER_DEVICE);
	if (p_cdio == (CdIo_t *) NULL) {
		cdw_vdm ("ERROR: failed to open cdio disc #%d using DRIVER_DEVICE\n", i);
		return CDW_ERROR;
	}

	const char *driver = cdio_get_driver_name(p_cdio);
	cdw_vdm ("INFO: device #%i is \"%s\", driver is \"%s\"\n",
		 i, cdw_cdio_drives[i].fullpath, driver);
	cdio_get_drive_cap(p_cdio,
			   &(cdw_cdio_drives[i].read_cap),
			   &(cdw_cdio_drives[i].write_cap),
			   &(cdw_cdio_drives[i].misc_cap));

	bool has_info = cdio_get_hwinfo(p_cdio, &(cdw_cdio_drives[i].hw_info));
	if (has_info) {
		cdw_string_rtrim(cdw_cdio_drives[i].hw_info.psz_vendor);
		cdw_string_rtrim(cdw_cdio_drives[i].hw_info.psz_model);
		cdw_vdm ("INFO: vendor = \"%s\"\n", cdw_cdio_drives[i].hw_info.psz_vendor);
		cdw_vdm ("INFO:  model = \"%s\"\n", cdw_cdio_drives[i].hw_info.psz_model);
	} else {
		cdw_vdm ("WARNING: no hw info for drive #%d\n", i);
	}

	cdw_vdm ("INFO: misc capabilities:\n");
	cdw_vdm ("INFO: is%s a file\n",         cdw_cdio_drives[i].misc_cap & CDIO_DRIVE_CAP_MISC_FILE ? "" : "n't");
	cdw_vdm ("INFO: can%s do reset\n",      cdw_cdio_drives[i].misc_cap & CDIO_DRIVE_CAP_MISC_RESET ? "" : "'t");
	cdw_vdm ("INFO: can%s select disc\n\n", cdw_cdio_drives[i].misc_cap & CDIO_DRIVE_CAP_MISC_SELECT_DISC ? "" : "'t");

	cdio_destroy(p_cdio);

	return CDW_OK;
}





int cdw_cdio_drives_search(const char *fullpath)
{
	for (int i = 0; i < CDW_CDIO_DRIVES_N_DEVICES_MAX; i++) {
		if (cdw_cdio_drives[i].fullpath == (char *) NULL) {
			return -1;
		}
		if (! strcmp(fullpath, cdw_cdio_drives[i].fullpath)) {
			return i;
		}
	}
	return -1;
}



#if 0
cdw_rv_t cdw_cdio_drives_resolve(char **config_fullpath)
{
	if (*config_fullpath == (char *) NULL) {
		cdw_vdm ("INFO: currently there is no drive configured\n");
	} else {
		cdw_vdm ("INFO: currently configured drive is \"%s\"\n", *config_fullpath);
		for (int d = 0; d < CDW_CDIO_DRIVES_N_DEVICES_MAX; d++) {
			if (cdw_cdio_drives[d].fullpath == (char *) NULL) {
				break;
			}
			if (! strcmp(*config_fullpath, cdw_cdio_drives[d].fullpath)) {
				cdw_vdm ("INFO: matched config device and device on drives list: device #%d = \"%s\"\n",
					 d, cdw_cdio_drives[d].fullpath);
				cdw_cdio_drives_config_drive_ind = d;
			}
		}
	}

	if (cdw_cdio_drives_config_drive_ind == -1) {
		/* drive path from config module doesn't match any
		   entry in table of discs detected by cdio */
		int i = cdw_cdio_drives_cdio_default_drive_ind;
		cdw_string_set(config_fullpath, cdw_cdio_drives[i].fullpath);
	}

	return CDW_OK;
}
#endif



int cdw_cdio_drives_get_cdio_default_drive_ind(void)
{
	cdw_assert (cdw_cdio_drives_cdio_default_drive_ind >= 0, "ERROR: default drive index is negative\n");
	return cdw_cdio_drives_cdio_default_drive_ind;
}





const char *cdw_cdio_drives_get_cdio_default_drive(void)
{
	int i = cdw_cdio_drives_cdio_default_drive_ind;
	if (i == -1) {
		cdw_sdm ("WARNING: no cdio default device\n");
		return (const char *) NULL;
	} else {
		const char *drive = cdw_cdio_drives[i].fullpath;
		cdw_sdm ("INFO: cdio default drive is \"%s\"\n", drive);
		return drive;
	}
}


const char *cdw_cdio_drives_get_config_drive(void)
{
	int i = cdw_cdio_drives_config_drive_ind;
	if (i == -1) {
		cdw_sdm ("WARNING: no configured device\n");
		return (const char *) NULL;
	} else {
		const char *drive = cdw_cdio_drives[i].fullpath;
		cdw_sdm ("INFO: configured drive is \"%s\"\n", drive);
		return drive;
	}
}




int cdw_cdio_drives_get_n_drives(void)
{
	int i = 0;
	for (i = 0; i < CDW_CDIO_DRIVES_N_DEVICES_MAX + 1; i++) {
		if (cdw_cdio_drives[i].fullpath == (char *) NULL) {
			break;
		}
	}
	if (i == CDW_CDIO_DRIVES_N_DEVICES_MAX + 1) {
		cdw_vdm ("ERROR: too many items in drives table: %d\n", i);
		return -1;
	}

	if (i == 0) {
		cdw_vdm ("WARNING: no drives detected by cdw_cdio_drives module\n");
	} else {
		cdw_vdm ("INFO: cdw_cdio_drives module detected %d drive(s)\n", i);
	}
	return i;
}
