/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file cdw_regex_dispatch.c

   \brief File with top-level functions controlling how and when data
   printed by external tools to stdout and stderr is processed with
   regular expressions

   Data printed by external tools to stdout and stderr contains useful
   information that needs to be captured and processed. This capturing
   and processing is done by execute() functions defined in
   cdw_<tool_name>_regex.c files. The functions use some additional
   data that needs to be prepared prior to execution of the functions,
   and that needs to be cleaned up (destroyed) after execution of the
   functions. This is done with prepare() and destroy() functions
   respectively.

   Every time an execute() function is called, a corresponding prepare()
   and destroy() functions need to be called too.

   Every external tool has a pair of execute() functions, a pair of
   prepare() functions, and a pair of destroy() functions: one function
   for stdout stream, and one function for stderr stream of data. So
   every tool has six functions that need to be called to correctly
   process all data printed by the tool to stdout and stderr.

   There are N tools with 6 functions to be called. Luckily there is a
   common pattern, common flow of execution that can be implemented as
   one function. The function is cdw_regex_caller(). Called with correct
   arguments, it can ensure that proper prepare(), execute() and
   destroy() functions are called in proper order and timing.

   These generic functions:
   cdw_regex_stdout_prepare()
   cdw_regex_stdout_execute()
   cdw_regex_stdout_destroy()
   cdw_regex_stderr_prepare()
   cdw_regex_stderr_execute()
   cdw_regex_stderr_destroy()
   make sure that cdw_regex_caller() is called with right arguments.


   Information about prepare(), execute() and destroy() functions
   (that are defined for every external tool separately):

   prepare() functions call callocs() and regcomp() to prepare variables
   for holding regex matches and to compile specific (char *) strings
   into regex variables.

   execute() functions call regexec() to compare content of stdout or
   stderr streams against precompiled regex variables; if there is
   positive match, then proper handler is called to extract a specific
   information from matched stream data.

   destroy() functions call regfree() and free() to clean up variables
   prepared by prepare() functions.


   regexec() operates on (compares content of) buffers that store
   information sent via UNIX pipes from child processes. These processes
   do something interesting to us and send information about their
   activities to their stdout and stderr (which is read from pipes and
   put into pipe buffers by threading code). See thread.c for more
   information about the pipes.

   Information read from pipes is then compared against set of regular
   expressions and and depending on results of this comparisons one
   of handle functions is called.

   One of arguments of such function is matchesX - table of offset
   pairs, each pair pointing to beginning and to end of consecutive
   (I guess: non-white) substrings in captured buffer.
   Each substring (each "word" or lexem) in process' output line (and,
   therefore, in pipe buffer) contains some useful information, for
   example output of cdrecord writing files to disc looks like this:

   "Track 02:    7 of   52 MB written (fifo 100%) [buf 100%]   4.1x.".

   If we add parentheses to this expression to mark patterns that we want
   to capture, the string will look like this:
   Track ([0-9]+): ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written ..."

   Pairs of offsets allow us to grab useful information from such
   string, e.g. using offsets pair number 2 we can extract second
   substring / lexem, which is in this example information about
   amount of data already written to disc - quite a useful information.

   Each handle function knows format of only one kind of data that can
   be found in pipe buffer. If we want to capture and analyze data in
   pipe, we have to write regular expression, compile it with regcomp(),
   compare our line from child process (stored in pipe buffer) against
   this regular expression, and (if data matches regex) call some specific
   handle function, which will analyze this one line by operating on
   selected words / lexems / substrings from the line.


   This file contains top-level dispatch code, regular expressions code
   itself is in cdw_<tool_name>_regex.c files. We gain three things with
   such organization:
   \li code for different tools is in separate files, so it is easier to
       maintain and modify code for specific tool, without dealing or
       interfering with code for other tools;
   \li it is easier to add code with support for new tool: just define new
       value of tool id in cdw_ext_tools.h, update fields of cdw_task_t
       data type definition (if necessary)  add another condition branch
       in code below, and define new set of functions in new
       cdw_<tool_name>_regex.c file.
   \li the code should run faster: checking value of some fields of task
       variable  and calling only subset of all prepare() or execute()
       or destroy() functions should be faster than calling all of those
       functions (most of the calls would be totally unnecessary).

   If you want to add code for capturing new string, and you are not sure
   if the code should operate on stdout pipe buffer or stderr pipe buffer,
   then activate debug messages on top of execute() functions (the messages
   with "INFO: STDOUT PIPE:" and "INFO: STDERR PIPE:" strings), redirect
   stderr output of cdw to file and search this file for debug messages with
   "INFO: STDOUT PIPE:" and "INFO: STDERR PIPE:" strings.

   This file also defines two utility functions used by functions in other
   cdw_<tool_name>_regex.c files:
   \li void cdw_regex_regcomp_error_handler(const char *caller_name, int errcode, const regex_t *regex, int id)
   \li int cdw_regex_get_submatch(regmatch_t *matches, unsigned int sub, char *pipe_buffer, char *submatch)
*/


/* small cheat sheet on data structures and functions for regex;
   I didn't want to put it in any particular *_regex.c file */
/*

regex_t->re_nsub: Number of subexpressions found by the compiler, does not count whole captured string as subexpression

 */


#include <stdlib.h> /* exit() */
#include <string.h>
#include <regex.h>

#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */

#include "cdw_regex_dispatch.h"
#include "cdw_growisofs_regex.h"
#include "cdw_cdrecord_regex.h"
#include "cdw_mkisofs_regex.h"
#include "cdw_dvd_rw_mediainfo_regex.h"
#include "cdw_dvd_rw_format_regex.h"
#include "cdw_digest_regex.h"
#include "cdw_xorriso_regex.h"
#include "cdw_ext_tools.h"



extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];


typedef void (* cdw_regex_function_t)(void);


static const int cdw_stdout = 1;
static const int cdw_stderr = 2;
static void cdw_regex_caller(const cdw_task_t *task, cdw_regex_function_t functions[], int file);



static cdw_regex_function_t stdout_preparers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stdout_regexp_prepare,
	cdrecord_stdout_regexp_prepare,
	growisofs_stdout_regexp_prepare,
	(cdw_regex_function_t) NULL, /* we don't capture data on dvd+rw-format stdout */
	dvd_rw_mediainfo_stdout_regexp_prepare,
	cdw_xorriso_stdout_regex_prepare,
	cdw_digest_stdout_regex_prepare
};

static cdw_regex_function_t stdout_executers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stdout_regexp_execute,
	cdrecord_stdout_regexp_execute,
	growisofs_stdout_regexp_execute,
	(cdw_regex_function_t) NULL, /* we don't capture data on dvd+rw-format stdout */
	dvd_rw_mediainfo_stdout_regexp_execute,
	cdw_xorriso_stdout_regex_execute,
	cdw_digest_stdout_regex_execute };

static cdw_regex_function_t stdout_destroyers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stdout_regexp_destroy,
	cdrecord_stdout_regexp_destroy,
	growisofs_stdout_regexp_destroy,
	(cdw_regex_function_t) NULL, /* we don't capture data on dvd+rw-format stdout */
	dvd_rw_mediainfo_stdout_regexp_destroy,
	cdw_xorriso_stdout_regex_destroy,
	cdw_digest_stdout_regex_destroy };

static cdw_regex_function_t stderr_preparers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stderr_regexp_prepare,
	cdrecord_stderr_regexp_prepare,
	growisofs_stderr_regexp_prepare,
	cdw_dvd_rw_format_stderr_regexp_prepare,
	(cdw_regex_function_t) NULL,     /* we don't capture data on dvd+rw-mediainfo stderr */
	cdw_xorriso_stderr_regex_prepare,
	(cdw_regex_function_t) NULL };   /* we don't capture data on digest tool stderr */


static cdw_regex_function_t stderr_executers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stderr_regexp_execute,
	cdrecord_stderr_regexp_execute,
	growisofs_stderr_regexp_execute,
	cdw_dvd_rw_format_stderr_regexp_execute,
	(cdw_regex_function_t) NULL,     /* we don't capture data on dvd+rw-mediainfo stderr */
	cdw_xorriso_stderr_regex_execute,
	(cdw_regex_function_t) NULL };   /* we don't capture data on digest tool stderr */

static cdw_regex_function_t stderr_destroyers[CDW_EXT_TOOLS_N_TOOLS] = {
	mkisofs_stderr_regexp_destroy,
	cdrecord_stderr_regexp_destroy,
	growisofs_stderr_regexp_destroy,
	cdw_dvd_rw_format_stderr_regexp_destroy,
	(cdw_regex_function_t) NULL,     /* we don't capture data on dvd+rw-mediainfo stderr */
	cdw_xorriso_stderr_regex_destroy,
	(cdw_regex_function_t) NULL };   /* we don't capture data on digest tool stderr */



void cdw_regex_stdout_prepare(const cdw_task_t *task)
{
	cdw_regex_caller(task, stdout_preparers, cdw_stdout);
	return;
}
void cdw_regex_stdout_execute(const cdw_task_t *task)
{
	cdw_sdm ("INFO: STDOUT PIPE: %s\n", stdout_pipe_buffer);
	cdw_regex_caller(task, stdout_executers, cdw_stdout);
	return;
}
void cdw_regex_stdout_destroy(const cdw_task_t *task)
{
	cdw_regex_caller(task, stdout_destroyers, cdw_stdout);
	return;
}
void cdw_regex_stderr_prepare(const cdw_task_t *task)
{
	cdw_regex_caller(task, stderr_preparers, cdw_stderr);
	return;
}
void cdw_regex_stderr_execute(const cdw_task_t *task)
{
	cdw_sdm ("INFO: STDERR PIPE: %s\n", stderr_pipe_buffer);
	cdw_regex_caller(task, stderr_executers, cdw_stderr);
	return;
}
void cdw_regex_stderr_destroy(const cdw_task_t *task)
{
	cdw_regex_caller(task, stderr_destroyers, cdw_stderr);
	return;
}





/**
   \brief Universal function controlling creation, execution and
   destroying of code related to regular expressions

   \p functions is a table with {prepare()|execute()|destroy()} functions
   for given stream (for stdout or for stderr). The table is indexed with
   tool id.

   \p file describes if function is called to operate on stdout or stderr.
   Allowed values are cdw_stdout and cdw_stderr. The parameter is useful
   in cases when a handler function is not defined for stdout or stderr.

   \param task - variable describing current task
   \param functions - table of functions to execute
   \param file - variable representing stdout or stderr
*/
void cdw_regex_caller(const cdw_task_t *task, cdw_regex_function_t functions[], int file)
{
	cdw_assert (file == cdw_stdout || file == cdw_stderr,
		    "ERROR: invalid file: %d\n", file);
	const char *file_label = file == cdw_stdout ? "stdout" : "stderr";

	if (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO: %s: dispatching to %s\n", file_label, cdw_ext_tools_get_tool_name(task->burn.tool.id));
		if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			functions[CDW_TOOL_GROWISOFS]();
		} else if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
			functions[CDW_TOOL_CDRECORD]();
		} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
			functions[CDW_TOOL_XORRISO]();
		} else {
			cdw_vdm ("ERROR: incorrect burning tool id %lld\n", task->burn.tool.id);
		}
		if (task->id == CDW_TASK_BURN_FROM_FILES
		    && task->burn.tool.id != CDW_TOOL_XORRISO) {
			/* mkisofs will be used during creation of
			   intermediate ISO file system, and this is
			   to catch errors reported during this process */
			cdw_vdm ("INFO: %s: dispatching to mkisofs\n", file_label);
			functions[CDW_TOOL_MKISOFS]();
		}
	} else if (task->id == CDW_TASK_CREATE_IMAGE) {
		if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
			cdw_vdm ("INFO: %s: dispatching to mkisofs\n", file_label);
			functions[CDW_TOOL_MKISOFS]();
		} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
			cdw_vdm ("INFO: %s: dispatching to xorriso\n", file_label);
			functions[CDW_TOOL_XORRISO]();
		} else {
			cdw_vdm ("ERROR: incorrect ISO9660 tool id %lld\n", task->create_image.tool.id);
		}
	} else if (task->id == CDW_TASK_MEDIA_INFO) {
		cdw_vdm ("INFO: %s: dispatching to %s\n", file_label, cdw_ext_tools_get_tool_name(task->media_info.tool.id));
		if (task->media_info.tool.id == CDW_TOOL_CDRECORD) {
			functions[CDW_TOOL_CDRECORD]();
		} else if (task->media_info.tool.id == CDW_TOOL_DVD_RW_MEDIAINFO) {
			if (file == cdw_stdout) {
				functions[CDW_TOOL_DVD_RW_MEDIAINFO]();
			} else {
				; /* we don't capture data on dvd+rw-mediainfo stderr */
			}
		} else if (task->media_info.tool.id == CDW_TOOL_XORRISO) {
			functions[CDW_TOOL_XORRISO]();
		} else {
			cdw_vdm ("ERROR: incorrect media info tool id %lld\n", task->media_info.tool.id);
		}
	} else if (task->id == CDW_TASK_ERASE_DISC) {
		cdw_vdm ("INFO: %s: dispatching to %s\n", file_label, cdw_ext_tools_get_tool_name(task->erase.tool.id));
		if (task->erase.tool.id == CDW_TOOL_CDRECORD) {
			functions[CDW_TOOL_CDRECORD]();
		} else if (task->erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {
			if (file == cdw_stderr) {
				functions[CDW_TOOL_DVD_RW_FORMAT]();
			} else {
				; /* we don't capture data on dvd+rw-format stdout */
			}
		} else if (task->erase.tool.id == CDW_TOOL_XORRISO) {
			functions[CDW_TOOL_XORRISO]();
		} else {
			cdw_vdm ("ERROR: incorrect erase tool id %lld\n", task->erase.tool.id);
		}
	} else if (task->id == CDW_TASK_CALCULATE_DIGEST) {
		cdw_vdm ("INFO: %s: dispatching to %s\n", file_label, cdw_ext_tools_get_tool_name(task->calculate_digest.tool.id));
		if (file == cdw_stdout) {
			functions[CDW_TOOL_MD5SUM]();
		} else {
			;
		} /* we don't capture data on md5sum stderr */
	} else {
		cdw_assert (0, "ERROR: unknown task id %lld\n", task->id);
	}

	return;
}




