/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE /* lstat() */
#define _GNU_SOURCE /* asprintf() */
#include <mntent.h> /* interface to /etc/mtab */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>


#include "cdw_fs.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_dll.h"
#include "cdw_logging.h"
#include "canonicalize.h"

/* remember that PATH_MAX already includes ending null:
   1. http://unix.derkeiler.com/Newsgroups/comp.unix.solaris/2005-01/2945.html
   "POSIX (at least IEEE Std 1003.1, 2004 Edition) is clear on this:
        PATH_MAX
	Maximum number of bytes in a pathname, including the terminating
	null character."

   2. http://lkml.indiana.edu/hypermail/linux/kernel/0111.2/1280.html
   "The POSIX.1a draft (the amendment to POSIX.1-1990)
   and XPG4 went with including the null byte in PATH_MAX, and the
   POSIX 1003.1-200x revision (Austin Group) and Single UNIX
   Specification Version 3 also continue this way." */



/* full path to user home directory, should be used by all other modules */
static char *home_dir_fullpath = (char *) NULL;
/* full path to temporary directory, should be used by all other modules */
static char *tmp_dir_fullpath = (char *) NULL;


static cdw_rv_t cdw_fs_init_home_dir_fullpath(void);
static cdw_rv_t cdw_fs_init_tmp_dir_fullpath(void);
static cdw_rv_t cdw_fs_traverse_dir(char *dirpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);
static cdw_rv_t cdw_fs_traverse_symlink(const char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data);

static int cdw_fs_errno_handler_get_index(int e);

/**
   \brief Initialize filesystem module

   Initialize home dir fullpath and tmp dir fullpath. Function first
   attempts to set path to home dir. If this is successful (and only then),
   then the function attempts to set path to tmp dir.

   Function returns CDW_GEN_ERROR if one of the paths can't be set properly
   for any reason.

   \return CDW_OK on success
   \return CDW_GEN_ERROR on failure
*/
cdw_rv_t cdw_fs_init(void)
{
	cdw_rv_t crv = CDW_OK;
	crv = cdw_fs_init_home_dir_fullpath();
	if (crv != CDW_OK) {
		return CDW_ERROR;
	} else {
		crv =  cdw_fs_init_tmp_dir_fullpath();
		if (crv != CDW_OK) {
			return CDW_ERROR;
		}
	}

	return CDW_OK;
}





/**
   \brief Deallocate resources allocated by fs module

   Function deallocates any resources allocated by cdw_fs_init()
   or other parts of fs module.
*/
void cdw_fs_clean(void)
{
	if (tmp_dir_fullpath != (char *) NULL
	    && (tmp_dir_fullpath != home_dir_fullpath)) {
		free(tmp_dir_fullpath);
		tmp_dir_fullpath = (char *) NULL;
	}

	if (home_dir_fullpath != (char *) NULL) {
		free(home_dir_fullpath);
		home_dir_fullpath = (char *) NULL;
	}

	return;
}





/**
   \brief Check existence of given file, check also permissions, if required

   Check if given file (regular file or dir) exists, has expected permissions
   and is of given type.

   \p permissions can be bit sum of symbolic constants: R_OK, W_OK, X_OK.
   Set \p permissions to F_OK if you don't want to check file permissions.

   \param fullpath - full path to given file
   \param permissions - file permissions
   \param filetype - type of file that you ask about (CDW_FILE or CDW_DIR)

   \return 0 if the path satisfies all constraints
   \return ENOENT if there is no such file as specified by fullpath
   \return EISDIR or ENOTDIR if file exists, but is of wrong kind (FILE / DIR)
   \return EACCES if file exists but permissions do not match
   \return EBADF if file is neither dir nor file
   \return errno number if lstat failed for some reason
*/
int cdw_fs_check_existing_path(char *fullpath, int permissions, int filetype)
{
	cdw_vdm ("checking path \"%s\"\n", fullpath);
	struct stat finfo;
	if (lstat(fullpath, &finfo) == -1) {
		int e = errno;
		return e;
	}

	/* check correctness of file type */
	if ( S_ISREG(finfo.st_mode) ) {
		if (filetype == CDW_FS_DIR) {
			return ENOTDIR;
		} else { /*  filetype == CDW_FS_DIR  */
			;
		}
	} else if ( S_ISDIR(finfo.st_mode) ) {
		if (filetype == CDW_FS_FILE) {
			return EISDIR;
		} else { /* filetype == CDW_FS_DIR */
			;
		}
	} else {
		return EBADF;
	}

	/* check permissions */
	if (permissions == F_OK) { /* don't ask for permissions */
		return 0;
	} else {
		int a = access(fullpath, permissions);
		/* at this point we know that fullpath points to existing file
		   of expected type, all that can fail now is permissions */
		if (a == -1) {
			int e = errno;
			perror("");
			cdw_vdm ("access() sets errno %d for %s\n", e, fullpath);

			return EACCES;
		} else { /* (a == 0) */
			cdw_vdm ("access() returns 0 for %s\n", fullpath);
			return 0;
		}
	}
}





/**
   \brief Policy for accepting selected file path

   Function returns binary sum of:
   CDW_FS_CHECK_IS_FILE,
   CDW_FS_CHECK_IS_DIR,
   CDW_FS_CHECK_EXISTS,
   CDW_FS_CHECK_DOESNT_EXIST,
   CDW_FS_CHECK_WRONG_PERM,
   CDW_FS_CHECK_SYS_ERROR
   depending on result of call of stat() and some checks

   \param fullpath - full path that we want to inspect
   \param expected_file_type - type of file, expected by caller of selection function: CDW_FS_FILE, CDW_FS_DIR
   \param expected_permissions - permissions of file, expected by caller of selection function: W_OK, R_OK, X_OK
   \param expected_new_or_existing - describes whether selected file has to already exist or not: CDW_FS_NEW, CDW_FS_EXISTING

   \return int of value representing state of file specified by fullpath
*/
int cdw_fs_check_fullpath(const char *fullpath, int expected_file_type, int expected_permissions, int expected_new_or_existing)
{
	int retval = 0;

	cdw_vdm ("INFO: checking path \"%s\"\n", fullpath);
	struct stat finfo;
	if (stat(fullpath, &finfo) == -1) {
		int e = errno;

		if (e == EFAULT              /* Bad address. */
		    || e == ENOMEM           /* Out of memory (i.e., kernel memory). */
		    || e == EOVERFLOW        /* (stat()) path refers to a file whose size cannot be represented in the type off_t. */
		    || e == ELOOP            /* Too many symbolic links encountered while traversing the path. */
		    || e == ENOTDIR          /*  A component of the path prefix of path is not a directory. */
		    || e == EACCES           /* Search permission is denied for one of the directories in the path prefix of path. */
		    || e == ENAMETOOLONG) {  /* File name too long. */

			/* in case of these errors we can't do nothing more
			   than show a dialog */
			cdw_fs_errno_handler(e);
			cdw_vdm ("INFO: path \"%s\" gives error \"%s\"\n", fullpath, strerror(e));
			return retval | CDW_FS_CHECK_SYS_ERROR;

		} else if (e == ENOENT) {     /* A component of path does not exist, or path is an empty string. */
			retval |= CDW_FS_CHECK_DOESNT_EXIST;
			/* this may work out if expected_new_or_existing == CDW_FS_NEW */
			ssize_t i = cdw_fs_get_filename_start(fullpath);
			char *parent = strndup (fullpath, (size_t) i);
			cdw_assert (parent != (char *) NULL, "ERROR: can't strdup() fullpath\n");
			if (stat(parent, &finfo) == -1) {
				retval |= CDW_FS_CHECK_NO_PARENT;
			} else {
				/* file doesn't exist, but check if a new file can be created
				   in existing parent dir */
				int a = access(parent, W_OK);
				if (a == -1) {
					retval |= CDW_FS_CHECK_WRONG_PERM;
					e = errno;
					cdw_vdm ("INFO: wrong PERMs of parent dir \"%s\"\n", parent);
					cdw_vdm ("INFO: access() sets errno %d  / %s for %s\n", e, strerror(e), parent);
				}
			}
			free(parent);
			parent = (char *) NULL;

			cdw_vdm ("INFO: path \"%s\" gives error ENOENT / \"%s\"\n", fullpath, strerror(e));
			return retval;
		} else {
			/* this shouldn't happen, all values were covered */
			cdw_vdm ("ERROR: unexpected errno %d / %s\n", e, strerror(e));
			return retval | CDW_FS_CHECK_SYS_ERROR;
		}
	} else {
		/* file exists in file system */
		retval |= CDW_FS_CHECK_EXISTS;
		if (expected_new_or_existing == CDW_FS_NEW) {
			/* ... but caller doesn't need existing file */
			cdw_vdm ("INFO: rejecting existing file expected only NEW: \"%s\"\n", fullpath);
			return retval;
		} else {
			/* expected_new_or_existing == CDW_FS_NEW | CDW_FS_EXISTING
			   or
			   expected_new_or_existing == CDW_FS_EXISTING */
			/* pass */
			cdw_vdm ("INFO: existing file \"%s\" passes \"new or existing\" criterion\n", fullpath);
		}

		/* check correctness of file type; we have used stat(), so
		   no need to check if this is link  */
		if ( S_ISREG(finfo.st_mode) ) {
			retval |= CDW_FS_CHECK_IS_FILE;
			if (expected_file_type & CDW_FS_DIR) {
				/* we wanted to pick dir, but current file is file */
				cdw_vdm ("INFO: rejecting existing file because is FILE, expected DIR: \"%s\"\n", fullpath);
				return retval;;
			} else { /*  expected_file_type == CDW_FS_FILE  */
				/* we wanted to pick file and current file is file */
				cdw_vdm ("INFO: correct file type for FILE \"%s\"\n", fullpath);
			}
		} else if ( S_ISDIR(finfo.st_mode) ) {
			retval |= CDW_FS_CHECK_IS_DIR;
			if (expected_file_type & CDW_FS_FILE) {
				/* we wanted to pick file, but current file is dir */
				cdw_vdm ("INFO: rejecting existing file because is DIR, expected FILE: \"%s\"\n", fullpath);
				return retval;
			} else { /* expected_file_type == CDW_FS_DIR */
				/* we wanted to pick dir and current file is dir */
				cdw_vdm ("INFO: correct file type for DIR \"%s\"\n", fullpath);
			}
		} else {
			cdw_vdm ("WARNING: rejecting file because neither FILE nor DIR: \"%s\"\n", fullpath);
			return retval | CDW_FS_CHECK_SYS_ERROR;
		}

		/* now check permissions of file */
		if (expected_permissions == F_OK) {
			/* we don't need to check permissions, and nothing more to check */
			cdw_vdm ("INFO: existing file \"%s\" passes \"perms = none\" criterion\n", fullpath);
			return retval;
		} else {
			/* at this point we know that fullpath points to existing file
			   of expected type, all that can fail now is permissions */
			int a = access(fullpath, expected_permissions);
			if (a == -1) {
				int e = errno;
				cdw_vdm ("INFO: rejecting existing file because of PERMS: \"%s\"\n", fullpath);
				cdw_vdm ("INFO: access() sets errno %d  / %s for %s\n", e, strerror(e), fullpath);

				return retval | CDW_FS_CHECK_WRONG_PERM;
			} else { /* (a == 0) */
				cdw_vdm ("INFO: correct PERMS for \"%s\"\n", fullpath);
				return retval;
			}
		}
	}

}





/**
   \brief Initialize variable with full path to home directory

   Function searches for user's home directory and puts full (non-relative)
   path to the directory into home_dir_fullpath variable.

   First place to search for user's home dir is HOME shell variable. If this
   fails (or in case of other problems) user is asked to enter path to
   home dir manually. If this fails too, or user cancels entering path,
   function returns CDW_ERROR

   \return CDW_OK on success
   \return CDW_ERROR if the variable with path can't be initialized for any reason
*/
cdw_rv_t cdw_fs_init_home_dir_fullpath(void)
{
	cdw_assert (home_dir_fullpath == (char *) NULL, "ERROR: calling init function when path != NULL\n");
	const char *home = getenv("HOME");
	if (home != (char *) NULL) {
		size_t len = strlen(home);
		if (len != 0 && len < PATH_MAX - 1) {
			if (home[len - 1] == '/') {
				cdw_vdm ("WARNING: getenv(\"HOME\") returns path with ending slash\n");
				home_dir_fullpath = strdup(home);
			} else {
				home_dir_fullpath = cdw_string_concat(home, "/", (char *) NULL);
			}
			if (home_dir_fullpath == (char *) NULL) {
				cdw_vdm ("ERROR: XX failed to initialize home_dir_fullpath with home = \"%s\", len = %zd\n", home, len);
			}
		} else {
			cdw_vdm ("WARNING: XX strlen(home) out of bounds: len = %zd, PATH_MAX - 1 = %d\n", len, PATH_MAX - 1);
		}
	} else {
		cdw_vdm ("WARNING: getenv(\"HOME\") returns NULL\n");
	}
	/* at this point home_dir_fullpath is either initialized
	   with correct HOME dirpath or is NULL */
	if (home_dir_fullpath == (char *) NULL) {
		/* no sane $HOME, but we need some value in
		   home_dir_fullpath for further function calls */
		home_dir_fullpath = strdup("");
	}

	cdw_vdm ("INFO: XX $HOME = \"%s\", home_dir_fullpath = \"%s\", strlen(home_dir_fullpath) = %zd\n",
		 home, home_dir_fullpath, strlen(home_dir_fullpath));

	int rv = cdw_fs_check_existing_path(home_dir_fullpath, W_OK, CDW_FS_DIR);
	if (rv == 0) {
		cdw_vdm ("INFO: XX home directory is initialized as \"%s\" (1)\n", home_dir_fullpath);
		return CDW_OK;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t crv = cdw_fs_ui_file_picker(_("Select directory"),
						     /* 2TRANS: this is message in dialog window. Please keep '\n'. */
						     _("Please select Home directory.\n   Press ESC to cancel."),
						     &home_dir_fullpath,
						     CDW_FS_DIR, W_OK|X_OK, CDW_FS_EXISTING);

		if (crv == CDW_OK) {
			cdw_vdm ("INFO: XX home directory is initialized as \"%s\" (2)\n", home_dir_fullpath);
			return CDW_OK;
		} else {
			if (home_dir_fullpath != (char *) NULL) {
				free(home_dir_fullpath);
				home_dir_fullpath = (char *) NULL;
			}

			cdw_vdm ("ERROR: XX failed to initialize home dir fullpath\n");

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog
					      window, "Home" is user's home dir,
					      "closing" refers to closing
					      application */
					   _("Path to Home directory not initialized properly, closing."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		}
	}
}





/**
   \brief Initialize variable with full path to temporary directory

   Function searches for system temporary directory and puts full (non-relative)
   path to the directory into tmp_dir_fullpath variable.

   First, default value is "/tmp". If this fails user is asked to enter path to
   tmp dir manually. If this fails too, or user cancels entering path,
   function sets tmp_dir_path to home_dir_path - these two pointers have the
   same value.

   \return CDW_OK
*/
cdw_rv_t cdw_fs_init_tmp_dir_fullpath(void)
{
	/* I'm using this assertion here for two reasons:
	   1. if setting home dirpath failed, then setting tmp dirpath
	   will probably fail as well; if setting home dirpath failed
	   and I still call this function to init tmp dirpat, then I made
	   an error somewhere;
	   2. home dir path may be used in this funciton, it is important
	   that home dir is correct */
	cdw_assert (home_dir_fullpath != (char *) NULL,
		    "you wan't to initialize tmp dirpath when home is not initialized (yet)\n");

	/* some sane initial value */
	char *tmp = strdup("/tmp/");
	if (tmp == (char *) NULL) {
		return CDW_ERROR;
	}

	int rv = cdw_fs_check_existing_path(tmp, W_OK, CDW_FS_DIR);
	if (rv == 0) {
		tmp_dir_fullpath = tmp;
		return CDW_OK;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_rv_t crv = cdw_fs_ui_file_picker(_("Select directory"),
						     /* 2TRANS: this is message in dialog window. Please keep '\n'. */
						     _("Please select temporary directory.\n   Press ESC to cancel."),
						     &tmp,
						     CDW_FS_DIR, W_OK|X_OK, CDW_FS_EXISTING);

		if (crv == CDW_OK) {
			tmp_dir_fullpath = tmp;
			cdw_sdm ("tmp directory is initialized as \"%s\"\n", tmp_dir_fullpath);

			return CDW_OK;
		} else {
			if (tmp != (char *) NULL) {
				cdw_vdm ("tmp is not null, free()ing\n");
				free(tmp);
				tmp = (char *) NULL;
			}

			tmp_dir_fullpath = home_dir_fullpath;
			cdw_sdm ("initialized tmp dir fullpath as home dir fullpath\n");

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Warning"),
					   /* 2TRANS: this is message in dialog window; */
					   _("Path to tmp directory is the same as path to home directory. This is not a bad thing, but some things might work differently."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			return CDW_OK;
		}
	}
}





/**
   \brief Getter function for tmp_dir_fullpath variable

   Use this function only after initializing fs module (calling
   cdw_fs_init()).

   \return pointer to string with tmp directory
*/
const char *cdw_fs_get_tmp_dir_fullpath(void)
{
	cdw_assert (tmp_dir_fullpath != (char *) NULL, "did you call cdw_fs_init()?\n");
	size_t len = strlen(tmp_dir_fullpath);
	cdw_assert (len != 0, "len of tmp dir fullpath is 0\n");

	return tmp_dir_fullpath;
}





/**
   \brief Getter function for home_dir_fullpath variable

   Use this function only after initializing fs module (calling
   cdw_fs_init()).

   \return pointer to string with home directory
*/
const char *cdw_fs_get_home_dir_fullpath(void)
{
	cdw_assert (home_dir_fullpath != (char *) NULL, "did you call cdw_fs_init()?\n");
	size_t len = strlen(home_dir_fullpath);
	cdw_assert (len != 0, "len of home dir fullpath is 0\n");

	return home_dir_fullpath;
}





char *cdw_fs_get_initial_dirpath(void)
{
	/*
	  1. try to return current working directory
	  2. if this fails try to return home directory
	  3. if this fails try to return tmp directory
	  4. if this fails try to return root directory */

	char *dirpath = get_current_dir_name();
	if (dirpath != (char *) NULL) {
		cdw_rv_t crv = cdw_fs_correct_dir_path_ending(&dirpath);
		if (crv == CDW_OK) {
			return dirpath;
		} else {
			cdw_vdm ("ERROR: failed to make correct dir path ending for cwd\n");
		}
	} else {
		cdw_vdm ("ERROR: can't get cwd, error = \"%s\"\n", strerror(errno));
	}

	dirpath = cdw_fs_get_home_or_tmp_dirpath();
	if (dirpath != (char *) NULL) {
		return dirpath;
	}

	dirpath = strdup("/");
	if (dirpath != (char *) NULL) {
		return dirpath;
	} else {
		cdw_vdm ("ERROR: can't strdup() root dir path\n");
	}

	return (char *) NULL;
}





char *cdw_fs_get_home_or_tmp_dirpath(void)
{
	char *dirpath = (char *) NULL;
	const char *h = cdw_fs_get_home_dir_fullpath();
	if (h != (char *) NULL) {
		dirpath = strdup(h);
		if (dirpath != (char *) NULL) {
			return dirpath;
		} else {
			cdw_vdm ("ERROR: can't strdup() home dir path\n");
		}
	} else {
		cdw_vdm ("ERROR: can't get home dir path\n");
	}

	const char *t = cdw_fs_get_tmp_dir_fullpath();
	if (t != (char *) NULL) {
		dirpath = strdup(t);
		if (dirpath != (char *) NULL) {
			return dirpath;
		} else {
			cdw_vdm ("ERROR: can't strdup() tmp dir path\n");
		}
	} else {
		cdw_vdm ("ERROR: can't get home tmp path\n");
	}

	return dirpath;
}





/**
   \brief Check if device of given path is mounted in file system

   Use GNU libc interface to /etc/mtab file to determine if given device
   is mounted. Officially you have to pass to a function a string with path
   to device file (like '/dev/xxx') which has to be checked, but
   mount point is also (implicitly) accepted.

   Function should work even if given \p device_fullpath is a symbolic
   link to real device.

   You most probably want to use this function to check if cdrom
   drive is mounted.

   \param device_fullpath - path to device file which has to be checked. No trailing slashes are accepted.

   \return CDW_OK if device is mounted
   \return CDW_NO if device is not mounted
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_fs_check_device_mounted(const char *device_fullpath)
{
	cdw_assert (device_fullpath != (char *) NULL, "device fullpath is null\n");
	struct stat finfo;
	int z = lstat(device_fullpath, &finfo);
	if (z == -1) {
		cdw_vdm ("ERROR: stat() returns !0 (%d) for device_fullpath %s\n", z, device_fullpath);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Path to drive specified in configuration is incorrect. Please check your configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	char *fullpath = (char *) NULL;
	if (!S_ISLNK(finfo.st_mode)) {
		/* device_fullpath is not a link */
		cdw_vdm ("INFO: file specified by fullpath \"%s\" is not a symlink\n", device_fullpath);
		fullpath = strdup(device_fullpath);

	} else { /* S_ISLNK(finfo.st_mode) */
		/* second argument == NULL is standardized in POSIX.1-2008 */
		fullpath = realpath(device_fullpath, (char *) NULL);
		if (fullpath == (char *) NULL) {
			int e = errno;
			cdw_vdm ("ERROR: realpath() can't resolve real path for device_fullpath link %s, strerror = \"%s\"\n", device_fullpath, strerror(e));
			/* 2TRANS: this is message printed in log file,
			   %s is full path to a device file */
			cdw_logging_write(_("cdw can't check device specified in configuration as \"%s\"\n"), device_fullpath);
		}
	}
	if (fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set final fullpath\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("cdw has problems with searching for your device. Please restart cdw."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	FILE *mtab_file = setmntent(_PATH_MOUNTED, "r");
	if (mtab_file == (FILE *) NULL) {
		free(fullpath);
		fullpath = (char *) NULL;
		cdw_vdm ("ERROR: call to setmnent() failed\n");
		return CDW_ERROR;
	}

	bool is_mounted = false;

	struct mntent *mnt = (struct mntent *) NULL;
	while ( (mnt = getmntent(mtab_file)) != (struct mntent *) NULL ) {
		int a = strcmp(fullpath, mnt->mnt_fsname);
		int b = strcmp(fullpath, mnt->mnt_dir);

		if (a == 0 || b == 0) {
			is_mounted = true;
			cdw_vdm ("INFO: device %s (%s) is mounted:\n", device_fullpath, fullpath);
			cdw_vdm ("name of mounted file system: %s\n", mnt->mnt_fsname); /* device name, e.g. /dev/scd0 */
			cdw_vdm ("file system path prefix:     %s\n", mnt->mnt_dir);    /* mount point, e.g. /mnt/cdrom (no ending slash) */
			cdw_vdm ("mount type:                  %s\n", mnt->mnt_type);   /* file system type (ext3, iso9660 etc.) */
			cdw_vdm ("mount options:               %s\n\n", mnt->mnt_opts); /* rw,noexec,nosuid,nodev etc. */
			break;
		}
	}

	endmntent(mtab_file);
	mtab_file = (FILE *) NULL;
	/* can't do that, apparently endmntent takes care of this */
	/* free(mnt);
	   mnt = (struct mntent *) NULL; */
	free(fullpath);
	fullpath = (char *) NULL;

	if (is_mounted) {
		cdw_vdm ("INFO: device %s is mounted\n", device_fullpath);
		return CDW_OK;
	} else {
		cdw_vdm ("INFO: device %s is not mounted\n", device_fullpath);
		return CDW_NO;
	}
}





/**
   \brief Ensure that given string ends with exactly one '/' char

   Check if given string ends with exactly one '/' char. If there are more
   '/' chars at the end, (without any other chars in between) then function
   removes all of them but one. If there is no ending '/' char, the function
   adds one. Before performing any of these actions, function may modify
   it's argument by removing any white chars from end of it.

   The function can't deal with cases such as "/path/  /", where there are
   two '/' chars at the end, but separated by spaces - function treats this
   as correctly ended string.

   The function can be used to ensure that directory path is properly
   ended with '/'.

   'path' cannot be NULL or empty string. It can be only "/" - function will
   recognize it as correct string.

   If malloc() fails and function returns CDW_ERROR, path is modified
   no more than by removing ending white chars.

   Function may realloc 'path'!

   \param path - directory path that you want to check

   \return CDW_OK if success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_fs_correct_dir_path_ending(char **path)
{
	if (*path == NULL) {
		cdw_vdm ("ERROR: path is NULL\n");
		return CDW_ERROR;
	}

	size_t len = strlen(*path);

	if (len == 0) {
		cdw_vdm ("ERROR: path has zero length\n");
		return CDW_ERROR;
	}

#if 0	/* an experiment, don't enable */
	size_t missing_chars = 0;
	bool missing_root = false;
	if (*(*path + 0) != '/'
	    || ((*(*path + 0) != '.') && (*(*path + 1) != '/'))
	    || ((*(*path + 0) != '.') && (*(*path + 1) != '.') && (*(*path + 2) != '/'))) {
		/* neither relative path nor path in root dir */

		cdw_vdm ("INFO: missing root\n");
		missing_root = true;
		missing_chars += 2; /* for two chars: "./" at the beginning of dir path */
	} else {
		cdw_vdm ("INFO: path %s has root\n", *path);
	}
	bool missing_end = false;
	if (file_type == CDW_FS_DIR &&
	    *(*path + len - 1) != '/') {
		cdw_vdm ("INFO: missing end\n");
		missing_end = true;
		missing_chars += 1; /* for one char: "/" at the end of dir path */
	} else {
		cdw_vdm ("INFO: path %s has end\n", *path);
	}

	if (missing_root || missing_end) {
		char *tmp = (char *) malloc(len + missing_chars + 1);
		if (tmp == (char *) NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for new path\n");
			return CDW_ERROR;
		} else {
			size_t offset = 0;
			if (missing_root) {
				snprintf(tmp + offset, 2 + 1, "./");
				offset += 2;
			}
			snprintf(tmp + offset, len + 1 + 1, "%s", *path);
			offset += len;
			if (missing_end) {
				  snprintf(tmp + offset, 1 + 1, "/");
				  offset += 1;
			}
			free(*path);
			*path = tmp;
		  }
	}

	ssize_t i = cdw_fs_get_filename_start(*path);
	cdw_assert (i > 0, "ERROR: can't find last dir name in dirpath \"%s\"\n", *path);
	len = strlen(*path);
	for (; (size_t) i < len; i++) {
		if ((*path)[i] == '/') { /* this is the slash that should end dirpath */
			if ((*path)[i + 1] != '\0') {
				/* and it is not the last char in the string */
				(*path)[i + 1] = '\0';
				break;
			} else {
				/* the slash was already last char in
				   the string, no need to do anything */
				break;
			}
		}
	}

#else

	if (*(*path + len - 1) != '/') {
		char *tmp = realloc(*path, len + 2);
		if (tmp == NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for new path\n");
			return CDW_ERROR;
		}
		*path = tmp;
		*(*path + len) = '/';
		*(*path + len + 1) = '\0';
	} else { /* there is at least one ending '/', but perhaps more */
		ssize_t i = cdw_fs_get_filename_start(*path);
		cdw_sdm ("INFO: path is \"%s\", file name start = %zd\n", *path, i);

		/* the condition is 'i >= 0' because paths can be relative, e.g.
		   "dir3//" - there is no root dir, and i = 0 */

		cdw_assert (i >= 0, "ERROR: can't find last dir name in dirpath \"%s\"\n", *path);

		/* special case, a hack: path is in form of "//" or similar */
		if (i == 1 && (*path)[i] == '/') {
			(*path)[i] = '\0';
			return CDW_OK;
		}

		len = strlen(*path);
		for (; (size_t) i < len; i++) {
			if ((*path)[i] == '/') { /* this is the slash that should end dirpath */
				if ((*path)[i + 1] != '\0') {
					/* and it is not the last char in the string */
					(*path)[i + 1] = '\0';
					break;
				} else {
					/* the slash was already last char in
					   the string, no need to do anything */
					break;
				}
			}
		}
	}
#endif
	return CDW_OK;
}





int cdw_fs_check_dirpath(const char *dirpath)
{
	struct stat stbuf;
	if (stat(dirpath, &stbuf) == -1) {
		cdw_vdm ("ERROR: stat() on \"%s\" failed\n", dirpath);
		return CDW_FS_CHECK_SYS_ERROR;
	}

	/* make sure that this is a dir */
	if ((stbuf.st_mode & S_IFMT) != S_IFDIR) {
		/* pressing ENTER on non-directory should have no effect;
		   link to directory is regarded as directory, so this
		   condition won't be true for link to dir */
		cdw_vdm ("ERROR: \"%s\" is not a dir\n", dirpath)
		return CDW_ERROR;
	}

	/* so this is a dir path - check permissions */
	if (access(dirpath, X_OK) == -1) {
		/* can't change directory, no point in checking if we can
		   read its content */
		return CDW_FS_CHECK_E_PERM_NO_X;
	} else {
		if (access(dirpath, R_OK) == -1) {
			/* technically it would be possible to visit
			   this dir, but caller won't have too much
			   use of this path - scandir will return -1 for it */
			return CDW_FS_CHECK_E_PERM_NO_R;
		} else {
			/* we have execute and read prems for new dir:
			   we can go into this dir and can read its content */
			return 0;
		}
	}
}





/**
   \brief Filter function for scandir()

   Filter function for scandir() - returns zero value for current
   directory ('.') entry. Entries, for which filter returns zero value,
   are omited by scandir().

   You needn't worry about passing any argumets to the function - just
   use this function's name as 3rd argument to scandir() call.

   \param entry - item from listing of directory

   \return 0 if entry is '.',
   \return 1 otherwise
*/
int cdw_scandir_filter_one(const struct dirent *entry)
{
	if (strcmp(entry->d_name, ".")) {
		return 1;
	} else {
		return 0;
	}
}





/**
   \brief Filter function for scandir()

   Filter function for scandir() - returns zero value for
   current directory ('.') and parent directory ('..') entries.
   Entries, for which filter returns zero value, are omited by scandir()

   You needn't worry about passing any argumets to the function - just
   use this function's name as 3rd argument to scandir() call.

   \param entry - item from listing of directory

   \return 0 if entry is '.' or '..',
   \return 1 otherwise
*/
int cdw_scandir_filter_two(const struct dirent *entry)
{
	if (strncmp(entry->d_name, ".", 2) && strncmp(entry->d_name, "..", 2)) {
		return 1;
	} else {
		return 0;
	}
}





/**
   \brief Get file of size specified by full path

   \return -1 on error
   \return size of file on success
*/
long long cdw_fs_get_file_size(const char *fullpath)
{
	struct stat stbuf;
	int rv = lstat(fullpath, &stbuf);
	if (rv != 0) {
		int e = errno;
		cdw_fs_errno_handler(e);

		return -1;
	} else {
		return (long long) stbuf.st_size;
	}
}





/**
   \brief Fill \p list with file names, types and sizes of files in given dir

   Read from \p eps file names of files in directory specified by \p dirpath,
   store information about each file in directory in given \p list of files.

   Caller has to make sure that \p eps does not hold reference to '.' dir.

   \p list must be empty, NULL pointer to doubly-linked list. The pointer, and
   whole list, will be initialized and filled by this function with pointers
   to cdw_file_t data structures.

   \p dirpath must be non-null string terminated with '\0', with valid
   path to a directory. This is the same dirpath which was passed to scandir()
   to get \p eps.

   \p eps must be result of call of scandir() with given \p dirpath

   \p n_files is a number of file names stored in \p eps, a value returned
   by scandir() when called with given \dirpath. Make sure that \p n_files is
   non-zero, therefore \p eps has to have at least one file name. It can't be
   "." - current dir, but it can be ".." - parent dir.

   \p list won't include hidden files if \p include_hidden is false.

   \param list - will be set to list of files in given dir
   \param dirpath - path to directory that you scan
   \param eps - list of file names in given dir
   \param n_files - number of file names on \p eps, or in given dir
   \param include_hidden - controls if hidden UNIX file should be copied from eps to list

   \return number of items copied to the list on success (there should be at least one: ".")
   \return 0 on errors
*/
unsigned long long cdw_fs_copy_dirent_to_list(cdw_dll_item_t **list, const char *dirpath, struct dirent **eps, unsigned long long n_items, bool include_hidden)
{
	cdw_assert (*list == (cdw_dll_item_t *) NULL, "ERROR: can't use non-null list\n");
	cdw_assert (dirpath != (char *) NULL, "ERROR: can't use null dirpath\n");
	cdw_assert (n_items > 0, "ERROR: you want to create list of zero files\n");

	cdw_dll_item_t *dirs = (cdw_dll_item_t *) NULL;
	cdw_dll_item_t *files = (cdw_dll_item_t *) NULL;

	unsigned long long n_dirs = 0;
	unsigned long long n_files = 0;
	unsigned long long n_hidden = 0;
	bool success = true;

	for (unsigned long long i = 0; i < n_items; i++) {
		if (cdw_fs_is_hidden(eps[i]->d_name) && !include_hidden) {
			n_hidden++;
			continue;
		}
		cdw_file_t *file = cdw_file_new(dirpath, eps[i]->d_name);
		if (file == (cdw_file_t *) NULL) {
			success = false;
			break;
		}
		if (file->type == CDW_FS_DIR) {
			cdw_dll_append(&dirs, (void *) file, cdw_file_equal);
			n_dirs++;
		} else { /* CDW_FS_FILE or CDW_FS_OTHER */
			cdw_dll_append(&files, (void *) file, cdw_file_equal);
			n_files++;
		}
	}

	if (success) {
		/* connect 'dirs' list and 'files' list into one */
		cdw_assert (n_dirs + n_files + n_hidden == n_items, "ERROR: numbers of dirs/files doesn't add up\n");
		if (n_dirs > 0) {
			*list = dirs;

			if (n_files > 0) {
				cdw_dll_item_t *last_dir = cdw_dll_ith_item(dirs, (size_t) (n_dirs - 1));
				last_dir->next = files;
				files->prev = dirs;
			}
		} else {
			if (n_files > 0) {
				*list = files;
			}
		}
		return n_files + n_dirs;
	} else {
		if (dirs != (cdw_dll_item_t *) NULL) {
			cdw_file_dealloc_files_from_list(dirs);
			cdw_dll_clean(dirs);
			dirs = (cdw_dll_item_t *) NULL;
		}
		if (files != (cdw_dll_item_t *) NULL) {
			cdw_file_dealloc_files_from_list(files);
			cdw_dll_clean(files);
			dirs = (cdw_dll_item_t *) NULL;
		}
		return 0;
	}
}





ssize_t cdw_fs_get_filename_start(const char *fullpath)
{
	if (fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: argument is NULL\n");
		return -1;
	}
	char *path = strdup(fullpath);
	cdw_assert (path != (char *) NULL, "ERROR: failed to strdup() input string\n");

	size_t len = strlen(path);
	if (len == 0) {
		free(path);
		path = (char *) NULL;
		cdw_vdm ("ERROR: argument has length = 0\n");
		return -1;
	}

	/* put "end" at last char in string that is not '/' */
	ssize_t end = (ssize_t) len - 1;
	while (end > 0 && path[end] == '/') {
		end--;
	}

	if (end == 0) {
		if (path[0] == '/') {
			/* path is "/" or "///" or "// // " or sth like that,
			   just slashes and spaces; in either case file name
			   start is at position 1 */
			end = 1;
		} else {
			/* path is "a/" or "a " */
			end = 0;
		}
	} else {
		/* "end" is now at the end of path, but before ending '/';
		   search for '/' that stands before last name in fullpath */
		while (end >= 0 && path[end] != '/') {
			end--;
		}
		end++;
	}

	free(path);
	path = (char *) NULL;
	return end;
}





cdw_rv_t cdw_fs_traverse_path(char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	/* WARNING: don't use stat() - it yelds bad results for symlinks;
	   at this point we don't want to follow symbolic links, we only
	   need to know if file pointed by path is a symlink or not;
	   we will examine symlinks more closely below */
	struct stat finfo;
	int ok = lstat(fullpath, &finfo);
	int e = errno;
	if (ok != 0) {
		/* TODO: that is weird, but lstat() fails on broken links;
		   this is not a bad thing in this context that we return
		   here, but I thought that lstat won't fail on broken links */
		/* TODO: perhaps we could write skipped paths to log file
		   without returning? */
		cdw_vdm ("ERROR: lstat fails for \"%s\", err = \"%s\"\n", fullpath, strerror(e));
		/* 2TRANS: this is message printed to log file; %s is
		   a full path to file */
		cdw_logging_write(_("Error: cdw can't get any information about this file: \"%s\"\n"), fullpath);
		/* 2TRANS: this is message printed to log file; %s is
		   an additional error message string */
		cdw_logging_write(_("Error: perhaps this information can help resolve problem: \"%s\"\n"), strerror(e));
		// assert (0);
		return CDW_ERROR;
	}

	cdw_rv_t retval = CDW_NO;

	if ( S_ISREG(finfo.st_mode) ) { /* regular file */
		visitor(fullpath, &finfo, visitor_data);
		retval = CDW_OK;
	} else if ( S_ISLNK(finfo.st_mode) ) { /* symbolic link */
		if (visitor_data->follow_symlinks) {
			cdw_vdm ("INFO: following symlinks\n");
			retval = cdw_fs_traverse_symlink(fullpath, visitor, visitor_data);
		} else {
			cdw_vdm ("INFO: NOT following link\n");
			visitor(fullpath, &finfo, visitor_data);
			retval = CDW_OK;
		}
	} else if ( S_ISDIR(finfo.st_mode) ) { /* directory */
		visitor(fullpath, &finfo, visitor_data);
		retval = cdw_fs_traverse_dir(fullpath, visitor, visitor_data);
	} else { /* other, non-appendable file type */
		retval = CDW_NO;
	}

	if (retval == CDW_ERROR) {
		/* 2TRANS: this is message printed to log file;
		   %s is a full path to file */
		cdw_logging_write(_("Error: cdw can't get any information about this file: \"%s\"\n"), fullpath);
		if (e != 0) {
			/* 2TRANS: this is message printed to log file; %s is
			   an additional error message string */
			cdw_logging_write(_("Error: perhaps this information can help resolve problem: \"%s\"\n"), strerror(e));
		}
	}

	return retval;
}





cdw_rv_t cdw_fs_traverse_dir(char *dirpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	struct dirent **eps = (struct dirent **) NULL;
	/* get directory listing - this allocates memory for **eps */
	int n = scandir(dirpath, &eps, cdw_scandir_filter_two, &alphasort);
	int e = errno;
	/* scandir(3): "The scandir() function returns the number of
	   directory entries selected or -1 if an error occurs." */
	cdw_rv_t retval = CDW_OK;
	if (n == -1) {
		cdw_vdm ("ERROR: scandir() returns -1 (%s) for dirpath \"%s\"\n", strerror(e), dirpath);
	       retval = CDW_ERROR;
	} else if (n == 0) {
		/* TODO: empty dir, is there anything more to do? */
		retval = CDW_OK;
	} else {
		for (int i = 0; i < n; i++) {
			/* for every dir entry create new full path to this entry and scan it */
			char *newpath = (char *) NULL;
			/* TODO : why use '/' here if dir paths should
			   end with '/' ? */
			int rv = asprintf(&newpath, "%s/%s", dirpath, eps[i]->d_name);
			if (rv == -1) {
				cdw_vdm ("ERROR: failed to allocate memory for new path\n");
				retval = CDW_ERROR;
				break;
			} else {
				cdw_rv_t crv = cdw_fs_traverse_path(newpath, visitor, visitor_data);
				free(newpath);
				newpath = (char *) NULL;
				if (crv != CDW_OK) {
					cdw_vdm ("ERROR: function failed to traverse path \"%s\"\n", newpath);
					retval = CDW_ERROR;
					break;
				}
			}
		}

	}

	if (eps != (struct dirent **) NULL) {
		for (int i = 0; i < n; i++) {
			free(eps[i]);
			eps[i] = (struct dirent *) NULL;
		}
		free(eps);
		eps = (struct dirent **) NULL;
	}

	return retval;
}







cdw_rv_t cdw_fs_traverse_symlink(const char *fullpath, cdw_fs_visitor_t visitor, cdw_fs_visitor_data_t *visitor_data)
{
	/* CAN_EXISTING - all elements of a path must exist */
	char *resolved_fullpath = canonicalize_filename_mode(fullpath, CAN_EXISTING);
	int e = errno;
	if (resolved_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to fetch target for link \"%s\"\n", fullpath);
		cdw_vdm ("ERROR: error = \"%s\"\n", strerror(e));

#if 0 /* enable after checking behavior of stat() and lstat() on broken links */
		if (errno == ENOENT) {

			struct stat link_info;
			int sym1 = lstat(new_fullpath, &link_info);
			int sym2 = stat(new_fullpath, &link_info);
			if (sym1 == 0 && sym2 != 0) {
				cdw_vdm ("  errno == ENOENT && the link is broken\n");
				/* we can broken link and happily scan dir further */
			} else {
				cdw_vdm ("  errno == ENOENT, but reason is other than broken link\n");
				/* TODO: some more information for user pls */
				return CDW_ERROR;
			}
		}
#endif

		return CDW_ERROR;
	} else { /* resolved_fullpath != NULL */
		cdw_rv_t retval = CDW_OK;
		if (!strcmp(resolved_fullpath, ".")) {
			cdw_vdm ("INFO: link \"%s\" resolved as \".\", skipping\n", fullpath);
			retval = CDW_OK;
		} else {
			cdw_vdm ("INFO: link \"%s\" resolved as new fullpath = \"%s\", checking it...\n",
				 fullpath, resolved_fullpath);

			retval = cdw_fs_traverse_path(resolved_fullpath, visitor, visitor_data);
			if (retval != CDW_OK) {
				cdw_vdm ("ERROR: failed to traverse path \"%s\"\n", resolved_fullpath);
			} else {
				cdw_vdm ("INFO: size of symlink resolved as %lld\n", visitor_data->size);
			}
		}

		free(resolved_fullpath);
		resolved_fullpath = (char *) NULL;
		return retval;
	}
}






void cdw_fs_visitor(const char *fullpath, struct stat *finfo, cdw_fs_visitor_data_t *data)
{
	if ( S_ISREG(finfo->st_mode) ) { /* regular file */
		/* st_size is file size, in bytes */
		cdw_vdm ("INFO: file, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
		data->size += (long long) finfo->st_size;

		if (finfo->st_size >= 4294967296) { /* 4 GB, TODO: ">" or ">=" ?? */
			cdw_vdm ("WARNING: file larger than 4 GB\n");
			data->has_file_over_4GB = true;
		}
	} else if ( S_ISLNK(finfo->st_mode) ) { /* symbolic link */
		if (data->follow_symlinks) {
			;
		} else {
			cdw_vdm ("INFO: link, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
			/* st_size is length of path without terminating null */
			data->size += finfo->st_size;

		}
	} else if ( S_ISDIR(finfo->st_mode) ) { /* directory */
		cdw_vdm ("INFO:  dir, st_size = %10zd, name = \"%s\"\n", (size_t) finfo->st_size, fullpath);
		/* experimental results: using "finfo->st_blocks * finfo->st_blksize"
		   gives results closer to reality than "finfo->st_size" */
		/* FIXME: don't guess why it works, KNOW why it works */
		data->size += (finfo->st_blocks * finfo->st_blksize);
	} else {
		;
	}

#ifndef NDEBUG
	if (data->size < 1024) {
		cdw_vdm ("INFO: accumulated data->size = %lld B\n", data->size);
	} else if (data->size < 1024 * 1024) {
		cdw_vdm ("INFO: accumulated data->size = %.2f kB\n", (double) data->size / 1024.0);
	} else {
		cdw_vdm ("INFO: accumulated data->size = %.2f MB\n", (double) data->size / (1024.0 * 1024.0));
	}
#endif

	return;
}







/* 2TRANS: this is title of dialog window */
const char *title_error = "Error";
/* 2TRANS: this is title of dialog window */
const char *title_warning = "Warning";

struct {
	int e;
	const char *title;
	const char *message;
	int dialog_type;
	int colors;
	int crv;
} cdw_fs_errno_table[] = {
	{ EEXIST, /* pathname already exists and O_CREAT and O_EXCL were used */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Warning"),
	  /* 2TRANS: this is message dialog window, user can
	     press Yes, No or Cancel button */
	  gettext_noop("File already exists. Overwrite?"),
	  /* 0 - special case, return value will be taken from cdw_buttons_dialog */
	  CDW_BUTTONS_YES_NO_CANCEL, CDW_COLORS_WARNING, 0 },

	{ EISDIR, /* pathname refers to a directory and the access requested
		     involved writing (that is, O_WRONLY or O_RDWR is set) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("The path points to directory."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EACCES, /* The requested access to the file is not allowed, or
		     search permission is denied for one of the directories
		     in the path prefix of pathname, or the file did not
		     exist yet and write access to the parent directory is
		     not allowed */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Wrong access rights to file or parent directory."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENAMETOOLONG, /* pathname was too long */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is too long."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOENT, /* O_CREAT is not set and the named file does not exist.
		     Or, a directory component in pathname does not exist
		     or is a dangling symbolic link */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File or part of directory path does not exist."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EBADF,
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is of incorrect type."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOTDIR, /* A component used as a directory in pathname is not,
		      in fact, a directory, or O_DIRECTORY was specified and
		      pathname was not a directory */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("This is not path to directory or part of a path name is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EPERM, /* The O_NOATIME flag was specified, but the effective user
		    ID of the caller did not match the  owner of the file
		    and the caller was not privileged (CAP_FOWNER) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Wrong permissions for the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENXIO, /* O_NONBLOCK | O_WRONLY is set, the named file is a FIFO
		    and no process has the file open for reading. Or, the
		    file is a device special file and no corresponding
		    device exists */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid or points to invalid file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENODEV, /* pathname refers to a device special file and no
		     corresponding device exists. (This is a Linux kernel bug;
		     in this situation ENXIO must be returned.) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid or points to invalid file"),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EROFS, /* pathname refers to a file on a read-only filesystem and
		    write access was requested */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path to file in read-only file system."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ETXTBSY, /* pathname refers to an executable image which is currently
		      being executed and write access was requested */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("The file pointed by path is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EFAULT, /* pathname points outside your accessible address space */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path is invalid."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EFBIG, /* pathname  refers to a regular file, too large to be opened */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is too big to open."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ EOVERFLOW, /* pathname  refers to a regular file, too large to be opened (POSIX.1-2001) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("File is too big to open."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ELOOP, /* Too many symbolic links were encountered in resolving
		    pathname, or O_NOFOLLOW was specified but pathname was
		    a symbolic link */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Path uses too many symbolic links."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOSPC, /* pathname was to be created but the device containing
		     pathname has no room for the new file */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("Error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("There is no space on this device for the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_NO },
	{ ENOMEM, /* Insufficient kernel memory was available */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("No system memory. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ EMFILE, /* The process already has the maximum number of files open */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Too many files opened. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ ENFILE, /* The system limit on the total number of open files has been reached */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Too many files opened. You should close application."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ EWOULDBLOCK, /* The O_NONBLOCK flag was specified,and an incompatible lease
			  was held on the file (see fcntl(2)) */
	  /* 2TRANS: this is title of dialog window */
	  gettext_noop("System error"),
	  /* 2TRANS: this is message in dialog window, user can press OK button */
	  gettext_noop("Unknown error while accessing the file."),
	  CDW_BUTTONS_OK, CDW_COLORS_ERROR, CDW_ERROR },
	{ 0, /* arbitrary value in guard element */
	  (char *) NULL, /* guard value */
	  (char *) NULL, /* guard value */
	  0, 0, 0 /* arbitrary values in guard element */ }};



int cdw_fs_errno_handler_get_index(int e)
{
	int i = 0;
	while (cdw_fs_errno_table[i].title != (char *) NULL) {
		if (cdw_fs_errno_table[i].e == e) {
			return i;
		}
		i++;
	}
	return -1;
}



/**
   \brief Wrapper for code checking errno value and informing user about results

   This code checks value of errno and puts dialog window informing user
   about current situation. The function doesn't take any actions to fix
   a problem that caused some function to set errno. Caller must take care
   of this.

   \param e - errno value to examine

   \return CDW_SYS_ERROR if \p e signifies some system error
   \return CDW_NO if path is invalid (because of some non-critical error)
           or if file already exists, but user don't want to overwrite it
           (caller should ask for another path)
   \return CDW_CANCEL  if file already exists and user don't want to
           overwrite nor enter new path
   \return CDW_OK if file already exists and user wants to overwrite it;
           caller has to check the file permissions before attempting to
           overwrite;
*/
cdw_rv_t cdw_fs_errno_handler(int e)
{
	int i = cdw_fs_errno_handler_get_index(e);
	if (i != -1) {
		cdw_rv_t crv = cdw_buttons_dialog(cdw_fs_errno_table[i].title,
						  cdw_fs_errno_table[i].message,
						  cdw_fs_errno_table[i].dialog_type,
						  cdw_fs_errno_table[i].colors);
		if (e == EEXIST) {
			/* EEXIST - pathname already exists and O_CREAT and O_EXCL were used */
			return crv;
		} else {
			return cdw_fs_errno_table[i].crv;
		}
	} else {
		/* "e" was not found in errors table */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("System error"),
				   /* 2TRANS: this is message in
				      dialog window, user can
				      press OK button */
				   _("Unknown error while accessing the file."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}
}





bool cdw_fs_is_hidden(const char *file_name)
{
	if (file_name[0] != '.') {
		return false;
	}

	if (file_name[0] == '.' && file_name[1] == '.' && file_name[2] == '\0') {
		/* reference to parent dir */
		return false;
	} else if (file_name[0] == '.' && file_name[1] == '\0') {
		/* reference to current dir */
		return false;
	} else {
		return true;
	}
}





char *cdw_fs_shorten_fullpath(cdw_file_t *file)
{
	ssize_t pos = cdw_fs_get_filename_start(file->fullpath);
	if (pos == -1) {
		return (char *) NULL;
	} else {
		char *new_fullpath = strndup(file->fullpath, (size_t) pos);
		cdw_vdm ("INFO: original fullpath  = \"%s\"\n", file->fullpath);
		cdw_vdm ("INFO: shortened fullpath = \"%s\"\n", new_fullpath);
		return new_fullpath;
	}
}





/**
   \brief Check if given fullpath is a symlink, and return target path for it

   This is a simple version of canonicalize_filename_mode(), only
   resolving symbolic links. It's basically a wrapper around readlink().

   \para fullpath - fullpath to inspect/resolve

   \return freshly allocated buffer with resolved path if \p fullpath is a symbolic link
   \return NULL pointer on errors or when \p fullpath is not a symbolic link
*/
char *cdw_fs_get_target_path_simple(const char *fullpath)
{
	char *target_path = (char *) NULL;

	char link_path[PATH_MAX - 2];
	ssize_t r = readlink(fullpath, link_path, PATH_MAX - 2);
	if (r != -1) {
		link_path[r] = '\0';

		target_path = strdup(link_path);
		cdw_sdm ("INFO: \"%s\" -> \"%s\"\n", fullpath, target_path);
	} else if (r == EINVAL) {
		/* "The named file is not a symbolic link." */
	} else {
		; /* TODO: check rest of values of r, some of them might
		     indicate serious problems */
	}

	return target_path;
}





// #define CDW_UNIT_TEST_CODE /* definition used during development of unit tests code */
#ifdef CDW_UNIT_TEST_CODE

/* *** unit test functions *** */




static void test_cdw_fs_correct_dir_path_ending(void);
static void test_cdw_fs_get_filename_start(void);
static void test_gnulib_canonicalize_filename_mode(void);



void cdw_fs_run_tests(void)
{
	fprintf(stderr, "testing cdw_fs.c\n");

	/* cdw_fs_correct_dir_path_ending() depends on cdw_fs_get_filename_start(),
	   test cdw_fs_get_filename_start() first */
	test_cdw_fs_get_filename_start();
	test_cdw_fs_correct_dir_path_ending();
	test_gnulib_canonicalize_filename_mode();

	fprintf(stderr, "done\n\n");

	return;
}





/* I'm sure that canonicalize_filename_mode() works just fine,
   but I want to test how exactly it behaves;

   Note that only CAN_MISSING has been used */
void test_gnulib_canonicalize_filename_mode(void)
{
	fprintf(stderr, "\ttesting cdw_fs_canonicalize_path()...\n");

	struct {
		int type;
		const char *input;
		const char *expected_result;
	} test_data[] = {
		/* test with only root dir */
		{ CAN_MISSING, "/", "/"},
		/* test normal situation, where path is not modified */
		{ CAN_MISSING, "/my/simple/test/dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out "/./" elements */
		{ CAN_MISSING, "/my/./simple/test/dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out mix of "/./" and "//" elements */
		{ CAN_MISSING, "/my///simple/////test/././//dirs/string", "/my/simple/test/dirs/string"},
		/* test cutting out mix of more "/./" and "//" elements */
		{ CAN_MISSING, "/.//./././/my////.//simple/./test/dirs/string", "/my/simple/test/dirs/string"},
		/* test behavior with reference to parent dir - simple case */
		{ CAN_MISSING, "/my/simple/../test/dirs/string", "/my/test/dirs/string"},
		/* test behavior with reference to parent dir - more complicated case */
		{ CAN_MISSING, "/../../simple/../../test/dirs/string", "/test/dirs/string"},
		/* test behavior with mix of "/../", "/./" and "//" */
		{ CAN_MISSING, "/..///../base/dir///.././/of/.//./my/simple/../././//././/.//..//test/dirs/string", "/base/of/test/dirs/string"},
		/* "...", "..of..", "..my." are valid dir names, not special entities */
		{ CAN_MISSING, "/base/.../..of..//..my./simple/...//./..//././/.//..//test/dirs../string.../////// ", "/base/.../..of../..my./test/dirs../string.../ "},
		/* test removing "." from end of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string/.", "/my/simple/test/dirs/string"},
		/* test removing ending ".." and last dir from of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string/..", "/my/simple/test/dirs"},
		/* test removing ending ".." and last dir from of path - second case */
		{ CAN_MISSING, "/my/simple/test/dirs/string/../..//", "/my/simple/test"},
		/* test removing ending ".."/"." and last dir from of path - third case */
		{ CAN_MISSING, "/my/simple/test/dirs/string/..//../.", "/my/simple/test"},
		/* test removing duplicated "/" from end of path */
		{ CAN_MISSING, "/my/simple/test/dirs/string//", "/my/simple/test/dirs/string"},
		{ CAN_MISSING, (char *) NULL, (char *) NULL }
	};

	int i = 0;
	while (test_data[i].input != (char *) NULL) {
		char *result = canonicalize_filename_mode(test_data[i].input, test_data[i].type);
		cdw_assert_test (!strcmp(result, test_data[i].expected_result),
				 "ERROR: failed test #%d: \"%s\"\n", i, result);
		free(result);
		result = (char *) NULL;
		i++;
	}

	fprintf(stderr, "OK\n");

	return;
}





void test_cdw_fs_correct_dir_path_ending(void)
{
	fprintf(stderr, "\ttesting cdw_fs_correct_dir_path_ending()... ");

	cdw_rv_t crv = CDW_OK;
	char *path = (char *) NULL;

	/* first test basic cases */

	crv = cdw_fs_correct_dir_path_ending(&path);
	assert(crv == CDW_ERROR);

	crv = cdw_string_set(&path, "");
	assert(crv == CDW_OK);
	crv = cdw_fs_correct_dir_path_ending(&path);
	assert(crv == CDW_ERROR);

	struct {
		const char *input;
		const char *output;
	} test_data[] = {
		/* some cases with more or less correct input */
		{ "/home",           "/home/" },
		{ "/home ",          "/home /" },
		{ "/home/",          "/home/" },
		{ "/home/ ",         "/home/ /" },

		/* the function can handle (i.e. remove) any number of slashes */
		{ "/home///////",    "/home/" },

		/* some tests with short strings */
		{ "/ho///",          "/ho/" },
		{ "h/////",          "h/" },
		{ "// ///",          "// /" },
		{ "/",               "/" },
		{ "/ ",              "/ /" },

		{ (char *) NULL,     (char *) NULL }
	};

	int i = 0;
	while (test_data[i].input != (char *) NULL) {

		crv = cdw_string_set(&path, test_data[i].input);
		cdw_assert_test (crv == CDW_OK, "ERROR: failed test #%d\n", i);
		crv = cdw_fs_correct_dir_path_ending(&path);
		cdw_assert_test (crv == CDW_OK, "ERROR: failed test #%d\n", i);
		int d = strcmp(path, test_data[i].output);
		cdw_assert_test (d == 0, "ERROR: test #%d failed, path should be \"%s\", is \"%s\"\n",
				 i, test_data[i].output, path);

		i++;
	}

	free(path);
	path = (char *) NULL;

	fprintf(stderr, "OK\n");
	return;

}





void test_cdw_fs_get_filename_start(void)
{
	fprintf(stderr, "\ttesting cdw_fs_get_filename_start()... ");

	struct {
		int expected_result;
		const char *path;
	} test_data[] = {

		/* first test simple cases without root dir */
		{ -1,  "" },
		{  0,  "n" },
		{  0,  "name" },
		{  0,  "name " },
		{  0,  "na me " },
		{  0,  " name" },

		/* more complicated cases without root dir */

		{  0,  "name/" },
		{  0,  "n/" },
		{  6,  "name / " },
		{  7,  "na me / " },
		{  4,  " na/me//" },
		{  6,  "my na/me of file////////" },

		/* test simple cases with root dir */

		{  1,  "/" },
		{  1,  "/a" },
		{  1,  "/file name /" },
		/* path is not canonicalized (i.e. leading "//" aren't compressed
		   into one), so file name starts at 2 */
		{  2,  "//file name /" },
		{  12,  "/file name / other /////" },

		/* test few unusual cases */

		{  1,  "/////" },
		/* TODO: is this a correct test case at all? */
		{  8,  " / / / / " },
		{  0,  "     " },

		{  0,  (char *) NULL }
	};

	int i = 0;
	while (test_data[i].path != (char *) NULL) {
		ssize_t pos = cdw_fs_get_filename_start(test_data[i].path);
		cdw_assert_test (pos == test_data[i].expected_result,
				 "ERROR: test #%d failed: input path = \"%s\", pos = %zd, expected %d\n",
				 i, test_data[i].path, pos, test_data[i].expected_result);
		i++;
	}

	fprintf(stderr, "OK\n");
	return;
}


#endif /* CDW_UNIT_TEST_CODE */

