/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>


#include "cdw_processwin.h"
#include "cdw_regex_dispatch.h"
#include "cdw_dvd_rw_format_regex.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_debug.h"



/* extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1]; */
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];

bool dvd_rw_format_pipe_regexp_performed;

static regex_t eregex1;
static regmatch_t *ematches1 = NULL;

/* there is one operation that needs to be done the first time
   a handler is called, and this is a flag for it */
static bool first_call_to_handler = true;

static void cdw_dvd_rw_format_handle_blanking(regex_t *eregex, regmatch_t *ematches);




/* regular expressions setup code */
void cdw_dvd_rw_format_stderr_regexp_prepare(void)
{
	dvd_rw_format_pipe_regexp_performed = false;
	first_call_to_handler = true;

	/* "blanking 85.1%" */
	/* "formatting 85.1%" */

	/* currently cdw utilizes dvd+rw-format only to blank media;
	   dvd+rw-format erases DVD-RW by normal blanking ("blanking 85.1%"
	   is displayed); DVD+RW is erased by formatting ("formatting 85.1%" is
	   displayed */

	/* don't put 'blanking' nor 'formatting' into regexp because
	   dvd+rw-format puts '^H' bytes between it and percent value */
	int rv = regcomp(&eregex1, "([0-9]+)([.]*)([0-9]*)%", REG_EXTENDED);
	ematches1 = (regmatch_t *) calloc(1, (eregex1.re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		cdw_regex_regcomp_error_handler(__func__, rv, &eregex1, 501);
	}

	return;
}





void cdw_dvd_rw_format_stderr_regexp_execute(void)
{
	cdw_vdm ("INFO: stderr_pipe_buffer before cleaning = \"%s\"\n", stderr_pipe_buffer);

	/* remove funny characters from buffer, but don't operate on whole
	   string - only few first chars are interesting */
	int line_width = 80;
	for (int i = 0; i < line_width && i < PIPE_BUFFER_SIZE; i++) {
		if (!isprint(stderr_pipe_buffer[i])) {
			if (stderr_pipe_buffer[i] == '\0') {
				break;
			} else if (stderr_pipe_buffer[i] == '\r'
				   || stderr_pipe_buffer[i] == '\n'
				   || stderr_pipe_buffer[i] == '\b') {

				stderr_pipe_buffer[i] = '\0';
				break;
			} else {
				stderr_pipe_buffer[i] = ' ';
			}
		}
	}
	cdw_vdm ("INFO: stderr_pipe_buffer after cleaning = \"%s\"\n", stderr_pipe_buffer);

	int rv = regexec(&eregex1, stderr_pipe_buffer, eregex1.re_nsub + 1, ematches1, 0);
	if (rv == 0) {
		if (first_call_to_handler) {
			first_call_to_handler = false;
			/* this erases "preparing, please wait" message displayed by
			   dvd_rw_format_interface.c module */
			cdw_processwin_display_sub_info("");
		}
		cdw_sdm ("calling cdw_dvd_rw_format_handle_blanking() (stderr)\n");
		cdw_dvd_rw_format_handle_blanking(&eregex1, ematches1);
	}

	return;
}





void cdw_dvd_rw_format_stderr_regexp_destroy(void)
{
	regfree(&eregex1);
	free(ematches1);
	ematches1 = NULL;

	return;
}





void cdw_dvd_rw_format_handle_blanking(regex_t *regex, regmatch_t *matches)
{
	cdw_sdm ("called\n");
	dvd_rw_format_pipe_regexp_performed = true;
	/* regexp for catching blanking / formatting is
	     1      2      3
	  ([0-9]+)([.]*)([0-9]*)% */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	int perc_decimal = 0, perc_fract = 0;

	for (unsigned int i = 1; i <= regex->re_nsub; i++) {
		if (i == 2) {
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stderr_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}
		if (i == 1) { /* percentage - decimal part */
			if (!strcmp(submatch, "??")) {
				perc_decimal = 0;
			} else {
				perc_decimal = atoi((char *) &submatch);
			}
			cdw_vdm ("INFO: submatch %d, decimal part = \"%s\" -> %d\n", i, submatch, perc_decimal);
		} else if (i == 3) { /* percentage - fractional part */
			if (!strcmp(submatch, "??")) {
				perc_fract = 0;
			} else {
				perc_fract = atoi((char *) &submatch);
			}
			cdw_vdm ("INFO: submatch %d, fractional part = \"%s\" -> %d\n", i, submatch, perc_fract);
		} else {
			;
		}
	}

	/* float percent = perc_a + 0.1 * perc_b; */
	/* NOTE: consider implementing
	   conditional_processwin_display_progress_float(float done, long total, char *string) */

	/* display data */
	cdw_processwin_display_progress_conditional(perc_decimal, 100, (char *) NULL);

	return;
}

