(*:Name: Geometry3D *)

(*:Title: Geometry3D *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 3.0 *)


(*:History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Summary:
	This package provides functions to perform a variety of
	three-dimensional geometric computations.
*)



BeginPackage[ "ExtendGraphics`Geometry3D`",
					"ExtendGraphics`Geometry`",
					"ExtendGraphics`SimpleHull`", 
					"ExtendGraphics`NonConvexTriangulate`"]

Plane::usage =
	"Plane[ c, n] represents the plane line which passes
	 through c and is normal to n."


OrthogonalVectors::usage = 
	"OrthogonalVectors[ v] returns a pair of orthogonal vectors
	 to the three-dimensional vector v."
	 
ToPolygon::usage = 
	"ToPolygon[ Plane[ c, n]] returns a polygon that lies in
	 the plane Plane[ c, n]."
	 
PlaneComponents::usage = 
	"PlaneComponents[ Plane[ c, n]] returns a graphical representation 
	 of the plane Plane[ c, n]."
	 
PointInPlaneQ::usage = 
	"PointInPlaneQ[ pt, pl] tests whether or not the point pt
	 lies in the plane pl."
	 
ClosestPointInPlane::usage = 
	"ClosestPointInPlane[ pt, pl] returns the point in the plane
	 pl closest the point pt."
	
SideOfPlaneQ::usage = 
	"SideOfPlaneQ[ pt, pl] tests which side of the plane pl 
	 the point pt lies."
	 
ToPlane::usage = 
	"ToPlane[ {p1, p2, p3...}] returns the plane of best fit through
	 the points p1, p2, p3, ..."
	 
ParallelProject::usage = 
	"ParallelProject[ {p1, p2, p3...}, pl] returns the parallel 
	 projection of the points {p1, p2, p3, ...} in the plane pl."
	 
ParallelProjectTo2D::usage = 
	"ParallelProjectTo2D[ {p1, p2, p3...}, pl] returns the 
	 two-dimensional coordinates of the
	 parallel projection of the points {p1, p2, p3, ...} 
	 in the plane pl."
	 
EmbedIn3D::usage = 
	"EmbedIn3D[ {p1, p2, p3...}, pl] embeds the two-dimensional
	 points {p1, p2, p3, ...} in the three-dimensional plane pl."
	 
Clip3D::usage = 
	"Clip3D[ prims, pl] clips the primitives prims with the
	 plane pl."
	 
ConvexPolygon::usage = 
	"ConvexPolygon[ poly] returns the convex polygon representation
	 of poly."
	 
NonConvexPolygon::usage = 
	"NonConvexPolygon[ poly] returns a non-convex polygon representation
	 of poly."
	 
Extrude::usage =
	"Extrude[ poly, d] returns the solid object formed by extruding the
	 polygon poly distances d and -d.  If poly is a two-dimensional polygon
	 it will first be embedded in the xy plane.  Extrude[ poly, d, False]
	 will not first triangulate the polygon."
	
	
Begin[ "`Private`"]



If[ Not[ $VersionNumber > 2.2],
	Unprotect[ Polygon];
	Polygon[a_,b__] := Polygon[a];
	Protect[ Polygon]];

OrthogonalVectors[ norm:{_,_,_}] :=
	Module[{pos, a, b, v1, v2},
		pos = If[ VectorQ[ norm, NumberQ],
				Abs[N[ norm]], norm] ; 
		pos = Sort[ Transpose[ {pos, Range[ 3]}]] ;
		{pos, a,b} = Map[ Last, pos] ;
		v1 = ReplacePart[ {0,0,0}, -Part[ norm, a], b] ;
		v1 = ReplacePart[ v1, Part[ norm, b], a] ;
		v2 = Cross[ norm, v1] ;
		{v1, v2}
	]

	
ToPolygon[ Plane[ c:{_,_,_}, n:{_,_,_}], size_:1] :=
    Block[{ v1, v2},
        {v1, v2} = OrthogonalVectors[ n] ;
        v1 = v1/VectorLength[ v1] ;
        v2 = v2/VectorLength[ v2] ;
        Polygon[
              {c + size v1, c + size v2, 
               c - size v1, c - size v2}]
    ]

PlaneComponents[ Plane[ c:{_,_,_}, n:{_,_,_}], size_:1] :=
    Block[{ v1, v2},
        poly = ToPolygon[ Plane[ c, n], size] ;
        {
		AbsolutePointSize[ 4],
        Point[ c],
        Line[ {c, c+size n/VectorLength[n]}],
        poly
        }
    ]

IntersectionPoint[ Line[ {a1_, a2_}], Line[ {b1_, b2_}]] :=
   Block[{v1, v2, cross, s, t},
   	  v1 = (a2-a1) ;
   	  v2 = (b2-b1) ;
   	  cross = Cross[ v1,v2] ;
   	  len = Chop[ VectorLength[ cross]^2] ;
      If[ len === 0,
			Print[ "Parallel Lines"];
			Return[ {Infinity, Infinity}]] ;
   	  {
   	  a1 + (a2 - a1) Det[ {b1-a1, v2, cross}]/len,
   	  b1 + (b2 - b1) Det[ {b1-a1, v1, cross}]/len
   	  }
   	  ]

PointInPlaneQ[ p_, Plane[ c_, n_]] :=
    Chop[(p-c).n] === 0

ClosestPointInPlane[ p_, Plane[ c_, n_]] :=
    p - (p-c).n/n.n n

SideOfPlaneQ[ p_, Plane[ c_, n_]] :=
    N[ (p - c).n] > 0

IntersectionPoint[ Line[ {p1_, p2_}], Plane[ c_, n_]] :=
    Block[ {tst},
        tst = Chop[(p2 - p1)].n ;
        If[ tst === 0,
			Print[ "Parallel Primitives"];
			Table[Infinity, {Length[ p1]}],
	    	p1 - (p2 - p1) (p1 - c).n/tst
			]
		]

ToPlane[ pts_ /; MatrixQ[ pts] && 
				 Length[ pts] > 2 && 
				 Length[ First[ pts]] === 3] :=
    Block[{x,y,z,xl,yl,zl, norm},
        {x,y,z} = Transpose[ pts] ;
        {xl,yl,zl} = Map[ RotateLeft, {x,y,z}] ;
		norm = {
            	Dot[ y-yl, z+zl],
            	Dot[ z-zl, x+xl],
            	Dot[ x-xl, y+yl]
            	} ;
		If[ norm == {0,0,0}, 
			norm = Cross[ Part[ pts,2]-First[ pts], Part[ pts, 3]-First[ pts]]];
        Plane[ Apply[ Plus, pts]/Length[ pts], norm]
        ]

ParallelProject[ pts_ /;
       MatrixQ[ pts] && Length[ First[ pts]] === 3, 
                 Plane[ c_, n_]] :=
    Map[ 
        ClosestPointInPlane[ #, 
                                      Plane[c,n]]&, 
        pts]

ParallelProjectTo2D[ pts_ /; 
       MatrixQ[ pts] && Length[ First[ pts]] === 3, 
	   Plane[ c_, n_]] :=
    Block[{npts},
		{v1, v2} = N[ OrthogonalVectors[ n]] ;
		v1 = v1/VectorLength[ v1] ;
		v2 = v2/VectorLength[ v2] ;
		npts = ParallelProject[ pts, Plane[ c, n]] ;
		npts = Map[ (#-c)&, npts] ;
		Map[ {#.v1, #.v2}&, npts]
		]

EmbedIn3D[ pts_ /; 
       MatrixQ[ pts] && Length[ First[ pts]] === 2, 
	   Plane[ c_, n_]] :=
    Block[{npts},
		{v1, v2} = N[ OrthogonalVectors[ n]] ;
		v1 = v1/VectorLength[ v1] ;
		v2 = v2/VectorLength[ v2] ;
		Map[ (c + #.{v1, v2})&, pts]
		]

EmbedIn3D[ pts_ /; 
       MatrixQ[ pts] && Length[ First[ pts]] === 2, 
	   Plane[ c_, n_], {x_, y_}] :=
    Block[{npts, v1, v2, st, ct, xh, yh, z1, z2, tst},
        {v1, v2} = N[ OrthogonalVectors[ n]] ;
        v1 = v1/VectorLength[ v1] ;
        v2 = v2/VectorLength[ v2] ;
		z = Map[ Last, {v1, v2}] ;
		z = z/VectorLength[ z] ;
        {xh, yh} = {-y, x}/VectorLength[ {x, y}] ;
		{st, ct} = {{-xh, -yh}, {yh, -xh}}.z ;
		If[ ({{ct, st}, {-st, ct}}.{x, y}).z < 0, 
					ct = -ct ; st = -st] ;
        npts = Map[ {{ct, st}, {-st, ct}}.#&, pts] ;
        Map[ (c + #.{v1, v2})&, npts]
        ]


Clip3D[ (head:Line|Polygon)[ pts_], Plane[ c_, n_]] :=
    Block[{tst}, 
		tst = Map[ SideOfPlaneQ[ #, Plane[c,n]]&, pts] ;
		If[ Apply[ SameQ, tst],
			If[ First[ tst],
				head[ pts],
				{}],
			ComplexClip3D[ head, pts, tst, Plane[ c, n]]]
		]


ComplexClip3D[ head_, pts_, test_, plane_] :=
    Block[{work, len},
		work = Transpose[ {pts, test}] ;
		work = Transpose[ {work, RotateLeft[ work]}] ;
		work = Map[ TestPairs[#, plane]&, work] ;
		If[ head === Line && Last[ test], 
			work = ReplacePart[ work, {Last[ pts]}, Length[ pts]]] ;
		work = 
		    Fold[ If[ #2 === {},
				Append[ #1, {}],
				Append[ Drop[#1, -1], Join[ Last[#1], #2]]]&,
				{{}}, work] ;
		work = DeleteCases[ work, {}] ;
		If[ head === Line,
			Map[ Line, work],
			Polygon[ Flatten[ work, 1]]]
	]

TestPairs[ {{p1_, t1_}, {p2_, t2_}}, plane_] :=
	Which[ 
		t1  && t2 , 
				{p1},
		!t1  && !t2 , 
				{},
		t1  && !t2 , 
				{p1, IntersectionPoint[ Line[{p2, p1}], plane]},
		!t1  && t2 , {IntersectionPoint[ Line[{p1, p2}], plane]}
	]

Clip3D[ Point[ pt_], Plane[ c_, n_]] :=
	If[ SideOfPlaneQ[ pt, Plane[ c, n]],
		Point[ pt], {}]

Clip3D[ prims_List, Plane[ c_, n_]] :=
	Map[ Clip3D[#, Plane[ c, n]]&, prims]

Clip3D[ Graphics3D[ prims_, opts___], Plane[ c_, n_]] :=
	Graphics3D[ Clip3D[prims, Plane[ c, n]], opts]

Clip3D[ unknown_, Plane[ c_, n_]] := unknown


ConvexPolygon[ Polygon[ pts_]] :=
	Block[{plane, pts2d},
		plane = ToPlane[ pts] ;
		pts2d = ParallelProjectTo2D[ pts, plane] ;
		Polygon[ Part[ pts, SimpleConvexHull[ pts2d]]]
		]


NonConvexPolygon[ Polygon[ pts_]] :=
	Block[{plane, pts2d},
		plane = ToPlane[ pts] ;
		pts2d = ParallelProjectTo2D[ pts, plane] ;
		Map[ Polygon[ Part[ pts, #]]&, NonConvexTriangulate[ pts2d]]
		]

NonConvexPolygon[ Polygon[ pts_]] :=
	Block[{plane, pts2d, tri},
		plane = ToPlane[ pts] ;
		pts2d = ParallelProjectTo2D[ pts, plane] ;
		tri = NonConvexTriangulate[ pts2d] ;
		sides = Map[ Transpose[ {#, RotateLeft[ #]}]&, tri] ;
		sides = Map[ (t = Abs[Apply[ Subtract, #]]; 
						t === 1 || t === Length[ pts]-1)&, sides, {2}] ;
		tri = Transpose[ {tri, sides}] ;
		Map[ Polygon[ Part[ pts, First[ #]], Automatic, Last[#]]&, tri]
		]


Extrude[ Polygon[ pts_] /; 
           MatrixQ[ pts] && Length[ First[ pts]] === 2, 
	     d_, tri_:True] :=
	Extrude[ Polygon[ Map[ Append[ #, 0]&, pts]], d, tri]


Extrude[ Polygon[ pts_, extra___] /; 
           MatrixQ[ pts] && Length[ First[ pts]] === 3, 
	     d_, tri_:True] :=
    Block[{poly, side, add},
		poly = 
		  If[ tri, 
		      NonConvexPolygon[ Polygon[ pts]], 
			  {Polygon[ pts,extra]}];
		poly = Map[ SplitPolygon[#, {0,0,d}]&, poly] ;
		side = Transpose[ {pts, RotateLeft[ pts]}] ;
		add = { {0,0,d}, {0,0,d}} ;
		side = Map[ Polygon[ Join[ # + add, Reverse[#] - add]]&, side] ;
		{side, poly}
    ]

Extrude[ pl:{__Polygon}, d_, tri_:True] :=
	Map[ Extrude[ #, d, tri]&, pl]


SplitPolygon[ Polygon[ pts_, extra___], d_] :=
	Block[{ ext},
		ext = Table[ d, {Length[ pts]}] ;
		{
		Polygon[ pts + ext, extra], 
		Polygon[ pts - ext, extra]
		}]


End[]

EndPackage[]


(*:Examples:

<<ExtendGraphics`Geometry3D`
		
*)

